import sys
import shutil
from urllib.parse import urlencode
from contextlib import contextmanager
import os
import pathlib
import io

import requests

from .exceptions import ResponseError


@contextmanager
def outfile(file=None):
    if file is None:
        yield sys.stdout.buffer
    elif isinstance(file, int):
        yield os.fdopen(file, "wb")
    elif isinstance(file, io.IOBase):
        yield file
    else:
        path = file
        if isinstance(file, str):
            path = pathlib.Path(file)
        with path.open("wb") as _file:
            yield _file


def put(uri, filename,
        redundancy=None, split=None, chunk_size=None, timeout=None):
    args = {}
    if redundancy:
        args['redundancy'] = redundancy
    if split:
        args['split'] = split
    if chunk_size:
        args['chunk_size'] = chunk_size
    if timeout:
        args['timeout'] = timeout

    uri += '?' + urlencode(args)

    with open(filename, 'rb') as _file:
        response = requests.put(uri, data=_file)

    if response.status_code == 200:
        return response.text

    raise ResponseError(response.status_code, response.text)


def get(uri, info_hash, file=None, byte_range=None):
    uri += info_hash
    headers = {}
    if byte_range:
        headers = {
            "RANGE": "bytes={}".format(byte_range)
        }
    response = requests.get(uri, stream=True, headers=headers)

    if response.status_code not in (200, 206):
        raise ResponseError(response.status_code, response.text)

    with outfile(file) as _file:
        shutil.copyfileobj(response.raw, _file)

    return True


def delete(uri, info_hash):
    uri += info_hash
    response = requests.delete(uri)

    if response.status_code == 200:
        return True

    raise ResponseError(response.status_code, response.text)


def head(uri, info_hash):
    uri += info_hash
    response = requests.head(uri)

    if response.status_code == 200:
        return response.headers

    raise ResponseError(response.status_code, response.text)
