#!/usr/local/env python
# coding: utf-8

# pylint: disable=line-too-long, invalid-name
"""deeptorrent client api.

Usage:
    deeptorrent-client put <filename> [--ip=<ip>] [--port=<port>] [--redundancy=<r>] [--split=<s>] [--timeout=<timeout>] [--chunk_size=<chunk_size>]
    deeptorrent-client get <info_hash> [--ip=<ip>] [--port=<port>] [--outfile=<outfile>] [--byte_range=<byte_range>]
    deeptorrent-client head <info_hash> [--ip=<ip>] [--port=<port>]
    deeptorrent-client delete <info_hash> [--ip=<ip>] [--port=<port>]
    deeptorrent-client --version

Options:
    -h --help                   Show this screen.
    --redundancy=<r>            Number of redundancy chunk
    --split=<s>                 Number of split chunk
    --timeout=<timeout>         Set timeout.
    --ip=<ip>                   Set ip to Deeptorrent Server [default: 127.0.0.1]
    --port=<port>               Set port to Deeptorrent Server [default: 5555]
    --outfile=<outfile>         Output File
    --chunk_size=<chunk_size>   Chunk Size in Byte
    --byte_range=<byte_range>   Byte range
    --version                   Show version of client
"""

import os
import sys
from pprint import pprint

from docopt import docopt

from .client import put, get, delete, head
from .exceptions import ResponseError

__VERSION__ = "1.3.1"


def _put(uri, arguments):
    filename = os.path.abspath(arguments['<filename>'])
    redundancy = None
    if arguments['--redundancy']:
        redundancy = int(arguments['--redundancy'])
    split = None
    if arguments['--split']:
        split = int(arguments['--split'])
    chunk_size = None
    if arguments['--chunk_size']:
        chunk_size = int(arguments['--chunk_size'])
    timeout = None
    if arguments['--timeout']:
        timeout = int(arguments['--timeout'])

    try:
        infohash = put(uri, filename, redundancy, split, chunk_size, timeout)
    except ResponseError as exc:
        print(exc.status_code, exc.message, file=sys.stderr)
        sys.exit(1)
    except Exception as exc:  # pylint: disable=broad-except
        print(str(exc), file=sys.stderr)
        sys.exit(2)
    else:
        print(infohash)
        sys.exit(0)


def _get(uri, arguments):
    info_hash = arguments['<info_hash>']
    filename = arguments['--outfile'] if arguments['--outfile'] else None
    byte_range = None
    if arguments['--byte_range']:
        byte_range = arguments['--byte_range']

    try:
        get(uri, info_hash, filename, byte_range)
    except ResponseError as exc:
        print(exc.status_code, exc.message, file=sys.stderr)
        sys.exit(1)
    except Exception as exc:  # pylint: disable=broad-except
        print(str(exc), file=sys.stderr)
        sys.exit(2)
    else:
        sys.exit(0)


def _delete(uri, arguments):
    info_hash = arguments['<info_hash>']

    try:
        delete(uri, info_hash)
    except ResponseError as exc:
        print(exc.status_code, exc.message, file=sys.stderr)
        sys.exit(1)
    except Exception as exc:  # pylint: disable=broad-except
        print(str(exc), file=sys.stderr)
        sys.exit(2)
    else:
        sys.exit(0)


def _head(uri, arguments):
    info_hash = arguments['<info_hash>']

    try:
        result = head(uri, info_hash)
    except ResponseError as exc:
        print(exc.status_code, exc.message, file=sys.stderr)
        sys.exit(1)
    except Exception as exc:  # pylint: disable=broad-except
        print(str(exc), file=sys.stderr)
        sys.exit(2)
    else:
        pprint(result)
        sys.exit(0)


def main():
    arguments = docopt(
        __doc__,
        version='deeptorrent-client {}'.format(__VERSION__))

    ip = arguments['--ip']
    port = arguments['--port']

    uri = "http://{}:{}/".format(ip, port)

    if arguments['--version']:
        print(__VERSION__)
        sys.exit(0)

    if arguments['put']:
        _put(uri, arguments)

    elif arguments['get']:
        _get(uri, arguments)

    elif arguments['head']:
        _head(uri, arguments)

    elif arguments['delete']:
        _delete(uri, arguments)


if __name__ == '__main__':
    main()
