"""Data sources parsers."""
from typing import Generator
import hashlib
from abc import ABC, abstractstaticmethod
from enum import Enum, auto
from io import StringIO
import numpy as np
import pandas as pd
from pandas import DataFrame
from pandas_schema import Column, Schema
from pandas_schema.validation import IsDtypeValidation, MatchesPatternValidation
from geniepy.datamgmt.scrapers import BaseScraper, CtdScraper, PubMedScraper
from geniepy.errors import ParserError
from geniepy.pubmed import PubMedArticle


class DataType(Enum):
    """Possible parsable datatypes."""

    CSV_STR = auto()
    XML = auto()


class BaseParser(ABC):
    """Abstract base parser class."""

    scraper: BaseScraper
    schema: Schema
    default_type: DataType = None

    @classmethod
    def validate(cls, payload: DataFrame) -> [str]:
        """
        Check if payload is valid schema.

        Arguments:
            payload {DataFrame} -- The data to be checked against parser schema.

        Returns:
            bool -- true if payload conforms to schema, false otherwise.
        """
        if payload is None:
            return ["Cannot validate None object"]
        return cls.schema.validate(payload)

    @abstractstaticmethod
    def parse(data, dtype: DataType = None) -> DataFrame:
        """
        Parse data and convert according to parser schema.

        Arguments:
            data {Implementation dependent} -- Data to be parsed

        Keyword Arguments:
            dtype {DataType} -- Type of data to be parsed (default: {DataType.CSV})

        Returns:
            DataFrame -- The parsed dataframe.
        """

    def fetch(self, chunksize: int) -> Generator[DataFrame, None, None]:
        """
        Fetch new data, if available from online sources.

        Keyword Arguments:
            chunksize {int} -- the returned generator chunk size

        Returns:
            Generator[DataFrame, None, None] -- Generator yielding fetched data
        """
        raw_gen = self.scraper.scrape(chunksize)
        for data_chunk in raw_gen:
            parsed_df = self.parse(data_chunk, self.default_type)
            yield parsed_df


class CtdParser(BaseParser):
    """
    Implementation of CTD Database Parser.

    Comparative Toxicogenomics Gene-Disease Associations Database Parser.
    http://ctdbase.org/
    """

    default_type: DataType = DataType.CSV_STR
    scraper: CtdScraper = CtdScraper()
    schema: Schema = Schema(
        [
            Column("digest"),
            Column("genesymbol"),
            Column("geneid", [IsDtypeValidation(np.int64)]),
            Column("diseasename"),
            Column(
                "diseaseid", [MatchesPatternValidation("^D[0-9]+$")]
            ),  # i.e. D000014
            Column("pmids"),
        ]
    )

    @staticmethod
    def hash_record(record: pd.Series) -> str:
        """
        Hash the ctd record to generate digest column.

        Arguments:
            record {pd.Series} -- The ctd record in form of pandas Series

        Returns:
            str -- the hex string of the computed digest
        """
        message = str.encode(str(record.geneid) + record.diseaseid)
        hexdigest = hashlib.sha256(message).hexdigest()
        return str(hexdigest)

    @staticmethod
    def parse(data, dtype=DataType.CSV_STR) -> DataFrame:
        """
        Parse data and convert according to parser schema.

        Arguments:
            data {Implementation dependent} -- Data to be parsed

        Keyword Arguments:
            dtype {DataType} -- Type of data to be parsed (default: {DataType.CSV})

        Returns:
            DataFrame -- The parsed dataframe.

        Raises:
            ParserError -- If unable to parse data
        """
        try:
            parsed_df = pd.read_csv(StringIO(data))
            # Remove unused columns
            parsed_df = parsed_df.drop(
                columns=[
                    "DirectEvidence",
                    "InferenceChemicalName",
                    "InferenceScore",
                    "OmimIDs",
                ]
            )
            # Remove prefix 'MESH:' from DiseaseIDs
            parsed_df["DiseaseID"] = parsed_df.apply(
                lambda x: x.DiseaseID.replace("MESH:", ""), axis=1
            )
            # Rename columns based on schema
            parsed_df.rename(
                columns={
                    "GeneSymbol": "genesymbol",
                    "GeneID": "geneid",
                    "DiseaseName": "diseasename",
                    "DiseaseID": "diseaseid",
                    "PubMedIDs": "pmids",
                },
                inplace=True,
            )
            # Compute and add the digest
            parsed_df["digest"] = parsed_df.apply(CtdParser.hash_record, axis=1)
            errors = CtdParser.validate(parsed_df)
            if errors:
                raise ParserError(errors)
            return parsed_df
        except Exception as parse_exp:
            raise ParserError(parse_exp)


class PubMedParser(BaseParser):
    """
    Implementation of PubMed Articles Parser.

    https://www.ncbi.nlm.nih.gov/pubmed/
    https://www.nlm.nih.gov/bsd/licensee/elements_descriptions.html
    """

    default_type: DataType = DataType.XML
    scraper: PubMedScraper()
    schema: Schema = Schema(
        [
            Column("pmid", [IsDtypeValidation(np.int64)]),
            Column("date_completed"),
            Column("pub_model"),
            Column("title"),
            Column("iso_abbreviation"),
            Column("article_title"),
            Column("abstract"),
            Column("authors"),
            Column("language"),
            Column("chemicals"),
            Column("mesh_list"),
        ]
    )

    @staticmethod
    def parse(data, dtype: DataType = None) -> DataFrame:
        """
        Parse data and convert according to parser schema.

        Arguments:
            data {Implementation dependent} -- Data to be parsed

        Keyword Arguments:
            dtype {DataType} -- Type of data to be parsed (default: {DataType.CSV})

        Returns:
            DataFrame -- The parsed dataframe.
        """
        # Data passed in should be a list of xml element trees
        xml_list = data

        # The keys of the dataframe
        keys = [
            "pmid",
            "date_completed",
            "pub_model",
            "title",
            "iso_abbreviation",
            "article_title",
            "abstract",
            "authors",
            "language",
            "chemicals",
            "mesh_list",
        ]

        # Temp array variables to store from each xml element tree
        pmid_list = []
        date_completed_list = []
        pub_model_list = []
        title_list = []
        iso_abbreviation_list = []
        article_title_list = []
        abstract_list = []
        authors_list = []
        language_list = []
        chemicals_list = []
        mesh_list_list = []

        try:
            # General XML Tags
            for xml_article in xml_list:
                article = PubMedArticle(xml_article)
                pmid_list.append(np.int64(article.pmid))
                date_completed_list.append(article.date_completed)
                pub_model_list.append(article.pub_model)
                title_list.append(article.title)
                iso_abbreviation_list.append(article.iso_abbreviation)
                article_title_list.append(article.article_title)
                abstract_list.append(article.abstract)
                authors_list.append(str(article.authors).strip("[]"))
                language_list.append(article.language)
                chemicals_list.append(str(article.chemicals).strip("[]"))
                mesh_list_list.append(str(article.mesh_list).strip("[]"))

            # Create the array of arrays of values in dataframe
            values = [
                pmid_list,
                date_completed_list,
                pub_model_list,
                title_list,
                iso_abbreviation_list,
                article_title_list,
                abstract_list,
                authors_list,
                language_list,
                chemicals_list,
                mesh_list_list,
            ]

            # Zip df keys and values and create dataframe
            zipped = list(zip(keys, values))
            parsed_df = pd.DataFrame(dict(zipped))

            errors = PubMedParser.validate(parsed_df)
            if errors:
                raise ParserError(errors)  # pragma: no cover - should never reach
            return parsed_df
        except Exception as parse_exp:
            raise ParserError(parse_exp)


class ClassifierParser(BaseParser):
    """
    Implementation of classifier dao Parser.

    The classifier output tables contain the output data from geniepy after the
    classifiers have calculated desired predictions.
    """

    default_type: DataType = None
    scraper: None
    """No online sources for classifiers output."""
    schema: Schema = Schema(
        [
            Column("digest"),
            Column("pub_score", [IsDtypeValidation(np.float64)]),
            Column("ct_score", [IsDtypeValidation(np.float64)]),
        ]
    )

    def fetch(self, chunksize: int) -> Generator[DataFrame, None, None]:
        """No online sources to fetch from for classifiers outputs."""
        raise NotImplementedError("Classifier Output Parser has no Scrapers")

    @staticmethod
    def parse(data, dtype=DataType.CSV_STR) -> DataFrame:
        """
        Parser function from base class.

        Raises:
            NotImplementedError -- Function not implemented since classifiers return
                dataframes that only need to be validated.
        """
        raise NotImplementedError("Classifier Output Parser has no Scrapers")
