# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['fastapi_opa',
 'fastapi_opa.auth',
 'fastapi_opa.opa',
 'fastapi_opa.opa.enrichment']

package_data = \
{'': ['*']}

install_requires = \
['PyJWT[crypto]', 'fastapi>=0.65.2', 'itsdangerous', 'requests']

extras_require = \
{'graphql': ['graphene>=2,<3'], 'saml': ['python3-saml', 'python-multipart']}

setup_kwargs = {
    'name': 'fastapi-opa',
    'version': '1.3.3',
    'description': 'Fastapi OPA middleware incl. auth flow.',
    'long_description': '# Open Policy Agent Middleware for FastAPI\n\n## Table of Contents\n- [Contributors](#contributors)\n- [What does fastapi-opa do](#about)\n- [Installation](#installation)\n- [How to get started](#getting-started)\n- [Open Policy Agent](#opa)\n- [Authentication Flow](#auth-flow)\n  - [OIDC Authentication](#oidc-auth)\n  - [SAML Authentication](#saml-auth)\n- [Custom Payload Enrichment](#custom-payload-enrichment)\n  - [Graphql Enrichment](#gql-enrichment)\n\n<a name="contributors"/>\n\n## Contributors\n\nThanks to all our contributors! There is no specific order and hopefully nobody was left out.\n\n<a href="https://github.com/morestanna">\n  <img src="https://avatars.githubusercontent.com/morestanna" width="60" height="60" />\n</a>\n<a href="https://github.com/busykoala">\n  <img src="https://avatars.githubusercontent.com/busykoala" width="60" height="60" />\n</a>\n<a href="https://github.com/TracyWR">\n  <img src="https://avatars.githubusercontent.com/TracyWR" width="60" height="60" />\n</a>\n<a href="https://github.com/loikki">\n  <img src="https://avatars.githubusercontent.com/loikki" width="60" height="60" />\n</a>\n\n<a name="about"/>\n\n## What does fastapi-opa do\n\n`fastapi-opa` is an extension to FastAPI that allows you to add a login flow\nto your application within minutes using open policy agent and your favourite\nidentity provider.\n\n![Flow Diagram](https://raw.githubusercontent.com/busykoala/fastapi-opa/master/assets/diagram.png)\n\nWhen a user tries to get a response from an endpoint he/she will be redirected\nto the identity provider for authorization.\nAfter the authentication the app validates the token provided. Once it was\nvalidated the user information is used to get an OPA decision whether\nthe user is allowed to get any information from the endpoint.\n\n<a name="installation"/>\n\n## Installation\n\n```bash\npoetry add [--extras "graphql"] [--extras "saml"] fastapi-opa \n```\n\n<a name="getting-started"/>\n\n## How to get started\n\n:bulb: Checkout the wiki for a complete environment setup with Keycloak and Open Policy Agent:  \n[Getting Started with FastAPI app with Authentication and Authorization](https://github.com/busykoala/fastapi-opa/wiki#dev-setup)\n\nThe package provides a very easy way to integrate authentication and\nauthorization. We can decide what authentication flow we inject into the\nOPAMiddleware to be able choosing between different flows.\n\nThere are \n - one example for oidc : fastapi_opa.example_oidc.py,\n - one example for saml: fastapi_opa.example_saml.py\n\n## Open Policy Agent\n\nThe (validated/authenticated) user token is sent to the Open Policy Agent\nwith the additional attributes `request_method` and `request_path`.\n\n```json\n{\n    "input": {\n        "exp": 1617466243,\n        "iat": 1617465943,\n        "auth_time": 1617465663,\n        "jti": "9aacb638-70c6-4f0a-b0c8-dbc67f92e3d1",\n        "iss": "http://localhost:8080/auth/realms/example-realm",\n        "aud": "example-client",\n        "sub": "ccf78dc0-e1d6-4606-99d4-9009e74e3ab4",\n        "typ": "ID",\n        "azp": "david",\n        "session_state": "41640fe7-39d2-44bc-818c-a3360b36fb87",\n        "at_hash": "2IGw-B9f5910Sll1tnfQRg",\n        "acr": "0",\n        "email_verified": false,\n        "hr": "true",\n        "preferred_username": "david",\n        "user": "david",\n        "subordinates": [],\n        "request_method": "GET",\n        "request_path": ["finance", "salary", "david"]\n    }\n}\n```\n\nIn open policy agent you can now easily create policies using user roles,\nroutes, or request methods etc.\n\nAn example policy (from [the official OPA docs](https://www.openpolicyagent.org/docs/v0.11.0/http-api-authorization/))\nfor this setup could be like:\n\n```rego\npackage httpapi.authz\n\n# bob is alice\'s manager, and betty is charlie\'s.\nsubordinates = {"alice": [], "charlie": [], "bob": ["alice"], "betty": ["charlie"]}\n\n# HTTP API request\nimport input\n\ndefault allow = false\n\n# Allow users to get their own salaries.\nallow {\n  some username\n  input.request_method == "GET"\n  input.request_path = ["finance", "salary", username]\n  input.user == username\n}\n\n# Allow managers to get their subordinates\' salaries.\nallow {\n  some username\n  input.request_method == "GET"\n  input.request_path = ["finance", "salary", username]\n  subordinates[input.user][_] == username\n}\n```\n\n<a name="auth-flow"/>\n\n## Authentication Flow\n\nThere is an interface provided to easily implement the desired authentication\nflow and inject it into OPAMiddleware\n(`fastapi_opa.auth.auth_interface.AuthInterface`), or you can open a pull\nrequest if you would like to contribute to the package.\n\nAlso there are implementations ready to use.\n\n<a name="oidc-auth"/>\n\n### OIDC Authentication\n\nThe example in [How to get started](#getting-started) provides an example for\nthe implementation of the OIDC Authentication.\n\n<a name="saml-auth"/>\n\n### SAML Authentication\n\nFor the saml implementation create your certs using\n`openssl req -new -x509 -days 3652 -nodes -out sp.crt -keyout sp.key` and\nadd the keys to the sp section of your `settings.json`. Checkout the test\nsettings to get an idea (`tests/test_data/saml/*.json`). The path to your\nown `settings.json` and `advanced_settings.json` has to be provided in the\n`SAMLAuthConfig` like in the example below (do not use the test data in\nproduction).\n\n```python\nfrom fastapi_opa import OPAConfig\nfrom fastapi_opa.auth.auth_saml import SAMLAuthentication\nfrom fastapi_opa.auth.auth_saml import SAMLConfig\n\nopa_host = "http://localhost:8181"\n\nsaml_config = SAMLConfig(settings_directory="./tests/test_data/saml")\nsaml_auth = SAMLAuthentication(saml_config)\n\nopa_config = OPAConfig(authentication=saml_auth, opa_host=opa_host,\n                       accepted_methods=["id_token", "access_token"])\n```\n\nThe cert has to be uploaded to your identity provider. Using Keycloak as an\nidp you need to configure `encrypt assertion`, `client signature required`,\n`force POST bindings` on creating the client.\nAlso configure: `Client Scopes` -> `role_list (saml)` -> `Mappers tab` ->\n`role list` -> `Single Role Attribute`\n\n<a name="custom-payload-enrichment"/>\n\n## Custom Payload Enrichment\n\nIn `fastapi_opa.opa.opa_config.Injectable` an interface is provided to add\nmore information to the payload sent to OPA.\n\nThe injectables can be added to the `OPAConfig`. Let\'s look at an example:\n\n```python\nclass FancyInjectable(Injectable):\n    async def extract(self, request: Request) -> List:\n        return ["some", "custom", "stuff"]\n\nfancy_inj = FancyInjectable("fancy_key", skip_endpoints=["/health", "/api/[^/]*/test])\n\nopa_config = OPAConfig(\n    authentication=oidc_auth, opa_host=opa_host, injectables=[fancy_inj]\n)\n```\n\nWith `skip_endpoints`, you can define some endpoints where the injectable\nwill not be applied. The endpoints can be defined either directly or through some\nregex.\n\n\n<a name="gql-enrichment"/>\n\n### Graphql Enrichment\n\nFor GraphQL there is a ready to use injectable:\n\n```python\nfrom fastapi_opa.opa.enrichment.graphql_enrichment import GraphQLInjectable`\n\ngraphql = GraphQLInjectable("gql_injectable")\nopa_config = OPAConfig(authentication=oidc_auth, opa_host=opa_host, injectables=[graphql])\n```\n',
    'author': 'Matthias Osswald',
    'author_email': 'm@osswald.li',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/busykoala/fastapi-opa',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.6.2,<4.0',
}


setup(**setup_kwargs)
