# -*- coding: utf-8 -*-

# PLEASE DO NOT EDIT THIS FILE, IT IS GENERATED AND WILL BE OVERWRITTEN:
# https://github.com/ccxt/ccxt/blob/master/CONTRIBUTING.md#how-to-contribute-code

from ccxt.base.exchange import Exchange
from ccxt.base.errors import ExchangeError
from ccxt.base.errors import AuthenticationError
from ccxt.base.errors import PermissionDenied
from ccxt.base.errors import AccountSuspended
from ccxt.base.errors import BadRequest
from ccxt.base.errors import BadSymbol
from ccxt.base.errors import BadResponse
from ccxt.base.errors import InsufficientFunds
from ccxt.base.errors import InvalidAddress
from ccxt.base.errors import InvalidOrder
from ccxt.base.errors import OrderNotFound
from ccxt.base.errors import NotSupported
from ccxt.base.errors import NetworkError
from ccxt.base.errors import DDoSProtection
from ccxt.base.errors import RateLimitExceeded
from ccxt.base.errors import InvalidNonce
from ccxt.base.decimal_to_precision import TICK_SIZE
from ccxt.base.precise import Precise


class digifinex(Exchange):

    def describe(self):
        return self.deep_extend(super(digifinex, self).describe(), {
            'id': 'digifinex',
            'name': 'DigiFinex',
            'countries': ['SG'],
            'version': 'v3',
            'rateLimit': 900,  # 300 for posts
            'has': {
                'CORS': None,
                'spot': True,
                'margin': True,
                'swap': None,  # has but unimplemented
                'future': False,
                'option': False,
                'cancelOrder': True,
                'cancelOrders': True,
                'createOrder': True,
                'createStopLimitOrder': False,
                'createStopMarketOrder': False,
                'createStopOrder': False,
                'fetchBalance': True,
                'fetchBorrowInterest': True,
                'fetchBorrowRate': True,
                'fetchBorrowRateHistories': False,
                'fetchBorrowRateHistory': False,
                'fetchBorrowRates': True,
                'fetchCurrencies': True,
                'fetchDepositAddress': True,
                'fetchDeposits': True,
                'fetchFundingRate': True,
                'fetchFundingRateHistory': True,
                'fetchLedger': True,
                'fetchMarginMode': False,
                'fetchMarkets': True,
                'fetchMyTrades': True,
                'fetchOHLCV': True,
                'fetchOpenOrders': True,
                'fetchOrder': True,
                'fetchOrderBook': True,
                'fetchOrders': True,
                'fetchPositionMode': False,
                'fetchStatus': True,
                'fetchTicker': True,
                'fetchTickers': True,
                'fetchTime': True,
                'fetchTrades': True,
                'fetchTradingFee': True,
                'fetchTradingFees': False,
                'fetchWithdrawals': True,
                'setMarginMode': False,
                'transfer': True,
                'withdraw': True,
            },
            'timeframes': {
                '1m': '1',
                '5m': '5',
                '15m': '15',
                '30m': '30',
                '1h': '60',
                '4h': '240',
                '12h': '720',
                '1d': '1D',
                '1w': '1W',
            },
            'urls': {
                'logo': 'https://user-images.githubusercontent.com/51840849/87443315-01283a00-c5fe-11ea-8628-c2a0feaf07ac.jpg',
                'api': {
                    'rest': 'https://openapi.digifinex.com',
                },
                'www': 'https://www.digifinex.com',
                'doc': [
                    'https://docs.digifinex.com',
                ],
                'fees': 'https://digifinex.zendesk.com/hc/en-us/articles/360000328422-Fee-Structure-on-DigiFinex',
                'referral': 'https://www.digifinex.com/en-ww/from/DhOzBg?channelCode=ljaUPp',
            },
            'api': {
                'public': {
                    'spot': {
                        'get': [
                            '{market}/symbols',
                            'kline',
                            'margin/currencies',
                            'margin/symbols',
                            'markets',
                            'order_book',
                            'ping',
                            'spot/symbols',
                            'time',
                            'trades',
                            'trades/symbols',
                            'ticker',
                            'currencies',
                        ],
                    },
                    'swap': {
                        'get': [
                            'public/api_weight',
                            'public/candles',
                            'public/candles_history',
                            'public/depth',
                            'public/funding_rate',
                            'public/funding_rate_history',
                            'public/instrument',
                            'public/instruments',
                            'public/ticker',
                            'public/tickers',
                            'public/time',
                            'public/trades',
                        ],
                    },
                },
                'private': {
                    'spot': {
                        'get': [
                            '{market}/financelog',
                            '{market}/mytrades',
                            '{market}/order',
                            '{market}/order/detail',
                            '{market}/order/current',
                            '{market}/order/history',
                            'margin/assets',
                            'margin/financelog',
                            'margin/mytrades',
                            'margin/order',
                            'margin/order/current',
                            'margin/order/history',
                            'margin/positions',
                            'otc/financelog',
                            'spot/assets',
                            'spot/financelog',
                            'spot/mytrades',
                            'spot/order',
                            'spot/order/current',
                            'spot/order/history',
                            'deposit/address',
                            'deposit/history',
                            'withdraw/history',
                        ],
                        'post': [
                            '{market}/order/cancel',
                            '{market}/order/new',
                            '{market}/order/batch_new',
                            'margin/order/cancel',
                            'margin/order/new',
                            'margin/position/close',
                            'spot/order/cancel',
                            'spot/order/new',
                            'transfer',
                            'withdraw/new',
                            'withdraw/cancel',
                        ],
                    },
                    'swap': {
                        'get': [
                            'account/balance',
                            'account/positions',
                            'account/finance_record',
                            'account/trading_fee_rate',
                            'account/transfer_record',
                            'trade/history_orders',
                            'trade/history_trades',
                            'trade/open_orders',
                            'trade/order_info',
                        ],
                        'post': [
                            'account/leverage',
                            'trade/batch_cancel_order',
                            'trade/batch_order',
                            'trade/cancel_order',
                            'trade/order_place',
                        ],
                    },
                },
            },
            'fees': {
                'trading': {
                    'tierBased': True,
                    'percentage': True,
                    'maker': self.parse_number('0.002'),
                    'taker': self.parse_number('0.002'),
                },
            },
            'precisionMode': TICK_SIZE,
            'exceptions': {
                'exact': {
                    '10001': [BadRequest, "Wrong request method, please check it's a GET ot POST request"],
                    '10002': [AuthenticationError, 'Invalid ApiKey'],
                    '10003': [AuthenticationError, "Sign doesn't match"],
                    '10004': [BadRequest, 'Illegal request parameters'],
                    '10005': [DDoSProtection, 'Request frequency exceeds the limit'],
                    '10006': [PermissionDenied, 'Unauthorized to execute self request'],
                    '10007': [PermissionDenied, 'IP address Unauthorized'],
                    '10008': [InvalidNonce, 'Timestamp for self request is invalid, timestamp must within 1 minute'],
                    '10009': [NetworkError, 'Unexist endpoint, please check endpoint URL'],
                    '10011': [AccountSuspended, 'ApiKey expired. Please go to client side to re-create an ApiKey'],
                    '20001': [PermissionDenied, 'Trade is not open for self trading pair'],
                    '20002': [PermissionDenied, 'Trade of self trading pair is suspended'],
                    '20003': [InvalidOrder, 'Invalid price or amount'],
                    '20007': [InvalidOrder, 'Price precision error'],
                    '20008': [InvalidOrder, 'Amount precision error'],
                    '20009': [InvalidOrder, 'Amount is less than the minimum requirement'],
                    '20010': [InvalidOrder, 'Cash Amount is less than the minimum requirement'],
                    '20011': [InsufficientFunds, 'Insufficient balance'],
                    '20012': [BadRequest, 'Invalid trade type, valid value: buy/sell)'],
                    '20013': [InvalidOrder, 'No order info found'],
                    '20014': [BadRequest, 'Invalid date, Valid format: 2018-07-25)'],
                    '20015': [BadRequest, 'Date exceeds the limit'],
                    '20018': [PermissionDenied, 'Your trading rights have been banned by the system'],
                    '20019': [BadSymbol, 'Wrong trading pair symbol. Correct format:"usdt_btc". Quote asset is in the front'],
                    '20020': [DDoSProtection, "You have violated the API operation trading rules and temporarily forbid trading. At present, we have certain restrictions on the user's transaction rate and withdrawal rate."],
                    '50000': [ExchangeError, 'Exception error'],
                    '20021': [BadRequest, 'Invalid currency'],
                    '20022': [BadRequest, 'The ending timestamp must be larger than the starting timestamp'],
                    '20023': [BadRequest, 'Invalid transfer type'],
                    '20024': [BadRequest, 'Invalid amount'],
                    '20025': [BadRequest, 'This currency is not transferable at the moment'],
                    '20026': [InsufficientFunds, 'Transfer amount exceed your balance'],
                    '20027': [PermissionDenied, 'Abnormal account status'],
                    '20028': [PermissionDenied, 'Blacklist for transfer'],
                    '20029': [PermissionDenied, 'Transfer amount exceed your daily limit'],
                    '20030': [BadRequest, 'You have no position on self trading pair'],
                    '20032': [PermissionDenied, 'Withdrawal limited'],
                    '20033': [BadRequest, 'Wrong Withdrawal ID'],
                    '20034': [PermissionDenied, 'Withdrawal service of self crypto has been closed'],
                    '20035': [PermissionDenied, 'Withdrawal limit'],
                    '20036': [ExchangeError, 'Withdrawal cancellation failed'],
                    '20037': [InvalidAddress, 'The withdrawal address, Tag or chain type is not included in the withdrawal management list'],
                    '20038': [InvalidAddress, 'The withdrawal address is not on the white list'],
                    '20039': [ExchangeError, "Can't be canceled in current status"],
                    '20040': [RateLimitExceeded, 'Withdraw too frequently; limitation: 3 times a minute, 100 times a day'],
                    '20041': [PermissionDenied, 'Beyond the daily withdrawal limit'],
                    '20042': [BadSymbol, 'Current trading pair does not support API trading'],
                    '400002': [BadRequest, 'Invalid Parameter'],
                },
                'broad': {
                },
            },
            'options': {
                'defaultType': 'spot',
                'types': ['spot', 'margin', 'otc'],
                'accountsByType': {
                    'spot': '1',
                    'margin': '2',
                    'OTC': '3',
                },
            },
            'commonCurrencies': {
                'BHT': 'Black House Test',
                'EPS': 'Epanus',
                'FREE': 'FreeRossDAO',
                'MBN': 'Mobilian Coin',
                'TEL': 'TEL666',
            },
        })

    def safe_network(self, networkId):
        if networkId is None:
            return None
        else:
            return networkId.upper()

    def fetch_currencies(self, params={}):
        """
        fetches all available currencies on an exchange
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: an associative dictionary of currencies
        """
        response = self.publicSpotGetCurrencies(params)
        #
        #     {
        #         "data":[
        #             {
        #                 "deposit_status":1,
        #                 "min_deposit_amount":10,
        #                 "withdraw_fee_rate":0,
        #                 "min_withdraw_amount":10,
        #                 "min_withdraw_fee":5,
        #                 "currency":"USDT",
        #                 "withdraw_status":0,
        #                 "chain":"OMNI"
        #             },
        #             {
        #                 "deposit_status":1,
        #                 "min_deposit_amount":10,
        #                 "withdraw_fee_rate":0,
        #                 "min_withdraw_amount":10,
        #                 "min_withdraw_fee":3,
        #                 "currency":"USDT",
        #                 "withdraw_status":1,
        #                 "chain":"ERC20"
        #             },
        #             {
        #                 "deposit_status":0,
        #                 "min_deposit_amount":0,
        #                 "withdraw_fee_rate":0,
        #                 "min_withdraw_amount":0,
        #                 "min_withdraw_fee":0,
        #                 "currency":"DGF13",
        #                 "withdraw_status":0,
        #                 "chain":""
        #             },
        #         ],
        #         "code":200
        #     }
        #
        data = self.safe_value(response, 'data', [])
        result = {}
        for i in range(0, len(data)):
            currency = data[i]
            id = self.safe_string(currency, 'currency')
            code = self.safe_currency_code(id)
            depositStatus = self.safe_integer(currency, 'deposit_status', 1)
            withdrawStatus = self.safe_integer(currency, 'withdraw_status', 1)
            deposit = depositStatus > 0
            withdraw = withdrawStatus > 0
            active = deposit and withdraw
            fee = self.safe_number(currency, 'min_withdraw_fee')  # withdraw_fee_rate was zero for all currencies, so self was the worst case scenario
            minWithdraw = self.safe_number(currency, 'min_withdraw_amount')
            minDeposit = self.safe_number(currency, 'min_deposit_amount')
            networkId = self.safe_string(currency, 'chain')
            network = {
                'id': networkId,
                'network': self.safe_network(networkId),
                'name': None,
                'active': active,
                'fee': fee,
                'precision': self.parse_number('0.00000001'),  # todo fix hardcoded value
                'deposit': deposit,
                'withdraw': withdraw,
                'limits': {
                    'amount': {
                        'min': None,
                        'max': None,
                    },
                    'withdraw': {
                        'min': minWithdraw,
                        'max': None,
                    },
                    'deposit': {
                        'min': minDeposit,
                        'max': None,
                    },
                },
                'info': currency,
            }
            if code in result:
                if isinstance(result[code]['info'], list):
                    result[code]['info'].append(currency)
                else:
                    result[code]['info'] = [result[code]['info'], currency]
                if withdraw:
                    result[code]['withdraw'] = True
                    result[code]['limits']['withdraw']['min'] = min(result[code]['limits']['withdraw']['min'], minWithdraw)
                if deposit:
                    result[code]['deposit'] = True
                    result[code]['limits']['deposit']['min'] = min(result[code]['limits']['deposit']['min'], minDeposit)
                if active:
                    result[code]['active'] = True
            else:
                result[code] = {
                    'id': id,
                    'code': code,
                    'info': currency,
                    'type': None,
                    'name': None,
                    'active': active,
                    'deposit': deposit,
                    'withdraw': withdraw,
                    'fee': fee,
                    'precision': self.parse_number('0.00000001'),  # todo fix hardcoded value
                    'limits': {
                        'amount': {
                            'min': None,
                            'max': None,
                        },
                        'withdraw': {
                            'min': minWithdraw,
                            'max': None,
                        },
                        'deposit': {
                            'min': minDeposit,
                            'max': None,
                        },
                    },
                    'networks': {},
                }
            result[code]['networks'][networkId] = network
        return result

    def fetch_markets(self, params={}):
        """
        retrieves data on all markets for digifinex
        :param dict params: extra parameters specific to the exchange api endpoint
        :returns [dict]: an array of objects representing market data
        """
        options = self.safe_value(self.options, 'fetchMarkets', {})
        method = self.safe_string(options, 'method', 'fetch_markets_v2')
        return getattr(self, method)(params)

    def fetch_markets_v2(self, params={}):
        defaultType = self.safe_string(self.options, 'defaultType')
        marginMode, query = self.handle_margin_mode_and_params('fetchMarketsV2', params)
        method = 'publicSpotGetMarginSymbols' if (marginMode is not None) else 'publicSpotGetTradesSymbols'
        promises = [getattr(self, method)(query), self.publicSwapGetPublicInstruments(params)]
        spotMarkets = promises[0]
        swapMarkets = promises[1]
        #
        # Spot
        #
        #     {
        #         "symbol_list":[
        #             {
        #                 "order_types":["LIMIT","MARKET"],
        #                 "quote_asset":"USDT",
        #                 "minimum_value":2,
        #                 "amount_precision":4,
        #                 "status":"TRADING",
        #                 "minimum_amount":0.0001,
        #                 "symbol":"BTC_USDT",
        #                 "is_allow":1,
        #                 "zone":"MAIN",
        #                 "base_asset":"BTC",
        #                 "price_precision":2
        #             }
        #         ],
        #         "code":0
        #     }
        #
        # Margin
        #
        #     {
        #         "symbol_list":[
        #             {
        #                     "order_types":["LIMIT"],
        #                     "quote_asset":"USDT",
        #                     "minimum_value":0,
        #                     "amount_precision":2,
        #                     "status":"TRADING",
        #                     "minimum_amount":22,
        #                     "liquidation_rate":0.3,
        #                     "symbol":"TRX_USDT",
        #                     "zone":"MAIN",
        #                     "base_asset":"TRX",
        #                     "price_precision":6
        #             },
        #         ],
        #         "code":0
        #     }
        #
        # Swap
        #
        #     {
        #         "code": 0,
        #         "data": [
        #             {
        #                 "instrument_id": "BTCUSDTPERP",
        #                 "type": "REAL",
        #                 "contract_type": "PERPETUAL",
        #                 "base_currency": "BTC",
        #                 "quote_currency": "USDT",
        #                 "clear_currency": "USDT",
        #                 "contract_value": "0.001",
        #                 "contract_value_currency": "BTC",
        #                 "is_inverse": False,
        #                 "is_trading": True,
        #                 "status": "ONLINE",
        #                 "price_precision": 4,
        #                 "tick_size": "0.0001",
        #                 "min_order_amount": 1,
        #                 "open_max_limits": [
        #                     {
        #                         "leverage": "50",
        #                         "max_limit": "1000000"
        #                     }
        #                 ]
        #             },
        #         ]
        #     }
        #
        spotData = self.safe_value(spotMarkets, 'symbol_list', [])
        swapData = self.safe_value(swapMarkets, 'data', [])
        response = self.array_concat(spotData, swapData)
        result = []
        for i in range(0, len(response)):
            market = response[i]
            id = self.safe_string_2(market, 'symbol', 'instrument_id')
            baseId = self.safe_string_2(market, 'base_asset', 'base_currency')
            quoteId = self.safe_string_2(market, 'quote_asset', 'quote_currency')
            settleId = self.safe_string(market, 'clear_currency')
            base = self.safe_currency_code(baseId)
            quote = self.safe_currency_code(quoteId)
            settle = self.safe_currency_code(settleId)
            #
            # The status is documented in the exchange API docs as follows:
            # TRADING, HALT(delisted), BREAK(trading paused)
            # https://docs.digifinex.vip/en-ww/v3/#/public/spot/symbols
            # However, all spot markets actually have status == 'HALT'
            # despite that they appear to be active on the exchange website.
            # Apparently, we can't trust self status.
            # status = self.safe_string(market, 'status')
            # active = (status == 'TRADING')
            #
            isAllowed = self.safe_integer(market, 'is_allow', 1)
            type = 'margin' if (defaultType == 'margin') else 'spot'
            spot = settle is None
            swap = not spot
            margin = True if (marginMode is not None) else None
            symbol = base + '/' + quote
            isInverse = None
            isLinear = None
            if swap:
                type = 'swap'
                symbol = base + '/' + quote + ':' + settle
                isInverse = self.safe_value(market, 'is_inverse')
                isLinear = True if (not isInverse) else False
                isTrading = self.safe_value(market, 'isTrading')
                if isTrading:
                    isAllowed = True
            result.append({
                'id': id,
                'symbol': symbol,
                'base': base,
                'quote': quote,
                'settle': settle,
                'baseId': baseId,
                'quoteId': quoteId,
                'settleId': settleId,
                'type': type,
                'spot': spot,
                'margin': margin,
                'swap': swap,
                'future': False,
                'option': False,
                'active': True if isAllowed else None,
                'contract': swap,
                'linear': isLinear,
                'inverse': isInverse,
                'contractSize': self.safe_number(market, 'contract_value'),
                'expiry': None,
                'expiryDatetime': None,
                'strike': None,
                'optionType': None,
                'precision': {
                    'amount': self.parse_number(self.parse_precision(self.safe_string(market, 'amount_precision'))),
                    'price': self.parse_number(self.parse_precision(self.safe_string(market, 'price_precision'))),
                },
                'limits': {
                    'leverage': {
                        'min': None,
                        'max': None,
                    },
                    'amount': {
                        'min': self.safe_number_2(market, 'minimum_amount', 'min_order_amount'),
                        'max': None,
                    },
                    'price': {
                        'min': self.safe_number(market, 'tick_size'),
                        'max': None,
                    },
                    'cost': {
                        'min': self.safe_number(market, 'minimum_value'),
                        'max': None,
                    },
                },
                'info': market,
            })
        return result

    def fetch_markets_v1(self, params={}):
        response = self.publicSpotGetMarkets(params)
        #
        #     {
        #         "data": [
        #             {
        #                 "volume_precision":4,
        #                 "price_precision":2,
        #                 "market":"btc_usdt",
        #                 "min_amount":2,
        #                 "min_volume":0.0001
        #             },
        #         ],
        #         "date":1564507456,
        #         "code":0
        #     }
        #
        markets = self.safe_value(response, 'data', [])
        result = []
        for i in range(0, len(markets)):
            market = markets[i]
            id = self.safe_string(market, 'market')
            baseId, quoteId = id.split('_')
            base = self.safe_currency_code(baseId)
            quote = self.safe_currency_code(quoteId)
            result.append({
                'id': id,
                'symbol': base + '/' + quote,
                'base': base,
                'quote': quote,
                'settle': None,
                'baseId': baseId,
                'quoteId': quoteId,
                'settleId': None,
                'type': 'spot',
                'spot': True,
                'margin': None,
                'swap': False,
                'future': False,
                'option': False,
                'active': None,
                'contract': False,
                'linear': None,
                'inverse': None,
                'contractSize': None,
                'expiry': None,
                'expiryDatetime': None,
                'strike': None,
                'optionType': None,
                'precision': {
                    'price': self.parse_number(self.parse_precision(self.safe_string(market, 'price_precision'))),
                    'amount': self.parse_number(self.parse_precision(self.safe_string(market, 'volume_precision'))),
                },
                'limits': {
                    'leverage': {
                        'min': None,
                        'max': None,
                    },
                    'amount': {
                        'min': self.safe_number(market, 'min_volume'),
                        'max': None,
                    },
                    'price': {
                        'min': None,
                        'max': None,
                    },
                    'cost': {
                        'min': self.safe_number(market, 'min_amount'),
                        'max': None,
                    },
                },
                'info': market,
            })
        return result

    def parse_balance(self, response):
        balances = self.safe_value(response, 'list', [])
        result = {'info': response}
        for i in range(0, len(balances)):
            balance = balances[i]
            currencyId = self.safe_string(balance, 'currency')
            code = self.safe_currency_code(currencyId)
            account = self.account()
            account['used'] = self.safe_string(balance, 'frozen')
            account['free'] = self.safe_string(balance, 'free')
            account['total'] = self.safe_string(balance, 'total')
            result[code] = account
        return self.safe_balance(result)

    def fetch_balance(self, params={}):
        """
        query for balance and get the amount of funds available for trading or funds locked in orders
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: a `balance structure <https://docs.ccxt.com/en/latest/manual.html?#balance-structure>`
        """
        defaultType = self.safe_string(self.options, 'defaultType', 'spot')
        type = self.safe_string(params, 'type', defaultType)
        params = self.omit(params, 'type')
        method = 'privateSpotGet' + self.capitalize(type) + 'Assets'
        response = getattr(self, method)(params)
        #
        #     {
        #         "code": 0,
        #         "list": [
        #             {
        #                 "currency": "BTC",
        #                 "free": 4723846.89208129,
        #                 "total": 0
        #             }
        #         ]
        #     }
        return self.parse_balance(response)

    def fetch_order_book(self, symbol, limit=None, params={}):
        """
        fetches information on open orders with bid(buy) and ask(sell) prices, volumes and other data
        :param str symbol: unified symbol of the market to fetch the order book for
        :param int|None limit: the maximum amount of order book entries to return
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: A dictionary of `order book structures <https://docs.ccxt.com/en/latest/manual.html#order-book-structure>` indexed by market symbols
        """
        self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
        }
        if limit is not None:
            request['limit'] = limit  # default 10, max 150
        response = self.publicSpotGetOrderBook(self.extend(request, params))
        #
        #     {
        #         "bids": [
        #             [9605.77,0.0016],
        #             [9605.46,0.0003],
        #             [9602.04,0.0127],
        #         ],
        #         "asks": [
        #             [9627.22,0.025803],
        #             [9627.12,0.168543],
        #             [9626.52,0.0011529],
        #         ],
        #         "date":1564509499,
        #         "code":0
        #     }
        #
        timestamp = self.safe_timestamp(response, 'date')
        return self.parse_order_book(response, market['symbol'], timestamp)

    def fetch_tickers(self, symbols=None, params={}):
        """
        fetches price tickers for multiple markets, statistical calculations with the information calculated over the past 24 hours each market
        :param [str]|None symbols: unified symbols of the markets to fetch the ticker for, all market tickers are returned if not assigned
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: an array of `ticker structures <https://docs.ccxt.com/en/latest/manual.html#ticker-structure>`
        """
        self.load_markets()
        symbols = self.market_symbols(symbols)
        response = self.publicSpotGetTicker(params)
        #
        #    {
        #        "ticker": [{
        #            "vol": 40717.4461,
        #            "change": -1.91,
        #            "base_vol": 392447999.65374,
        #            "sell": 9592.23,
        #            "last": 9592.22,
        #            "symbol": "btc_usdt",
        #            "low": 9476.24,
        #            "buy": 9592.03,
        #            "high": 9793.87
        #        }],
        #        "date": 1589874294,
        #        "code": 0
        #    }
        #
        result = {}
        tickers = self.safe_value(response, 'ticker', [])
        date = self.safe_integer(response, 'date')
        for i in range(0, len(tickers)):
            rawTicker = self.extend({
                'date': date,
            }, tickers[i])
            ticker = self.parse_ticker(rawTicker)
            symbol = ticker['symbol']
            result[symbol] = ticker
        return self.filter_by_array(result, 'symbol', symbols)

    def fetch_ticker(self, symbol, params={}):
        """
        fetches a price ticker, a statistical calculation with the information calculated over the past 24 hours for a specific market
        :param str symbol: unified symbol of the market to fetch the ticker for
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: a `ticker structure <https://docs.ccxt.com/en/latest/manual.html#ticker-structure>`
        """
        self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
        }
        response = self.publicSpotGetTicker(self.extend(request, params))
        #
        #    {
        #        "ticker": [{
        #            "vol": 40717.4461,
        #            "change": -1.91,
        #            "base_vol": 392447999.65374,
        #            "sell": 9592.23,
        #            "last": 9592.22,
        #            "symbol": "btc_usdt",
        #            "low": 9476.24,
        #            "buy": 9592.03,
        #            "high": 9793.87
        #        }],
        #        "date": 1589874294,
        #        "code": 0
        #    }
        #
        date = self.safe_integer(response, 'date')
        tickers = self.safe_value(response, 'ticker', [])
        firstTicker = self.safe_value(tickers, 0, {})
        result = self.extend({'date': date}, firstTicker)
        return self.parse_ticker(result, market)

    def parse_ticker(self, ticker, market=None):
        #
        # fetchTicker, fetchTickers
        #
        #     {
        #         "last":0.021957,
        #         "symbol": "btc_usdt",
        #         "base_vol":2249.3521732227,
        #         "change":-0.6,
        #         "vol":102443.5111,
        #         "sell":0.021978,
        #         "low":0.021791,
        #         "buy":0.021946,
        #         "high":0.022266,
        #         "date"1564518452,  # injected from fetchTicker/fetchTickers
        #     }
        #
        marketId = self.safe_string_upper(ticker, 'symbol')
        symbol = self.safe_symbol(marketId, market, '_')
        timestamp = self.safe_timestamp(ticker, 'date')
        last = self.safe_string(ticker, 'last')
        percentage = self.safe_string(ticker, 'change')
        return self.safe_ticker({
            'symbol': symbol,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'high': self.safe_string(ticker, 'high'),
            'low': self.safe_string(ticker, 'low'),
            'bid': self.safe_string(ticker, 'buy'),
            'bidVolume': None,
            'ask': self.safe_string(ticker, 'sell'),
            'askVolume': None,
            'vwap': None,
            'open': None,
            'close': last,
            'last': last,
            'previousClose': None,
            'change': None,
            'percentage': percentage,
            'average': None,
            'baseVolume': self.safe_string(ticker, 'vol'),
            'quoteVolume': self.safe_string(ticker, 'base_vol'),
            'info': ticker,
        }, market)

    def parse_trade(self, trade, market=None):
        #
        # fetchTrades(public)
        #
        #     {
        #         "date":1564520003,
        #         "id":1596149203,
        #         "amount":0.7073,
        #         "type":"buy",
        #         "price":0.02193,
        #     }
        #
        # fetchMyTrades(private)
        #
        #     {
        #         "symbol": "BTC_USDT",
        #         "order_id": "6707cbdcda0edfaa7f4ab509e4cbf966",
        #         "id": 28457,
        #         "price": 0.1,
        #         "amount": 0,
        #         "fee": 0.096,
        #         "fee_currency": "USDT",
        #         "timestamp": 1499865549,
        #         "side": "buy",  # or "side": "sell_market"
        #         "is_maker": True
        #     }
        #
        id = self.safe_string(trade, 'id')
        orderId = self.safe_string(trade, 'order_id')
        timestamp = self.safe_timestamp_2(trade, 'date', 'timestamp')
        side = self.safe_string_2(trade, 'type', 'side')
        parts = side.split('_')
        side = self.safe_string(parts, 0)
        type = self.safe_string(parts, 1)
        priceString = self.safe_string(trade, 'price')
        amountString = self.safe_string(trade, 'amount')
        marketId = self.safe_string(trade, 'symbol')
        symbol = self.safe_symbol(marketId, market, '_')
        takerOrMaker = self.safe_value(trade, 'is_maker')
        feeCostString = self.safe_string(trade, 'fee')
        fee = None
        if feeCostString is not None:
            feeCurrencyId = self.safe_string(trade, 'fee_currency')
            feeCurrencyCode = self.safe_currency_code(feeCurrencyId)
            fee = {
                'cost': feeCostString,
                'currency': feeCurrencyCode,
            }
        return self.safe_trade({
            'id': id,
            'info': trade,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'symbol': symbol,
            'type': type,
            'order': orderId,
            'side': side,
            'price': priceString,
            'amount': amountString,
            'cost': None,
            'takerOrMaker': takerOrMaker,
            'fee': fee,
        }, market)

    def fetch_time(self, params={}):
        """
        fetches the current integer timestamp in milliseconds from the exchange server
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns int: the current integer timestamp in milliseconds from the exchange server
        """
        response = self.publicSpotGetTime(params)
        #
        #     {
        #         "server_time": 1589873762,
        #         "code": 0
        #     }
        #
        return self.safe_timestamp(response, 'server_time')

    def fetch_status(self, params={}):
        """
        the latest known information on the availability of the exchange API
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: a `status structure <https://docs.ccxt.com/en/latest/manual.html#exchange-status-structure>`
        """
        response = self.publicSpotGetPing(params)
        #
        #     {
        #         "msg": "pong",
        #         "code": 0
        #     }
        #
        code = self.safe_integer(response, 'code')
        status = 'ok' if (code == 0) else 'maintenance'
        return {
            'status': status,
            'updated': None,
            'eta': None,
            'url': None,
            'info': response,
        }

    def fetch_trades(self, symbol, since=None, limit=None, params={}):
        """
        get the list of most recent trades for a particular symbol
        :param str symbol: unified symbol of the market to fetch trades for
        :param int|None since: timestamp in ms of the earliest trade to fetch
        :param int|None limit: the maximum amount of trades to fetch
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns [dict]: a list of `trade structures <https://docs.ccxt.com/en/latest/manual.html?#public-trades>`
        """
        self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
        }
        if limit is not None:
            request['limit'] = limit  # default 100, max 500
        response = self.publicSpotGetTrades(self.extend(request, params))
        #
        #     {
        #         "data":[
        #             {
        #                 "date":1564520003,
        #                 "id":1596149203,
        #                 "amount":0.7073,
        #                 "type":"buy",
        #                 "price":0.02193,
        #             },
        #             {
        #                 "date":1564520002,
        #                 "id":1596149165,
        #                 "amount":0.3232,
        #                 "type":"sell",
        #                 "price":0.021927,
        #             },
        #         ],
        #         "code": 0,
        #         "date": 1564520003,
        #     }
        #
        data = self.safe_value(response, 'data', [])
        return self.parse_trades(data, market, since, limit)

    def parse_ohlcv(self, ohlcv, market=None):
        #
        #     [
        #         1556712900,
        #         2205.899,
        #         0.029967,
        #         0.02997,
        #         0.029871,
        #         0.029927
        #     ]
        #
        return [
            self.safe_timestamp(ohlcv, 0),
            self.safe_number(ohlcv, 5),  # open
            self.safe_number(ohlcv, 3),  # high
            self.safe_number(ohlcv, 4),  # low
            self.safe_number(ohlcv, 2),  # close
            self.safe_number(ohlcv, 1),  # volume
        ]

    def fetch_ohlcv(self, symbol, timeframe='1m', since=None, limit=None, params={}):
        """
        fetches historical candlestick data containing the open, high, low, and close price, and the volume of a market
        :param str symbol: unified symbol of the market to fetch OHLCV data for
        :param str timeframe: the length of time each candle represents
        :param int|None since: timestamp in ms of the earliest candle to fetch
        :param int|None limit: the maximum amount of candles to fetch
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns [[int]]: A list of candles ordered as timestamp, open, high, low, close, volume
        """
        self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
            'period': self.timeframes[timeframe],
            # 'start_time': 1564520003,  # starting timestamp, 200 candles before end_time by default
            # 'end_time': 1564520003,  # ending timestamp, current timestamp by default
        }
        if since is not None:
            startTime = int(since / 1000)
            request['start_time'] = startTime
            if limit is not None:
                duration = self.parse_timeframe(timeframe)
                request['end_time'] = self.sum(startTime, limit * duration)
        elif limit is not None:
            endTime = self.seconds()
            duration = self.parse_timeframe(timeframe)
            request['startTime'] = self.sum(endTime, -limit * duration)
        response = self.publicSpotGetKline(self.extend(request, params))
        #
        #     {
        #         "code":0,
        #         "data":[
        #             [1556712900,2205.899,0.029967,0.02997,0.029871,0.029927],
        #             [1556713800,1912.9174,0.029992,0.030014,0.029955,0.02996],
        #             [1556714700,1556.4795,0.029974,0.030019,0.029969,0.02999],
        #         ]
        #     }
        #
        data = self.safe_value(response, 'data', [])
        return self.parse_ohlcvs(data, market, timeframe, since, limit)

    def create_order(self, symbol, type, side, amount, price=None, params={}):
        """
        create a trade order
        :param str symbol: unified symbol of the market to create an order in
        :param str type: 'market' or 'limit'
        :param str side: 'buy' or 'sell'
        :param float amount: how much of currency you want to trade in units of base currency
        :param float|None price: the price at which the order is to be fullfilled, in units of the quote currency, ignored in market orders
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: an `order structure <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        self.load_markets()
        market = self.market(symbol)
        defaultType = self.safe_string(self.options, 'defaultType', 'spot')
        orderType = self.safe_string(params, 'type', defaultType)
        params = self.omit(params, 'type')
        request = {
            'market': orderType,
            'symbol': market['id'],
            'amount': self.amount_to_precision(market['symbol'], amount),
            # 'post_only': 0,  # 0 by default, if set to 1 the order will be canceled if it can be executed immediately, making sure there will be no market taking
        }
        suffix = ''
        if type == 'market':
            suffix = '_market'
        else:
            request['price'] = self.price_to_precision(market['symbol'], price)
        request['type'] = side + suffix
        response = self.privateSpotPostMarketOrderNew(self.extend(request, params))
        #
        #     {
        #         "code": 0,
        #         "order_id": "198361cecdc65f9c8c9bb2fa68faec40"
        #     }
        #
        result = self.parse_order(response, market)
        return self.extend(result, {
            'symbol': market['symbol'],
            'side': side,
            'type': type,
            'amount': amount,
            'price': price,
        })

    def cancel_order(self, id, symbol=None, params={}):
        """
        cancels an open order
        :param str id: order id
        :param str|None symbol: not used by digifinex cancelOrder()
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: An `order structure <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        self.load_markets()
        defaultType = self.safe_string(self.options, 'defaultType', 'spot')
        orderType = self.safe_string(params, 'type', defaultType)
        params = self.omit(params, 'type')
        request = {
            'market': orderType,
            'order_id': id,
        }
        response = self.privateSpotPostMarketOrderCancel(self.extend(request, params))
        #
        #     {
        #         "code": 0,
        #         "success": [
        #             "198361cecdc65f9c8c9bb2fa68faec40",
        #             "3fb0d98e51c18954f10d439a9cf57de0"
        #         ],
        #         "error": [
        #             "78a7104e3c65cc0c5a212a53e76d0205"
        #         ]
        #     }
        #
        canceledOrders = self.safe_value(response, 'success', [])
        numCanceledOrders = len(canceledOrders)
        if numCanceledOrders != 1:
            raise OrderNotFound(self.id + ' cancelOrder() ' + id + ' not found')
        return response

    def cancel_orders(self, ids, symbol=None, params={}):
        """
        cancel multiple orders
        :param [str] ids: order ids
        :param str|None symbol: not used by digifinex cancelOrders()
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: an list of `order structures <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        self.load_markets()
        defaultType = self.safe_string(self.options, 'defaultType', 'spot')
        orderType = self.safe_string(params, 'type', defaultType)
        params = self.omit(params, 'type')
        request = {
            'market': orderType,
            'order_id': ','.join(ids),
        }
        response = self.privateSpotPostCancelOrder(self.extend(request, params))
        #
        #     {
        #         "code": 0,
        #         "success": [
        #             "198361cecdc65f9c8c9bb2fa68faec40",
        #             "3fb0d98e51c18954f10d439a9cf57de0"
        #         ],
        #         "error": [
        #             "78a7104e3c65cc0c5a212a53e76d0205"
        #         ]
        #     }
        #
        canceledOrders = self.safe_value(response, 'success', [])
        numCanceledOrders = len(canceledOrders)
        if numCanceledOrders < 1:
            raise OrderNotFound(self.id + ' cancelOrders() error')
        return response

    def parse_order_status(self, status):
        statuses = {
            '0': 'open',
            '1': 'open',  # partially filled
            '2': 'closed',
            '3': 'canceled',
            '4': 'canceled',  # partially filled and canceled
        }
        return self.safe_string(statuses, status, status)

    def parse_order(self, order, market=None):
        #
        # createOrder
        #
        #     {
        #         "code": 0,
        #         "order_id": "198361cecdc65f9c8c9bb2fa68faec40"
        #     }
        #
        # fetchOrder, fetchOpenOrders, fetchOrders
        #
        #     {
        #         "symbol": "BTC_USDT",
        #         "order_id": "dd3164b333a4afa9d5730bb87f6db8b3",
        #         "created_date": 1562303547,
        #         "finished_date": 0,
        #         "price": 0.1,
        #         "amount": 1,
        #         "cash_amount": 1,
        #         "executed_amount": 0,
        #         "avg_price": 0,
        #         "status": 1,
        #         "type": "buy",
        #         "kind": "margin"
        #     }
        #
        id = self.safe_string(order, 'order_id')
        timestamp = self.safe_timestamp(order, 'created_date')
        lastTradeTimestamp = self.safe_timestamp(order, 'finished_date')
        side = self.safe_string(order, 'type')
        type = None
        if side is not None:
            parts = side.split('_')
            numParts = len(parts)
            if numParts > 1:
                side = parts[0]
                type = parts[1]
            else:
                type = 'limit'
        status = self.parse_order_status(self.safe_string(order, 'status'))
        marketId = self.safe_string(order, 'symbol')
        symbol = self.safe_symbol(marketId, market, '_')
        amountString = self.safe_string(order, 'amount')
        filledString = self.safe_string(order, 'executed_amount')
        priceString = self.safe_string(order, 'price')
        averageString = self.safe_string(order, 'avg_price')
        return self.safe_order({
            'info': order,
            'id': id,
            'clientOrderId': None,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'lastTradeTimestamp': lastTradeTimestamp,
            'symbol': symbol,
            'type': type,
            'timeInForce': None,
            'postOnly': None,
            'side': side,
            'price': priceString,
            'stopPrice': None,
            'amount': amountString,
            'filled': filledString,
            'remaining': None,
            'cost': None,
            'average': averageString,
            'status': status,
            'fee': None,
            'trades': None,
        }, market)

    def fetch_open_orders(self, symbol=None, since=None, limit=None, params={}):
        """
        fetch all unfilled currently open orders
        :param str|None symbol: unified market symbol
        :param int|None since: the earliest time in ms to fetch open orders for
        :param int|None limit: the maximum number of  open orders structures to retrieve
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns [dict]: a list of `order structures <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        defaultType = self.safe_string(self.options, 'defaultType', 'spot')
        orderType = self.safe_string(params, 'type', defaultType)
        params = self.omit(params, 'type')
        self.load_markets()
        market = None
        request = {
            'market': orderType,
        }
        if symbol is not None:
            market = self.market(symbol)
            request['symbol'] = market['id']
        response = self.privateSpotGetMarketOrderCurrent(self.extend(request, params))
        #
        #     {
        #         "code": 0,
        #         "data": [
        #             {
        #                 "symbol": "BTC_USDT",
        #                 "order_id": "dd3164b333a4afa9d5730bb87f6db8b3",
        #                 "created_date": 1562303547,
        #                 "finished_date": 0,
        #                 "price": 0.1,
        #                 "amount": 1,
        #                 "cash_amount": 1,
        #                 "executed_amount": 0,
        #                 "avg_price": 0,
        #                 "status": 1,
        #                 "type": "buy",
        #                 "kind": "margin"
        #             }
        #         ]
        #     }
        #
        data = self.safe_value(response, 'data', [])
        return self.parse_orders(data, market, since, limit)

    def fetch_orders(self, symbol=None, since=None, limit=None, params={}):
        """
        fetches information on multiple orders made by the user
        :param str|None symbol: unified market symbol of the market orders were made in
        :param int|None since: the earliest time in ms to fetch orders for
        :param int|None limit: the maximum number of  orde structures to retrieve
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns [dict]: a list of `order structures <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        defaultType = self.safe_string(self.options, 'defaultType', 'spot')
        orderType = self.safe_string(params, 'type', defaultType)
        params = self.omit(params, 'type')
        self.load_markets()
        market = None
        request = {
            'market': orderType,
        }
        if symbol is not None:
            market = self.market(symbol)
            request['symbol'] = market['id']
        if since is not None:
            request['start_time'] = int(since / 1000)  # default 3 days from now, max 30 days
        if limit is not None:
            request['limit'] = limit  # default 10, max 100
        response = self.privateSpotGetMarketOrderHistory(self.extend(request, params))
        #
        #     {
        #         "code": 0,
        #         "data": [
        #             {
        #                 "symbol": "BTC_USDT",
        #                 "order_id": "dd3164b333a4afa9d5730bb87f6db8b3",
        #                 "created_date": 1562303547,
        #                 "finished_date": 0,
        #                 "price": 0.1,
        #                 "amount": 1,
        #                 "cash_amount": 1,
        #                 "executed_amount": 0,
        #                 "avg_price": 0,
        #                 "status": 1,
        #                 "type": "buy",
        #                 "kind": "margin"
        #             }
        #         ]
        #     }
        #
        data = self.safe_value(response, 'data', [])
        return self.parse_orders(data, market, since, limit)

    def fetch_order(self, id, symbol=None, params={}):
        """
        fetches information on an order made by the user
        :param str|None symbol: unified symbol of the market the order was made in
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: An `order structure <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        defaultType = self.safe_string(self.options, 'defaultType', 'spot')
        orderType = self.safe_string(params, 'type', defaultType)
        params = self.omit(params, 'type')
        self.load_markets()
        market = None
        if symbol is not None:
            market = self.market(symbol)
        request = {
            'market': orderType,
            'order_id': id,
        }
        response = self.privateSpotGetMarketOrder(self.extend(request, params))
        #
        #     {
        #         "code": 0,
        #         "data": [
        #             {
        #                 "symbol": "BTC_USDT",
        #                 "order_id": "dd3164b333a4afa9d5730bb87f6db8b3",
        #                 "created_date": 1562303547,
        #                 "finished_date": 0,
        #                 "price": 0.1,
        #                 "amount": 1,
        #                 "cash_amount": 1,
        #                 "executed_amount": 0,
        #                 "avg_price": 0,
        #                 "status": 1,
        #                 "type": "buy",
        #                 "kind": "margin"
        #             }
        #         ]
        #     }
        #
        data = self.safe_value(response, 'data', [])
        order = self.safe_value(data, 0)
        if order is None:
            raise OrderNotFound(self.id + ' fetchOrder() order ' + id + ' not found')
        return self.parse_order(order, market)

    def fetch_my_trades(self, symbol=None, since=None, limit=None, params={}):
        """
        fetch all trades made by the user
        :param str|None symbol: unified market symbol
        :param int|None since: the earliest time in ms to fetch trades for
        :param int|None limit: the maximum number of trades structures to retrieve
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns [dict]: a list of `trade structures <https://docs.ccxt.com/en/latest/manual.html#trade-structure>`
        """
        defaultType = self.safe_string(self.options, 'defaultType', 'spot')
        orderType = self.safe_string(params, 'type', defaultType)
        params = self.omit(params, 'type')
        self.load_markets()
        market = None
        request = {
            'market': orderType,
        }
        if symbol is not None:
            market = self.market(symbol)
            request['symbol'] = market['id']
        if since is not None:
            request['start_time'] = int(since / 1000)  # default 3 days from now, max 30 days
        if limit is not None:
            request['limit'] = limit  # default 10, max 100
        response = self.privateSpotGetMarketMytrades(self.extend(request, params))
        #
        #      {
        #          "list":[
        #              {
        #                  "timestamp":1639506068,
        #                  "is_maker":false,
        #                  "id":"8975951332",
        #                  "amount":31.83,
        #                  "side":"sell_market",
        #                  "symbol":"DOGE_USDT",
        #                  "fee_currency":"USDT",
        #                  "fee":0.01163774826
        #                  ,"order_id":"32b169792f4a7a19e5907dc29fc123d4",
        #                  "price":0.182811
        #                }
        #             ],
        #           "code": 0
        #      }
        #
        data = self.safe_value(response, 'list', [])
        return self.parse_trades(data, market, since, limit)

    def parse_ledger_entry_type(self, type):
        types = {}
        return self.safe_string(types, type, type)

    def parse_ledger_entry(self, item, currency=None):
        #
        #     {
        #         "currency_mark": "BTC",
        #         "type": 100234,
        #         "num": 28457,
        #         "balance": 0.1,
        #         "time": 1546272000
        #     }
        #
        id = self.safe_string(item, 'num')
        account = None
        type = self.parse_ledger_entry_type(self.safe_string(item, 'type'))
        code = self.safe_currency_code(self.safe_string(item, 'currency_mark'), currency)
        timestamp = self.safe_timestamp(item, 'time')
        before = None
        after = self.safe_number(item, 'balance')
        status = 'ok'
        return {
            'info': item,
            'id': id,
            'direction': None,
            'account': account,
            'referenceId': None,
            'referenceAccount': None,
            'type': type,
            'currency': code,
            'amount': None,
            'before': before,
            'after': after,
            'status': status,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'fee': None,
        }

    def fetch_ledger(self, code=None, since=None, limit=None, params={}):
        """
        fetch the history of changes, actions done by the user or operations that altered balance of the user
        :param str|None code: unified currency code, default is None
        :param int|None since: timestamp in ms of the earliest ledger entry, default is None
        :param int|None limit: max number of ledger entrys to return, default is None
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: a `ledger structure <https://docs.ccxt.com/en/latest/manual.html#ledger-structure>`
        """
        defaultType = self.safe_string(self.options, 'defaultType', 'spot')
        orderType = self.safe_string(params, 'type', defaultType)
        params = self.omit(params, 'type')
        self.load_markets()
        request = {
            'market': orderType,
        }
        currency = None
        if code is not None:
            currency = self.currency(code)
            request['currency_mark'] = currency['id']
        if since is not None:
            request['start_time'] = int(since / 1000)
        if limit is not None:
            request['limit'] = limit  # default 100, max 1000
        response = self.privateSpotGetMarketFinancelog(self.extend(request, params))
        #
        #     {
        #         "code": 0,
        #         "data": {
        #             "total": 521,
        #             "finance": [
        #                 {
        #                     "currency_mark": "BTC",
        #                     "type": 100234,
        #                     "num": 28457,
        #                     "balance": 0.1,
        #                     "time": 1546272000
        #                 }
        #             ]
        #         }
        #     }
        #
        data = self.safe_value(response, 'data', {})
        items = self.safe_value(data, 'finance', [])
        return self.parse_ledger(items, currency, since, limit)

    def parse_deposit_address(self, depositAddress, currency=None):
        #
        #     {
        #         "addressTag":"",
        #         "address":"0xf1104d9f8624f89775a3e9d480fc0e75a8ef4373",
        #         "currency":"USDT",
        #         "chain":"ERC20"
        #     }
        #
        address = self.safe_string(depositAddress, 'address')
        tag = self.safe_string(depositAddress, 'addressTag')
        currencyId = self.safe_string_upper(depositAddress, 'currency')
        code = self.safe_currency_code(currencyId)
        return {
            'info': depositAddress,
            'currency': code,
            'address': address,
            'tag': tag,
            'network': None,
        }

    def fetch_deposit_address(self, code, params={}):
        """
        fetch the deposit address for a currency associated with self account
        :param str code: unified currency code
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: an `address structure <https://docs.ccxt.com/en/latest/manual.html#address-structure>`
        """
        self.load_markets()
        currency = self.currency(code)
        request = {
            'currency': currency['id'],
        }
        response = self.privateSpotGetDepositAddress(self.extend(request, params))
        #
        #     {
        #         "data":[
        #             {
        #                 "addressTag":"",
        #                 "address":"0xf1104d9f8624f89775a3e9d480fc0e75a8ef4373",
        #                 "currency":"USDT",
        #                 "chain":"ERC20"
        #             }
        #         ],
        #         "code":200
        #     }
        #
        data = self.safe_value(response, 'data', [])
        addresses = self.parse_deposit_addresses(data)
        address = self.safe_value(addresses, code)
        if address is None:
            raise InvalidAddress(self.id + ' fetchDepositAddress() did not return an address for ' + code + ' - create the deposit address in the user settings on the exchange website first.')
        return address

    def fetch_transactions_by_type(self, type, code=None, since=None, limit=None, params={}):
        self.load_markets()
        currency = None
        request = {
            # 'currency': currency['id'],
            # 'from': 'fromId',  # When direct is' prev ', from is 1, returning from old to new ascending, when direct is' next ', from is the ID of the most recent record, returned from the old descending order
            # 'size': 100,  # default 100, max 500
            # 'direct': 'prev',  # "prev" ascending, "next" descending
        }
        if code is not None:
            currency = self.currency(code)
            request['currency'] = currency['id']
        if limit is not None:
            request['size'] = min(500, limit)
        method = 'privateSpotGetDepositHistory' if (type == 'deposit') else 'privateSpotGetWithdrawHistory'
        response = getattr(self, method)(self.extend(request, params))
        #
        #     {
        #         "code": 200,
        #         "data": [
        #             {
        #                 "id": 1171,
        #                 "currency": "xrp",
        #                 "hash": "ed03094b84eafbe4bc16e7ef766ee959885ee5bcb265872baaa9c64e1cf86c2b",
        #                 "chain": "",
        #                 "amount": 7.457467,
        #                 "address": "rae93V8d2mdoUQHwBDBdM4NHCMehRJAsbm",
        #                 "memo": "100040",
        #                 "fee": 0,
        #                 "state": "safe",
        #                 "created_date": "2020-04-20 11:23:00",
        #                 "finished_date": "2020-04-20 13:23:00"
        #             },
        #         ]
        #     }
        #
        data = self.safe_value(response, 'data', [])
        return self.parse_transactions(data, currency, since, limit, {'type': type})

    def fetch_deposits(self, code=None, since=None, limit=None, params={}):
        """
        fetch all deposits made to an account
        :param str|None code: unified currency code
        :param int|None since: the earliest time in ms to fetch deposits for
        :param int|None limit: the maximum number of deposits structures to retrieve
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns [dict]: a list of `transaction structures <https://docs.ccxt.com/en/latest/manual.html#transaction-structure>`
        """
        return self.fetch_transactions_by_type('deposit', code, since, limit, params)

    def fetch_withdrawals(self, code=None, since=None, limit=None, params={}):
        """
        fetch all withdrawals made from an account
        :param str|None code: unified currency code
        :param int|None since: the earliest time in ms to fetch withdrawals for
        :param int|None limit: the maximum number of withdrawals structures to retrieve
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns [dict]: a list of `transaction structures <https://docs.ccxt.com/en/latest/manual.html#transaction-structure>`
        """
        return self.fetch_transactions_by_type('withdrawal', code, since, limit, params)

    def parse_transaction_status(self, status):
        # deposit state includes: 1(in deposit), 2(to be confirmed), 3(successfully deposited), 4(stopped)
        # withdrawal state includes: 1(application in progress), 2(to be confirmed), 3(completed), 4(rejected)
        statuses = {
            '1': 'pending',  # in Progress
            '2': 'pending',  # to be confirmed
            '3': 'ok',  # Completed
            '4': 'failed',  # Rejected
        }
        return self.safe_string(statuses, status, status)

    def parse_transaction(self, transaction, currency=None):
        #
        # withdraw
        #
        #     {
        #         "code": 200,
        #         "withdraw_id": 700
        #     }
        #
        # fetchDeposits, fetchWithdrawals
        #
        #     {
        #         "id": 1171,
        #         "currency": "xrp",
        #         "hash": "ed03094b84eafbe4bc16e7ef766ee959885ee5bcb265872baaa9c64e1cf86c2b",
        #         "chain": "",
        #         "amount": 7.457467,
        #         "address": "rae93V8d2mdoUQHwBDBdM4NHCMehRJAsbm",
        #         "memo": "100040",
        #         "fee": 0,
        #         "state": "safe",
        #         "created_date": "2020-04-20 11:23:00",
        #         "finished_date": "2020-04-20 13:23:00"
        #     }
        #
        id = self.safe_string_2(transaction, 'id', 'withdraw_id')
        address = self.safe_string(transaction, 'address')
        tag = self.safe_string(transaction, 'memo')
        txid = self.safe_string(transaction, 'hash')
        currencyId = self.safe_string_upper(transaction, 'currency')
        code = self.safe_currency_code(currencyId, currency)
        timestamp = self.parse8601(self.safe_string(transaction, 'created_date'))
        updated = self.parse8601(self.safe_string(transaction, 'finished_date'))
        status = self.parse_transaction_status(self.safe_string(transaction, 'state'))
        amount = self.safe_number(transaction, 'amount')
        feeCost = self.safe_number(transaction, 'fee')
        fee = None
        if feeCost is not None:
            fee = {'currency': code, 'cost': feeCost}
        network = self.safe_string(transaction, 'chain')
        return {
            'info': transaction,
            'id': id,
            'txid': txid,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'network': network,
            'address': address,
            'addressTo': address,
            'addressFrom': None,
            'tag': tag,
            'tagTo': tag,
            'tagFrom': None,
            'type': None,
            'amount': amount,
            'currency': code,
            'status': status,
            'updated': updated,
            'fee': fee,
        }

    def parse_transfer_status(self, status):
        statuses = {
            '0': 'ok',
        }
        return self.safe_string(statuses, status, status)

    def parse_transfer(self, transfer, currency=None):
        #
        #     {
        #         "code": 0
        #     }
        #
        return {
            'info': transfer,
            'id': None,
            'timestamp': None,
            'datetime': None,
            'currency': self.safe_currency_code(None, currency),
            'amount': self.safe_number(transfer, 'amount'),
            'fromAccount': self.safe_string(transfer, 'fromAccount'),
            'toAccount': self.safe_string(transfer, 'toAccount'),
            'status': self.parse_transfer_status(self.safe_string(transfer, 'code')),
        }

    def transfer(self, code, amount, fromAccount, toAccount, params={}):
        """
        transfer currency internally between wallets on the same account
        :param str code: unified currency code
        :param float amount: amount to transfer
        :param str fromAccount: account to transfer from
        :param str toAccount: account to transfer to
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: a `transfer structure <https://docs.ccxt.com/en/latest/manual.html#transfer-structure>`
        """
        self.load_markets()
        currency = self.currency(code)
        accountsByType = self.safe_value(self.options, 'accountsByType', {})
        fromId = self.safe_string(accountsByType, fromAccount, fromAccount)
        toId = self.safe_string(accountsByType, toAccount, toAccount)
        request = {
            'currency_mark': currency['id'],
            'num': float(self.currency_to_precision(code, amount)),
            'from': fromId,  # 1 = SPOT, 2 = MARGIN, 3 = OTC
            'to': toId,  # 1 = SPOT, 2 = MARGIN, 3 = OTC
        }
        response = self.privateSpotPostTransfer(self.extend(request, params))
        #
        #     {
        #         "code": 0
        #     }
        #
        transfer = self.parse_transfer(response, currency)
        return self.extend(transfer, {
            'amount': amount,
            'currency': code,
            'fromAccount': fromAccount,
            'toAccount': toAccount,
        })

    def withdraw(self, code, amount, address, tag=None, params={}):
        """
        make a withdrawal
        :param str code: unified currency code
        :param float amount: the amount to withdraw
        :param str address: the address to withdraw to
        :param str|None tag:
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: a `transaction structure <https://docs.ccxt.com/en/latest/manual.html#transaction-structure>`
        """
        tag, params = self.handle_withdraw_tag_and_params(tag, params)
        self.check_address(address)
        self.load_markets()
        currency = self.currency(code)
        request = {
            # 'chain': 'ERC20', 'OMNI', 'TRC20',  # required for USDT
            'address': address,
            'amount': float(amount),
            'currency': currency['id'],
        }
        if tag is not None:
            request['memo'] = tag
        response = self.privateSpotPostWithdrawNew(self.extend(request, params))
        #
        #     {
        #         "code": 200,
        #         "withdraw_id": 700
        #     }
        #
        return self.parse_transaction(response, currency)

    def fetch_borrow_interest(self, code=None, symbol=None, since=None, limit=None, params={}):
        self.load_markets()
        request = {}
        market = None
        if symbol is not None:
            market = self.market(symbol)
            request['symbol'] = market['id']
        response = self.privateSpotGetMarginPositions(self.extend(request, params))
        #
        #     {
        #         "margin": "45.71246418952618",
        #         "code": 0,
        #         "margin_rate": "7.141978570340037",
        #         "positions": [
        #             {
        #                 "amount": 0.0006103,
        #                 "side": "go_long",
        #                 "entry_price": 31428.72,
        #                 "liquidation_rate": 0.3,
        #                 "liquidation_price": 10225.335481159,
        #                 "unrealized_roe": -0.0076885829266987,
        #                 "symbol": "BTC_USDT",
        #                 "unrealized_pnl": -0.049158102631999,
        #                 "leverage_ratio": 3
        #             }
        #         ],
        #         "unrealized_pnl": "-0.049158102631998504"
        #     }
        #
        rows = self.safe_value(response, 'positions')
        interest = self.parse_borrow_interests(rows, market)
        return self.filter_by_currency_since_limit(interest, code, since, limit)

    def parse_borrow_interest(self, info, market):
        #
        #     {
        #         "amount": 0.0006103,
        #         "side": "go_long",
        #         "entry_price": 31428.72,
        #         "liquidation_rate": 0.3,
        #         "liquidation_price": 10225.335481159,
        #         "unrealized_roe": -0.0076885829266987,
        #         "symbol": "BTC_USDT",
        #         "unrealized_pnl": -0.049158102631999,
        #         "leverage_ratio": 3
        #     }
        #
        symbol = self.safe_string(info, 'symbol')
        amountString = self.safe_string(info, 'amount')
        leverageString = self.safe_string(info, 'leverage_ratio')
        amountInvested = Precise.string_div(amountString, leverageString)
        amountBorrowed = Precise.string_sub(amountString, amountInvested)
        currency = None if (market is None) else market['base']
        return {
            'account': self.safe_symbol(symbol, market),
            'currency': currency,
            'interest': None,
            'interestRate': 0.001,  # all interest rates on digifinex are 0.1%
            'amountBorrowed': self.parse_number(amountBorrowed),
            'timestamp': None,
            'datetime': None,
            'info': info,
        }

    def fetch_borrow_rate(self, code, params={}):
        self.load_markets()
        request = {}
        response = self.privateSpotGetMarginAssets(self.extend(request, params))
        #
        #     {
        #         "list": [
        #             {
        #                 "valuation_rate": 1,
        #                 "total": 1.92012186174,
        #                 "free": 1.92012186174,
        #                 "currency": "USDT"
        #             },
        #         ],
        #         "total": 45.133305540922,
        #         "code": 0,
        #         "unrealized_pnl": 0,
        #         "free": 45.133305540922,
        #         "equity": 45.133305540922
        #     }
        #
        data = self.safe_value(response, 'list', [])
        result = []
        for i in range(0, len(data)):
            entry = data[i]
            if self.safe_string(entry, 'currency') == code:
                result = entry
        currency = self.safe_string(result, 'currency')
        return self.parse_borrow_rate(result, currency)

    def fetch_borrow_rates(self, params={}):
        """
        fetch the borrow interest rates of all currencies
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: a list of `borrow rate structures <https://docs.ccxt.com/en/latest/manual.html#borrow-rate-structure>`
        """
        self.load_markets()
        response = self.privateSpotGetMarginAssets(params)
        #
        #     {
        #         "list": [
        #             {
        #                 "valuation_rate": 1,
        #                 "total": 1.92012186174,
        #                 "free": 1.92012186174,
        #                 "currency": "USDT"
        #             },
        #         ],
        #         "total": 45.133305540922,
        #         "code": 0,
        #         "unrealized_pnl": 0,
        #         "free": 45.133305540922,
        #         "equity": 45.133305540922
        #     }
        #
        result = self.safe_value(response, 'list')
        return self.parse_borrow_rates(result, 'currency')

    def parse_borrow_rate(self, info, currency=None):
        #
        #     {
        #         "valuation_rate": 1,
        #         "total": 1.92012186174,
        #         "free": 1.92012186174,
        #         "currency": "USDT"
        #     }
        #
        timestamp = self.milliseconds()
        currencyId = self.safe_string(info, 'currency')
        return {
            'currency': self.safe_currency_code(currencyId, currency),
            'rate': 0.001,  # all interest rates on digifinex are 0.1%
            'period': 86400000,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'info': info,
        }

    def parse_borrow_rates(self, info, codeKey):
        #
        #     {
        #         "valuation_rate": 1,
        #         "total": 1.92012186174,
        #         "free": 1.92012186174,
        #         "currency": "USDT"
        #     },
        #
        result = {}
        for i in range(0, len(info)):
            item = info[i]
            currency = self.safe_string(item, codeKey)
            code = self.safe_currency_code(currency)
            borrowRate = self.parse_borrow_rate(item, currency)
            result[code] = borrowRate
        return result

    def fetch_funding_rate(self, symbol, params={}):
        """
        fetch the current funding rate
        see https://docs.digifinex.com/en-ww/swap/v2/rest.html#currentfundingrate
        :param str symbol: unified market symbol
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: a `funding rate structure <https://docs.ccxt.com/en/latest/manual.html#funding-rate-structure>`
        """
        self.load_markets()
        market = self.market(symbol)
        if not market['swap']:
            raise BadSymbol(self.id + ' fetchFundingRate() supports swap contracts only')
        request = {
            'instrument_id': market['id'],
        }
        response = self.publicSwapGetPublicFundingRate(self.extend(request, params))
        #
        #     {
        #         "code": 0,
        #         "data": {
        #             "instrument_id": "BTCUSDTPERP",
        #             "funding_rate": "-0.00012",
        #             "funding_time": 1662710400000,
        #             "next_funding_rate": "0.0001049907085171607",
        #             "next_funding_time": 1662739200000
        #         }
        #     }
        #
        data = self.safe_value(response, 'data', {})
        return self.parse_funding_rate(data, market)

    def parse_funding_rate(self, contract, market=None):
        #
        #     {
        #         "instrument_id": "BTCUSDTPERP",
        #         "funding_rate": "-0.00012",
        #         "funding_time": 1662710400000,
        #         "next_funding_rate": "0.0001049907085171607",
        #         "next_funding_time": 1662739200000
        #     }
        #
        marketId = self.safe_string(contract, 'instrument_id')
        timestamp = self.safe_integer(contract, 'funding_time')
        nextTimestamp = self.safe_integer(contract, 'next_funding_time')
        return {
            'info': contract,
            'symbol': self.safe_symbol(marketId, market),
            'markPrice': None,
            'indexPrice': None,
            'interestRate': None,
            'estimatedSettlePrice': None,
            'timestamp': None,
            'datetime': None,
            'fundingRate': self.safe_string(contract, 'funding_rate'),
            'fundingTimestamp': timestamp,
            'fundingDatetime': self.iso8601(timestamp),
            'nextFundingRate': self.safe_string(contract, 'next_funding_rate'),
            'nextFundingTimestamp': nextTimestamp,
            'nextFundingDatetime': self.iso8601(nextTimestamp),
            'previousFundingRate': None,
            'previousFundingTimestamp': None,
            'previousFundingDatetime': None,
        }

    def fetch_funding_rate_history(self, symbol=None, since=None, limit=None, params={}):
        """
        fetches historical funding rate prices
        :param str|None symbol: unified symbol of the market to fetch the funding rate history for
        :param int|None since: timestamp in ms of the earliest funding rate to fetch
        :param int|None limit: the maximum amount of `funding rate structures <https://docs.ccxt.com/en/latest/manual.html?#funding-rate-history-structure>` to fetch
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns [dict]: a list of `funding rate structures <https://docs.ccxt.com/en/latest/manual.html?#funding-rate-history-structure>`
        """
        self.check_required_symbol('fetchFundingRateHistory', symbol)
        self.load_markets()
        market = self.market(symbol)
        if not market['swap']:
            raise BadSymbol(self.id + ' fetchFundingRateHistory() supports swap contracts only')
        request = {
            'instrument_id': market['id'],
        }
        if since is not None:
            request['start_timestamp'] = since
        if limit is not None:
            request['limit'] = limit
        response = self.publicSwapGetPublicFundingRateHistory(self.extend(request, params))
        #
        #     {
        #         "code": 0,
        #         "data": {
        #             "instrument_id": "BTCUSDTPERP",
        #             "funding_rates": [
        #                 {
        #                     "rate": "-0.00375",
        #                     "time": 1607673600000
        #                 },
        #                 ...
        #             ]
        #         }
        #     }
        #
        data = self.safe_value(response, 'data', {})
        result = self.safe_value(data, 'funding_rates', [])
        rates = []
        for i in range(0, len(result)):
            entry = result[i]
            marketId = self.safe_string(data, 'instrument_id')
            symbol = self.safe_symbol(marketId)
            timestamp = self.safe_integer(entry, 'time')
            rates.append({
                'info': entry,
                'symbol': symbol,
                'fundingRate': self.safe_string(entry, 'rate'),
                'timestamp': timestamp,
                'datetime': self.iso8601(timestamp),
            })
        sorted = self.sort_by(rates, 'timestamp')
        return self.filter_by_symbol_since_limit(sorted, symbol, since, limit)

    def fetch_trading_fee(self, symbol, params={}):
        """
        fetch the trading fees for a market
        see https://docs.digifinex.com/en-ww/swap/v2/rest.html#tradingfee
        :param str symbol: unified market symbol
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: a `fee structure <https://docs.ccxt.com/en/latest/manual.html#fee-structure>`
        """
        self.load_markets()
        market = self.market(symbol)
        if not market['swap']:
            raise BadRequest(self.id + ' fetchTradingFee() supports swap markets only')
        request = {
            'instrument_id': market['id'],
        }
        response = self.privateSwapGetAccountTradingFeeRate(self.extend(request, params))
        #
        #     {
        #         "code": 0,
        #         "data": {
        #             "instrument_id": "BTCUSDTPERP",
        #             "taker_fee_rate": "0.0005",
        #             "maker_fee_rate": "0.0003"
        #         }
        #     }
        #
        data = self.safe_value(response, 'data', {})
        return self.parse_trading_fee(data, market)

    def parse_trading_fee(self, fee, market=None):
        #
        #     {
        #         "instrument_id": "BTCUSDTPERP",
        #         "taker_fee_rate": "0.0005",
        #         "maker_fee_rate": "0.0003"
        #     }
        #
        marketId = self.safe_string(fee, 'instrument_id')
        symbol = self.safe_symbol(marketId, market)
        return {
            'info': fee,
            'symbol': symbol,
            'maker': self.safe_number(fee, 'maker_fee_rate'),
            'taker': self.safe_number(fee, 'taker_fee_rate'),
        }

    def handle_margin_mode_and_params(self, methodName, params={}):
        """
         * @ignore
        marginMode specified by params["marginMode"], self.options["marginMode"], self.options["defaultMarginMode"], params["margin"] = True or self.options["defaultType"] = 'margin'
        :param dict params: extra parameters specific to the exchange api endpoint
        :returns [str|None, dict]: the marginMode in lowercase
        """
        defaultType = self.safe_string(self.options, 'defaultType')
        isMargin = self.safe_value(params, 'margin', False)
        marginMode = None
        marginMode, params = super(digifinex, self).handle_margin_mode_and_params(methodName, params)
        if marginMode is not None:
            if marginMode != 'cross':
                raise NotSupported(self.id + ' only cross margin is supported')
        else:
            if (defaultType == 'margin') or (isMargin is True):
                marginMode = 'cross'
        return [marginMode, params]

    def sign(self, path, api=[], method='GET', params={}, headers=None, body=None):
        signed = api[0] == 'private'
        endpoint = api[1]
        pathPart = '/v3' if (endpoint == 'spot') else '/swap/v2'
        request = '/' + self.implode_params(path, params)
        payload = pathPart + request
        url = self.urls['api']['rest'] + payload
        query = self.omit(params, self.extract_params(path))
        urlencoded = self.urlencode(self.keysort(query))
        if signed:
            auth = None
            nonce = None
            if pathPart == '/swap/v2':
                nonce = str(self.milliseconds())
                auth = nonce + method + payload
                if method == 'GET':
                    if urlencoded:
                        auth += '?' + urlencoded
                elif method == 'POST':
                    swapPostParams = json.dumps(params)
                    urlencoded = swapPostParams
                    auth += swapPostParams
            else:
                nonce = str(self.nonce())
                auth = urlencoded
            signature = self.hmac(self.encode(auth), self.encode(self.secret))
            if method == 'GET':
                if urlencoded:
                    url += '?' + urlencoded
            elif method == 'POST':
                headers = {
                    'Content-Type': 'application/x-www-form-urlencoded',
                }
                if urlencoded:
                    body = urlencoded
            headers = {
                'ACCESS-KEY': self.apiKey,
                'ACCESS-SIGN': signature,
                'ACCESS-TIMESTAMP': nonce,
            }
        else:
            if urlencoded:
                url += '?' + urlencoded
        return {'url': url, 'method': method, 'body': body, 'headers': headers}

    def handle_errors(self, statusCode, statusText, url, method, responseHeaders, responseBody, response, requestHeaders, requestBody):
        if not response:
            return  # fall back to default error handler
        code = self.safe_string(response, 'code')
        if (code == '0') or (code == '200'):
            return  # no error
        feedback = self.id + ' ' + responseBody
        if code is None:
            raise BadResponse(feedback)
        unknownError = [ExchangeError, feedback]
        ExceptionClass, message = self.safe_value(self.exceptions['exact'], code, unknownError)
        raise ExceptionClass(message)
