from typing import Callable, Type, Any

from spark_auto_mapper.type_definitions.defined_types import AutoMapperTextInputType

from spark_auto_mapper_fhir.classproperty import genericclassproperty
from spark_auto_mapper_fhir.fhir_types.uri import FhirUri
from spark_auto_mapper_fhir.valuesets.FhirValueSetBase import FhirValueSetBase


# noinspection PyMethodParameters
# noinspection PyPep8Naming
class ObservationCategoryCode(FhirValueSetBase):
    """
    https://www.hl7.org/fhir/valueset-observation-category.html
    """

    def __init__(self, value: AutoMapperTextInputType):
        super().__init__(value=value)

    # noinspection PyPep8Naming,SpellCheckingInspection
    class classproperty(object):
        def __init__(self, f: Callable[..., "ObservationCategoryCode"]) -> None:
            self.f: Callable[..., "ObservationCategoryCode"] = f

        def __get__(
            self, obj: Any, owner: Type["ObservationCategoryCode"]
        ) -> "ObservationCategoryCode":
            return self.f(owner)

    @classproperty
    def SocialHistory(cls) -> "ObservationCategoryCode":
        """
        Social History Observations define the patient's occupational, personal (e.g., lifestyle), social, familial,
         and environmental history and health risk factors that may impact the patient's health.
        """
        # noinspection PyCallingNonCallable
        return ObservationCategoryCode("social-history")

    @classproperty
    def VitalSigns(cls) -> "ObservationCategoryCode":
        """
        Clinical observations measure the body's basic functions such as blood pressure, heart rate, respiratory rate,
         height, weight, body mass index, head circumference, pulse oximetry, temperature, and body surface area.
        """
        # noinspection PyCallingNonCallable
        return ObservationCategoryCode("vital-signs")

    @classproperty
    def Imaging(cls) -> "ObservationCategoryCode":
        """
        Observations generated by imaging. The scope includes observations regarding plain x-ray, ultrasound, CT,
        MRI, angiography, echocardiography, and nuclear medicine.
        """
        # noinspection PyCallingNonCallable
        return ObservationCategoryCode("imaging")

    @classproperty
    def Laboratory(cls) -> "ObservationCategoryCode":
        """
        The results of observations generated by laboratories. Laboratory results are typically generated by
        laboratories providing analytic services in areas such as chemistry, hematology, serology, histology,
        cytology, anatomic pathology (including digital pathology), microbiology, and/or virology.
        These observations are based on analysis of specimens obtained from the patient and submitted to the laboratory.
        """
        # noinspection PyCallingNonCallable
        return ObservationCategoryCode("laboratory")

    @classproperty
    def Procedure(cls) -> "ObservationCategoryCode":
        """
        Observations generated by other procedures. This category includes observations resulting from interventional
         and non-interventional procedures excluding laboratory and imaging (e.g., cardiology catheterization,
         endoscopy, electrodiagnostics, etc.). Procedure results are typically generated by a clinician to
         provide more granular information about component observations made during a procedure.
         An example would be when a gastroenterologist reports the size of a polyp observed during a colonoscopy.
        """
        # noinspection PyCallingNonCallable
        return ObservationCategoryCode("procedure")

    @classproperty
    def Survey(cls) -> "ObservationCategoryCode":
        """
        Assessment tool/survey instrument observations (e.g., Apgar Scores, Montreal Cognitive Assessment (MoCA)).
        """
        # noinspection PyCallingNonCallable
        return ObservationCategoryCode("survey")

    @classproperty
    def Exam(cls) -> "ObservationCategoryCode":
        """
        Observations generated by physical exam findings including direct observations made by a clinician and
        use of simple instruments and the result of simple maneuvers performed directly on the patient's body.
        """
        # noinspection PyCallingNonCallable
        return ObservationCategoryCode("exam")

    @classproperty
    def Therapy(cls) -> "ObservationCategoryCode":
        """
        Observations generated by non-interventional treatment protocols (e.g. occupational, physical,
        radiation, nutritional and medication therapy)
        """
        # noinspection PyCallingNonCallable
        return ObservationCategoryCode("therapy")

    @classproperty
    def Activity(cls) -> "ObservationCategoryCode":
        """
        Observations that measure or record any bodily activity that enhances or maintains physical fitness
        and overall health and wellness. Not under direct supervision of practitioner such as a physical therapist.
        (e.g., laps swum, steps, sleep data)
        """
        # noinspection PyCallingNonCallable
        return ObservationCategoryCode("activity")

    @genericclassproperty
    def codeset(cls) -> FhirUri:
        """
        http://terminology.hl7.org/CodeSystem/observation-category
        """
        return "http://terminology.hl7.org/CodeSystem/observation-category"

    @genericclassproperty
    def oid(cls) -> FhirUri:
        """
        2.16.840.1.113883.4.642.3.403
        """
        return "2.16.840.1.113883.4.642.3.403"
