"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CertManager = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
class CertManager extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const cluster_issuer_namespace_manifest = {
            apiVersion: 'v1',
            kind: 'Namespace',
            metadata: { name: 'aws-pca-issuer' },
        };
        const cert_issuer_namespace = props.cluster.addManifest('ClusterIssuerNamespaceManifest', cluster_issuer_namespace_manifest);
        const service_account = props.cluster.addServiceAccount('PrivateCASA');
        service_account.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            resources: [props.privateCaArn],
            actions: [
                'acm-pca:Get*',
                'acm-pca:Issue*',
                'acm-pca:Describe*',
            ],
        }));
        // TODO : Dont know why, but pod SA still wants node to lookup Describe* calls on PCA
        props.nodegroup.role.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            resources: [props.privateCaArn],
            actions: [
                'acm-pca:Get*',
                'acm-pca:Issue*',
                'acm-pca:Describe*',
            ],
        }));
        service_account.node.addDependency(cert_issuer_namespace);
        const deploy_cert_manager = props.cluster.addHelmChart('CertManager', {
            repository: 'https://charts.jetstack.io',
            chart: 'cert-manager',
            namespace: 'cert-manager',
            release: 'cert-manager',
            wait: true,
            version: 'v1.5.0',
            values: {
                installCRDs: true,
                webhook: {
                    timeoutSeconds: 30,
                },
            },
        });
        const priv_ca_issuer_helm = props.cluster.addHelmChart('PrivateCaIssuer', {
            repository: 'https://cert-manager.github.io/aws-privateca-issuer',
            chart: 'aws-pca-issuer',
            namespace: 'aws-pca-issuer',
            release: 'aws-pca-issuer',
            wait: true,
            values: {
                serviceAccount: {
                    create: 'false',
                    name: service_account.serviceAccountName,
                },
            },
        });
        priv_ca_issuer_helm.node.addDependency(deploy_cert_manager);
        const deploy_waiter = props.cluster.addHelmChart('DummyWaiterPostCertManager', {
            repository: 'https://anshrma.github.io/helm-charts',
            chart: 'helm-waiter',
            // namespace: 'default',
            release: 'helm-waiter',
            wait: true,
        });
        deploy_waiter.node.addDependency(priv_ca_issuer_helm);
        const cluster_issuer_manifest = {
            apiVersion: 'awspca.cert-manager.io/v1beta1',
            kind: 'AWSPCAClusterIssuer',
            metadata: { name: props.clusterIssuerName },
            spec: {
                arn: props.privateCaArn,
                // arn: cfnCertificateAuthority.attrArn,
                region: aws_cdk_lib_1.Stack.of(this).region,
            },
        };
        const cluster_issuer = props.cluster.addManifest('ClusterIssuerManifest', cluster_issuer_manifest);
        cluster_issuer.node.addDependency(deploy_waiter);
        const namespace_manifest = {
            apiVersion: 'v1',
            kind: 'Namespace',
            metadata: { name: props.namespace },
        };
        const ns = props.cluster.addManifest('NSManifests', namespace_manifest);
        const cert_manifest = {
            apiVersion: 'cert-manager.io/v1',
            kind: 'Certificate',
            metadata: {
                name: props.cacertname,
                namespace: props.namespace,
            },
            spec: {
                commonName: props.hostedZoneName,
                dnsNames: props.dnsNames,
                duration: '2160h0m0s',
                issuerRef: {
                    group: 'awspca.cert-manager.io',
                    kind: 'AWSPCAClusterIssuer',
                    name: props.clusterIssuerName,
                },
                renewBefore: '360h0m0s',
                secretName: `${props.clusterIssuerName}-secret`,
                usages: [
                    'server auth',
                    'client auth',
                ],
                privateKey: {
                    algorithm: 'RSA',
                    size: 2048,
                },
            },
        };
        const cert = props.cluster.addManifest('CertManifests', cert_manifest);
        cert.node.addDependency(ns, cluster_issuer);
    }
}
exports.CertManager = CertManager;
//# sourceMappingURL=data:application/json;base64,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