# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'vendor/wrapt/src'}

packages = \
['appmap',
 'appmap._implementation',
 'appmap.test',
 'appmap.test.data',
 'appmap.test.data.flask',
 'appmap.test.data.package1',
 'appmap.test.data.package1.package2',
 'appmap.test.data.pytest',
 'appmap.test.data.trial.init',
 'appmap.test.data.trial.test',
 'appmap.test.data.unittest.init',
 'appmap.test.data.unittest.simple',
 'wrapt']

package_data = \
{'': ['*'],
 'appmap.test.data': ['trial/*', 'unittest/*'],
 'appmap.test.data.pytest': ['tmp/appmap/pytest/*']}

install_requires = \
['PyYAML>=5.3.0,<5.4.0', 'importlib-metadata>=0.8', 'inflection>=0.3.0,<0.4.0']

entry_points = \
{'pytest11': ['appmap = appmap.pytest']}

setup_kwargs = {
    'name': 'appmap',
    'version': '0.10.0',
    'description': 'Create AppMap files by recording a Python application.',
    'long_description': '- [About](#about)\n  - [Supported versions](#supported-versions)\n- [Installation](#installation)\n- [Configuration](#configuration)\n  - [Environment Variables](#environment-variables)\n- [Labels](#labels)\n- [Test Frameworks](#test-frameworks)\n  - [pytest](#pytest)\n  - [unittest](#unittest)\n  - [Run your tests](#run-your-tests)\n- [Remote Recording [coming soon]](#remote-recording-coming-soon)\n  - [Django](#django)\n  - [Flask](#flask)\n  - [Run your web app](#run-your-web-app)\n- [Context manager](#context-manager)\n- [Development](#development)\n  - [Python version support](#python-version-support)\n  - [Dependency management](#dependency-management)\n  - [Linting](#linting)\n  - [Testing](#testing)\n    - [pytest](#pytest-1)\n    - [tox](#tox)\n  - [Code Coverage](#code-coverage)\n# About\n`appmap-python` is a Python package for recording\n[AppMaps](https://github.com/applandinc/appmap) of your code. "AppMap" is a data format\nwhich records code structure (modules, classes, and methods), code execution events\n(function calls and returns), and code metadata (repo name, repo URL, commit SHA, labels,\netc). It\'s more granular than a performance profile, but it\'s less granular than a full\ndebug trace. It\'s designed to be optimal for understanding the design intent and structure\nof code and key data flows.\n\nThere are several ways to record AppMaps of your Python program using the `appmap` package:\n\n* Run your tests (pytest, unittest) with the environment variable\n  `APPMAP=true`. An AppMap will be generated for each test case.\n\n* Use the `appmap.record` [context manager](#context-manager) to control recording. The context manager takes\n  an instance of an `appmap.Recording`, which can be used to generate the AppMap.\n\n* [coming soon] Run your application server with AppMap remote recording enabled, and use\n  the [AppLand browser extension](https://github.com/applandinc/appland-browser-extension)\n  to start, stop, and upload recordings.\n\nOnce you have made a recording, there are two ways to view automatically generated\ndiagrams of the AppMaps.\n\nThe first option is to load the diagrams directly in your IDE, using the [AppMap extension\nfor VSCode](https://marketplace.visualstudio.com/items?itemName=appland.appmap).\n\nThe second option is to upload them to the [AppLand server](https://app.land) using the\n[AppLand CLI](https://github.com/applandinc/appland-cli/releases).\n\n## Supported versions\n\n* Python >=3.6\n* Pytest >=6.1.2\n\nSupport for new versions is added frequently, please check back regularly for updates.\n\n# Installation\n\nIf your project uses `pip` for dependency management, add the `appmap` package to the requirements\nfile or install it directly with\n```shell\npip install appmap\n```\n\nFor projects that use `poetry` , add the `appmap` package to `pyproject.toml`.\n\n# Configuration\nAdd your modules as `path` entries in `appmap.yml`, and external packages\n(distributions) as `dist`:\n\n```yaml\nname: my_python_app\npackages:\n- path: app.mod1\n  shallow: true\n- path: app.mod2\n  exclude:\n  - MyClass\n  - MyOtherClass.my_instance_method\n  - MyOtherClass.my_class_method\n- dist: Django\n  exclude:\n  - django.db\n```\n\nNote that `exclude`s are resolved relative to the associated path. So, for example, this\nconfiguration excludes `app.mod2.MyClass`.\n\nFor external [distribution packages](https://packaging.python.org/glossary/#term-Distribution-Package)\nuse the `dist` specifier; the names are looked up in the\n[database of installed Python distributions](https://www.python.org/dev/peps/pep-0376/).\nThis is generally the same package name as you\'d give to `pip install` or put\nin `pyproject.toml`. You can additionally use `path` and `exclude` on `dist`\nentries to limit the capture to specific patterns.\n\nNote by default shallow capture is enabled on `dist` packages, supressing tracking\nof most internal execution flow, which allows you to capture the interaction without\ngetting bogged down with detail. If this isn\'t what you want, use `shallow: false`.\nYou can also use `shallow: true` on `path` entries.\n\n## Environment Variables\n\n* `APPMAP` if `true`, code will be instrumented and AppMaps will be generated. Not\n  case-sensitive, defaults to \'false\'.\n\n* `APPMAP_CONFIG` specifies the configuration file to use. Defaults to `appmap.yml` in the\n  current directory\n\n* `APPMAP_LOG_LEVEL` specifies log level to use, from the set `CRITICAL`, `ERROR`,\n  `WARNING`, `INFO`, `DEBUG`. Not case-sensitive, defaults to `WARNING`.\n\n* `APPMAP_OUTPUT_DIR` specifies the root directory for writing AppMaps. Defaults to\n  `tmp/appmap`.\n\n* `APPMAP_DISPLAY_PARAMS` enables rendering of parameters as strings. If `true` (the\n  default, not case-sensitive), parameters are rendered using `str` and/or `repr`. If\n  `false`, a generic string is used instead.\n\n# Labels\n\nThe [AppMap data format](https://github.com/applandinc/appmap) provides for class and\nfunction `labels`, which can be used to enhance the AppMap visualizations, and to\nprogramatically analyze the data.\n\nYou can apply function labels using the `appmap.labels` decorator in your Python code. To\napply a labels to a function, decorate the function with `@appmap.labels`.\n\nFor example\n\n```python\nimport appmap.labels\n\nclass ApiKey\n  @appmap.labels(\'provider.authentication\', \'security\')\n  def authenticate(self, key):\n      # logic to verify the key here...\n```\n\nThen the AppMap metadata section for this function will include:\n\n```json\n  {\n    "name": "authenticate",\n    "type": "function",\n    "labels": [ "provider.authentication", "security" ]\n  }\n```\n\n# Test Frameworks\n`appmap-python` supports recording [pytest](https://pytest.org) and\n[unittest](https://docs.python.org/3/library/unittest.html) test cases.\n\n## pytest\n`appmap-python` is a `pytest` plugin. When it\'s installed in a project that uses\n`pytest`, it will be available to generate AppMaps.\n\n```shell\nroot@e9987eaa93c8:/src/appmap/test/data/pytest# pip show appmap\nName: appmap\nVersion: 0.0.0\nSummary: Create AppMap files by recording a Python application.\nHome-page: None\nAuthor: Alan Potter\nAuthor-email: alan@app.land\nLicense: None\nLocation: /usr/local/lib/python3.9/site-packages\nRequires: orjson, PyYAML, inflection\nRequired-by:\nroot@e9987eaa93c8:/src/appmap/test/data/pytest# APPMAP=true APPMAP_LOG_LEVEL=info pytest -svv\n[2021-02-10 11:37:59,345] INFO root: appmap enabled: True\n[2021-02-10 11:37:59,350] INFO appmap._implementation.configuration: ConfigFilter, includes {\'simple\'}\n[2021-02-10 11:37:59,350] INFO appmap._implementation.configuration: ConfigFilter, excludes set()\n===================================================================== test session starts =====================================================================\nplatform linux -- Python 3.9.1, pytest-6.2.2, py-1.10.0, pluggy-0.13.1 -- /usr/local/bin/python\ncachedir: .pytest_cache\nrootdir: /src, configfile: pytest.ini\nplugins: appmap-0.0.0\ncollected 1 item\n\ntest_simple.py::test_hello_world [2021-02-10 11:37:59,482] INFO appmap.pytest: starting recording /tmp/pytest/test_hello_world.appmap.json\n[2021-02-10 11:37:59,484] INFO appmap._implementation.configuration: included class simple.Simple\n[2021-02-10 11:37:59,484] INFO appmap._implementation.configuration: included function simple.Simple.hello\n[2021-02-10 11:37:59,489] INFO appmap._implementation.configuration: included function simple.Simple.hello_world\n[2021-02-10 11:37:59,490] INFO appmap._implementation.configuration: included function simple.Simple.world\n[2021-02-10 11:37:59,828] INFO appmap.pytest: wrote recording /tmp/pytest/test_hello_world.appmap.json\nPASSED\n\n====================================================================== 1 passed in 0.45s ======================================================================\n```\n\n## unittest\n`import appmap.unittest`. Instruments subclasses of `unittest.TestCase` and records each\n`test_*` function in the subclasses. You can also use `python -m appmap.unittest` exactly like\n`python -m unittest` and leave your code unmodified (just remember to set the `APPMAP=true`\nenvironment variable).\n\n## Run your tests\nOnce you\'ve configured your tests to generate AppMaps, run the tests with the\n`APPMAP=true` in the environment. For example, to run a pytest test suite:\n\n```sh\n$ APPMAP=true pytest\n```\n\n\n# Remote Recording\n`appmap-python` supports remote recording of Django and Flask web applications. Import the\nappropriate remote recording support into your web app.\n\n## Django [coming soon]\n`import appmap.django`. Adds `/_appmap/record` routes to a Django app.\n\n## Flask\nFor projects that use a [Flask application\nfactory](https://flask.palletsprojects.com/en/1.1.x/patterns/appfactories/), installing\n`appmap-python` automatically configures the project for remote recording. No further\nmodifications are required. When the application initializes, `appmap-python` adds\nmiddleware that handles the `/_appmap/record` routes.\n\nFor projects that don\'t provide an application factory, `appmap-python` can be used as a\n[Flask extension](https://flask.palletsprojects.com/en/1.1.x/extensions/#extensions). \n\nFor example:\n```python\nfrom flask import Flask\n\nfrom appmap.flask import AppmapFlask\n\napp = Flask(__name__)\n\nappmap_flask = AppmapFlask(app)\n```\n\nThis will add the `/_appmap/record` routes your app.\n\n## Run your web app\nOnce you\'ve configured your web app to add the remote-recording routes, you can use the\nroutes to manage recordings. The [AppLand browser\nextension](https://github.com/applandinc/appland-browser-extension),\n[CLI](https://github.com/applandinc/appland-cli/), or just plain [cURL](https://curl.se/)\nwill all work for this.\n\nAs when running tests, start the web server with `APPMAP=true` in the environment. For\nexample, to start a Flask app:\n\n```sh\n$ APPMAP=true flask run\n```\n\nAn app with remote recording enabled supports these routes:\n\n* `POST /_appmap/record`\n  Starts a new recording\n\n  200 if the recording was started successfully\n  409 if there\'s already a recording in progress\n\n* `GET /_appmap/record`\n  Returns JSON describing current recording state\n  200 with body\n\n  ```json\n  {\n    "enabled": true\n  }\n  ```\n  `enabled` indicates whether recording has been enabled\n\n* `DELETE /_appmap/record`\n  Returns AppMap as JSON\n  200 with AppMap as body\n  404 if there\'s no recording in progress\n\n# Context manager\nYou can use `appmap.record` as a context manager to record your code.\n\nWith a file called `record_sample.py` like this\n\n```python\nimport os\nimport sys\n\nimport appmap\n\nr = appmap.Recording()\nwith r:\n    import sample\n    print(sample.C().hello_world(), file=sys.stderr)\n\nwith os.fdopen(sys.stdout.fileno(), "wb", closefd=False) as stdout:\n    stdout.write(appmap.generation.dump(r))\n    stdout.flush()\n```\n\nand a source file called `sample.py` like this\n\n```python\nclass C:\n    def make_str(self, s):\n        return s;\n\n    def hello_world(self):\n        return f\'{self.make_str("Hello")} {self.make_str("world!")}\'\n```\n\nas well as an `appmap.yml`\n\n```yaml\nname: sample\npackages:\n- path: sample\n```\n\nyou can generate a recording of the code\n\n```sh\n% APPMAP=true python record_sample.py > record_sample.appmap.json\n% jq \'.events | length\' record_sample.appmap.json\n6\n% jq < record_sample.appmap.json | head -10\n{\n  "version": "1.4",\n  "metadata": {\n    "language": {\n      "name": "python",\n      "engine": "CPython",\n      "version": "3.9.1"\n    },\n    "client": {\n      "name": "appmap",\n```\n\n# Development\n\n[![Build Status](https://travis-ci.com/applandinc/appmap-python.svg?branch=master)](https://travis-ci.com/applandinc/appmap-python)\n\n## Getting the code\nClone the repo to begin development. Note that vendored dependencies are included as\nsubmodules.\n\n```shell\n% g clone --recurse-submodules https://github.com/applandinc/appmap-python.git\nCloning into \'appmap-python\'...\nremote: Enumerating objects: 167, done.\nremote: Counting objects: 100% (167/167), done.\nremote: Compressing objects: 100% (100/100), done.\nremote: Total 962 (delta 95), reused 116 (delta 61), pack-reused 795\nReceiving objects: 100% (962/962), 217.31 KiB | 4.62 MiB/s, done.\nResolving deltas: 100% (653/653), done.\nSubmodule \'extern/wrapt\' (https://github.com/applandinc/wrapt.git) registered for path \'vendor/wrapt\'\nCloning into \'/private/tmp/appmap-python/vendor/wrapt\'...\nremote: Enumerating objects: 46, done.\nremote: Counting objects: 100% (46/46), done.\nremote: Compressing objects: 100% (39/39), done.\nremote: Total 2537 (delta 9), reused 19 (delta 4), pack-reused 2491\nReceiving objects: 100% (2537/2537), 755.94 KiB | 7.48 MiB/s, done.\nResolving deltas: 100% (1643/1643), done.\nSubmodule path \'vendor/wrapt\': checked out \'9bdfbe54b88a64069cba1f3c36e77edc3c1339c9\'\n\n% ls appmap-python/vendor/wrapt\nLICENSE\t\tMakefile\tappveyor.yml\tdocs\t\tsrc\t\ttests\nMANIFEST.in\tREADME.rst\tblog\t\tsetup.py\ttddium.yml\ttox.ini\n```\n\n## Python version support\nAs a package intended to be installed in as many environments as possible, `appmap-python`\nneeds to avoid using features of Python or the standard library that were added after the\noldest version currently supported (see [above](#supported-version)).\n\nThe Travis build uses [vermin](https://github.com/netromdk/vermin) to help weed out the\nuse of some invalid features. Additionally, tests are run using all supported versions of\nPython.\n\n\n## Dependency management\n\n[poetry](https://https://python-poetry.org/) for dependency management:\n\n```\n% brew install poetry\n% cd appmap-python\n% poetry install\n```\n\n## Linting\n[pylint](https://www.pylint.org/) for linting:\n\n```\n% cd appmap-python\n% poetry run pylint appmap\n\n--------------------------------------------------------------------\nYour code has been rated at 10.00/10 (previous run: 10.00/10, +0.00)\n\n```\n\n[Note that the current configuration requires a 10.0 for the Travis build to pass. To make\nthis easier to achieve, convention and refactoring checks have both been disabled. They\nshould be reenabled as soon as possible.]\n\n\n## Testing\n### pytest\n[pytest](https://docs.pytest.org/en/stable/) for testing:\n\n```\n% cd appmap-python\n% poetry run pytest\n```\n\n### tox\nAdditionally, the `tox` configuration provides the ability to run the tests for all\nsupported versions of Python and django:\n\n```sh\n% cd appmap-python\n% poetry run tox\n```\n\nNote that `tox` requires the correct version of Python to be installed before it can\ncreate a test environment. [pyenv](https://github.com/pyenv/pyenv) is an easy way to\nmanage multiple versions of Python.\n\n## Code Coverage\n[coverage](https://coverage.readthedocs.io/) for coverage:\n\n```\n% cd appmap-python\n% poetry run coverage run -m pytest\n% poetry run coverage html\n% open htmlcov/index.html\n```\n',
    'author': 'Alan Potter',
    'author_email': 'alan@app.land',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/applandinc/appmap-python',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
