// Copyright (c) Mito
import React, { useCallback, useState } from 'react';
import '../../../css/elements/MultiToggleBox.css';
import { classNames } from '../../utils/classNames';
import { fuzzyMatch } from '../../utils/strings';
import Row from '../spacing/Row';
import { ensureInView } from './Dropdown';
import Input from './Input';
import LoadingDots from './LoadingDots';
/*
    So that we don't crash the users browser, we show at most
    10k items to them at once
*/
const MAX_DISPLAYED = 10000;
const MultiToggleBoxMessage = (props) => {
    if (props.loading) {
        return (React.createElement(Row, { justify: 'center' },
            React.createElement("p", { className: 'text-body-1 text-align-center' },
                "Loading items",
                React.createElement(LoadingDots, null))));
    }
    else if (props.maxDisplayed || props.isSubset) {
        return (React.createElement(Row, { justify: 'center' },
            React.createElement("p", { className: 'text-body-1 text-align-center' }, "There are too many items to display. Search to filter down to the items you care about.")));
    }
    else if (props.numDisplayed === 0) {
        return (React.createElement(Row, { justify: 'center' },
            React.createElement("p", { className: 'text-body-1' }, "No items to display.")));
    }
    else if (props.message !== undefined) {
        return (React.createElement(Row, { justify: 'center' },
            React.createElement("p", { className: 'text-body-1 text-align-center' }, props.message)));
    }
    return (React.createElement(React.Fragment, null));
};
const MultiToggleSelectedMessage = (props) => {
    let text = `${props.numToggled} selected`;
    if (props.numToggled > 0 && props.numToggled === props.numToggledButNotDisplayed) {
        text = `${props.numToggled} selected and not displayed`;
    }
    else if (props.numToggledButNotDisplayed > 0) {
        text = `${props.numToggled} selected, of which ${props.numToggledButNotDisplayed} not displayed`;
    }
    return (React.createElement(React.Fragment, null,
        "Toggle ",
        props.searchString !== '' ? "Displayed" : "All",
        React.createElement("span", { className: 'text-color-medium-gray-important' },
            "\u00A0(",
            text,
            ")")));
};
/*
  A box that contains a variety of options that can be toggled on and off indivigually.
  
  If optional toggleAllOptions are passed, then a Toggle All button is also displayed
  that toggles all the buttons at once.
*/
const MultiToggleBox = (props) => {
    // We can store state in this element, or in the parent element if we want
    const [_searchString, _setSearchString] = useState('');
    const searchString = props.searchState !== undefined ? props.searchState.searchString : _searchString;
    const setSearchString = props.searchState !== undefined ? props.searchState.setSearchString : _setSearchString;
    // This hook runs when the multi toggle box renders, and makes sure that the first selected
    // element is visible in the toggle box. This is necessary to make sure that users aren't confused
    // by where their selection is.
    const setRef = useCallback((unsavedDropdownAnchor) => {
        if (unsavedDropdownAnchor !== null) {
            const firstSelectedChild = unsavedDropdownAnchor.querySelector('.multi-toggle-box-row-selected');
            if (firstSelectedChild !== null) {
                ensureInView(unsavedDropdownAnchor, firstSelectedChild, 0);
            }
        }
    }, []);
    const height = props.height || 'block';
    const width = props.width || 'block';
    const heightClass = `element-height-${height}`;
    const widthClass = `element-width-${width}`;
    let displayedNonDisabledAllToggled = true;
    const nonDisabledDisplayedIndexes = [];
    let numToggled = 0;
    let numToggledButNotDisplayed = 0;
    let numDisplayed = 0;
    let maxDisplayed = false;
    // Only display the options that we're searching for, and also collect
    // information about how many children are passed and displayed
    const childrenToDisplay = React.Children.map((props.children), (child) => {
        const title = child.props.title;
        const rightText = child.props.rightText;
        const toggled = child.props.toggled;
        if (toggled) {
            numToggled++;
        }
        const noTitleMatch = title === null || title === undefined || fuzzyMatch(title + '', searchString) < .8;
        const noRightTextMatch = title === null || title === undefined || fuzzyMatch(rightText + '', searchString) < .8;
        // Don't display if it doesn't match either of the title or the right text
        if (noTitleMatch && noRightTextMatch) {
            if (toggled) {
                numToggledButNotDisplayed++;
            }
            return null;
        }
        // Don't display if we've displayed enough already
        if (numDisplayed > MAX_DISPLAYED) {
            maxDisplayed = true;
            return null;
        }
        numDisplayed++;
        // Make sure that if multi-select is disabled entirely, each element is disabled
        const itemDisabled = child.props.disabled || props.disabled;
        if (!itemDisabled) {
            nonDisabledDisplayedIndexes.push(child.props.index);
            displayedNonDisabledAllToggled = displayedNonDisabledAllToggled && child.props.toggled;
        }
        const copiedChild = React.cloneElement(child, {
            disabled: itemDisabled
        });
        return copiedChild;
    });
    const { toggleAllIndexes } = props;
    return (React.createElement("div", { className: classNames('mutli-toggle-box-container', heightClass, widthClass, props.className) },
        props.searchable &&
            React.createElement(Input, { value: searchString, onChange: (e) => {
                    setSearchString(e.target.value);
                }, placeholder: 'Search', width: 'block', className: 'mb-2px' }),
        React.createElement("div", { className: classNames('multi-toggle-box'), 
            // It's hard to get the box to fill the rest of the container,
            // so we do a calculation if the search box is displayed
            style: { height: props.searchable ? 'calc(100% - 30px)' : '100%' }, ref: setRef },
            React.createElement(MultiToggleBoxMessage, { loading: props.loading, isSubset: props.isSubset, message: props.message, maxDisplayed: maxDisplayed, numDisplayed: numDisplayed }),
            toggleAllIndexes !== undefined && numDisplayed > 0 &&
                React.createElement("div", { key: 'Toggle All', className: classNames('multi-toggle-box-row', { 'multi-toggle-box-row-selected': displayedNonDisabledAllToggled }), onClick: () => {
                        if (props.disabled) {
                            return;
                        }
                        toggleAllIndexes(nonDisabledDisplayedIndexes, !displayedNonDisabledAllToggled);
                    } },
                    React.createElement("input", { key: 'Toggle All', type: "checkbox", name: 'Toggle All', checked: displayedNonDisabledAllToggled }),
                    React.createElement(MultiToggleSelectedMessage, { searchString: searchString, numToggled: numToggled, numToggledButNotDisplayed: numToggledButNotDisplayed })),
            childrenToDisplay)));
};
export default MultiToggleBox;
//# sourceMappingURL=MultiToggleBox.js.map