"""
Copyright 2021 nanato12

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
"""


import os
from ast import (
    AnnAssign,
    Attribute,
    ClassDef,
    Constant,
    ImportFrom,
    Load,
    Module,
    Name,
    Store,
    Subscript,
    Tuple,
    alias,
)
from datetime import datetime

import astor

from line_bot_tool.base.generator import BaseGenerator
from line_bot_tool.utils import convert_camel_case, extract_alphabet


class CommandGenerator(BaseGenerator):
    GEN_COMMENT = "# Generated by CommandGenerator on {date}\n\n"

    name: str
    ast_body: list
    output_dir: str

    def __init__(self, name: str) -> None:
        self.name = extract_alphabet(name)
        self.output_dir = os.environ.get("COMMAND_OUTPUT_PATH", "commands")

        os.makedirs(self.output_dir, exist_ok=True)

        if not os.path.exists(self.settings_path):
            with open(self.settings_path, "w") as settings_file:
                settings_file.write(
                    self.GEN_COMMENT.format(
                        date=datetime.now().strftime("%Y-%m-%d %H:%M:%S")
                    )
                    + astor.to_source(self.__create_settings_module())
                )

        if not os.path.exists(self.init_path):
            with open(self.init_path, "w") as settings_file:
                settings_file.write("")

    @property
    def settings_path(self) -> str:
        return f"{self.output_dir}/settings.py"

    @property
    def init_path(self) -> str:
        return f"{self.output_dir}/__init__.py"

    def __create_settings_module(self) -> Module:
        """create settings ast module

        Returns:
            Module: settings ast module
        """

        return Module(
            body=[
                ImportFrom(
                    module="typing",
                    names=[alias(name="Optional"), alias(name="Tuple")],
                    level=0,
                ),
                AnnAssign(
                    target=Name(id="PREFIXES", ctx=Store()),
                    annotation=Subscript(
                        value=Name(id="Optional", ctx=Load()),
                        slice=Subscript(
                            value=Name(id="Tuple", ctx=Load()),
                            slice=Tuple(
                                elts=[
                                    Name(id="str", ctx=Load()),
                                    Constant(value=Ellipsis),
                                ],
                                ctx=Load(),
                            ),
                            ctx=Load(),
                        ),
                        ctx=Load(),
                    ),
                    value=Tuple(
                        elts=[Constant(value="/"), Constant(value="@")], ctx=Load()
                    ),
                    simple=1,
                ),
            ],
            type_ignores=[],
        )

    def __ast_import_from(self) -> None:
        """append ImportFrom to ast body"""

        self.ast_body.append(
            ImportFrom(
                module="typing",
                names=[alias(name="Optional"), alias(name="Tuple")],
                level=0,
            ),
        )
        self.ast_body.append(
            ImportFrom(
                module="line_bot_tool.base.commnad",
                names=[alias(name="BaseCommand")],
                level=0,
            )
        )
        self.ast_body.append(
            ImportFrom(names=[alias(name="settings")], level=1),
        )

    def __ast_class_def(self) -> None:
        """append ClassDef to ast body"""

        self.ast_body.append(
            ClassDef(
                name=f"{extract_alphabet(convert_camel_case(self.name))}Command",
                bases=[Name(id="BaseCommand", ctx=Load())],
                keywords=[],
                body=[
                    AnnAssign(
                        target=Name(id="prefixes", ctx=Store()),
                        annotation=Subscript(
                            value=Name(id="Optional", ctx=Load()),
                            slice=Subscript(
                                value=Name(id="Tuple", ctx=Load()),
                                slice=Tuple(
                                    elts=[
                                        Name(id="str", ctx=Load()),
                                        Constant(value=Ellipsis),
                                    ],
                                    ctx=Load(),
                                ),
                                ctx=Load(),
                            ),
                            ctx=Load(),
                        ),
                        value=Attribute(
                            value=Name(id="settings", ctx=Load()),
                            attr="PREFIXES",
                            ctx=Load(),
                        ),
                        simple=1,
                    ),
                    AnnAssign(
                        target=Name(id="cmd", ctx=Store()),
                        annotation=Name(id="str", ctx=Load()),
                        value=Constant(value=self.name),
                        simple=1,
                    ),
                    AnnAssign(
                        target=Name(id="description", ctx=Store()),
                        annotation=Name(id="str", ctx=Load()),
                        value=Constant(value="Please input here command description."),
                        simple=1,
                    ),
                ],
                decorator_list=[],
            ),
        )

    def __create_ast_module(self) -> Module:
        """create ast module

        Returns:
            Module: ast module
        """

        self.ast_body = []
        self.__ast_import_from()
        self.__ast_class_def()
        return Module(body=self.ast_body, type_ignores=[])

    def generate(self) -> None:
        """generate command py file"""

        file_name = f"{self.output_dir}/{self.name}_command.py"
        if os.path.exists(file_name):
            raise Exception(f"already exists command: {self.name}")

        with open(file_name, "w") as cmd_py_file:
            cmd_py_file.write(
                self.GEN_COMMENT.format(
                    date=datetime.now().strftime("%Y-%m-%d %H:%M:%S")
                )
                + astor.to_source(self.__create_ast_module())
            )
        print(f"Generate success! {file_name}")
