# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['qbee']

package_data = \
{'': ['*']}

install_requires = \
['matplotlib>=3.6.2,<4.0.0',
 'memory_profiler',
 'numpy>=1,<2',
 'ordered-set',
 'pandas>=1,<2',
 'pyarrow>=10,<11',
 'sympy>=1.5.0,<2.0.0',
 'tqdm>=4,<5']

setup_kwargs = {
    'name': 'qbee',
    'version': '0.7.1',
    'description': 'Quadratization of differential equations in python',
    'long_description': '[![Build Status](https://travis-ci.com/AndreyBychkov/QBee.svg?branch=dev)](https://app.travis-ci.com/github/AndreyBychkov/QBee)\n[![Documentation Status](https://readthedocs.org/projects/qbee/badge/?version=latest)](https://qbee.readthedocs.io/en/latest/?badge=latest)\n# QBee\n\n### [Online playground](https://huggingface.co/spaces/Armaliltril/qbee)\n\nQBee is a Python library for transforming systems of differential equations into a systems with quadratic right-rand side.\n\n# Installation\n\n## PyPI\n`pip install qbee`\n\n## Manual\n1. Clone repository: `https://github.com/AndreyBychkov/QBee.git`\n   * Or, if you want our bleeding edge version, clone `https://github.com/AndreyBychkov/QBee/tree/dev`\n2. Change directory: `cd QBee`\n3. Install package: `pip install .`\n\nIf you use `poetry` you can alternately install if with\n`poetry install`\n\n# What is quadratization?\n\nThe problem of *quadratization* is, given a system of ODEs with polynomial right-hand side, reduce the system to a\nsystem with quadratic right-hand side by introducing as few new variables as possible. We will explain it using toy\nexample. Consider the system\n\n<img alt="\\begin{cases} x_1&#39; = x_1 x_2 \\\\ x_2&#39; = -x_1 x_2^3 \\end{cases}" height="135" src="https://latex.codecogs.com/png.latex?\\dpi{200}&amp;space;\\huge&amp;space;{\\color{DarkOrange}&amp;space;\\begin{cases}&amp;space;x_1&#39;&amp;space;=&amp;space;x_1&amp;space;x_2&amp;space;\\\\&amp;space;x_2&#39;&amp;space;=&amp;space;-x_1&amp;space;x_2^3&amp;space;\\end{cases}}" width="200"/>\n\nAn example of quadratization of this system will be a new variable\n\n<img alt="y = x_1 x_2^2" height="60" src="https://latex.codecogs.com/png.latex?\\dpi{200}&amp;amp;amp;space;\\huge&amp;amp;amp;space;{\\color{DarkOrange}y&amp;amp;amp;space;=&amp;amp;amp;space;x_1&amp;amp;amp;space;x_2^2}" width="150"/>\n\nleading to the following ODE\n\n<img alt="y&#39; = x_2 y - 2y^2" height="50" src="https://latex.codecogs.com/png.latex?\\dpi{200}&amp;space;\\huge&amp;space;{\\color{DarkOrange}y&#39;&amp;space;=&amp;space;x_2&amp;space;y&amp;space;-&amp;space;2y^2}" width="250"/>\n\nThus, we attained the system with quadratic right-hand side\n\n<img alt="\\begin{cases} x_1&#39; = x_1 x_2 \\\\ x_2&#39; = -x_2 y \\\\ y&#39; = x_2 y - 2y^2 \\end{cases}" height="202" src="https://latex.codecogs.com/png.latex?\\dpi{200}&amp;space;\\huge&amp;space;{\\color{DarkOrange}\\begin{cases}&amp;space;x_1&#39;&amp;space;=&amp;space;x_1&amp;space;x_2&amp;space;\\\\&amp;space;x_2&#39;&amp;space;=&amp;space;-x_2&amp;space;y&amp;space;\\\\&amp;space;y&#39;&amp;space;=&amp;space;x_2&amp;space;y&amp;space;-&amp;space;2y^2&amp;space;\\end{cases}}" width="300"/>\n\nWe used only one new variable, so we achieved an *optimal* quadratization.\n\n# Qbee usage\n\nQBee implements algorithms that **take** system of ODEs with elementary functions right-hand side and\n**return** *optimal monomial quadratization* - optimal quadratization constructed from monomial substitutions.\n\nWe will demonstrate usage of QBee on the example below. Other interactive examples you can find\nin [examples section](examples).\n\n### 1. Importing QBee\n\nQBee relies on Sympy for a high-level API.\n\n```python\nimport sympy\nfrom qbee import *\n\nsympy.init_printing()  # If you work in Jupyter notebook \n```\n\n### 2. System definition\n\nFor example, we will take the **A1** system from [Swischuk et al.\'2020](https://arxiv.org/abs/1908.03620)\n\n<img alt="{\\color{DarkOrange} \\begin{cases} c_1&#39; = -A \\exp(-E_a / (R_u T)) c_1 ^{0.2} c_2^{1.3}\\\\ c_2&#39; = 2c_1&#39; \\\\ c_3&#39; = -c_1&#39; \\\\ c_4&#39; = -2 c_1&#39; \\end{cases}}" height="250" src="https://latex.codecogs.com/png.latex?\\dpi{200}&amp;space;\\huge&amp;space;{\\color{DarkOrange}&amp;space;\\begin{cases}&amp;space;c_1&#39;&amp;space;=&amp;space;-A&amp;space;\\exp(-E_a&amp;space;/&amp;space;(R_u&amp;space;T))&amp;space;c_1&amp;space;^{0.2}&amp;space;c_2^{1.3}\\\\&amp;space;c_2&#39;&amp;space;=&amp;space;2c_1&#39;&amp;space;\\\\&amp;space;c_3&#39;&amp;space;=&amp;space;-c_1&#39;&amp;space;\\\\&amp;space;c_4&#39;&amp;space;=&amp;space;-2&amp;space;c_1&#39;&amp;space;\\end{cases}}" width="550"/>\n\nThe parameters in the system are `A, Ea and Ru`, and the others are either state variables or inputs.\nSo, let\'s define them with the system in code:\n```python\nA, Ea, Ru = parameters("A, Ea, Ru")\nc1, c2, c3, c4, T = functions("c1, c2, c3, c4, T")  \n\neq1 = -A * sp.exp(-Ea / (Ru * T)) * c1 ** 0.2 * c2 ** 1.3\nsystem = [\n    (c1, eq1),\n    (c2, 2 * eq1),\n    (c3, -eq1),\n    (c4, -2 * eq1)\n]\n```\n\n### 3. Polynomialization and Quadratization\n\nWhen we work with ODEs with the right-hand side being a general continuous function, \nwe utilize the following pipeline: \n```\nInput system -> Polynomial System -> Quadratic System\n```\nand the transformations are called *polynomialization* and *quadratization* accordingly. \n\nThe example system is not polynomial, so we use the most general method for achieving optimal monomial quadratization.\n\n```python\n# {T: 2} means than T can have a derivative of order at most two => T\'\'\nquadr_system = polynomialize_and_quadratize(system, input_der_orders={T: 2})\nif quadr_system:\n    print("Quadratized system:")\n    print(quadr_system)\n```\n\nSample output:\n\n```\nVariables introduced in polynomialization:\nw_{0} = c1**(-0.8)\nw_{1} = c2**(-0.7)\nw_{2} = 1/T\nw_{3} = exp(-Ea*w_{2}/Ru)\n\nElapsed time: 0.139s.\n==================================================\nQuadratization result\n==================================================\nNumber of introduced variables: 5\nNodes traversed: 117\nIntroduced variables:\nw_{4} = T\'*w_{2}\nw_{5} = T\'*w_{2}**2\nw_{6} = c2**2*w_{0}*w_{1}*w_{3}\nw_{7} = w_{2}**2\nw_{8} = c1*c2*w_{0}*w_{1}*w_{3}\n\nQuadratized system:\nc1\' = -A*c2*w_{8}\nc2\' = -2*A*c2*w_{8}\nc3\' = A*c2*w_{8}\nc4\' = 2*A*c2*w_{8}\nw_{0}\' = 4*A*w_{0}*w_{6}/5\nw_{1}\' = 7*A*w_{1}*w_{8}/5\nw_{2}\' = -T\'*w_{7}\nw_{3}\' = Ea*w_{3}*w_{5}/Ru\nT\' = T\'\nT\'\' = T\'\'\nT\'\'\' = 0\nw_{4}\' = -T\'*w_{5} + T\'\'*w_{2}\nw_{5}\' = T\'\'*w_{7} - 2*w_{4}*w_{5}\nw_{6}\' = 4*A*w_{6}**2/5 - 13*A*w_{6}*w_{8}/5 + Ea*w_{5}*w_{6}/Ru\nw_{7}\' = -2*w_{4}*w_{7}\nw_{8}\' = -A*w_{6}*w_{8}/5 - 3*A*w_{8}**2/5 + Ea*w_{5}*w_{8}/Ru\n\nProcess finished with exit code 0\n\n```\n\nIntroduced variables are the optimal monomial quadratization.\n\n### 4. Work inside of package\n\n#### 1. Configuration\n\nInside of `config.ini` you can change the following arguments:\n\n* `logging_enable = [True | False]`. If enabled, work of algorithm is logged into `logging_file` and `quad_systems_file`\n  . Requires memory to work. Is not recommended for long quadratizations.\n* `logging_file`: must be in Apache Arrow `.feather` format.\n* `quad_systems_file`: dump quadratic systems by using pickle. `.pkl` file format is recommended.\n* `progress_bar_enable`: enables progress bar during quadratization.\n\n#### 2. Visualization\n\nIn order to visualize work of an algorithm you can pass logging data to `qbee.visualize.visualize_pyvis`:\n\n```python\nvisualize_pyvis(\'log.feather\', \'quad_systems.pkl\')\n```\n\n## Papers\n\n* Optimal Monomial Quadratization for ODE systems: [arxiv](https://arxiv.org/abs/2103.08013), [Springer](https://link.springer.com/chapter/10.1007/978-3-030-79987-8_9)\n\n## Citation\n\nIf you find this code useful in your research, please consider citing the above paper that works best for you. \n\n\n\n\n\n',
    'author': 'Andrey Bychkov, Gleb Pogudin',
    'author_email': 'None',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
