# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['filterbox', 'filterbox.concurrent', 'filterbox.frozen', 'filterbox.mutable']

package_data = \
{'': ['*']}

install_requires = \
['cykhash>=2.0.0,<3.0.0',
 'numpy>=1.14,<2.0',
 'readerwriterlock>=1.0.9,<2.0.0',
 'sortednp>=0.4.0,<0.5.0']

setup_kwargs = {
    'name': 'filterbox',
    'version': '0.7.0',
    'description': 'Container for finding Python objects by matching attributes. Stores objects by attribute value for fast lookup.',
    'long_description': '# FilterBox\n\nContainer for finding Python objects by matching attributes. \n\n[![tests Actions Status](https://github.com/manimino/filterbox/workflows/tests/badge.svg)](https://github.com/manimino/filterbox/actions)\n[![Coverage - 100%](https://img.shields.io/static/v1?label=Coverage&message=100%&color=2ea44f)](test/cov.txt)\n[![license - MIT](https://img.shields.io/static/v1?label=license&message=MIT&color=2ea44f)](/LICENSE)\n![python - 3.7+](https://img.shields.io/static/v1?label=python&message=3.7%2B&color=2ea44f)\n\nFilterBox stores objects by the values you specify. \n\nFinds are very fast. [Finding objects using FilterBox can be 5-10x faster than SQLite.](https://github.com/manimino/filterbox/blob/main/examples/perf_demo.ipynb)\n\n### Install\n\n```\npip install filterbox\n```\n\n### Usage\n\nFind which day will be good for flying a kite. It needs to be windy and sunny.\n\n```\nfrom filterbox import FilterBox\n\ndays = [\n    {\'day\': \'Saturday\', \'wind_speed\': 1, \'sky\': \'sunny\',},\n    {\'day\': \'Sunday\', \'wind_speed\': 3, \'sky\': \'rainy\'},\n    {\'day\': \'Monday\', \'wind_speed\': 7, \'sky\': \'sunny\'},\n    {\'day\': \'Tuesday\', \'wind_speed\': 9, \'sky\': \'rainy\'}\n]\n\ndef is_windy(obj):\n    return obj[\'wind_speed\'] > 5\n\n# make a FilterBox\nfb = FilterBox(               # make a FilterBox\n    days,                     # add objects of any Python type\n    on=[is_windy, \'sky\']      # functions + attributes to find by\n)\n\n# find objects by function and / or attribute values\nfb.find({is_windy: True, \'sky\': \'sunny\'})  \n# result: [{\'day\': \'Monday\', \'wind_speed\': 7, \'sky\': \'sunny\'}]\n```\n\nThere are three classes available.\n - FilterBox: can add, remove, and update objects after creation.\n - ConcurrentFilterBox: Thread-safe version of FilterBox.\n - FrozenFilterBox: Cannot be changed after creation. Faster finds, lower memory usage, and thread-safe.\n\n## More Examples\n\nExpand for sample code.\n\n<details>\n<summary>Match and exclude multiple values</summary>\n<br>\n\n```\nfrom filterbox import FilterBox\n\nobjects = [\n    {\'item\': 1, \'size\': 10, \'flavor\': \'melon\'}, \n    {\'item\': 2, \'size\': 10, \'flavor\': \'lychee\'}, \n    {\'item\': 3, \'size\': 20, \'flavor\': \'peach\'},\n    {\'item\': 4, \'size\': 30, \'flavor\': \'apple\'}\n]\n\nfb = FilterBox(objects, on=[\'size\', \'flavor\'])\n\nfb.find(\n    match={\'size\': [10, 20]},                # match anything with size in [10, 20] \n    exclude={\'flavor\': [\'lychee\', \'peach\']}  # where flavor is not in [\'lychee\', \'peach\']\n)  \n# result: [{\'item\': 1, \'size\': 10, \'flavor\': \'melon\'}]\n```\n</details>\n\n<details>\n<summary>Accessing nested data using functions</summary>\n<br />\nUse functions to get values from nested data structures.\n\n```\nfrom filterbox import FilterBox\n\nobjs = [\n    {\'a\': {\'b\': [1, 2, 3]}},\n    {\'a\': {\'b\': [4, 5, 6]}}\n]\n\ndef get_nested(obj):\n    return obj[\'a\'][\'b\'][0]\n\nfb = FilterBox(objs, [get_nested])\nfb.find({get_nested: 4})  \n# result: {\'a\': {\'b\': [4, 5, 6]}}\n```\n</details>\n\n<details>\n<summary>Greater than, less than</summary>\n<br />\n\nFilterBox does <code>==</code> very well, but <code><</code> and <code><</code> take some extra effort.\n\nSuppose you need to find objects where x >= some number. If the number is constant, a function that returns \n<code>obj.x >= constant</code> will work. \n\nOtherwise, FilterBox and FrozenFilterBox have a method <code>get_values(attr)</code> which gets the set of \nunique values for an attribute. \n\nHere\'s how to use it to find objects having <code>x >= 3</code>.\n```\nfrom filterbox import FilterBox\n\ndata = [{\'x\': i} for i in [1, 1, 2, 3, 5]]\nfb = FilterBox(data, [\'x\'])\nvals = fb.get_values(\'x\')                # get the set of unique values: {1, 2, 3, 5}\nbig_vals = [x for x in vals if x >= 3]   # big_vals is [3, 5]\nfb.find({\'x\': big_vals})                 # result: [{\'x\': 3}, {\'x\': 5}\n```\n\nIf x is a float or has many unique values, consider making a function on x that rounds it or puts it\ninto a bin of similar values. Discretizing x in ths way will make lookups faster.\n</details>\n\n<details>\n<summary>Handling missing attributes</summary>\n<br />\n\nObjects don\'t need to have every attribute.\n\n - Objects that are missing an attribute will not be stored under that attribute. This saves lots of memory.\n - To find all objects that have an attribute, match the special value <code>ANY</code>. \n - To find objects missing the attribute, exclude <code>ANY</code>.\n - In functions, raise <code>MissingAttribute</code> to tell FilterBox the object is missing.\n\nExample:\n```\nfrom filterbox import FilterBox, ANY\nfrom filterbox.exceptions import MissingAttribute\n\ndef get_a(obj):\n    try:\n        return obj[\'a\']\n    except KeyError:\n        raise MissingAttribute  # tell FilterBox this attribute is missing\n\nobjs = [{\'a\': 1}, {\'a\': 2}, {}]\nfb = FilterBox(objs, [\'a\', get_a])\n\nfb.find({\'a\': ANY})          # result: [{\'a\': 1}, {\'a\': 2}]\nfb.find({get_a: ANY})        # result: [{\'a\': 1}, {\'a\': 2}]\nfb.find(exclude={\'a\': ANY})  # result: [{}]\n```\n</details>\n\n### Recipes\n \n - [Auto-updating](https://github.com/manimino/filterbox/blob/main/examples/update.py) - Keep FilterBox updated when attribute values change\n - [Wordle solver](https://github.com/manimino/filterbox/blob/main/examples/wordle.ipynb) - Solve string matching problems faster than regex\n - [Collision detection](https://github.com/manimino/filterbox/blob/main/examples/collision.py) - Find objects based on type and proximity (grid-based)\n - [Percentiles](https://github.com/manimino/filterbox/blob/main/examples/percentile.py) - Find by percentile (median, p99, etc.)\n\n____\n\n## How it works\n\nFor every attribute in FilterBox, it holds a dict that maps each unique value to the set of objects with that value. \n\nThis is the rough idea of the data structure: \n```\nFilterBox = {\n    \'attribute1\': {val1: set(some_objs), val2: set(other_objs)},\n    \'attribute2\': {val3: set(some_objs), val4: set(other_objs)}\n}\n```\n\nDuring `find()`, the object sets matching each query value are retrieved. Then set operations like `union`, \n`intersect`, and `difference` are applied to get the final result.\n\nThat\'s a simplified version; for way more detail, See the "how it works" pages for:\n - [FilterBox](filterbox/mutable/how_it_works.md)\n - [ConcurrentFilterBox](filterbox/concurrent/how_it_works.md)\n - [FrozenFilterBox](filterbox/frozen/how_it_works.md)\n\n### API documentation:\n - [FilterBox](https://filterbox.readthedocs.io/en/latest/filterbox.mutable.html#filterbox.mutable.main.FilterBox)\n - [ConcurrentFilterBox](https://filterbox.readthedocs.io/en/latest/filterbox.concurrent.html#filterbox.concurrent.main.ConcurrentFilterBox)\n - [FrozenFilterBox](https://filterbox.readthedocs.io/en/latest/filterbox.frozen.html#filterbox.frozen.main.FrozenFilterBox)\n\n### Related projects\n\nFilterBox is a type of inverted index. It is optimized for its goal of finding in-memory Python objects.\n\nOther Python inverted index implementations are aimed at things like [vector search](https://pypi.org/project/rii/) and\n[finding documents by words](https://pypi.org/project/nltk/). Outside of Python, ElasticSearch is a popular inverted\nindex search tool. Each of these has goals outside of FilterBox\'s niche; there are no plans to expand FilterBox towards\nthese functions.\n\n____\n',
    'author': 'Theo Walker',
    'author_email': 'theo.ca.walker@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/manimino/filterbox/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
