## How It Works, Frozen Edition

FrozenFilterBox can use arrays instead of sets, thanks to its immutability constraint. It stores a numpy array of objects. 
Attribute values map to indices in the object array. On `find()`, the array indices for each match are retrieved. 
Then, set operations provided by `sortednp` are used to get a final set of object array indices. Last, the objects are 
retrieved from the object array by index and returned.

Each FrozenAttrIndex contains three numpy arrays of length n_objects:
 - Array of object indices
 - Array of values
 - Array of value hashes (this is actually not stored, only used to compute the arrays below.)

These are stored in a carefully chosen order for fast lookup.
They are sorted by hash, then grouped within each hash by value, and within that they are sorted by object index.

There is a second set of parallel arrays, of shorter length.
These are generated by run-length encoding of the value hashes array above.
 - unique_hashes: contains one entry for each unique hash
 - hash_starts: stores an index for lookup in the three n_objects arrays
 - hash_run_lengths: stores the number of items with the same hash.

On find(val), we do:
 - val_hash = hash(val)
 - Bisect the array of unique hashes to find the matching val_hash. If none, return.
 - Get the start and length of the values matching that hash from hash_starts and hash_run_lengths.
 - Now we know the range of object indices that match on hash -- but not necessarily on value!
 - Initialize two pointers in the array of values at the start and end of the matching hash range.
 - Move pointers inwards until they both match the val.
 - Return the object indices that match the val. Note that they are already sorted.

Why not just sort and bisect on value? Because the values may not be sortable. Imagine
values like ``[(1, 3), 'kwyjibo', SomeWeirdHashableObject()]``. How you gonna sort those?
Defining a comparator between them would be wacky.
But hashes are nice; ``hash(obj)`` is just an int in the int64 range. Hashes are definitely comparable, and very likely
to be unique. So they're the best comparator we can hope for. We just need to handle the collisions, as detailed above.

And there's one last optimization in here, to handle a hash collision scenario.
Suppose we had two values, val_big and val_small, with the same hash.
val1 is associated with a million objects; val2 is associated with ten.
When we look up val_small, we don't want to have to crawl through all the val_big values -- that
would take tens of milliseconds! Every query involving val_small would be a performance disaster.
So for values with many objects, like val_big, we extract those off into their own arrays and give them
a dict lookup. It adds a little code complexity and initialization time, but it makes query times predictable.
Further, we don't need to store many copies of val1 in that case -- just one, for the dict lookup. So it saves
memory as well. Super worth it.
