# coding: utf-8
from importlib import import_module, resources

__all__ = ['dump', 'set_dump_styles']

# Set global dump_styles dict
dump_styles = {}

def dump(style, system, **kwargs):
    """
    Convert a System to another format.
    
    Parameters
    ----------
    style : str
        Indicates the format of the content to dump the atomman.System as.
    system : atomman.System
        The system to convert.
    kwargs : any, optional
        Any extra keyword arguments to pass to the underlying dump methods.
        
    Returns
    -------
    str, object or tuple
        Any content returned by the underlying dump methods.
    """
    
    if style in dump_styles:
        return dump_styles[style](system, **kwargs)

    else:
        raise ValueError(f'Unsupported Atoms dump style {style}')

def set_dump_styles():
    """
    Imports and sets the dump styles.  Should be called after importing the
    iprPy.load submodule.
    """
    # Define subfolders to ignore
    ignorelist = ['__pycache__']

    # Dynamically import calculation styles
    for style in resources.contents(__name__):

        # Only import subfolders not in ignorelist
        if '.' not in style and style not in ignorelist:
            
            # Import module and set to dump_styles
            module = import_module(f'.{style}', __name__)
            dump_styles[style] = module.dump