# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['robonomicsinterface']

package_data = \
{'': ['*']}

install_requires = \
['click>=8.0.3,<9.0.0', 'substrate-interface>=1.1.7,<2.0.0']

entry_points = \
{'console_scripts': ['robonomics_interface = '
                     'robonomicsinterface.robonomics_interface_io:cli']}

setup_kwargs = {
    'name': 'robonomics-interface',
    'version': '0.7.0',
    'description': 'Robonomics wrapper over https://github.com/polkascan/py-substrate-interface created to facilitate programming with Robonomics',
    'long_description': '# robonomics-interface\nThis is a simple wrapper over https://github.com/polkascan/py-substrate-interface used to facilitate writing code for applications using Robonomics.\n\nRobonomics project: https://robonomics.network/\n\nRobonomics parachain dapp: https://parachain.robonomics.network/\n_______\n# Installation \n```bash\npip3 install robonomics-interface\n```\n# Usage\n*More info may be found in docstrings in the source code*\n```python\nimport robonomicsinterface as RI\n```\n## Initialization\n```python\ninterface = RI.RobonomicsInterface()\n```\nBy default, you will only be able to fetch Chainstate info from Frontier parachain and use PubSub pattern.  \n\nYou can specify another `node address` (e.g. local), `seed` to sign extrinsics (more on that [later](#extrinsics)) \nand custom `registry types`.\n\nAddress of the device may be obtained using `define_address` method. If the interface was initialed with a seed/private key\nthis method will return `<ss58_addr>` of the device whose seed/private key was passed.\n\n## Simple case: fetch Chainstate\nHere, no need to pass any arguments, by\n```python\ninterface = RI.RobonomicsInterface()\n```\nyou will be able to read any Chainstate info from the Frontier parachain:\n```python\nnum_dt = interface.custom_chainstate("DigitalTwin", "Total")\n```\nyou can also specify an argument for the query. Several arguments should be put in a list. Block hash parameter is \nalso available via `block_hash` argument if you want to make a query as of a specified block.\n\nThere is a dedicated function to obtain **Datalog**:\n```python\nrecord = interface.fetch_datalog(<ss58_addr>)\n```\nThis will give you the latest datalog record of the specified account with its timestamp. You may pass an index argument to fetch specific record. If you create an interface with a provided seed, you\'ll be able to fetch self-datalog calling `fetch_datalog` with no arguments (or just the `index` argument). \n\n## Extrinsics\n**Providing seed** (any raw or mnemonic) while initializing **will let you create and submit extrinsics**:\n```python\ninterface = RI.RobonmicsInterface(seed:str = <seed>)\nhash = interface.custom_extrinsic("DigitalTwin", "create")\n```\n`hash` here is the transaction hash of the succeeded extrinsic. You can also specify arguments for the extrinsic as a dictionary.\n\nThere are dedicated functions for recording datalog and sending launch commands:\n```python\ninterface.record_datalog("Hello, Robonomics")\ninterface.send_launch(<target_addr>, True)\n```\nCurrent nonce definition and manual nonce setting is also possible.\n\n## Subscriptions\nThere is a subscriptions functional implemented. When initiated, blocks thread and processes new events with a user-passed \ncallback function. Pay attention that this callback may only accept one argument - the event data. Up to now, the only supported \nevents are `NewRecord`, `NewLaunch`, `Transfer`\n```python\nfrom robonomicsinterface import RobonomicsInterface, Subscriber, SubEvent\n\ndef callback(data):\n    print(data)\n\ninterface = RobonomicsInterface()\nsubscriber = Subscriber(interface, SubEvent.NewLaunch, callback, <ss58_addr>)\n```\nOne may also pass a list of addresses.\n\n## Digital Twins\n[Digital Twins](https://wiki.robonomics.network/docs/en/digital-twins/) functionality is also supported.\n```python\ndt_it, tr_hash = interface.dt_create()\ntopic_hashed, source_tr_hash = interface.dt_set_source(dt_id, <topic_name>, <ss58_source_addr>)\ninterface.dt_info(dt_id)\ninterface.dt_owner(dt_id)\ninterface.dt_total()\n```\nOne may also find topic source by\n```python\ninterface.dt_get_source(dt_id, <topic_name>)\n```\n\n## Liabilities\nThis package support Robonomics liability functionality. [Here](https://wiki.robonomics.network/docs/en/robonomics-how-it-works/)\nis a bit about the concept on Ethereum. It\'s slightly different in Substrate.\n\nWith this package one can create liabilities, sign technical parameters messages, report completed liabilities, sign \nreport messages, fetch information about current and completed liabilities:\n```python\npromisee = RobonomicsInterface(remote_ws="ws://127.0.0.1:9944", seed="<seed>")\npromisor = RobonomicsInterface(remote_ws="ws://127.0.0.1:9944", seed="<seed>")\n\ntask = "QmYA2fn8cMbVWo4v95RwcwJVyQsNtnEwHerfWR8UNtEwoE" # task parsing is on user side\nreward = 10 * 10 ** 9\npromisee = promisee.define_address()\npromisor = promisor.define_address()\n\npromisee_task_signature = promisee.sign_create_liability(task, reward)\npromisor_task_signature = promisor.sign_create_liability(task, reward)\n\nindex, tr_hash = promisee.create_liability(\n    task, reward, promisee, promisor, promisee_task_signature, promisor_task_signature\n)\n\nprint(index)\nprint(promisee.liability_info(index))\n\nreport = "Qmc5gCcjYypU7y28oCALwfSvxCBskLuPKWpK4qpterKC7z" # report parsing is on user side\npromisor.finalize_liability(index, report) # this one signs report message automatically if no signature provided\nprint(promisor.liability_report(index))\n```\nMore information and functionality may be found in the code.\n\n## Robonomics Web Services (RWS)\nThere are as well dedicated methods for convenient usage of RWS.\n- Chainstate functions `auctionQueue`, `auction`, `devices` to examine subscriptions auctions:\n```python\ninterface.rws_auction_queue()\ninteface.rws_auction(<auction_index>)\ninterface.rws_list_devices(<subscription_owner_addr>)\n```\n- Extrinsincs: `bid`, `set_devices` and, the most important, `call`\n```python\ninterface.rws_bid(<auction_index>, <amount_weiners>)\ninterface.rws_set_devices([<ss58_addr>, <ss58_addr>])\ninterface.rws_custom_call(<subscription_owner_addr>,\n                           <call_module>,\n                           <call_function>,\n                           <params_dict>)\n```\nThere are as well dedicated `datalog`, `launch` and [DigitalTwin](#Digital Twins) functions for RWS-based transactions.\n```python\ninterface.rws_record_datalog(<subscription_owner_addr>, <data>)\ninterface.rws_send_launch(<subscription_owner_addr>, <target_addr>, True)\ninterface.rws_dt_create(<subscription_owner_addr>)\ninterface.rws_dt_set_source(<subscription_owner_addr>, dt_id, <topic_name>, <ss58_source_addr>)\n```\n\n## IO\nThis package provides console prototyping tool such as [robonomics io](https://wiki.robonomics.network/docs/en/rio-overview/)\nwith slight differences:\n```bash\n$ robonomics_interface read datalog\n$ echo "Hello, Robonomics" | robonomics_interface write datalog -s <seed>\n$ robonomics_interface read launch\n$ echo "ON" | robonomics_interface write launch -s <seed> -r <target_addr>\n```\nMore info may be found with \n```bash\n$ robonomics_interface --help\n```\n\n## JSON RPC\n*WARNING: THIS MODULE IS UNDER CONSTRUCTIONS, USE AT YOUR OWN RISK! TO BE UPDATED SOON.*  \nThere is a way to implement robonomics pubsub rpc calls:\n\n```python3\ninterface = RI.RobonomicsInterface()\npubsub = PubSub(interface)\npubsub.peer()\n```\n\nThis is an evolving package, it may have errors and lack of functionality, fixes are coming.\nFeel free to open issues when faced a problem.',
    'author': 'Pavel Tarasov',
    'author_email': 'p040399@outlook.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/Multi-Agent-io/robonomics-interface',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>3.6,<4.0',
}


setup(**setup_kwargs)
