from typing import Any

import pytest
from flatten_dict import flatten

from typing import Any, get_type_hints
from flatten_dict import flatten


class Andi(dict):
    def __init__(self, **kwargs):
        # Initialize with class-level defaults first
        for anno in self.__annotations__:
            default_value = getattr(self.__class__, anno, None)
            self.__setattr__(anno, default_value)

        super().__init__()

        # Override with provided kwargs
        for kwarg in kwargs:
            setattr(self, kwarg, kwargs.get(kwarg))

    def __setattr__(self, key, value):
        if key.startswith('_'):
            super().__setattr__(key, value)
        else:
            self[key] = value

    def __getattribute__(self, item):
        try:
            annotations = object.__getattribute__(self, '__class__').__annotations__
            for anno in annotations:
                if anno in self:
                    object.__setattr__(self, anno, self[anno])

            # Also check if item exists in dict but not in annotations
            if item in object.__getattribute__(self, 'keys')():
                if item not in annotations:
                    return self[item]
        except (AttributeError, KeyError):
            pass
        return object.__getattribute__(self, item)

    @classmethod
    def filtered(cls, **kwargs) -> 'Andi':
        if not hasattr(cls, '__annotations__') or not cls.__annotations__:
            raise KeyError("Class has no annotations to filter for!")
        allowed_keys = set(cls.__annotations__.keys())
        return cls(**{k: v for k, v in kwargs.items() if k in allowed_keys})

    @classmethod
    def _get_all_annotations(cls) -> dict:
        """Get annotations from entire inheritance chain"""
        annotations = {}
        for base in reversed(cls.__mro__):
            if hasattr(base, '__annotations__'):
                annotations.update(base.__annotations__)
        return annotations

    def flatten(self,
                reducer: Any = "tuple",
                inverse: bool = False,
                max_flatten_depth: Any = None,
                enumerate_types: Any = (),
                keep_empty_types: Any = ()
                ) -> dict:
        return flatten(
            self,
            reducer,
            inverse,
            max_flatten_depth,
            enumerate_types,
            keep_empty_types
        )

class TestAndi:
    """Test suite for Andi class"""

    def test_basic_initialization(self):
        class Simple(Andi):
            name: str
            age: int

        obj = Simple(name="Alice", age=30)
        assert obj.name == "Alice"
        assert obj.age == 30
        assert obj["name"] == "Alice"
        assert obj["age"] == 30

    def test_default_values(self):
        class WithDefaults(Andi):
            name: str
            age: int = 25
            active: bool = True

        obj = WithDefaults(name="Bob")
        assert obj.name == "Bob"
        assert obj.age == 25
        assert obj.active == True

    def test_override_defaults(self):
        class WithDefaults(Andi):
            name: str
            age: int = 25
            active: bool = True

        obj = WithDefaults(name="Charlie", age=35, active=False)
        assert obj.name == "Charlie"
        assert obj.age == 35
        assert obj.active == False

    def test_dict_access(self):
        class Person(Andi):
            name: str
            age: int

        obj = Person(name="Diana", age=28)
        assert obj["name"] == "Diana"
        assert obj["age"] == 28

    def test_attribute_access(self):
        class Person(Andi):
            name: str
            age: int

        obj = Person(name="Eve", age=22)
        assert obj.name == "Eve"
        assert obj.age == 22

    def test_mixed_access(self):
        class Person(Andi):
            name: str
            age: int

        obj = Person(name="Frank", age=40)
        obj["city"] = "NYC"
        assert obj.city == "NYC"
        obj.country = "USA"
        assert obj["country"] == "USA"

    def test_none_default_when_not_specified(self):
        class NoDefaults(Andi):
            name: str
            age: int

        obj = NoDefaults(name="George")
        assert obj.name == "George"
        assert obj.age is None

    def test_underscore_attributes_not_in_dict(self):
        class WithPrivate(Andi):
            name: str
            _internal: str

        obj = WithPrivate(name="Helen")
        obj._internal = "secret"
        assert obj.name == "Helen"
        assert obj._internal == "secret"
        assert "_internal" not in obj

    def test_empty_initialization(self):
        class Empty(Andi):
            name: str = "default"
            age: int = 0

        obj = Empty()
        assert obj.name == "default"
        assert obj.age == 0

    def test_nested_defaults(self):
        class Config(Andi):
            host: str = "localhost"
            port: int = 8080
            debug: bool = False

        obj = Config(host="example.com")
        assert obj.host == "example.com"
        assert obj.port == 8080
        assert obj.debug == False

    def test_type_annotations_preserved(self):
        class Typed(Andi):
            name: str
            age: int
            score: float = 0.0

        obj = Typed(name="Ian", age=30)
        assert "name" in obj.__class__.__annotations__
        assert "age" in obj.__class__.__annotations__
        assert "score" in obj.__class__.__annotations__

    def test_flatten_basic(self):
        class Nested(Andi):
            name: str
            metadata: dict

        obj = Nested(name="Jack", metadata={"key": "value", "nested": {"deep": "data"}})
        flattened = obj.flatten()
        assert ("metadata", "key") in flattened
        assert ("metadata", "nested", "deep") in flattened

    def test_multiple_instances_independent(self):
        class Counter(Andi):
            count: int = 0

        obj1 = Counter(count=5)
        obj2 = Counter(count=10)
        assert obj1.count == 5
        assert obj2.count == 10

    def test_modify_after_creation(self):
        class Mutable(Andi):
            name: str
            age: int = 0

        obj = Mutable(name="Kate")
        obj.age = 25
        assert obj.age == 25
        assert obj["age"] == 25

        obj["name"] = "Katherine"
        assert obj.name == "Katherine"

    def test_string_representation(self):
        class Person(Andi):
            name: str
            age: int

        obj = Person(name="Leo", age=35)
        assert "name" in str(obj)
        assert "Leo" in str(obj)

    def test_iteration(self):
        class Data(Andi):
            x: int
            y: int
            z: int = 0

        obj = Data(x=1, y=2)
        keys = list(obj.keys())
        assert "x" in keys
        assert "y" in keys
        assert "z" in keys

    def test_length(self):
        class Data(Andi):
            a: int
            b: int
            c: int = 0

        obj = Data(a=1, b=2)
        assert len(obj) == 3

    def test_contains(self):
        class Data(Andi):
            name: str
            age: int = 0

        obj = Data(name="Mike")
        assert "name" in obj
        assert "age" in obj
        assert "unknown" not in obj

    def test_get_method(self):
        class Data(Andi):
            name: str
            age: int = 0

        obj = Data(name="Nina")
        assert obj.get("name") == "Nina"
        assert obj.get("age") == 0
        assert obj.get("missing", "default") == "default"

    def test_update_method(self):
        class Data(Andi):
            name: str
            age: int = 0

        obj = Data(name="Oscar")
        obj.update({"age": 30, "city": "LA"})
        assert obj.age == 30
        assert obj["city"] == "LA"

    def test_pop_method(self):
        class Data(Andi):
            name: str
            age: int = 0

        obj = Data(name="Paul", age=40)
        age = obj.pop("age")
        assert age == 40
        assert "age" not in obj

    def test_inheritance(self):
        class Base(Andi):
            name: str
            age: int = 0

        class Extended(Base):
            email: str

        obj = Extended(name="Quinn", email="quinn@example.com")
        assert obj.name == "Quinn"
        assert obj.age == 0
        assert obj.email == "quinn@example.com"


class TestAndiFiltered:

    def test_filtered_keeps_only_annotated_attributes(self):
        class Person(Andi):
            name: str
            age: int
            email: str

        person = Person.filtered(
            name="Alice",
            age=30,
            email="alice@example.com",
            password="secret123",
            admin=True
        )

        assert person.name == "Alice"
        assert person.age == 30
        assert person.email == "alice@example.com"
        assert "password" not in person
        assert "admin" not in person
        assert len(person) == 3

    def test_filtered_handles_missing_annotated_attributes(self):
        class User(Andi):
            username: str
            password: str
            email: str

        user = User.filtered(
            username="bob",
            email="bob@example.com",
            extra_field="ignored"
        )

        assert user.username == "bob"
        assert user.password is None
        assert user.email == "bob@example.com"
        assert "extra_field" not in user
        assert len(user) == 3

    def test_filtered_with_only_extra_kwargs(self):
        class Config(Andi):
            host: str
            port: int

        config = Config.filtered(
            database="mydb",
            timeout=30,
            retry=True
        )

        assert config.host is None
        assert config.port is None
        assert "database" not in config
        assert "timeout" not in config
        assert "retry" not in config
        assert len(config) == 2

    def test_filtered_with_empty_kwargs(self):
        class Empty(Andi):
            value: str
            count: int

        empty = Empty.filtered()

        assert empty.value is None
        assert empty.count is None
        assert len(empty) == 2

    def test_filtered_with_complex_types(self):
        class Request(Andi):
            url: str
            method: str
            params: dict
            headers: dict

        request = Request.filtered(
            url="https://api.example.com",
            method="GET",
            params={"key": "value"},
            headers={"Auth": "Bearer token"},
            timeout=30,
            verify_ssl=False
        )

        assert request.url == "https://api.example.com"
        assert request.method == "GET"
        assert request.params == {"key": "value"}
        assert request.headers == {"Auth": "Bearer token"}
        assert "timeout" not in request
        assert "verify_ssl" not in request
        assert len(request) == 4

    def test_filtered_raises_error_without_annotations(self):
        class NoAnnotations(Andi):
            pass

        with pytest.raises(KeyError, match="Class has no annotations to filter for!"):
            NoAnnotations.filtered(foo="bar")

    def test_filtered_preserves_none_values(self):
        class Data(Andi):
            field1: str
            field2: int
            field3: bool

        data = Data.filtered(
            field1=None,
            field2=None,
            field3=None,
            extra=None
        )

        assert data.field1 is None
        assert data.field2 is None
        assert data.field3 is None
        assert "extra" not in data
        assert len(data) == 3

    def test_filtered_vs_normal_init(self):
        class Item(Andi):
            name: str
            price: float

        normal = Item(name="Widget", price=9.99, stock=100, category="Tools")
        filtered = Item.filtered(name="Widget", price=9.99, stock=100, category="Tools")

        assert normal.name == filtered.name
        assert normal.price == filtered.price

        assert "stock" in normal
        assert "category" in normal
        assert "stock" not in filtered
        assert "category" not in filtered

        assert len(normal) == 4
        assert len(filtered) == 2

    def test_filtered_with_nested_andi_objects(self):
        class Address(Andi):
            street: str
            city: str

        class Person(Andi):
            name: str
            address: Address

        address = Address.filtered(street="123 Main St", city="NYC", country="USA")
        person = Person.filtered(
            name="Charlie",
            address=address,
            age=35,
            job="Engineer"
        )

        assert person.name == "Charlie"
        assert person.address.street == "123 Main St"
        assert person.address.city == "NYC"
        assert "country" not in person.address
        assert "age" not in person
        assert "job" not in person

if __name__ == "__main__":
    pytest.main([__file__, "-v"])