import numpy as np
import MDAnalysis.analysis.distances as distance

from numba import jit
from tqdm import tqdm


@jit
def cos_angle(v1, v2):
    """compute the cos angle of two giving vectors"""
    return np.dot(v1, v2) / (np.linalg.norm(v1) * np.linalg.norm(v2))


class Order:
    def __init__(self, u, sel):
        self.u = u
        self.sel = u.select_atoms(sel)
        self.outfile = "order_" + "-".join(sel.split())
        self.neighbors = None
        self.n_vectors = None

        self.Q = []
        self.S = []

    def run(self, oto=False, tto=False, avc=False):

        print("Calculating order parameters ...")

        progress = {"ncols": 80}
        for ts in tqdm(self.u.trajectory, **progress):

            d = distance.distance_array(
                self.sel.positions, self.sel.positions, ts.dimensions
            )

            # distance array
            d_array = np.sort(d)[:, 1:5]

            # self and four nearest neighbors indexs
            neighbors = self.sel.positions[np.argsort(d)[:, 1:5]]

            # get neighbor vectors
            n_vectors = np.empty_like(neighbors)

            box = self.sel.dimensions[:3]
            for i in range(len(neighbors)):
                # periodic boundry conditions
                dr = neighbors[i] - self.sel.positions[i]
                n_vectors[i] = dr - box * np.round(dr / box)

            # oto
            self.Q += self.oto(n_vectors)

            # tto
            self.S += self.tto(d_array)

        self.output("oto")
        self.output("tto")

    def oto(self, V):
        """Orientational Tetrahedral Order

        Arguments:
            V {numpy.ndarray} -- neighbor vectors

        Returns:
            list -- Q
        """

        Q = []

        for v in V:
            q = 0
            for i in range(3):
                for j in range(i + 1, 4):
                    cos_phi = cos_angle(v[i], v[j])
                    q += (cos_phi + 1 / 3) ** 2
            q = 1 - 3 / 8 * q

            Q.append(q)

        return Q

    def tto(self, N):
        """Translational Tetrahedral Order

        Arguments:
            N {numpy.ndarray} -- distance array

        Returns:
            list -- S
        """

        # FIXME: if may crash if not using 4 nbs
        mean = np.mean(N, axis=1).reshape(len(N), 1)
        S = np.sum((N - mean) ** 2 / (4 * mean ** 2), axis=1)
        S = 1 - 1 / 3 * S

        return list(S)

    def avc(self):
        # TODO: implement avc
        pass

    def output(self, taskname, bins=100):

        if taskname == "oto":
            raw_data = self.outfile + "_oto_raw.dat"
            dist_data = self.outfile + "_oto.dat"

            P = self.Q
            his, param = np.histogram(P, bins, range=(0, 1))

        if taskname == "tto":
            raw_data = self.outfile + "_tto_raw.dat"
            dist_data = self.outfile + "_tto.dat"

            P = self.S
            his, param = np.histogram(P, bins, range=(0, 1))

        with open(raw_data, "w") as f:
            f.write(
                "# "
                + taskname
                + " generated by MDA: Analysis Tools for MD Simulations\n"
            )
            f.write("# raw data\n")

            for p in P:
                f.write(str(p) + "\n")

        with open(dist_data, "w") as f:
            f.write(
                "# "
                + taskname
                + " generated by MDA: Analysis Tools for MD Simulations\n"
            )
            f.write("# p count\n")

            for q, c in zip(param[1:], his):
                f.write("{:.3f}\t{:d}\n".format(q, c))


if __name__ == "__main__":
    from MDAnalysis import Universe

    topo = "../examples/tip3p.prmtop"
    traj = "../examples/tip3p.ncdf"

    u = Universe(topo, traj)
    sel = "name O"

    o = Order(u, sel)
    o.run()
