import collections

EllipticCurve = collections.namedtuple('EllipticCurve', 'name p a b g n h')

curve = EllipticCurve(
    'secp256k1',
    # Field characteristic.
    p=0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffefffffc2f,
    # Curve coefficients.
    a=0,
    b=7,
    # Base point.
    g=(0x79be667ef9dcbbac55a06295ce870b07029bfcdb2dce28d959f2815b16f81798,
       0x483ada7726a3c4655da4fbfc0e1108a8fd17b448a68554199c47d08ffb10d4b8),
    # Subgroup order.
    n=0xfffffffffffffffffffffffffffffffebaaedce6af48a03bbfd25e8cd0364141,
    # Subgroup cofactor.
    h=1,
)

def inverse_mod(k, p):
    if k == 0:
        raise ZeroDivisionError('division by zero')

    if k < 0:
        # k ** -1 = p - (-k) ** -1  (mod p)
        return p - inverse_mod(-k, p)

    # Extended Euclidean algorithm.
    s, old_s = 0, 1
    t, old_t = 1, 0
    r, old_r = p, k

    while r != 0:
        quotient = old_r // r
        old_r, r = r, old_r - quotient * r
        old_s, s = s, old_s - quotient * s
        old_t, t = t, old_t - quotient * t

    gcd, x, y = old_r, old_s, old_t

    assert gcd == 1
    assert (k * x) % p == 1

    return x % p

def ECCadd(point1, point2):

    if point1 is None:
        # 0 + point2 = point2
        return point2
    if point2 is None:
        # point1 + 0 = point1
        return point1

    x1, y1 = point1
    x2, y2 = point2

    """
    if x1 == x2 and y1 != y2:
        # point1 + (-point1) = 0
        return None
    """
    if x1 == x2:
        # This is the case point1 == point2.
        m = (3 * x1 * x1 + curve.a) * inverse_mod(2 * y1, curve.p)
    else:
        # This is the case point1 != point2.
        m = (y1 - y2) * inverse_mod(x1 - x2, curve.p)

    x3 = m * m - x1 - x2
    y3 = y1 + m * (x3 - x1)
    result = (x3 % curve.p,
              -y3 % curve.p)


    return result

def point_neg(point):

    if point is None:
        # -0 = 0
        return None

    x, y = point
    result = (x, -y % curve.p)


    return result

def scalar_mult(k, point):
    """
    if k % curve.n == 0 or point is None:
        return None
    """
    if k < 0:
        # k * point = -k * (-point)
        return scalar_mult(-k, point_neg(point))

    result = None
    addend = point

    while k:
        if k & 1:
            # Add.
            result = ECCadd(result, addend)

        # Double.
        addend = ECCadd(addend, addend)

        k >>= 1


    return result

def ECCDouble(point):
    return ECCadd(point, point)

def ECCFlip(point):
    res = []
    res.append(point[0])
    res.append(curve.p - point[1])
    return res

def ECCSub(lastKey, pkPub): #first arg higher
    flipped = ECCFlip(pkPub)
    return ECCadd(flipped, lastKey)

def ECCDiv(point):
    mulKey = 0x7fffffffffffffffffffffffffffffff5d576e7357a4501ddfe92f46681b20a0 + 1
    return scalar_mult( mulKey, point)

def get_point(key):
    return scalar_mult(key, curve.g)
