"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceDeployment = exports.ServiceType = void 0;
const constructs_1 = require("constructs");
const k8s = require("./imports/k8s");
var ServiceType;
(function (ServiceType) {
    ServiceType["EXTERNAL_NAME"] = "ExternalName";
    ServiceType["CLUSTER_IP"] = "ClusterIP";
    ServiceType["NODE_PORT"] = "NodePort";
    ServiceType["LOAD_BALANCER"] = "LoadBalancer";
})(ServiceType = exports.ServiceType || (exports.ServiceType = {}));
class ServiceDeployment extends constructs_1.Construct {
    constructor(scope, id, options) {
        super(scope, id);
        const replicas = options.replicas ?? 1;
        const resources = options.resources ?? { requests: { cpu: '100m', memory: '100Mi' } };
        const externalPort = options.externalPort ?? 80;
        const containerPort = options.containerPort ?? 8080;
        const containerName = options.containerName ?? 'app';
        const serviceType = options.serviceType ?? ServiceType.CLUSTER_IP;
        const label = {
            ...options.labels,
            app: constructs_1.Node.of(this).uniqueId,
        };
        const service = new k8s.Service(this, 'service', {
            metadata: { labels: label },
            spec: {
                type: serviceType,
                ports: [{ port: externalPort, targetPort: containerPort }],
                selector: label,
            },
        });
        new k8s.Deployment(this, 'deployment', {
            spec: {
                selector: { matchLabels: label },
                replicas: replicas,
                template: {
                    metadata: { labels: label },
                    spec: {
                        containers: [
                            {
                                name: containerName,
                                image: options.image,
                                ports: [{ containerPort }],
                                resources: resources,
                                env: renderEnv(options.env),
                            },
                        ],
                    },
                },
            },
        });
        this.host = service.name;
    }
}
exports.ServiceDeployment = ServiceDeployment;
function renderEnv(env = {}) {
    const result = new Array();
    for (const [key, value] of Object.entries(env)) {
        result.push({
            name: key,
            value: value,
        });
    }
    return result;
}
//# sourceMappingURL=data:application/json;base64,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