"use strict";
/*
 * Unless explicitly stated otherwise all files in this repository are licensed
 * under the Apache License Version 2.0.
 *
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2021 Datadog, Inc.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.redirectHandlers = void 0;
const loglevel_1 = require("loglevel");
const constants_1 = require("./constants");
/**
 * To avoid modifying code in the user's lambda handler, redirect the handler to a Datadog
 * handler that initializes the Lambda Layers and then calls the original handler.
 * 'DD_LAMBDA_HANDLER' is set to the original handler in the lambda's environment for the
 * replacement handler to find.
 *
 * Unchanged aside from parameter type
 */
function redirectHandlers(lambdas, addLayers) {
    loglevel_1.default.debug(`Wrapping Lambda function handlers with Datadog handler...`);
    lambdas.forEach((lam) => {
        const runtime = lam.runtime.name;
        const lambdaRuntime = constants_1.runtimeLookup[runtime];
        if (lambdaRuntime === constants_1.RuntimeType.JAVA) {
            lam.addEnvironment(constants_1.AWS_JAVA_WRAPPER_ENV_VAR, constants_1.AWS_JAVA_WRAPPER_ENV_VAR_VALUE);
        }
        else {
            const cfnFunction = lam.node.defaultChild;
            if (cfnFunction === undefined) {
                loglevel_1.default.debug("Unable to get Lambda Function handler");
                return;
            }
            const originalHandler = cfnFunction.handler;
            lam.addEnvironment(constants_1.DD_HANDLER_ENV_VAR, originalHandler);
            const handler = getDDHandler(lambdaRuntime, addLayers);
            if (handler === null) {
                loglevel_1.default.debug("Unable to get Datadog handler");
                return;
            }
            cfnFunction.handler = handler;
        }
    });
}
exports.redirectHandlers = redirectHandlers;
function getDDHandler(lambdaRuntime, addLayers) {
    if (lambdaRuntime === undefined || lambdaRuntime === constants_1.RuntimeType.UNSUPPORTED) {
        loglevel_1.default.debug("Unsupported/undefined Lambda runtime");
        return;
    }
    switch (lambdaRuntime) {
        case constants_1.RuntimeType.NODE:
            return addLayers ? constants_1.JS_HANDLER_WITH_LAYERS : constants_1.JS_HANDLER;
        case constants_1.RuntimeType.PYTHON:
            return constants_1.PYTHON_HANDLER;
        case constants_1.RuntimeType.JAVA:
            return null;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVkaXJlY3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvcmVkaXJlY3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Ozs7R0FNRzs7O0FBR0gsdUNBQTJCO0FBQzNCLDJDQVNxQjtBQUVyQjs7Ozs7OztHQU9HO0FBQ0gsU0FBZ0IsZ0JBQWdCLENBQUMsT0FBMEIsRUFBRSxTQUFrQjtJQUM3RSxrQkFBRyxDQUFDLEtBQUssQ0FBQywyREFBMkQsQ0FBQyxDQUFDO0lBQ3ZFLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRTtRQUN0QixNQUFNLE9BQU8sR0FBVyxHQUFHLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQztRQUN6QyxNQUFNLGFBQWEsR0FBZ0IseUJBQWEsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUMxRCxJQUFJLGFBQWEsS0FBSyx1QkFBVyxDQUFDLElBQUksRUFBRTtZQUN0QyxHQUFHLENBQUMsY0FBYyxDQUFDLG9DQUF3QixFQUFFLDBDQUE4QixDQUFDLENBQUM7U0FDOUU7YUFBTTtZQUNMLE1BQU0sV0FBVyxHQUFHLEdBQUcsQ0FBQyxJQUFJLENBQUMsWUFBa0MsQ0FBQztZQUNoRSxJQUFJLFdBQVcsS0FBSyxTQUFTLEVBQUU7Z0JBQzdCLGtCQUFHLENBQUMsS0FBSyxDQUFDLHVDQUF1QyxDQUFDLENBQUM7Z0JBQ25ELE9BQU87YUFDUjtZQUNELE1BQU0sZUFBZSxHQUFHLFdBQVcsQ0FBQyxPQUFpQixDQUFDO1lBQ3RELEdBQUcsQ0FBQyxjQUFjLENBQUMsOEJBQWtCLEVBQUUsZUFBZSxDQUFDLENBQUM7WUFDeEQsTUFBTSxPQUFPLEdBQUcsWUFBWSxDQUFDLGFBQWEsRUFBRSxTQUFTLENBQUMsQ0FBQztZQUN2RCxJQUFJLE9BQU8sS0FBSyxJQUFJLEVBQUU7Z0JBQ3BCLGtCQUFHLENBQUMsS0FBSyxDQUFDLCtCQUErQixDQUFDLENBQUM7Z0JBQzNDLE9BQU87YUFDUjtZQUNELFdBQVcsQ0FBQyxPQUFPLEdBQUcsT0FBTyxDQUFDO1NBQy9CO0lBQ0gsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDO0FBdkJELDRDQXVCQztBQUVELFNBQVMsWUFBWSxDQUFDLGFBQTBCLEVBQUUsU0FBa0I7SUFDbEUsSUFBSSxhQUFhLEtBQUssU0FBUyxJQUFJLGFBQWEsS0FBSyx1QkFBVyxDQUFDLFdBQVcsRUFBRTtRQUM1RSxrQkFBRyxDQUFDLEtBQUssQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO1FBQ2xELE9BQU87S0FDUjtJQUNELFFBQVEsYUFBYSxFQUFFO1FBQ3JCLEtBQUssdUJBQVcsQ0FBQyxJQUFJO1lBQ25CLE9BQU8sU0FBUyxDQUFDLENBQUMsQ0FBQyxrQ0FBc0IsQ0FBQyxDQUFDLENBQUMsc0JBQVUsQ0FBQztRQUN6RCxLQUFLLHVCQUFXLENBQUMsTUFBTTtZQUNyQixPQUFPLDBCQUFjLENBQUM7UUFDeEIsS0FBSyx1QkFBVyxDQUFDLElBQUk7WUFDbkIsT0FBTyxJQUFJLENBQUM7S0FDZjtBQUNILENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKlxuICogVW5sZXNzIGV4cGxpY2l0bHkgc3RhdGVkIG90aGVyd2lzZSBhbGwgZmlsZXMgaW4gdGhpcyByZXBvc2l0b3J5IGFyZSBsaWNlbnNlZFxuICogdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlIFZlcnNpb24gMi4wLlxuICpcbiAqIFRoaXMgcHJvZHVjdCBpbmNsdWRlcyBzb2Z0d2FyZSBkZXZlbG9wZWQgYXQgRGF0YWRvZyAoaHR0cHM6Ly93d3cuZGF0YWRvZ2hxLmNvbS8pLlxuICogQ29weXJpZ2h0IDIwMjEgRGF0YWRvZywgSW5jLlxuICovXG5cbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLWxhbWJkYVwiO1xuaW1wb3J0IGxvZyBmcm9tIFwibG9nbGV2ZWxcIjtcbmltcG9ydCB7XG4gIFJ1bnRpbWVUeXBlLFxuICBydW50aW1lTG9va3VwLFxuICBERF9IQU5ETEVSX0VOVl9WQVIsXG4gIEFXU19KQVZBX1dSQVBQRVJfRU5WX1ZBUixcbiAgQVdTX0pBVkFfV1JBUFBFUl9FTlZfVkFSX1ZBTFVFLFxuICBKU19IQU5ETEVSX1dJVEhfTEFZRVJTLFxuICBKU19IQU5ETEVSLFxuICBQWVRIT05fSEFORExFUixcbn0gZnJvbSBcIi4vY29uc3RhbnRzXCI7XG5cbi8qKlxuICogVG8gYXZvaWQgbW9kaWZ5aW5nIGNvZGUgaW4gdGhlIHVzZXIncyBsYW1iZGEgaGFuZGxlciwgcmVkaXJlY3QgdGhlIGhhbmRsZXIgdG8gYSBEYXRhZG9nXG4gKiBoYW5kbGVyIHRoYXQgaW5pdGlhbGl6ZXMgdGhlIExhbWJkYSBMYXllcnMgYW5kIHRoZW4gY2FsbHMgdGhlIG9yaWdpbmFsIGhhbmRsZXIuXG4gKiAnRERfTEFNQkRBX0hBTkRMRVInIGlzIHNldCB0byB0aGUgb3JpZ2luYWwgaGFuZGxlciBpbiB0aGUgbGFtYmRhJ3MgZW52aXJvbm1lbnQgZm9yIHRoZVxuICogcmVwbGFjZW1lbnQgaGFuZGxlciB0byBmaW5kLlxuICpcbiAqIFVuY2hhbmdlZCBhc2lkZSBmcm9tIHBhcmFtZXRlciB0eXBlXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiByZWRpcmVjdEhhbmRsZXJzKGxhbWJkYXM6IGxhbWJkYS5GdW5jdGlvbltdLCBhZGRMYXllcnM6IGJvb2xlYW4pIHtcbiAgbG9nLmRlYnVnKGBXcmFwcGluZyBMYW1iZGEgZnVuY3Rpb24gaGFuZGxlcnMgd2l0aCBEYXRhZG9nIGhhbmRsZXIuLi5gKTtcbiAgbGFtYmRhcy5mb3JFYWNoKChsYW0pID0+IHtcbiAgICBjb25zdCBydW50aW1lOiBzdHJpbmcgPSBsYW0ucnVudGltZS5uYW1lO1xuICAgIGNvbnN0IGxhbWJkYVJ1bnRpbWU6IFJ1bnRpbWVUeXBlID0gcnVudGltZUxvb2t1cFtydW50aW1lXTtcbiAgICBpZiAobGFtYmRhUnVudGltZSA9PT0gUnVudGltZVR5cGUuSkFWQSkge1xuICAgICAgbGFtLmFkZEVudmlyb25tZW50KEFXU19KQVZBX1dSQVBQRVJfRU5WX1ZBUiwgQVdTX0pBVkFfV1JBUFBFUl9FTlZfVkFSX1ZBTFVFKTtcbiAgICB9IGVsc2Uge1xuICAgICAgY29uc3QgY2ZuRnVuY3Rpb24gPSBsYW0ubm9kZS5kZWZhdWx0Q2hpbGQgYXMgbGFtYmRhLkNmbkZ1bmN0aW9uO1xuICAgICAgaWYgKGNmbkZ1bmN0aW9uID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgbG9nLmRlYnVnKFwiVW5hYmxlIHRvIGdldCBMYW1iZGEgRnVuY3Rpb24gaGFuZGxlclwiKTtcbiAgICAgICAgcmV0dXJuO1xuICAgICAgfVxuICAgICAgY29uc3Qgb3JpZ2luYWxIYW5kbGVyID0gY2ZuRnVuY3Rpb24uaGFuZGxlciBhcyBzdHJpbmc7XG4gICAgICBsYW0uYWRkRW52aXJvbm1lbnQoRERfSEFORExFUl9FTlZfVkFSLCBvcmlnaW5hbEhhbmRsZXIpO1xuICAgICAgY29uc3QgaGFuZGxlciA9IGdldERESGFuZGxlcihsYW1iZGFSdW50aW1lLCBhZGRMYXllcnMpO1xuICAgICAgaWYgKGhhbmRsZXIgPT09IG51bGwpIHtcbiAgICAgICAgbG9nLmRlYnVnKFwiVW5hYmxlIHRvIGdldCBEYXRhZG9nIGhhbmRsZXJcIik7XG4gICAgICAgIHJldHVybjtcbiAgICAgIH1cbiAgICAgIGNmbkZ1bmN0aW9uLmhhbmRsZXIgPSBoYW5kbGVyO1xuICAgIH1cbiAgfSk7XG59XG5cbmZ1bmN0aW9uIGdldERESGFuZGxlcihsYW1iZGFSdW50aW1lOiBSdW50aW1lVHlwZSwgYWRkTGF5ZXJzOiBib29sZWFuKSB7XG4gIGlmIChsYW1iZGFSdW50aW1lID09PSB1bmRlZmluZWQgfHwgbGFtYmRhUnVudGltZSA9PT0gUnVudGltZVR5cGUuVU5TVVBQT1JURUQpIHtcbiAgICBsb2cuZGVidWcoXCJVbnN1cHBvcnRlZC91bmRlZmluZWQgTGFtYmRhIHJ1bnRpbWVcIik7XG4gICAgcmV0dXJuO1xuICB9XG4gIHN3aXRjaCAobGFtYmRhUnVudGltZSkge1xuICAgIGNhc2UgUnVudGltZVR5cGUuTk9ERTpcbiAgICAgIHJldHVybiBhZGRMYXllcnMgPyBKU19IQU5ETEVSX1dJVEhfTEFZRVJTIDogSlNfSEFORExFUjtcbiAgICBjYXNlIFJ1bnRpbWVUeXBlLlBZVEhPTjpcbiAgICAgIHJldHVybiBQWVRIT05fSEFORExFUjtcbiAgICBjYXNlIFJ1bnRpbWVUeXBlLkpBVkE6XG4gICAgICByZXR1cm4gbnVsbDtcbiAgfVxufVxuIl19