"use strict";
/*
 * Unless explicitly stated otherwise all files in this repository are licensed
 * under the Apache License Version 2.0.
 *
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2021 Datadog, Inc.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateLayerId = exports.generateExtensionLayerId = exports.generateLambdaLayerId = exports.getMissingLayerVersionErrorMsg = exports.getExtensionLayerArn = exports.getLambdaLayerArn = exports.applyLayers = void 0;
const crypto = require("crypto");
const lambda = require("aws-cdk-lib/aws-lambda");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const loglevel_1 = require("loglevel");
const index_1 = require("./index");
const layers = new Map();
function applyLayers(scope, region, lambdas, pythonLayerVersion, nodeLayerVersion, javaLayerVersion, extensionLayerVersion, useLayersFromAccount) {
    // TODO: check region availability
    const errors = [];
    loglevel_1.default.debug("Applying layers to Lambda functions...");
    lambdas.forEach((lam) => {
        const runtime = lam.runtime.name;
        const lambdaRuntimeType = index_1.runtimeLookup[runtime];
        const isARM = lam.architecture?.dockerPlatform !== undefined &&
            lam.architecture.dockerPlatform === aws_lambda_1.Architecture.ARM_64.dockerPlatform;
        const isNode = lambdaRuntimeType === index_1.RuntimeType.NODE;
        if (lambdaRuntimeType === undefined || lambdaRuntimeType === index_1.RuntimeType.UNSUPPORTED) {
            loglevel_1.default.debug(`Unsupported runtime: ${runtime}`);
            return;
        }
        const accountId = useLayersFromAccount;
        let lambdaLayerArn;
        let extensionLayerArn;
        if (lambdaRuntimeType === index_1.RuntimeType.PYTHON) {
            if (pythonLayerVersion === undefined) {
                const errorMessage = getMissingLayerVersionErrorMsg(lam.node.id, "Python", "python");
                loglevel_1.default.error(errorMessage);
                errors.push(errorMessage);
                return;
            }
            lambdaLayerArn = getLambdaLayerArn(region, pythonLayerVersion, runtime, isARM, isNode, accountId);
            loglevel_1.default.debug(`Using Python Lambda layer: ${lambdaLayerArn}`);
            addLayer(lambdaLayerArn, false, scope, lam, runtime);
        }
        if (lambdaRuntimeType === index_1.RuntimeType.NODE) {
            if (nodeLayerVersion === undefined) {
                const errorMessage = getMissingLayerVersionErrorMsg(lam.node.id, "Node.js", "node");
                loglevel_1.default.error(errorMessage);
                errors.push(errorMessage);
                return;
            }
            lambdaLayerArn = getLambdaLayerArn(region, nodeLayerVersion, runtime, isARM, isNode, accountId);
            loglevel_1.default.debug(`Using Node Lambda layer: ${lambdaLayerArn}`);
            addLayer(lambdaLayerArn, false, scope, lam, runtime);
        }
        if (lambdaRuntimeType === index_1.RuntimeType.JAVA) {
            if (javaLayerVersion === undefined) {
                const errorMessage = getMissingLayerVersionErrorMsg(lam.node.id, "Java", "java");
                loglevel_1.default.error(errorMessage);
                errors.push(errorMessage);
                return;
            }
            lambdaLayerArn = getLambdaLayerArn(region, javaLayerVersion, runtime, isARM, isNode, accountId);
            loglevel_1.default.debug(`Using dd-trace-java layer: ${lambdaLayerArn}`);
            addLayer(lambdaLayerArn, false, scope, lam, runtime);
        }
        if (extensionLayerVersion !== undefined) {
            extensionLayerArn = getExtensionLayerArn(region, extensionLayerVersion, isARM, accountId);
            loglevel_1.default.debug(`Using extension layer: ${extensionLayerArn}`);
            addLayer(extensionLayerArn, true, scope, lam, runtime);
        }
    });
    return errors;
}
exports.applyLayers = applyLayers;
function addLayer(layerArn, isExtensionLayer, scope, lam, runtime) {
    const layerId = generateLayerId(isExtensionLayer, lam.functionArn, runtime);
    if (layerArn !== undefined) {
        let lambdaLayer = layers.get(layerArn);
        if (lambdaLayer === undefined) {
            lambdaLayer = lambda.LayerVersion.fromLayerVersionArn(scope, layerId, layerArn);
            layers.set(layerArn, lambdaLayer);
        }
        loglevel_1.default.debug(`Adding layer ${lambdaLayer} to Lambda: ${lam.functionName}`);
        lam.addLayers(lambdaLayer);
    }
}
function getLambdaLayerArn(region, version, runtime, isArm, isNode, accountId) {
    const baseLayerName = index_1.runtimeToLayerName[runtime];
    const layerName = isArm && !isNode ? `${baseLayerName}-ARM` : baseLayerName;
    const partition = getAWSPartitionFromRegion(region);
    // TODO: edge case where gov cloud is the region, but they are using a token so we can't resolve it.
    const isGovCloud = index_1.govCloudRegions.includes(region);
    // if this is a GovCloud region, use the GovCloud lambda layer
    if (isGovCloud) {
        loglevel_1.default.debug("GovCloud region detected, using the GovCloud lambda layer");
        return `arn:${partition}:lambda:${region}:${accountId ?? index_1.DD_GOV_ACCOUNT_ID}:layer:${layerName}:${version}`;
    }
    return `arn:${partition}:lambda:${region}:${accountId ?? index_1.DD_ACCOUNT_ID}:layer:${layerName}:${version}`;
}
exports.getLambdaLayerArn = getLambdaLayerArn;
function getExtensionLayerArn(region, version, isArm, accountId) {
    const baseLayerName = "Datadog-Extension";
    const layerName = isArm ? `${baseLayerName}-ARM` : baseLayerName;
    const partition = getAWSPartitionFromRegion(region);
    const isGovCloud = index_1.govCloudRegions.includes(region);
    if (isGovCloud) {
        loglevel_1.default.debug("GovCloud region detected, using the GovCloud extension layer");
        return `arn:${partition}:lambda:${region}:${accountId ?? index_1.DD_GOV_ACCOUNT_ID}:layer:${layerName}:${version}`;
    }
    return `arn:${partition}:lambda:${region}:${accountId ?? index_1.DD_ACCOUNT_ID}:layer:${layerName}:${version}`;
}
exports.getExtensionLayerArn = getExtensionLayerArn;
function getMissingLayerVersionErrorMsg(functionKey, formalRuntime, paramRuntime) {
    return (`Resource ${functionKey} has a ${formalRuntime} runtime, but no ${formalRuntime} Lambda Library version was provided. ` +
        `Please add the '${paramRuntime}LayerVersion' parameter for the Datadog serverless macro.`);
}
exports.getMissingLayerVersionErrorMsg = getMissingLayerVersionErrorMsg;
function generateLambdaLayerId(lambdaFunctionArn, runtime) {
    loglevel_1.default.debug("Generating construct Id for Datadog Lambda layer");
    const layerValue = crypto.createHash("sha256").update(lambdaFunctionArn).digest("hex");
    return `${index_1.LAYER_PREFIX}-${runtime}-${layerValue}`;
}
exports.generateLambdaLayerId = generateLambdaLayerId;
function generateExtensionLayerId(lambdaFunctionArn) {
    loglevel_1.default.debug("Generating construct Id for Datadog Extension layer");
    const layerValue = crypto.createHash("sha256").update(lambdaFunctionArn).digest("hex");
    return `${index_1.EXTENSION_LAYER_PREFIX}-${layerValue}`;
}
exports.generateExtensionLayerId = generateExtensionLayerId;
function generateLayerId(isExtensionLayer, functionArn, runtime) {
    if (isExtensionLayer) {
        return generateExtensionLayerId(functionArn);
    }
    return generateLambdaLayerId(functionArn, runtime);
}
exports.generateLayerId = generateLayerId;
function getAWSPartitionFromRegion(region) {
    if (region.startsWith("us-gov-")) {
        return "aws-us-gov";
    }
    if (region.startsWith("cn-")) {
        return "aws-cn";
    }
    return "aws";
}
//# sourceMappingURL=data:application/json;base64,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