"use strict";
/*
 * Unless explicitly stated otherwise all files in this repository are licensed
 * under the Apache License Version 2.0.
 *
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2021 Datadog, Inc.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateSubscriptionFilterName = exports.generateForwarderConstructId = exports.addForwarderToLogGroups = exports.addForwarder = void 0;
const crypto = require("crypto");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_logs_destinations_1 = require("aws-cdk-lib/aws-logs-destinations");
const loglevel_1 = require("loglevel");
const index_1 = require("./index");
function getForwarder(scope, forwarderArn) {
    const forwarderConstructId = generateForwarderConstructId(forwarderArn);
    if (scope.node.tryFindChild(forwarderConstructId)) {
        return scope.node.tryFindChild(forwarderConstructId);
    }
    else {
        return lambda.Function.fromFunctionArn(scope, forwarderConstructId, forwarderArn);
    }
}
function addForwarder(scope, lambdaFunctions, forwarderArn, createForwarderPermissions) {
    const forwarder = getForwarder(scope, forwarderArn);
    const forwarderDestination = new aws_logs_destinations_1.LambdaDestination(forwarder, { addPermissions: createForwarderPermissions });
    lambdaFunctions.forEach((lam) => {
        const subscriptionFilterName = generateSubscriptionFilterName(aws_cdk_lib_1.Names.uniqueId(lam), forwarderArn);
        loglevel_1.default.debug(`Adding log subscription ${subscriptionFilterName} for ${lam.functionName}`);
        lam.logGroup.addSubscriptionFilter(subscriptionFilterName, {
            destination: forwarderDestination,
            filterPattern: aws_logs_1.FilterPattern.allEvents(),
        });
    });
}
exports.addForwarder = addForwarder;
function addForwarderToLogGroups(scope, logGroups, forwarderArn, createForwarderPermissions) {
    const forwarder = getForwarder(scope, forwarderArn);
    const forwarderDestination = new aws_logs_destinations_1.LambdaDestination(forwarder, { addPermissions: createForwarderPermissions });
    logGroups.forEach((group) => {
        const subscriptionFilterName = generateSubscriptionFilterName(aws_cdk_lib_1.Names.nodeUniqueId(group.node), forwarderArn);
        group.addSubscriptionFilter(subscriptionFilterName, {
            destination: forwarderDestination,
            filterPattern: aws_logs_1.FilterPattern.allEvents(),
        });
    });
}
exports.addForwarderToLogGroups = addForwarderToLogGroups;
function generateForwarderConstructId(forwarderArn) {
    loglevel_1.default.debug("Generating construct Id for Datadog Lambda Forwarder");
    return "forwarder" + crypto.createHash("sha256").update(forwarderArn).digest("hex");
}
exports.generateForwarderConstructId = generateForwarderConstructId;
function generateSubscriptionFilterName(functionUniqueId, forwarderArn) {
    const subscriptionFilterValue = crypto
        .createHash("sha256")
        .update(functionUniqueId)
        .update(forwarderArn)
        .digest("hex");
    const subscriptionFilterValueLength = subscriptionFilterValue.length;
    const subscriptionFilterName = index_1.SUBSCRIPTION_FILTER_PREFIX +
        subscriptionFilterValue.substring(subscriptionFilterValueLength - 8, subscriptionFilterValueLength);
    return subscriptionFilterName;
}
exports.generateSubscriptionFilterName = generateSubscriptionFilterName;
//# sourceMappingURL=data:application/json;base64,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