"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.handleSettingPropDefaults = exports.checkForMultipleApiKeys = exports.validateProps = exports.addCdkConstructVersionTag = exports.Datadog = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
 * Unless explicitly stated otherwise all files in this repository are licensed
 * under the Apache License Version 2.0.
 *
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2021 Datadog, Inc.
 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const constructs_1 = require("constructs");
const loglevel_1 = require("loglevel");
const index_1 = require("./index");
const versionJson = require("../version.json");
class Datadog extends constructs_1.Construct {
    constructor(scope, id, props) {
        if (process.env.DD_CONSTRUCT_DEBUG_LOGS?.toLowerCase() == "true") {
            loglevel_1.default.setLevel("debug");
        }
        super(scope, id);
        this.scope = scope;
        this.props = props;
        let apiKeySecretArn = this.props.apiKeySecretArn;
        if (this.props.apiKeySecret !== undefined) {
            apiKeySecretArn = this.props.apiKeySecret.secretArn;
        }
        validateProps(this.props, this.props.apiKeySecret !== undefined);
        this.transport = new index_1.Transport(this.props.flushMetricsToLogs, this.props.site, this.props.apiKey, apiKeySecretArn, this.props.apiKmsKey, this.props.extensionLayerVersion);
    }
    addLambdaFunctions(lambdaFunctions, construct) {
        // baseProps contains all properties set by the user, with default values for properties
        // defined in DefaultDatadogProps (if not set by user)
        const baseProps = handleSettingPropDefaults(this.props);
        const extractedLambdaFunctions = extractSingletonFunctions(lambdaFunctions);
        if (this.props !== undefined && extractedLambdaFunctions.length > 0) {
            if (this.props.apiKeySecret !== undefined) {
                grantReadLambdas(this.props.apiKeySecret, extractedLambdaFunctions);
            }
            else if (this.props.apiKeySecretArn !== undefined &&
                construct !== undefined &&
                baseProps.grantSecretReadAccess) {
                loglevel_1.default.debug("Granting read access to the provided Secret ARN for all your lambda functions.");
                grantReadLambdasFromSecretArn(construct, this.props.apiKeySecretArn, extractedLambdaFunctions);
            }
            const region = `${extractedLambdaFunctions[0].env.region}`;
            loglevel_1.default.debug(`Using region: ${region}`);
            if (baseProps.addLayers) {
                index_1.applyLayers(this.scope, region, extractedLambdaFunctions, this.props.pythonLayerVersion, this.props.nodeLayerVersion, this.props.javaLayerVersion, this.props.extensionLayerVersion, this.props.useLayersFromAccount);
            }
            if (baseProps.redirectHandler) {
                index_1.redirectHandlers(extractedLambdaFunctions, baseProps.addLayers);
            }
            if (this.props.forwarderArn !== undefined) {
                if (this.props.extensionLayerVersion !== undefined) {
                    loglevel_1.default.debug(`Skipping adding subscriptions to the lambda log groups since the extension is enabled`);
                }
                else {
                    loglevel_1.default.debug(`Adding log subscriptions using provided Forwarder ARN: ${this.props.forwarderArn}`);
                    index_1.addForwarder(this.scope, extractedLambdaFunctions, this.props.forwarderArn, this.props.createForwarderPermissions === true);
                }
            }
            else {
                loglevel_1.default.debug("Forwarder ARN not provided, no log group subscriptions will be added");
            }
            addCdkConstructVersionTag(extractedLambdaFunctions);
            index_1.applyEnvVariables(extractedLambdaFunctions, baseProps);
            index_1.setDDEnvVariables(extractedLambdaFunctions, this.props);
            setTags(extractedLambdaFunctions, this.props);
            this.transport.applyEnvVars(extractedLambdaFunctions);
            if (baseProps.sourceCodeIntegration) {
                this.addGitCommitMetadata(extractedLambdaFunctions);
            }
        }
    }
    // unused parameters gitCommitSha and gitRepoUrl are kept for backwards compatibility
    addGitCommitMetadata(lambdaFunctions, 
    // eslint-disable-next-line @typescript-eslint/ban-ts-comment
    // @ts-ignore
    gitCommitSha, 
    // eslint-disable-next-line @typescript-eslint/ban-ts-comment
    // @ts-ignore
    gitRepoUrl) {
        const extractedLambdaFunctions = extractSingletonFunctions(lambdaFunctions);
        index_1.setGitEnvironmentVariables(extractedLambdaFunctions);
    }
    addForwarderToNonLambdaLogGroups(logGroups) {
        if (this.props.forwarderArn !== undefined) {
            index_1.addForwarderToLogGroups(this.scope, logGroups, this.props.forwarderArn, this.props.createForwarderPermissions === true);
        }
        else {
            loglevel_1.default.debug("Forwarder ARN not provided, no non lambda log group subscriptions will be added");
        }
    }
}
exports.Datadog = Datadog;
_a = JSII_RTTI_SYMBOL_1;
Datadog[_a] = { fqn: "datadog-cdk-constructs-v2.Datadog", version: "1.8.1" };
function addCdkConstructVersionTag(lambdaFunctions) {
    loglevel_1.default.debug(`Adding CDK Construct version tag: ${versionJson.version}`);
    lambdaFunctions.forEach((functionName) => {
        aws_cdk_lib_1.Tags.of(functionName).add(index_1.TagKeys.CDK, `v${versionJson.version}`, {
            includeResourceTypes: ["AWS::Lambda::Function"],
        });
    });
}
exports.addCdkConstructVersionTag = addCdkConstructVersionTag;
function setTags(lambdaFunctions, props) {
    loglevel_1.default.debug(`Adding datadog tags`);
    lambdaFunctions.forEach((functionName) => {
        if (props.forwarderArn) {
            if (props.env) {
                aws_cdk_lib_1.Tags.of(functionName).add(index_1.TagKeys.ENV, props.env);
            }
            if (props.service) {
                aws_cdk_lib_1.Tags.of(functionName).add(index_1.TagKeys.SERVICE, props.service);
            }
            if (props.version) {
                aws_cdk_lib_1.Tags.of(functionName).add(index_1.TagKeys.VERSION, props.version);
            }
            if (props.tags) {
                const tagsArray = props.tags.split(",");
                tagsArray.forEach((tag) => {
                    const [key, value] = tag.split(":");
                    if (key && value) {
                        aws_cdk_lib_1.Tags.of(functionName).add(key, value);
                    }
                });
            }
        }
    });
}
function grantReadLambdas(secret, lambdaFunctions) {
    lambdaFunctions.forEach((functionName) => {
        secret.grantRead(functionName);
    });
}
function grantReadLambdasFromSecretArn(construct, arn, lambdaFunctions) {
    const secret = aws_secretsmanager_1.Secret.fromSecretPartialArn(construct, "DatadogApiKeySecret", arn);
    lambdaFunctions.forEach((functionName) => {
        secret.grantRead(functionName);
    });
}
function extractSingletonFunctions(lambdaFunctions) {
    // extract lambdaFunction property from Singleton Function
    // using bracket notation here since lambdaFunction is a private property
    const extractedLambdaFunctions = lambdaFunctions.map((fn) => {
        // eslint-disable-next-line dot-notation
        return isSingletonFunction(fn) ? fn["lambdaFunction"] : fn;
    });
    return extractedLambdaFunctions;
}
function isSingletonFunction(fn) {
    return fn.hasOwnProperty("lambdaFunction");
}
function validateProps(props, apiKeyArnOverride = false) {
    loglevel_1.default.debug("Validating props...");
    checkForMultipleApiKeys(props, apiKeyArnOverride);
    const siteList = [
        "datadoghq.com",
        "datadoghq.eu",
        "us3.datadoghq.com",
        "us5.datadoghq.com",
        "ap1.datadoghq.com",
        "ddog-gov.com",
    ];
    if (props.site !== undefined &&
        !siteList.includes(props.site.toLowerCase()) &&
        !(props.site.startsWith("${Token[") && props.site.endsWith("]}")) &&
        !process.env.DD_CDK_BYPASS_SITE_VALIDATION) {
        throw new Error("Warning: Invalid site URL. Must be either datadoghq.com, datadoghq.eu, us3.datadoghq.com, us5.datadoghq.com, ap1.datadoghq.com, or ddog-gov.com.");
    }
    if (props.apiKey === undefined &&
        props.apiKmsKey === undefined &&
        props.apiKeySecretArn === undefined &&
        props.flushMetricsToLogs === false &&
        !apiKeyArnOverride) {
        throw new Error("When `flushMetricsToLogs` is false, `apiKey`, `apiKeySecretArn`, or `apiKmsKey` must also be set.");
    }
    if (props.extensionLayerVersion !== undefined) {
        if (props.apiKey === undefined &&
            props.apiKeySecretArn === undefined &&
            props.apiKmsKey === undefined &&
            !apiKeyArnOverride) {
            throw new Error("When `extensionLayer` is set, `apiKey`, `apiKeySecretArn`, or `apiKmsKey` must also be set.");
        }
    }
    if (props.enableDatadogTracing === false && props.enableDatadogASM === true) {
        throw new Error("When `enableDatadogASM` is enabled, `enableDatadogTracing` must also be enabled.");
    }
}
exports.validateProps = validateProps;
function checkForMultipleApiKeys(props, apiKeyArnOverride = false) {
    let multipleApiKeysMessage;
    const apiKeyArnOrOverride = props.apiKeySecretArn !== undefined || apiKeyArnOverride;
    if (props.apiKey !== undefined && props.apiKmsKey !== undefined && apiKeyArnOrOverride) {
        multipleApiKeysMessage = "`apiKey`, `apiKmsKey`, and `apiKeySecretArn`";
    }
    else if (props.apiKey !== undefined && props.apiKmsKey !== undefined) {
        multipleApiKeysMessage = "`apiKey` and `apiKmsKey`";
    }
    else if (props.apiKey !== undefined && apiKeyArnOrOverride) {
        multipleApiKeysMessage = "`apiKey` and `apiKeySecretArn`";
    }
    else if (props.apiKmsKey !== undefined && apiKeyArnOrOverride) {
        multipleApiKeysMessage = "`apiKmsKey` and `apiKeySecretArn`";
    }
    if (multipleApiKeysMessage) {
        throw new Error(`${multipleApiKeysMessage} should not be set at the same time.`);
    }
}
exports.checkForMultipleApiKeys = checkForMultipleApiKeys;
function handleSettingPropDefaults(props) {
    let addLayers = props.addLayers;
    let enableDatadogTracing = props.enableDatadogTracing;
    let enableDatadogASM = props.enableDatadogASM;
    let enableMergeXrayTraces = props.enableMergeXrayTraces;
    let injectLogContext = props.injectLogContext;
    const logLevel = props.logLevel;
    let enableDatadogLogs = props.enableDatadogLogs;
    let captureLambdaPayload = props.captureLambdaPayload;
    let sourceCodeIntegration = props.sourceCodeIntegration;
    let redirectHandler = props.redirectHandler;
    let grantSecretReadAccess = props.grantSecretReadAccess;
    const extensionLayerVersion = props.extensionLayerVersion;
    if (addLayers === undefined) {
        loglevel_1.default.debug(`No value provided for addLayers, defaulting to ${index_1.DefaultDatadogProps.addLayers}`);
        addLayers = index_1.DefaultDatadogProps.addLayers;
    }
    if (enableDatadogTracing === undefined) {
        loglevel_1.default.debug(`No value provided for enableDatadogTracing, defaulting to ${index_1.DefaultDatadogProps.enableDatadogTracing}`);
        enableDatadogTracing = index_1.DefaultDatadogProps.enableDatadogTracing;
    }
    if (enableDatadogASM === undefined) {
        loglevel_1.default.debug(`No value provided for enableDatadogASM, defaulting to ${index_1.DefaultDatadogProps.enableDatadogASM}`);
        enableDatadogASM = index_1.DefaultDatadogProps.enableDatadogASM;
    }
    if (enableMergeXrayTraces === undefined) {
        loglevel_1.default.debug(`No value provided for enableMergeXrayTraces, defaulting to ${index_1.DefaultDatadogProps.enableMergeXrayTraces}`);
        enableMergeXrayTraces = index_1.DefaultDatadogProps.enableMergeXrayTraces;
    }
    if (injectLogContext === undefined) {
        loglevel_1.default.debug(`No value provided for injectLogContext, defaulting to ${index_1.DefaultDatadogProps.injectLogContext}`);
        injectLogContext = index_1.DefaultDatadogProps.injectLogContext;
    }
    if (logLevel === undefined) {
        loglevel_1.default.debug(`No value provided for logLevel`);
    }
    if (enableDatadogLogs === undefined) {
        loglevel_1.default.debug(`No value provided for enableDatadogLogs, defaulting to ${index_1.DefaultDatadogProps.enableDatadogLogs}`);
        enableDatadogLogs = index_1.DefaultDatadogProps.enableDatadogLogs;
    }
    if (captureLambdaPayload === undefined) {
        loglevel_1.default.debug(`No value provided for captureLambdaPayload, default to ${index_1.DefaultDatadogProps.captureLambdaPayload}`);
        captureLambdaPayload = index_1.DefaultDatadogProps.captureLambdaPayload;
    }
    if (sourceCodeIntegration === undefined) {
        loglevel_1.default.debug(`No value provided for sourceCodeIntegration, default to ${index_1.DefaultDatadogProps.sourceCodeIntegration}`);
        sourceCodeIntegration = index_1.DefaultDatadogProps.sourceCodeIntegration;
    }
    if (redirectHandler === undefined) {
        loglevel_1.default.debug(`No value provided for redirectHandler, default to ${index_1.DefaultDatadogProps.redirectHandler}`);
        redirectHandler = index_1.DefaultDatadogProps.redirectHandler;
    }
    if (grantSecretReadAccess === undefined) {
        loglevel_1.default.debug(`No value provided for grantSecretReadAccess, default to ${index_1.DefaultDatadogProps.grantSecretReadAccess}`);
        grantSecretReadAccess = index_1.DefaultDatadogProps.grantSecretReadAccess;
    }
    return {
        addLayers: addLayers,
        enableDatadogTracing: enableDatadogTracing,
        enableDatadogASM,
        enableMergeXrayTraces: enableMergeXrayTraces,
        injectLogContext: injectLogContext,
        logLevel: logLevel,
        enableDatadogLogs: enableDatadogLogs,
        captureLambdaPayload: captureLambdaPayload,
        sourceCodeIntegration: sourceCodeIntegration,
        redirectHandler: redirectHandler,
        grantSecretReadAccess: grantSecretReadAccess,
        extensionLayerVersion: extensionLayerVersion,
    };
}
exports.handleSettingPropDefaults = handleSettingPropDefaults;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGF0YWRvZy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9kYXRhZG9nLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUE7Ozs7OztHQU1HO0FBRUgsNkNBQW1DO0FBR25DLHVFQUFpRTtBQUNqRSwyQ0FBdUM7QUFDdkMsdUNBQTJCO0FBQzNCLG1DQWFpQjtBQUdqQixNQUFNLFdBQVcsR0FBRyxPQUFPLENBQUMsaUJBQWlCLENBQUMsQ0FBQztBQUUvQyxNQUFhLE9BQVEsU0FBUSxzQkFBUztJQUlwQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQW1CO1FBQzNELElBQUksT0FBTyxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsRUFBRSxXQUFXLEVBQUUsSUFBSSxNQUFNLEVBQUU7WUFDaEUsa0JBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUM7U0FDdkI7UUFDRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ2pCLElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDO1FBQ25CLElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDO1FBQ25CLElBQUksZUFBZSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsZUFBZSxDQUFDO1FBQ2pELElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZLEtBQUssU0FBUyxFQUFFO1lBQ3pDLGVBQWUsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxTQUFTLENBQUM7U0FDckQ7UUFDRCxhQUFhLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksS0FBSyxTQUFTLENBQUMsQ0FBQztRQUNqRSxJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksaUJBQVMsQ0FDNUIsSUFBSSxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsRUFDN0IsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQ2YsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLEVBQ2pCLGVBQWUsRUFDZixJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFDcEIsSUFBSSxDQUFDLEtBQUssQ0FBQyxxQkFBcUIsQ0FDakMsQ0FBQztJQUNKLENBQUM7SUFFTSxrQkFBa0IsQ0FBQyxlQUFpQyxFQUFFLFNBQXFCO1FBQ2hGLHdGQUF3RjtRQUN4RixzREFBc0Q7UUFDdEQsTUFBTSxTQUFTLEdBQXVCLHlCQUF5QixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUU1RSxNQUFNLHdCQUF3QixHQUFHLHlCQUF5QixDQUFDLGVBQWUsQ0FBQyxDQUFDO1FBRTVFLElBQUksSUFBSSxDQUFDLEtBQUssS0FBSyxTQUFTLElBQUksd0JBQXdCLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUNuRSxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxLQUFLLFNBQVMsRUFBRTtnQkFDekMsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZLEVBQUUsd0JBQXdCLENBQUMsQ0FBQzthQUNyRTtpQkFBTSxJQUNMLElBQUksQ0FBQyxLQUFLLENBQUMsZUFBZSxLQUFLLFNBQVM7Z0JBQ3hDLFNBQVMsS0FBSyxTQUFTO2dCQUN2QixTQUFTLENBQUMscUJBQXFCLEVBQy9CO2dCQUNBLGtCQUFHLENBQUMsS0FBSyxDQUFDLGdGQUFnRixDQUFDLENBQUM7Z0JBQzVGLDZCQUE2QixDQUFDLFNBQVMsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGVBQWUsRUFBRSx3QkFBd0IsQ0FBQyxDQUFDO2FBQ2hHO1lBRUQsTUFBTSxNQUFNLEdBQUcsR0FBRyx3QkFBd0IsQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsTUFBTSxFQUFFLENBQUM7WUFDM0Qsa0JBQUcsQ0FBQyxLQUFLLENBQUMsaUJBQWlCLE1BQU0sRUFBRSxDQUFDLENBQUM7WUFDckMsSUFBSSxTQUFTLENBQUMsU0FBUyxFQUFFO2dCQUN2QixtQkFBVyxDQUNULElBQUksQ0FBQyxLQUFLLEVBQ1YsTUFBTSxFQUNOLHdCQUF3QixFQUN4QixJQUFJLENBQUMsS0FBSyxDQUFDLGtCQUFrQixFQUM3QixJQUFJLENBQUMsS0FBSyxDQUFDLGdCQUFnQixFQUMzQixJQUFJLENBQUMsS0FBSyxDQUFDLGdCQUFnQixFQUMzQixJQUFJLENBQUMsS0FBSyxDQUFDLHFCQUFxQixFQUNoQyxJQUFJLENBQUMsS0FBSyxDQUFDLG9CQUFvQixDQUNoQyxDQUFDO2FBQ0g7WUFFRCxJQUFJLFNBQVMsQ0FBQyxlQUFlLEVBQUU7Z0JBQzdCLHdCQUFnQixDQUFDLHdCQUF3QixFQUFFLFNBQVMsQ0FBQyxTQUFTLENBQUMsQ0FBQzthQUNqRTtZQUVELElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZLEtBQUssU0FBUyxFQUFFO2dCQUN6QyxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMscUJBQXFCLEtBQUssU0FBUyxFQUFFO29CQUNsRCxrQkFBRyxDQUFDLEtBQUssQ0FBQyx1RkFBdUYsQ0FBQyxDQUFDO2lCQUNwRztxQkFBTTtvQkFDTCxrQkFBRyxDQUFDLEtBQUssQ0FBQywwREFBMEQsSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZLEVBQUUsQ0FBQyxDQUFDO29CQUMvRixvQkFBWSxDQUNWLElBQUksQ0FBQyxLQUFLLEVBQ1Ysd0JBQXdCLEVBQ3hCLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxFQUN2QixJQUFJLENBQUMsS0FBSyxDQUFDLDBCQUEwQixLQUFLLElBQUksQ0FDL0MsQ0FBQztpQkFDSDthQUNGO2lCQUFNO2dCQUNMLGtCQUFHLENBQUMsS0FBSyxDQUFDLHNFQUFzRSxDQUFDLENBQUM7YUFDbkY7WUFFRCx5QkFBeUIsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1lBRXBELHlCQUFpQixDQUFDLHdCQUF3QixFQUFFLFNBQVMsQ0FBQyxDQUFDO1lBQ3ZELHlCQUFpQixDQUFDLHdCQUF3QixFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUN4RCxPQUFPLENBQUMsd0JBQXdCLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBRTlDLElBQUksQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDLHdCQUF3QixDQUFDLENBQUM7WUFFdEQsSUFBSSxTQUFTLENBQUMscUJBQXFCLEVBQUU7Z0JBQ25DLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO2FBQ3JEO1NBQ0Y7SUFDSCxDQUFDO0lBRUQscUZBQXFGO0lBQzlFLG9CQUFvQixDQUN6QixlQUFpQztJQUNqQyw2REFBNkQ7SUFDN0QsYUFBYTtJQUNiLFlBQXFCO0lBQ3JCLDZEQUE2RDtJQUM3RCxhQUFhO0lBQ2IsVUFBbUI7UUFFbkIsTUFBTSx3QkFBd0IsR0FBRyx5QkFBeUIsQ0FBQyxlQUFlLENBQUMsQ0FBQztRQUM1RSxrQ0FBMEIsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQ3ZELENBQUM7SUFFTSxnQ0FBZ0MsQ0FBQyxTQUEyQjtRQUNqRSxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxLQUFLLFNBQVMsRUFBRTtZQUN6QywrQkFBdUIsQ0FDckIsSUFBSSxDQUFDLEtBQUssRUFDVixTQUFTLEVBQ1QsSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZLEVBQ3ZCLElBQUksQ0FBQyxLQUFLLENBQUMsMEJBQTBCLEtBQUssSUFBSSxDQUMvQyxDQUFDO1NBQ0g7YUFBTTtZQUNMLGtCQUFHLENBQUMsS0FBSyxDQUFDLGlGQUFpRixDQUFDLENBQUM7U0FDOUY7SUFDSCxDQUFDOztBQXZISCwwQkF3SEM7OztBQUVELFNBQWdCLHlCQUF5QixDQUFDLGVBQWtDO0lBQzFFLGtCQUFHLENBQUMsS0FBSyxDQUFDLHFDQUFxQyxXQUFXLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQztJQUN0RSxlQUFlLENBQUMsT0FBTyxDQUFDLENBQUMsWUFBWSxFQUFFLEVBQUU7UUFDdkMsa0JBQUksQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLENBQUMsR0FBRyxDQUFDLGVBQU8sQ0FBQyxHQUFHLEVBQUUsSUFBSSxXQUFXLENBQUMsT0FBTyxFQUFFLEVBQUU7WUFDaEUsb0JBQW9CLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQztTQUNoRCxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUM7QUFQRCw4REFPQztBQUVELFNBQVMsT0FBTyxDQUFDLGVBQWtDLEVBQUUsS0FBbUI7SUFDdEUsa0JBQUcsQ0FBQyxLQUFLLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUNqQyxlQUFlLENBQUMsT0FBTyxDQUFDLENBQUMsWUFBWSxFQUFFLEVBQUU7UUFDdkMsSUFBSSxLQUFLLENBQUMsWUFBWSxFQUFFO1lBQ3RCLElBQUksS0FBSyxDQUFDLEdBQUcsRUFBRTtnQkFDYixrQkFBSSxDQUFDLEVBQUUsQ0FBQyxZQUFZLENBQUMsQ0FBQyxHQUFHLENBQUMsZUFBTyxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7YUFDbkQ7WUFDRCxJQUFJLEtBQUssQ0FBQyxPQUFPLEVBQUU7Z0JBQ2pCLGtCQUFJLENBQUMsRUFBRSxDQUFDLFlBQVksQ0FBQyxDQUFDLEdBQUcsQ0FBQyxlQUFPLENBQUMsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQzthQUMzRDtZQUNELElBQUksS0FBSyxDQUFDLE9BQU8sRUFBRTtnQkFDakIsa0JBQUksQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLENBQUMsR0FBRyxDQUFDLGVBQU8sQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO2FBQzNEO1lBQ0QsSUFBSSxLQUFLLENBQUMsSUFBSSxFQUFFO2dCQUNkLE1BQU0sU0FBUyxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO2dCQUN4QyxTQUFTLENBQUMsT0FBTyxDQUFDLENBQUMsR0FBVyxFQUFFLEVBQUU7b0JBQ2hDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztvQkFDcEMsSUFBSSxHQUFHLElBQUksS0FBSyxFQUFFO3dCQUNoQixrQkFBSSxDQUFDLEVBQUUsQ0FBQyxZQUFZLENBQUMsQ0FBQyxHQUFHLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxDQUFDO3FCQUN2QztnQkFDSCxDQUFDLENBQUMsQ0FBQzthQUNKO1NBQ0Y7SUFDSCxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUM7QUFFRCxTQUFTLGdCQUFnQixDQUFDLE1BQWUsRUFBRSxlQUFrQztJQUMzRSxlQUFlLENBQUMsT0FBTyxDQUFDLENBQUMsWUFBWSxFQUFFLEVBQUU7UUFDdkMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxZQUFZLENBQUMsQ0FBQztJQUNqQyxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUM7QUFFRCxTQUFTLDZCQUE2QixDQUFDLFNBQW9CLEVBQUUsR0FBVyxFQUFFLGVBQWtDO0lBQzFHLE1BQU0sTUFBTSxHQUFHLDJCQUFNLENBQUMsb0JBQW9CLENBQUMsU0FBUyxFQUFFLHFCQUFxQixFQUFFLEdBQUcsQ0FBQyxDQUFDO0lBQ2xGLGVBQWUsQ0FBQyxPQUFPLENBQUMsQ0FBQyxZQUFZLEVBQUUsRUFBRTtRQUN2QyxNQUFNLENBQUMsU0FBUyxDQUFDLFlBQVksQ0FBQyxDQUFDO0lBQ2pDLENBQUMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQztBQUVELFNBQVMseUJBQXlCLENBQUMsZUFBaUM7SUFDbEUsMERBQTBEO0lBQzFELHlFQUF5RTtJQUN6RSxNQUFNLHdCQUF3QixHQUFzQixlQUFlLENBQUMsR0FBRyxDQUFDLENBQUMsRUFBRSxFQUFFLEVBQUU7UUFDN0Usd0NBQXdDO1FBQ3hDLE9BQU8sbUJBQW1CLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7SUFDN0QsQ0FBQyxDQUFDLENBQUM7SUFFSCxPQUFPLHdCQUF3QixDQUFDO0FBQ2xDLENBQUM7QUFFRCxTQUFTLG1CQUFtQixDQUFDLEVBQWtCO0lBQzdDLE9BQU8sRUFBRSxDQUFDLGNBQWMsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO0FBQzdDLENBQUM7QUFFRCxTQUFnQixhQUFhLENBQUMsS0FBbUIsRUFBRSxpQkFBaUIsR0FBRyxLQUFLO0lBQzFFLGtCQUFHLENBQUMsS0FBSyxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFFakMsdUJBQXVCLENBQUMsS0FBSyxFQUFFLGlCQUFpQixDQUFDLENBQUM7SUFDbEQsTUFBTSxRQUFRLEdBQWE7UUFDekIsZUFBZTtRQUNmLGNBQWM7UUFDZCxtQkFBbUI7UUFDbkIsbUJBQW1CO1FBQ25CLG1CQUFtQjtRQUNuQixjQUFjO0tBQ2YsQ0FBQztJQUNGLElBQ0UsS0FBSyxDQUFDLElBQUksS0FBSyxTQUFTO1FBQ3hCLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDO1FBQzVDLENBQUMsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxVQUFVLENBQUMsSUFBSSxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNqRSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsNkJBQTZCLEVBQzFDO1FBQ0EsTUFBTSxJQUFJLEtBQUssQ0FDYixrSkFBa0osQ0FDbkosQ0FBQztLQUNIO0lBRUQsSUFDRSxLQUFLLENBQUMsTUFBTSxLQUFLLFNBQVM7UUFDMUIsS0FBSyxDQUFDLFNBQVMsS0FBSyxTQUFTO1FBQzdCLEtBQUssQ0FBQyxlQUFlLEtBQUssU0FBUztRQUNuQyxLQUFLLENBQUMsa0JBQWtCLEtBQUssS0FBSztRQUNsQyxDQUFDLGlCQUFpQixFQUNsQjtRQUNBLE1BQU0sSUFBSSxLQUFLLENBQ2IsbUdBQW1HLENBQ3BHLENBQUM7S0FDSDtJQUNELElBQUksS0FBSyxDQUFDLHFCQUFxQixLQUFLLFNBQVMsRUFBRTtRQUM3QyxJQUNFLEtBQUssQ0FBQyxNQUFNLEtBQUssU0FBUztZQUMxQixLQUFLLENBQUMsZUFBZSxLQUFLLFNBQVM7WUFDbkMsS0FBSyxDQUFDLFNBQVMsS0FBSyxTQUFTO1lBQzdCLENBQUMsaUJBQWlCLEVBQ2xCO1lBQ0EsTUFBTSxJQUFJLEtBQUssQ0FBQyw2RkFBNkYsQ0FBQyxDQUFDO1NBQ2hIO0tBQ0Y7SUFDRCxJQUFJLEtBQUssQ0FBQyxvQkFBb0IsS0FBSyxLQUFLLElBQUksS0FBSyxDQUFDLGdCQUFnQixLQUFLLElBQUksRUFBRTtRQUMzRSxNQUFNLElBQUksS0FBSyxDQUFDLGtGQUFrRixDQUFDLENBQUM7S0FDckc7QUFDSCxDQUFDO0FBL0NELHNDQStDQztBQUVELFNBQWdCLHVCQUF1QixDQUFDLEtBQW1CLEVBQUUsaUJBQWlCLEdBQUcsS0FBSztJQUNwRixJQUFJLHNCQUFzQixDQUFDO0lBQzNCLE1BQU0sbUJBQW1CLEdBQUcsS0FBSyxDQUFDLGVBQWUsS0FBSyxTQUFTLElBQUksaUJBQWlCLENBQUM7SUFDckYsSUFBSSxLQUFLLENBQUMsTUFBTSxLQUFLLFNBQVMsSUFBSSxLQUFLLENBQUMsU0FBUyxLQUFLLFNBQVMsSUFBSSxtQkFBbUIsRUFBRTtRQUN0RixzQkFBc0IsR0FBRyw4Q0FBOEMsQ0FBQztLQUN6RTtTQUFNLElBQUksS0FBSyxDQUFDLE1BQU0sS0FBSyxTQUFTLElBQUksS0FBSyxDQUFDLFNBQVMsS0FBSyxTQUFTLEVBQUU7UUFDdEUsc0JBQXNCLEdBQUcsMEJBQTBCLENBQUM7S0FDckQ7U0FBTSxJQUFJLEtBQUssQ0FBQyxNQUFNLEtBQUssU0FBUyxJQUFJLG1CQUFtQixFQUFFO1FBQzVELHNCQUFzQixHQUFHLGdDQUFnQyxDQUFDO0tBQzNEO1NBQU0sSUFBSSxLQUFLLENBQUMsU0FBUyxLQUFLLFNBQVMsSUFBSSxtQkFBbUIsRUFBRTtRQUMvRCxzQkFBc0IsR0FBRyxtQ0FBbUMsQ0FBQztLQUM5RDtJQUVELElBQUksc0JBQXNCLEVBQUU7UUFDMUIsTUFBTSxJQUFJLEtBQUssQ0FBQyxHQUFHLHNCQUFzQixzQ0FBc0MsQ0FBQyxDQUFDO0tBQ2xGO0FBQ0gsQ0FBQztBQWhCRCwwREFnQkM7QUFFRCxTQUFnQix5QkFBeUIsQ0FBQyxLQUFtQjtJQUMzRCxJQUFJLFNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO0lBQ2hDLElBQUksb0JBQW9CLEdBQUcsS0FBSyxDQUFDLG9CQUFvQixDQUFDO0lBQ3RELElBQUksZ0JBQWdCLEdBQUcsS0FBSyxDQUFDLGdCQUFnQixDQUFDO0lBQzlDLElBQUkscUJBQXFCLEdBQUcsS0FBSyxDQUFDLHFCQUFxQixDQUFDO0lBQ3hELElBQUksZ0JBQWdCLEdBQUcsS0FBSyxDQUFDLGdCQUFnQixDQUFDO0lBQzlDLE1BQU0sUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUM7SUFDaEMsSUFBSSxpQkFBaUIsR0FBRyxLQUFLLENBQUMsaUJBQWlCLENBQUM7SUFDaEQsSUFBSSxvQkFBb0IsR0FBRyxLQUFLLENBQUMsb0JBQW9CLENBQUM7SUFDdEQsSUFBSSxxQkFBcUIsR0FBRyxLQUFLLENBQUMscUJBQXFCLENBQUM7SUFDeEQsSUFBSSxlQUFlLEdBQUcsS0FBSyxDQUFDLGVBQWUsQ0FBQztJQUM1QyxJQUFJLHFCQUFxQixHQUFHLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQztJQUN4RCxNQUFNLHFCQUFxQixHQUFHLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQztJQUUxRCxJQUFJLFNBQVMsS0FBSyxTQUFTLEVBQUU7UUFDM0Isa0JBQUcsQ0FBQyxLQUFLLENBQUMsa0RBQWtELDJCQUFtQixDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUM7UUFDN0YsU0FBUyxHQUFHLDJCQUFtQixDQUFDLFNBQVMsQ0FBQztLQUMzQztJQUNELElBQUksb0JBQW9CLEtBQUssU0FBUyxFQUFFO1FBQ3RDLGtCQUFHLENBQUMsS0FBSyxDQUFDLDZEQUE2RCwyQkFBbUIsQ0FBQyxvQkFBb0IsRUFBRSxDQUFDLENBQUM7UUFDbkgsb0JBQW9CLEdBQUcsMkJBQW1CLENBQUMsb0JBQW9CLENBQUM7S0FDakU7SUFDRCxJQUFJLGdCQUFnQixLQUFLLFNBQVMsRUFBRTtRQUNsQyxrQkFBRyxDQUFDLEtBQUssQ0FBQyx5REFBeUQsMkJBQW1CLENBQUMsZ0JBQWdCLEVBQUUsQ0FBQyxDQUFDO1FBQzNHLGdCQUFnQixHQUFHLDJCQUFtQixDQUFDLGdCQUFnQixDQUFDO0tBQ3pEO0lBQ0QsSUFBSSxxQkFBcUIsS0FBSyxTQUFTLEVBQUU7UUFDdkMsa0JBQUcsQ0FBQyxLQUFLLENBQ1AsOERBQThELDJCQUFtQixDQUFDLHFCQUFxQixFQUFFLENBQzFHLENBQUM7UUFDRixxQkFBcUIsR0FBRywyQkFBbUIsQ0FBQyxxQkFBcUIsQ0FBQztLQUNuRTtJQUNELElBQUksZ0JBQWdCLEtBQUssU0FBUyxFQUFFO1FBQ2xDLGtCQUFHLENBQUMsS0FBSyxDQUFDLHlEQUF5RCwyQkFBbUIsQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDLENBQUM7UUFDM0csZ0JBQWdCLEdBQUcsMkJBQW1CLENBQUMsZ0JBQWdCLENBQUM7S0FDekQ7SUFDRCxJQUFJLFFBQVEsS0FBSyxTQUFTLEVBQUU7UUFDMUIsa0JBQUcsQ0FBQyxLQUFLLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztLQUM3QztJQUNELElBQUksaUJBQWlCLEtBQUssU0FBUyxFQUFFO1FBQ25DLGtCQUFHLENBQUMsS0FBSyxDQUFDLDBEQUEwRCwyQkFBbUIsQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLENBQUM7UUFDN0csaUJBQWlCLEdBQUcsMkJBQW1CLENBQUMsaUJBQWlCLENBQUM7S0FDM0Q7SUFDRCxJQUFJLG9CQUFvQixLQUFLLFNBQVMsRUFBRTtRQUN0QyxrQkFBRyxDQUFDLEtBQUssQ0FBQywwREFBMEQsMkJBQW1CLENBQUMsb0JBQW9CLEVBQUUsQ0FBQyxDQUFDO1FBQ2hILG9CQUFvQixHQUFHLDJCQUFtQixDQUFDLG9CQUFvQixDQUFDO0tBQ2pFO0lBQ0QsSUFBSSxxQkFBcUIsS0FBSyxTQUFTLEVBQUU7UUFDdkMsa0JBQUcsQ0FBQyxLQUFLLENBQUMsMkRBQTJELDJCQUFtQixDQUFDLHFCQUFxQixFQUFFLENBQUMsQ0FBQztRQUNsSCxxQkFBcUIsR0FBRywyQkFBbUIsQ0FBQyxxQkFBcUIsQ0FBQztLQUNuRTtJQUVELElBQUksZUFBZSxLQUFLLFNBQVMsRUFBRTtRQUNqQyxrQkFBRyxDQUFDLEtBQUssQ0FBQyxxREFBcUQsMkJBQW1CLENBQUMsZUFBZSxFQUFFLENBQUMsQ0FBQztRQUN0RyxlQUFlLEdBQUcsMkJBQW1CLENBQUMsZUFBZSxDQUFDO0tBQ3ZEO0lBRUQsSUFBSSxxQkFBcUIsS0FBSyxTQUFTLEVBQUU7UUFDdkMsa0JBQUcsQ0FBQyxLQUFLLENBQUMsMkRBQTJELDJCQUFtQixDQUFDLHFCQUFxQixFQUFFLENBQUMsQ0FBQztRQUNsSCxxQkFBcUIsR0FBRywyQkFBbUIsQ0FBQyxxQkFBcUIsQ0FBQztLQUNuRTtJQUVELE9BQU87UUFDTCxTQUFTLEVBQUUsU0FBUztRQUNwQixvQkFBb0IsRUFBRSxvQkFBb0I7UUFDMUMsZ0JBQWdCO1FBQ2hCLHFCQUFxQixFQUFFLHFCQUFxQjtRQUM1QyxnQkFBZ0IsRUFBRSxnQkFBZ0I7UUFDbEMsUUFBUSxFQUFFLFFBQVE7UUFDbEIsaUJBQWlCLEVBQUUsaUJBQWlCO1FBQ3BDLG9CQUFvQixFQUFFLG9CQUFvQjtRQUMxQyxxQkFBcUIsRUFBRSxxQkFBcUI7UUFDNUMsZUFBZSxFQUFFLGVBQWU7UUFDaEMscUJBQXFCLEVBQUUscUJBQXFCO1FBQzVDLHFCQUFxQixFQUFFLHFCQUFxQjtLQUM3QyxDQUFDO0FBQ0osQ0FBQztBQTVFRCw4REE0RUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKlxuICogVW5sZXNzIGV4cGxpY2l0bHkgc3RhdGVkIG90aGVyd2lzZSBhbGwgZmlsZXMgaW4gdGhpcyByZXBvc2l0b3J5IGFyZSBsaWNlbnNlZFxuICogdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlIFZlcnNpb24gMi4wLlxuICpcbiAqIFRoaXMgcHJvZHVjdCBpbmNsdWRlcyBzb2Z0d2FyZSBkZXZlbG9wZWQgYXQgRGF0YWRvZyAoaHR0cHM6Ly93d3cuZGF0YWRvZ2hxLmNvbS8pLlxuICogQ29weXJpZ2h0IDIwMjEgRGF0YWRvZywgSW5jLlxuICovXG5cbmltcG9ydCB7IFRhZ3MgfSBmcm9tIFwiYXdzLWNkay1saWJcIjtcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLWxhbWJkYVwiO1xuaW1wb3J0ICogYXMgbG9ncyBmcm9tIFwiYXdzLWNkay1saWIvYXdzLWxvZ3NcIjtcbmltcG9ydCB7IElTZWNyZXQsIFNlY3JldCB9IGZyb20gXCJhd3MtY2RrLWxpYi9hd3Mtc2VjcmV0c21hbmFnZXJcIjtcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gXCJjb25zdHJ1Y3RzXCI7XG5pbXBvcnQgbG9nIGZyb20gXCJsb2dsZXZlbFwiO1xuaW1wb3J0IHtcbiAgYXBwbHlMYXllcnMsXG4gIHJlZGlyZWN0SGFuZGxlcnMsXG4gIGFkZEZvcndhcmRlcixcbiAgYWRkRm9yd2FyZGVyVG9Mb2dHcm91cHMsXG4gIGFwcGx5RW52VmFyaWFibGVzLFxuICBUYWdLZXlzLFxuICBEYXRhZG9nU3RyaWN0UHJvcHMsXG4gIHNldEdpdEVudmlyb25tZW50VmFyaWFibGVzLFxuICBzZXREREVudlZhcmlhYmxlcyxcbiAgRGVmYXVsdERhdGFkb2dQcm9wcyxcbiAgRGF0YWRvZ1Byb3BzLFxuICBUcmFuc3BvcnQsXG59IGZyb20gXCIuL2luZGV4XCI7XG5pbXBvcnQgeyBMYW1iZGFGdW5jdGlvbiB9IGZyb20gXCIuL2ludGVyZmFjZXNcIjtcblxuY29uc3QgdmVyc2lvbkpzb24gPSByZXF1aXJlKFwiLi4vdmVyc2lvbi5qc29uXCIpO1xuXG5leHBvcnQgY2xhc3MgRGF0YWRvZyBleHRlbmRzIENvbnN0cnVjdCB7XG4gIHNjb3BlOiBDb25zdHJ1Y3Q7XG4gIHByb3BzOiBEYXRhZG9nUHJvcHM7XG4gIHRyYW5zcG9ydDogVHJhbnNwb3J0O1xuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogRGF0YWRvZ1Byb3BzKSB7XG4gICAgaWYgKHByb2Nlc3MuZW52LkREX0NPTlNUUlVDVF9ERUJVR19MT0dTPy50b0xvd2VyQ2FzZSgpID09IFwidHJ1ZVwiKSB7XG4gICAgICBsb2cuc2V0TGV2ZWwoXCJkZWJ1Z1wiKTtcbiAgICB9XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICB0aGlzLnNjb3BlID0gc2NvcGU7XG4gICAgdGhpcy5wcm9wcyA9IHByb3BzO1xuICAgIGxldCBhcGlLZXlTZWNyZXRBcm4gPSB0aGlzLnByb3BzLmFwaUtleVNlY3JldEFybjtcbiAgICBpZiAodGhpcy5wcm9wcy5hcGlLZXlTZWNyZXQgIT09IHVuZGVmaW5lZCkge1xuICAgICAgYXBpS2V5U2VjcmV0QXJuID0gdGhpcy5wcm9wcy5hcGlLZXlTZWNyZXQuc2VjcmV0QXJuO1xuICAgIH1cbiAgICB2YWxpZGF0ZVByb3BzKHRoaXMucHJvcHMsIHRoaXMucHJvcHMuYXBpS2V5U2VjcmV0ICE9PSB1bmRlZmluZWQpO1xuICAgIHRoaXMudHJhbnNwb3J0ID0gbmV3IFRyYW5zcG9ydChcbiAgICAgIHRoaXMucHJvcHMuZmx1c2hNZXRyaWNzVG9Mb2dzLFxuICAgICAgdGhpcy5wcm9wcy5zaXRlLFxuICAgICAgdGhpcy5wcm9wcy5hcGlLZXksXG4gICAgICBhcGlLZXlTZWNyZXRBcm4sXG4gICAgICB0aGlzLnByb3BzLmFwaUttc0tleSxcbiAgICAgIHRoaXMucHJvcHMuZXh0ZW5zaW9uTGF5ZXJWZXJzaW9uLFxuICAgICk7XG4gIH1cblxuICBwdWJsaWMgYWRkTGFtYmRhRnVuY3Rpb25zKGxhbWJkYUZ1bmN0aW9uczogTGFtYmRhRnVuY3Rpb25bXSwgY29uc3RydWN0PzogQ29uc3RydWN0KSB7XG4gICAgLy8gYmFzZVByb3BzIGNvbnRhaW5zIGFsbCBwcm9wZXJ0aWVzIHNldCBieSB0aGUgdXNlciwgd2l0aCBkZWZhdWx0IHZhbHVlcyBmb3IgcHJvcGVydGllc1xuICAgIC8vIGRlZmluZWQgaW4gRGVmYXVsdERhdGFkb2dQcm9wcyAoaWYgbm90IHNldCBieSB1c2VyKVxuICAgIGNvbnN0IGJhc2VQcm9wczogRGF0YWRvZ1N0cmljdFByb3BzID0gaGFuZGxlU2V0dGluZ1Byb3BEZWZhdWx0cyh0aGlzLnByb3BzKTtcblxuICAgIGNvbnN0IGV4dHJhY3RlZExhbWJkYUZ1bmN0aW9ucyA9IGV4dHJhY3RTaW5nbGV0b25GdW5jdGlvbnMobGFtYmRhRnVuY3Rpb25zKTtcblxuICAgIGlmICh0aGlzLnByb3BzICE9PSB1bmRlZmluZWQgJiYgZXh0cmFjdGVkTGFtYmRhRnVuY3Rpb25zLmxlbmd0aCA+IDApIHtcbiAgICAgIGlmICh0aGlzLnByb3BzLmFwaUtleVNlY3JldCAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIGdyYW50UmVhZExhbWJkYXModGhpcy5wcm9wcy5hcGlLZXlTZWNyZXQsIGV4dHJhY3RlZExhbWJkYUZ1bmN0aW9ucyk7XG4gICAgICB9IGVsc2UgaWYgKFxuICAgICAgICB0aGlzLnByb3BzLmFwaUtleVNlY3JldEFybiAhPT0gdW5kZWZpbmVkICYmXG4gICAgICAgIGNvbnN0cnVjdCAhPT0gdW5kZWZpbmVkICYmXG4gICAgICAgIGJhc2VQcm9wcy5ncmFudFNlY3JldFJlYWRBY2Nlc3NcbiAgICAgICkge1xuICAgICAgICBsb2cuZGVidWcoXCJHcmFudGluZyByZWFkIGFjY2VzcyB0byB0aGUgcHJvdmlkZWQgU2VjcmV0IEFSTiBmb3IgYWxsIHlvdXIgbGFtYmRhIGZ1bmN0aW9ucy5cIik7XG4gICAgICAgIGdyYW50UmVhZExhbWJkYXNGcm9tU2VjcmV0QXJuKGNvbnN0cnVjdCwgdGhpcy5wcm9wcy5hcGlLZXlTZWNyZXRBcm4sIGV4dHJhY3RlZExhbWJkYUZ1bmN0aW9ucyk7XG4gICAgICB9XG5cbiAgICAgIGNvbnN0IHJlZ2lvbiA9IGAke2V4dHJhY3RlZExhbWJkYUZ1bmN0aW9uc1swXS5lbnYucmVnaW9ufWA7XG4gICAgICBsb2cuZGVidWcoYFVzaW5nIHJlZ2lvbjogJHtyZWdpb259YCk7XG4gICAgICBpZiAoYmFzZVByb3BzLmFkZExheWVycykge1xuICAgICAgICBhcHBseUxheWVycyhcbiAgICAgICAgICB0aGlzLnNjb3BlLFxuICAgICAgICAgIHJlZ2lvbixcbiAgICAgICAgICBleHRyYWN0ZWRMYW1iZGFGdW5jdGlvbnMsXG4gICAgICAgICAgdGhpcy5wcm9wcy5weXRob25MYXllclZlcnNpb24sXG4gICAgICAgICAgdGhpcy5wcm9wcy5ub2RlTGF5ZXJWZXJzaW9uLFxuICAgICAgICAgIHRoaXMucHJvcHMuamF2YUxheWVyVmVyc2lvbixcbiAgICAgICAgICB0aGlzLnByb3BzLmV4dGVuc2lvbkxheWVyVmVyc2lvbixcbiAgICAgICAgICB0aGlzLnByb3BzLnVzZUxheWVyc0Zyb21BY2NvdW50LFxuICAgICAgICApO1xuICAgICAgfVxuXG4gICAgICBpZiAoYmFzZVByb3BzLnJlZGlyZWN0SGFuZGxlcikge1xuICAgICAgICByZWRpcmVjdEhhbmRsZXJzKGV4dHJhY3RlZExhbWJkYUZ1bmN0aW9ucywgYmFzZVByb3BzLmFkZExheWVycyk7XG4gICAgICB9XG5cbiAgICAgIGlmICh0aGlzLnByb3BzLmZvcndhcmRlckFybiAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIGlmICh0aGlzLnByb3BzLmV4dGVuc2lvbkxheWVyVmVyc2lvbiAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgbG9nLmRlYnVnKGBTa2lwcGluZyBhZGRpbmcgc3Vic2NyaXB0aW9ucyB0byB0aGUgbGFtYmRhIGxvZyBncm91cHMgc2luY2UgdGhlIGV4dGVuc2lvbiBpcyBlbmFibGVkYCk7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgbG9nLmRlYnVnKGBBZGRpbmcgbG9nIHN1YnNjcmlwdGlvbnMgdXNpbmcgcHJvdmlkZWQgRm9yd2FyZGVyIEFSTjogJHt0aGlzLnByb3BzLmZvcndhcmRlckFybn1gKTtcbiAgICAgICAgICBhZGRGb3J3YXJkZXIoXG4gICAgICAgICAgICB0aGlzLnNjb3BlLFxuICAgICAgICAgICAgZXh0cmFjdGVkTGFtYmRhRnVuY3Rpb25zLFxuICAgICAgICAgICAgdGhpcy5wcm9wcy5mb3J3YXJkZXJBcm4sXG4gICAgICAgICAgICB0aGlzLnByb3BzLmNyZWF0ZUZvcndhcmRlclBlcm1pc3Npb25zID09PSB0cnVlLFxuICAgICAgICAgICk7XG4gICAgICAgIH1cbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIGxvZy5kZWJ1ZyhcIkZvcndhcmRlciBBUk4gbm90IHByb3ZpZGVkLCBubyBsb2cgZ3JvdXAgc3Vic2NyaXB0aW9ucyB3aWxsIGJlIGFkZGVkXCIpO1xuICAgICAgfVxuXG4gICAgICBhZGRDZGtDb25zdHJ1Y3RWZXJzaW9uVGFnKGV4dHJhY3RlZExhbWJkYUZ1bmN0aW9ucyk7XG5cbiAgICAgIGFwcGx5RW52VmFyaWFibGVzKGV4dHJhY3RlZExhbWJkYUZ1bmN0aW9ucywgYmFzZVByb3BzKTtcbiAgICAgIHNldERERW52VmFyaWFibGVzKGV4dHJhY3RlZExhbWJkYUZ1bmN0aW9ucywgdGhpcy5wcm9wcyk7XG4gICAgICBzZXRUYWdzKGV4dHJhY3RlZExhbWJkYUZ1bmN0aW9ucywgdGhpcy5wcm9wcyk7XG5cbiAgICAgIHRoaXMudHJhbnNwb3J0LmFwcGx5RW52VmFycyhleHRyYWN0ZWRMYW1iZGFGdW5jdGlvbnMpO1xuXG4gICAgICBpZiAoYmFzZVByb3BzLnNvdXJjZUNvZGVJbnRlZ3JhdGlvbikge1xuICAgICAgICB0aGlzLmFkZEdpdENvbW1pdE1ldGFkYXRhKGV4dHJhY3RlZExhbWJkYUZ1bmN0aW9ucyk7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgLy8gdW51c2VkIHBhcmFtZXRlcnMgZ2l0Q29tbWl0U2hhIGFuZCBnaXRSZXBvVXJsIGFyZSBrZXB0IGZvciBiYWNrd2FyZHMgY29tcGF0aWJpbGl0eVxuICBwdWJsaWMgYWRkR2l0Q29tbWl0TWV0YWRhdGEoXG4gICAgbGFtYmRhRnVuY3Rpb25zOiBMYW1iZGFGdW5jdGlvbltdLFxuICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBAdHlwZXNjcmlwdC1lc2xpbnQvYmFuLXRzLWNvbW1lbnRcbiAgICAvLyBAdHMtaWdub3JlXG4gICAgZ2l0Q29tbWl0U2hhPzogc3RyaW5nLFxuICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBAdHlwZXNjcmlwdC1lc2xpbnQvYmFuLXRzLWNvbW1lbnRcbiAgICAvLyBAdHMtaWdub3JlXG4gICAgZ2l0UmVwb1VybD86IHN0cmluZyxcbiAgKSB7XG4gICAgY29uc3QgZXh0cmFjdGVkTGFtYmRhRnVuY3Rpb25zID0gZXh0cmFjdFNpbmdsZXRvbkZ1bmN0aW9ucyhsYW1iZGFGdW5jdGlvbnMpO1xuICAgIHNldEdpdEVudmlyb25tZW50VmFyaWFibGVzKGV4dHJhY3RlZExhbWJkYUZ1bmN0aW9ucyk7XG4gIH1cblxuICBwdWJsaWMgYWRkRm9yd2FyZGVyVG9Ob25MYW1iZGFMb2dHcm91cHMobG9nR3JvdXBzOiBsb2dzLklMb2dHcm91cFtdKSB7XG4gICAgaWYgKHRoaXMucHJvcHMuZm9yd2FyZGVyQXJuICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIGFkZEZvcndhcmRlclRvTG9nR3JvdXBzKFxuICAgICAgICB0aGlzLnNjb3BlLFxuICAgICAgICBsb2dHcm91cHMsXG4gICAgICAgIHRoaXMucHJvcHMuZm9yd2FyZGVyQXJuLFxuICAgICAgICB0aGlzLnByb3BzLmNyZWF0ZUZvcndhcmRlclBlcm1pc3Npb25zID09PSB0cnVlLFxuICAgICAgKTtcbiAgICB9IGVsc2Uge1xuICAgICAgbG9nLmRlYnVnKFwiRm9yd2FyZGVyIEFSTiBub3QgcHJvdmlkZWQsIG5vIG5vbiBsYW1iZGEgbG9nIGdyb3VwIHN1YnNjcmlwdGlvbnMgd2lsbCBiZSBhZGRlZFwiKTtcbiAgICB9XG4gIH1cbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGFkZENka0NvbnN0cnVjdFZlcnNpb25UYWcobGFtYmRhRnVuY3Rpb25zOiBsYW1iZGEuRnVuY3Rpb25bXSkge1xuICBsb2cuZGVidWcoYEFkZGluZyBDREsgQ29uc3RydWN0IHZlcnNpb24gdGFnOiAke3ZlcnNpb25Kc29uLnZlcnNpb259YCk7XG4gIGxhbWJkYUZ1bmN0aW9ucy5mb3JFYWNoKChmdW5jdGlvbk5hbWUpID0+IHtcbiAgICBUYWdzLm9mKGZ1bmN0aW9uTmFtZSkuYWRkKFRhZ0tleXMuQ0RLLCBgdiR7dmVyc2lvbkpzb24udmVyc2lvbn1gLCB7XG4gICAgICBpbmNsdWRlUmVzb3VyY2VUeXBlczogW1wiQVdTOjpMYW1iZGE6OkZ1bmN0aW9uXCJdLFxuICAgIH0pO1xuICB9KTtcbn1cblxuZnVuY3Rpb24gc2V0VGFncyhsYW1iZGFGdW5jdGlvbnM6IGxhbWJkYS5GdW5jdGlvbltdLCBwcm9wczogRGF0YWRvZ1Byb3BzKSB7XG4gIGxvZy5kZWJ1ZyhgQWRkaW5nIGRhdGFkb2cgdGFnc2ApO1xuICBsYW1iZGFGdW5jdGlvbnMuZm9yRWFjaCgoZnVuY3Rpb25OYW1lKSA9PiB7XG4gICAgaWYgKHByb3BzLmZvcndhcmRlckFybikge1xuICAgICAgaWYgKHByb3BzLmVudikge1xuICAgICAgICBUYWdzLm9mKGZ1bmN0aW9uTmFtZSkuYWRkKFRhZ0tleXMuRU5WLCBwcm9wcy5lbnYpO1xuICAgICAgfVxuICAgICAgaWYgKHByb3BzLnNlcnZpY2UpIHtcbiAgICAgICAgVGFncy5vZihmdW5jdGlvbk5hbWUpLmFkZChUYWdLZXlzLlNFUlZJQ0UsIHByb3BzLnNlcnZpY2UpO1xuICAgICAgfVxuICAgICAgaWYgKHByb3BzLnZlcnNpb24pIHtcbiAgICAgICAgVGFncy5vZihmdW5jdGlvbk5hbWUpLmFkZChUYWdLZXlzLlZFUlNJT04sIHByb3BzLnZlcnNpb24pO1xuICAgICAgfVxuICAgICAgaWYgKHByb3BzLnRhZ3MpIHtcbiAgICAgICAgY29uc3QgdGFnc0FycmF5ID0gcHJvcHMudGFncy5zcGxpdChcIixcIik7XG4gICAgICAgIHRhZ3NBcnJheS5mb3JFYWNoKCh0YWc6IHN0cmluZykgPT4ge1xuICAgICAgICAgIGNvbnN0IFtrZXksIHZhbHVlXSA9IHRhZy5zcGxpdChcIjpcIik7XG4gICAgICAgICAgaWYgKGtleSAmJiB2YWx1ZSkge1xuICAgICAgICAgICAgVGFncy5vZihmdW5jdGlvbk5hbWUpLmFkZChrZXksIHZhbHVlKTtcbiAgICAgICAgICB9XG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgfSk7XG59XG5cbmZ1bmN0aW9uIGdyYW50UmVhZExhbWJkYXMoc2VjcmV0OiBJU2VjcmV0LCBsYW1iZGFGdW5jdGlvbnM6IGxhbWJkYS5GdW5jdGlvbltdKSB7XG4gIGxhbWJkYUZ1bmN0aW9ucy5mb3JFYWNoKChmdW5jdGlvbk5hbWUpID0+IHtcbiAgICBzZWNyZXQuZ3JhbnRSZWFkKGZ1bmN0aW9uTmFtZSk7XG4gIH0pO1xufVxuXG5mdW5jdGlvbiBncmFudFJlYWRMYW1iZGFzRnJvbVNlY3JldEFybihjb25zdHJ1Y3Q6IENvbnN0cnVjdCwgYXJuOiBzdHJpbmcsIGxhbWJkYUZ1bmN0aW9uczogbGFtYmRhLkZ1bmN0aW9uW10pIHtcbiAgY29uc3Qgc2VjcmV0ID0gU2VjcmV0LmZyb21TZWNyZXRQYXJ0aWFsQXJuKGNvbnN0cnVjdCwgXCJEYXRhZG9nQXBpS2V5U2VjcmV0XCIsIGFybik7XG4gIGxhbWJkYUZ1bmN0aW9ucy5mb3JFYWNoKChmdW5jdGlvbk5hbWUpID0+IHtcbiAgICBzZWNyZXQuZ3JhbnRSZWFkKGZ1bmN0aW9uTmFtZSk7XG4gIH0pO1xufVxuXG5mdW5jdGlvbiBleHRyYWN0U2luZ2xldG9uRnVuY3Rpb25zKGxhbWJkYUZ1bmN0aW9uczogTGFtYmRhRnVuY3Rpb25bXSkge1xuICAvLyBleHRyYWN0IGxhbWJkYUZ1bmN0aW9uIHByb3BlcnR5IGZyb20gU2luZ2xldG9uIEZ1bmN0aW9uXG4gIC8vIHVzaW5nIGJyYWNrZXQgbm90YXRpb24gaGVyZSBzaW5jZSBsYW1iZGFGdW5jdGlvbiBpcyBhIHByaXZhdGUgcHJvcGVydHlcbiAgY29uc3QgZXh0cmFjdGVkTGFtYmRhRnVuY3Rpb25zOiBsYW1iZGEuRnVuY3Rpb25bXSA9IGxhbWJkYUZ1bmN0aW9ucy5tYXAoKGZuKSA9PiB7XG4gICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGRvdC1ub3RhdGlvblxuICAgIHJldHVybiBpc1NpbmdsZXRvbkZ1bmN0aW9uKGZuKSA/IGZuW1wibGFtYmRhRnVuY3Rpb25cIl0gOiBmbjtcbiAgfSk7XG5cbiAgcmV0dXJuIGV4dHJhY3RlZExhbWJkYUZ1bmN0aW9ucztcbn1cblxuZnVuY3Rpb24gaXNTaW5nbGV0b25GdW5jdGlvbihmbjogTGFtYmRhRnVuY3Rpb24pOiBmbiBpcyBsYW1iZGEuU2luZ2xldG9uRnVuY3Rpb24ge1xuICByZXR1cm4gZm4uaGFzT3duUHJvcGVydHkoXCJsYW1iZGFGdW5jdGlvblwiKTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIHZhbGlkYXRlUHJvcHMocHJvcHM6IERhdGFkb2dQcm9wcywgYXBpS2V5QXJuT3ZlcnJpZGUgPSBmYWxzZSkge1xuICBsb2cuZGVidWcoXCJWYWxpZGF0aW5nIHByb3BzLi4uXCIpO1xuXG4gIGNoZWNrRm9yTXVsdGlwbGVBcGlLZXlzKHByb3BzLCBhcGlLZXlBcm5PdmVycmlkZSk7XG4gIGNvbnN0IHNpdGVMaXN0OiBzdHJpbmdbXSA9IFtcbiAgICBcImRhdGFkb2docS5jb21cIixcbiAgICBcImRhdGFkb2docS5ldVwiLFxuICAgIFwidXMzLmRhdGFkb2docS5jb21cIixcbiAgICBcInVzNS5kYXRhZG9naHEuY29tXCIsXG4gICAgXCJhcDEuZGF0YWRvZ2hxLmNvbVwiLFxuICAgIFwiZGRvZy1nb3YuY29tXCIsXG4gIF07XG4gIGlmIChcbiAgICBwcm9wcy5zaXRlICE9PSB1bmRlZmluZWQgJiZcbiAgICAhc2l0ZUxpc3QuaW5jbHVkZXMocHJvcHMuc2l0ZS50b0xvd2VyQ2FzZSgpKSAmJlxuICAgICEocHJvcHMuc2l0ZS5zdGFydHNXaXRoKFwiJHtUb2tlbltcIikgJiYgcHJvcHMuc2l0ZS5lbmRzV2l0aChcIl19XCIpKSAmJlxuICAgICFwcm9jZXNzLmVudi5ERF9DREtfQllQQVNTX1NJVEVfVkFMSURBVElPTlxuICApIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICBcIldhcm5pbmc6IEludmFsaWQgc2l0ZSBVUkwuIE11c3QgYmUgZWl0aGVyIGRhdGFkb2docS5jb20sIGRhdGFkb2docS5ldSwgdXMzLmRhdGFkb2docS5jb20sIHVzNS5kYXRhZG9naHEuY29tLCBhcDEuZGF0YWRvZ2hxLmNvbSwgb3IgZGRvZy1nb3YuY29tLlwiLFxuICAgICk7XG4gIH1cblxuICBpZiAoXG4gICAgcHJvcHMuYXBpS2V5ID09PSB1bmRlZmluZWQgJiZcbiAgICBwcm9wcy5hcGlLbXNLZXkgPT09IHVuZGVmaW5lZCAmJlxuICAgIHByb3BzLmFwaUtleVNlY3JldEFybiA9PT0gdW5kZWZpbmVkICYmXG4gICAgcHJvcHMuZmx1c2hNZXRyaWNzVG9Mb2dzID09PSBmYWxzZSAmJlxuICAgICFhcGlLZXlBcm5PdmVycmlkZVxuICApIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICBcIldoZW4gYGZsdXNoTWV0cmljc1RvTG9nc2AgaXMgZmFsc2UsIGBhcGlLZXlgLCBgYXBpS2V5U2VjcmV0QXJuYCwgb3IgYGFwaUttc0tleWAgbXVzdCBhbHNvIGJlIHNldC5cIixcbiAgICApO1xuICB9XG4gIGlmIChwcm9wcy5leHRlbnNpb25MYXllclZlcnNpb24gIT09IHVuZGVmaW5lZCkge1xuICAgIGlmIChcbiAgICAgIHByb3BzLmFwaUtleSA9PT0gdW5kZWZpbmVkICYmXG4gICAgICBwcm9wcy5hcGlLZXlTZWNyZXRBcm4gPT09IHVuZGVmaW5lZCAmJlxuICAgICAgcHJvcHMuYXBpS21zS2V5ID09PSB1bmRlZmluZWQgJiZcbiAgICAgICFhcGlLZXlBcm5PdmVycmlkZVxuICAgICkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFwiV2hlbiBgZXh0ZW5zaW9uTGF5ZXJgIGlzIHNldCwgYGFwaUtleWAsIGBhcGlLZXlTZWNyZXRBcm5gLCBvciBgYXBpS21zS2V5YCBtdXN0IGFsc28gYmUgc2V0LlwiKTtcbiAgICB9XG4gIH1cbiAgaWYgKHByb3BzLmVuYWJsZURhdGFkb2dUcmFjaW5nID09PSBmYWxzZSAmJiBwcm9wcy5lbmFibGVEYXRhZG9nQVNNID09PSB0cnVlKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKFwiV2hlbiBgZW5hYmxlRGF0YWRvZ0FTTWAgaXMgZW5hYmxlZCwgYGVuYWJsZURhdGFkb2dUcmFjaW5nYCBtdXN0IGFsc28gYmUgZW5hYmxlZC5cIik7XG4gIH1cbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGNoZWNrRm9yTXVsdGlwbGVBcGlLZXlzKHByb3BzOiBEYXRhZG9nUHJvcHMsIGFwaUtleUFybk92ZXJyaWRlID0gZmFsc2UpIHtcbiAgbGV0IG11bHRpcGxlQXBpS2V5c01lc3NhZ2U7XG4gIGNvbnN0IGFwaUtleUFybk9yT3ZlcnJpZGUgPSBwcm9wcy5hcGlLZXlTZWNyZXRBcm4gIT09IHVuZGVmaW5lZCB8fCBhcGlLZXlBcm5PdmVycmlkZTtcbiAgaWYgKHByb3BzLmFwaUtleSAhPT0gdW5kZWZpbmVkICYmIHByb3BzLmFwaUttc0tleSAhPT0gdW5kZWZpbmVkICYmIGFwaUtleUFybk9yT3ZlcnJpZGUpIHtcbiAgICBtdWx0aXBsZUFwaUtleXNNZXNzYWdlID0gXCJgYXBpS2V5YCwgYGFwaUttc0tleWAsIGFuZCBgYXBpS2V5U2VjcmV0QXJuYFwiO1xuICB9IGVsc2UgaWYgKHByb3BzLmFwaUtleSAhPT0gdW5kZWZpbmVkICYmIHByb3BzLmFwaUttc0tleSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgbXVsdGlwbGVBcGlLZXlzTWVzc2FnZSA9IFwiYGFwaUtleWAgYW5kIGBhcGlLbXNLZXlgXCI7XG4gIH0gZWxzZSBpZiAocHJvcHMuYXBpS2V5ICE9PSB1bmRlZmluZWQgJiYgYXBpS2V5QXJuT3JPdmVycmlkZSkge1xuICAgIG11bHRpcGxlQXBpS2V5c01lc3NhZ2UgPSBcImBhcGlLZXlgIGFuZCBgYXBpS2V5U2VjcmV0QXJuYFwiO1xuICB9IGVsc2UgaWYgKHByb3BzLmFwaUttc0tleSAhPT0gdW5kZWZpbmVkICYmIGFwaUtleUFybk9yT3ZlcnJpZGUpIHtcbiAgICBtdWx0aXBsZUFwaUtleXNNZXNzYWdlID0gXCJgYXBpS21zS2V5YCBhbmQgYGFwaUtleVNlY3JldEFybmBcIjtcbiAgfVxuXG4gIGlmIChtdWx0aXBsZUFwaUtleXNNZXNzYWdlKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKGAke211bHRpcGxlQXBpS2V5c01lc3NhZ2V9IHNob3VsZCBub3QgYmUgc2V0IGF0IHRoZSBzYW1lIHRpbWUuYCk7XG4gIH1cbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGhhbmRsZVNldHRpbmdQcm9wRGVmYXVsdHMocHJvcHM6IERhdGFkb2dQcm9wcyk6IERhdGFkb2dTdHJpY3RQcm9wcyB7XG4gIGxldCBhZGRMYXllcnMgPSBwcm9wcy5hZGRMYXllcnM7XG4gIGxldCBlbmFibGVEYXRhZG9nVHJhY2luZyA9IHByb3BzLmVuYWJsZURhdGFkb2dUcmFjaW5nO1xuICBsZXQgZW5hYmxlRGF0YWRvZ0FTTSA9IHByb3BzLmVuYWJsZURhdGFkb2dBU007XG4gIGxldCBlbmFibGVNZXJnZVhyYXlUcmFjZXMgPSBwcm9wcy5lbmFibGVNZXJnZVhyYXlUcmFjZXM7XG4gIGxldCBpbmplY3RMb2dDb250ZXh0ID0gcHJvcHMuaW5qZWN0TG9nQ29udGV4dDtcbiAgY29uc3QgbG9nTGV2ZWwgPSBwcm9wcy5sb2dMZXZlbDtcbiAgbGV0IGVuYWJsZURhdGFkb2dMb2dzID0gcHJvcHMuZW5hYmxlRGF0YWRvZ0xvZ3M7XG4gIGxldCBjYXB0dXJlTGFtYmRhUGF5bG9hZCA9IHByb3BzLmNhcHR1cmVMYW1iZGFQYXlsb2FkO1xuICBsZXQgc291cmNlQ29kZUludGVncmF0aW9uID0gcHJvcHMuc291cmNlQ29kZUludGVncmF0aW9uO1xuICBsZXQgcmVkaXJlY3RIYW5kbGVyID0gcHJvcHMucmVkaXJlY3RIYW5kbGVyO1xuICBsZXQgZ3JhbnRTZWNyZXRSZWFkQWNjZXNzID0gcHJvcHMuZ3JhbnRTZWNyZXRSZWFkQWNjZXNzO1xuICBjb25zdCBleHRlbnNpb25MYXllclZlcnNpb24gPSBwcm9wcy5leHRlbnNpb25MYXllclZlcnNpb247XG5cbiAgaWYgKGFkZExheWVycyA9PT0gdW5kZWZpbmVkKSB7XG4gICAgbG9nLmRlYnVnKGBObyB2YWx1ZSBwcm92aWRlZCBmb3IgYWRkTGF5ZXJzLCBkZWZhdWx0aW5nIHRvICR7RGVmYXVsdERhdGFkb2dQcm9wcy5hZGRMYXllcnN9YCk7XG4gICAgYWRkTGF5ZXJzID0gRGVmYXVsdERhdGFkb2dQcm9wcy5hZGRMYXllcnM7XG4gIH1cbiAgaWYgKGVuYWJsZURhdGFkb2dUcmFjaW5nID09PSB1bmRlZmluZWQpIHtcbiAgICBsb2cuZGVidWcoYE5vIHZhbHVlIHByb3ZpZGVkIGZvciBlbmFibGVEYXRhZG9nVHJhY2luZywgZGVmYXVsdGluZyB0byAke0RlZmF1bHREYXRhZG9nUHJvcHMuZW5hYmxlRGF0YWRvZ1RyYWNpbmd9YCk7XG4gICAgZW5hYmxlRGF0YWRvZ1RyYWNpbmcgPSBEZWZhdWx0RGF0YWRvZ1Byb3BzLmVuYWJsZURhdGFkb2dUcmFjaW5nO1xuICB9XG4gIGlmIChlbmFibGVEYXRhZG9nQVNNID09PSB1bmRlZmluZWQpIHtcbiAgICBsb2cuZGVidWcoYE5vIHZhbHVlIHByb3ZpZGVkIGZvciBlbmFibGVEYXRhZG9nQVNNLCBkZWZhdWx0aW5nIHRvICR7RGVmYXVsdERhdGFkb2dQcm9wcy5lbmFibGVEYXRhZG9nQVNNfWApO1xuICAgIGVuYWJsZURhdGFkb2dBU00gPSBEZWZhdWx0RGF0YWRvZ1Byb3BzLmVuYWJsZURhdGFkb2dBU007XG4gIH1cbiAgaWYgKGVuYWJsZU1lcmdlWHJheVRyYWNlcyA9PT0gdW5kZWZpbmVkKSB7XG4gICAgbG9nLmRlYnVnKFxuICAgICAgYE5vIHZhbHVlIHByb3ZpZGVkIGZvciBlbmFibGVNZXJnZVhyYXlUcmFjZXMsIGRlZmF1bHRpbmcgdG8gJHtEZWZhdWx0RGF0YWRvZ1Byb3BzLmVuYWJsZU1lcmdlWHJheVRyYWNlc31gLFxuICAgICk7XG4gICAgZW5hYmxlTWVyZ2VYcmF5VHJhY2VzID0gRGVmYXVsdERhdGFkb2dQcm9wcy5lbmFibGVNZXJnZVhyYXlUcmFjZXM7XG4gIH1cbiAgaWYgKGluamVjdExvZ0NvbnRleHQgPT09IHVuZGVmaW5lZCkge1xuICAgIGxvZy5kZWJ1ZyhgTm8gdmFsdWUgcHJvdmlkZWQgZm9yIGluamVjdExvZ0NvbnRleHQsIGRlZmF1bHRpbmcgdG8gJHtEZWZhdWx0RGF0YWRvZ1Byb3BzLmluamVjdExvZ0NvbnRleHR9YCk7XG4gICAgaW5qZWN0TG9nQ29udGV4dCA9IERlZmF1bHREYXRhZG9nUHJvcHMuaW5qZWN0TG9nQ29udGV4dDtcbiAgfVxuICBpZiAobG9nTGV2ZWwgPT09IHVuZGVmaW5lZCkge1xuICAgIGxvZy5kZWJ1ZyhgTm8gdmFsdWUgcHJvdmlkZWQgZm9yIGxvZ0xldmVsYCk7XG4gIH1cbiAgaWYgKGVuYWJsZURhdGFkb2dMb2dzID09PSB1bmRlZmluZWQpIHtcbiAgICBsb2cuZGVidWcoYE5vIHZhbHVlIHByb3ZpZGVkIGZvciBlbmFibGVEYXRhZG9nTG9ncywgZGVmYXVsdGluZyB0byAke0RlZmF1bHREYXRhZG9nUHJvcHMuZW5hYmxlRGF0YWRvZ0xvZ3N9YCk7XG4gICAgZW5hYmxlRGF0YWRvZ0xvZ3MgPSBEZWZhdWx0RGF0YWRvZ1Byb3BzLmVuYWJsZURhdGFkb2dMb2dzO1xuICB9XG4gIGlmIChjYXB0dXJlTGFtYmRhUGF5bG9hZCA9PT0gdW5kZWZpbmVkKSB7XG4gICAgbG9nLmRlYnVnKGBObyB2YWx1ZSBwcm92aWRlZCBmb3IgY2FwdHVyZUxhbWJkYVBheWxvYWQsIGRlZmF1bHQgdG8gJHtEZWZhdWx0RGF0YWRvZ1Byb3BzLmNhcHR1cmVMYW1iZGFQYXlsb2FkfWApO1xuICAgIGNhcHR1cmVMYW1iZGFQYXlsb2FkID0gRGVmYXVsdERhdGFkb2dQcm9wcy5jYXB0dXJlTGFtYmRhUGF5bG9hZDtcbiAgfVxuICBpZiAoc291cmNlQ29kZUludGVncmF0aW9uID09PSB1bmRlZmluZWQpIHtcbiAgICBsb2cuZGVidWcoYE5vIHZhbHVlIHByb3ZpZGVkIGZvciBzb3VyY2VDb2RlSW50ZWdyYXRpb24sIGRlZmF1bHQgdG8gJHtEZWZhdWx0RGF0YWRvZ1Byb3BzLnNvdXJjZUNvZGVJbnRlZ3JhdGlvbn1gKTtcbiAgICBzb3VyY2VDb2RlSW50ZWdyYXRpb24gPSBEZWZhdWx0RGF0YWRvZ1Byb3BzLnNvdXJjZUNvZGVJbnRlZ3JhdGlvbjtcbiAgfVxuXG4gIGlmIChyZWRpcmVjdEhhbmRsZXIgPT09IHVuZGVmaW5lZCkge1xuICAgIGxvZy5kZWJ1ZyhgTm8gdmFsdWUgcHJvdmlkZWQgZm9yIHJlZGlyZWN0SGFuZGxlciwgZGVmYXVsdCB0byAke0RlZmF1bHREYXRhZG9nUHJvcHMucmVkaXJlY3RIYW5kbGVyfWApO1xuICAgIHJlZGlyZWN0SGFuZGxlciA9IERlZmF1bHREYXRhZG9nUHJvcHMucmVkaXJlY3RIYW5kbGVyO1xuICB9XG5cbiAgaWYgKGdyYW50U2VjcmV0UmVhZEFjY2VzcyA9PT0gdW5kZWZpbmVkKSB7XG4gICAgbG9nLmRlYnVnKGBObyB2YWx1ZSBwcm92aWRlZCBmb3IgZ3JhbnRTZWNyZXRSZWFkQWNjZXNzLCBkZWZhdWx0IHRvICR7RGVmYXVsdERhdGFkb2dQcm9wcy5ncmFudFNlY3JldFJlYWRBY2Nlc3N9YCk7XG4gICAgZ3JhbnRTZWNyZXRSZWFkQWNjZXNzID0gRGVmYXVsdERhdGFkb2dQcm9wcy5ncmFudFNlY3JldFJlYWRBY2Nlc3M7XG4gIH1cblxuICByZXR1cm4ge1xuICAgIGFkZExheWVyczogYWRkTGF5ZXJzLFxuICAgIGVuYWJsZURhdGFkb2dUcmFjaW5nOiBlbmFibGVEYXRhZG9nVHJhY2luZyxcbiAgICBlbmFibGVEYXRhZG9nQVNNLFxuICAgIGVuYWJsZU1lcmdlWHJheVRyYWNlczogZW5hYmxlTWVyZ2VYcmF5VHJhY2VzLFxuICAgIGluamVjdExvZ0NvbnRleHQ6IGluamVjdExvZ0NvbnRleHQsXG4gICAgbG9nTGV2ZWw6IGxvZ0xldmVsLFxuICAgIGVuYWJsZURhdGFkb2dMb2dzOiBlbmFibGVEYXRhZG9nTG9ncyxcbiAgICBjYXB0dXJlTGFtYmRhUGF5bG9hZDogY2FwdHVyZUxhbWJkYVBheWxvYWQsXG4gICAgc291cmNlQ29kZUludGVncmF0aW9uOiBzb3VyY2VDb2RlSW50ZWdyYXRpb24sXG4gICAgcmVkaXJlY3RIYW5kbGVyOiByZWRpcmVjdEhhbmRsZXIsXG4gICAgZ3JhbnRTZWNyZXRSZWFkQWNjZXNzOiBncmFudFNlY3JldFJlYWRBY2Nlc3MsXG4gICAgZXh0ZW5zaW9uTGF5ZXJWZXJzaW9uOiBleHRlbnNpb25MYXllclZlcnNpb24sXG4gIH07XG59XG4iXX0=