"use strict";
/*
 * Unless explicitly stated otherwise all files in this repository are licensed
 * under the Apache License Version 2.0.
 *
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2021 Datadog, Inc.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transport = exports.transportDefaults = exports.FLUSH_METRICS_TO_LOGS_ENV_VAR = exports.SITE_URL_ENV_VAR = exports.KMS_API_KEY_ENV_VAR = exports.API_KEY_SECRET_ARN_ENV_VAR = exports.API_KEY_ENV_VAR = void 0;
const loglevel_1 = require("loglevel");
const index_1 = require("./index");
exports.API_KEY_ENV_VAR = "DD_API_KEY";
exports.API_KEY_SECRET_ARN_ENV_VAR = "DD_API_KEY_SECRET_ARN";
exports.KMS_API_KEY_ENV_VAR = "DD_KMS_API_KEY";
exports.SITE_URL_ENV_VAR = "DD_SITE";
exports.FLUSH_METRICS_TO_LOGS_ENV_VAR = "DD_FLUSH_TO_LOG";
exports.transportDefaults = {
    site: "datadoghq.com",
    flushMetricsToLogs: true,
    enableDatadogTracing: true,
};
class Transport {
    constructor(flushMetricsToLogs, site, apiKey, apiKeySecretArn, apiKmsKey, extensionLayerVersion) {
        if (flushMetricsToLogs === undefined) {
            loglevel_1.default.debug(`No value provided for flushMetricsToLogs, defaulting to ${exports.transportDefaults.flushMetricsToLogs}`);
            this.flushMetricsToLogs = exports.transportDefaults.flushMetricsToLogs;
        }
        else {
            this.flushMetricsToLogs = flushMetricsToLogs;
        }
        this.extensionLayerVersion = extensionLayerVersion;
        // If the extension is used, metrics will be submitted via the extension.
        if (this.extensionLayerVersion !== undefined) {
            loglevel_1.default.debug(`Using extension version ${this.extensionLayerVersion}, metrics will be submitted via the extension`);
            this.flushMetricsToLogs = false;
        }
        if (site === undefined) {
            loglevel_1.default.debug(`No value provided for site, defaulting to ${exports.transportDefaults.site}`);
            this.site = exports.transportDefaults.site;
        }
        else {
            this.site = site;
        }
        this.apiKey = apiKey;
        this.apiKeySecretArn = apiKeySecretArn;
        this.apiKmsKey = apiKmsKey;
    }
    applyEnvVars(lambdas) {
        loglevel_1.default.debug(`Setting Datadog transport environment variables...`);
        lambdas.forEach((lam) => {
            lam.addEnvironment(exports.FLUSH_METRICS_TO_LOGS_ENV_VAR, this.flushMetricsToLogs.toString());
            if (this.site !== undefined && this.flushMetricsToLogs === false) {
                lam.addEnvironment(exports.SITE_URL_ENV_VAR, this.site);
            }
            if (this.apiKey !== undefined) {
                lam.addEnvironment(exports.API_KEY_ENV_VAR, this.apiKey);
            }
            if (this.apiKeySecretArn !== undefined) {
                const isNode = index_1.runtimeLookup[lam.runtime.name] === index_1.RuntimeType.NODE;
                const isSendingSynchronousMetrics = this.extensionLayerVersion === undefined && !this.flushMetricsToLogs;
                if (isSendingSynchronousMetrics && isNode) {
                    throw new Error(`\`apiKeySecretArn\` is not supported for Node runtimes when using Synchronous Metrics. Use either \`apiKey\` or \`apiKmsKey\`.`);
                }
                lam.addEnvironment(exports.API_KEY_SECRET_ARN_ENV_VAR, this.apiKeySecretArn);
            }
            if (this.apiKmsKey !== undefined) {
                lam.addEnvironment(exports.KMS_API_KEY_ENV_VAR, this.apiKmsKey);
            }
        });
    }
}
exports.Transport = Transport;
//# sourceMappingURL=data:application/json;base64,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