"use strict";
/*
 * Unless explicitly stated otherwise all files in this repository are licensed
 * under the Apache License Version 2.0.
 *
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2021 Datadog, Inc.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateLayerId = exports.generateExtensionLayerId = exports.generateLambdaLayerId = exports.getMissingLayerVersionErrorMsg = exports.getExtensionLayerArn = exports.getLambdaLayerArn = exports.applyLayers = void 0;
const crypto = require("crypto");
const lambda = require("aws-cdk-lib/aws-lambda");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const loglevel_1 = require("loglevel");
const index_1 = require("./index");
const layers = new Map();
function applyLayers(scope, region, lambdas, pythonLayerVersion, nodeLayerVersion, javaLayerVersion, extensionLayerVersion, useLayersFromAccount) {
    // TODO: check region availability
    const errors = [];
    loglevel_1.default.debug("Applying layers to Lambda functions...");
    lambdas.forEach((lam) => {
        var _a;
        const runtime = lam.runtime.name;
        const lambdaRuntimeType = index_1.runtimeLookup[runtime];
        const isARM = ((_a = lam.architecture) === null || _a === void 0 ? void 0 : _a.dockerPlatform) !== undefined &&
            lam.architecture.dockerPlatform === aws_lambda_1.Architecture.ARM_64.dockerPlatform;
        const isNode = lambdaRuntimeType === index_1.RuntimeType.NODE;
        if (lambdaRuntimeType === undefined || lambdaRuntimeType === index_1.RuntimeType.UNSUPPORTED) {
            loglevel_1.default.debug(`Unsupported runtime: ${runtime}`);
            return;
        }
        const accountId = useLayersFromAccount;
        let lambdaLayerArn;
        let extensionLayerArn;
        if (lambdaRuntimeType === index_1.RuntimeType.PYTHON) {
            if (pythonLayerVersion === undefined) {
                const errorMessage = getMissingLayerVersionErrorMsg(lam.node.id, "Python", "python");
                loglevel_1.default.error(errorMessage);
                errors.push(errorMessage);
                return;
            }
            lambdaLayerArn = getLambdaLayerArn(region, pythonLayerVersion, runtime, isARM, isNode, accountId);
            loglevel_1.default.debug(`Using Python Lambda layer: ${lambdaLayerArn}`);
            addLayer(lambdaLayerArn, false, scope, lam, runtime);
        }
        if (lambdaRuntimeType === index_1.RuntimeType.NODE) {
            if (nodeLayerVersion === undefined) {
                const errorMessage = getMissingLayerVersionErrorMsg(lam.node.id, "Node.js", "node");
                loglevel_1.default.error(errorMessage);
                errors.push(errorMessage);
                return;
            }
            lambdaLayerArn = getLambdaLayerArn(region, nodeLayerVersion, runtime, isARM, isNode, accountId);
            loglevel_1.default.debug(`Using Node Lambda layer: ${lambdaLayerArn}`);
            addLayer(lambdaLayerArn, false, scope, lam, runtime);
        }
        if (lambdaRuntimeType === index_1.RuntimeType.JAVA) {
            if (javaLayerVersion === undefined) {
                const errorMessage = getMissingLayerVersionErrorMsg(lam.node.id, "Java", "java");
                loglevel_1.default.error(errorMessage);
                errors.push(errorMessage);
                return;
            }
            lambdaLayerArn = getLambdaLayerArn(region, javaLayerVersion, runtime, isARM, isNode, accountId);
            loglevel_1.default.debug(`Using dd-trace-java layer: ${lambdaLayerArn}`);
            addLayer(lambdaLayerArn, false, scope, lam, runtime);
        }
        if (extensionLayerVersion !== undefined) {
            extensionLayerArn = getExtensionLayerArn(region, extensionLayerVersion, isARM, accountId);
            loglevel_1.default.debug(`Using extension layer: ${extensionLayerArn}`);
            addLayer(extensionLayerArn, true, scope, lam, runtime);
        }
    });
    return errors;
}
exports.applyLayers = applyLayers;
function addLayer(layerArn, isExtensionLayer, scope, lam, runtime) {
    const layerId = generateLayerId(isExtensionLayer, lam.functionArn, runtime);
    if (layerArn !== undefined) {
        let lambdaLayer = layers.get(layerArn);
        if (lambdaLayer === undefined) {
            lambdaLayer = lambda.LayerVersion.fromLayerVersionArn(scope, layerId, layerArn);
            layers.set(layerArn, lambdaLayer);
        }
        loglevel_1.default.debug(`Adding layer ${lambdaLayer} to Lambda: ${lam.functionName}`);
        lam.addLayers(lambdaLayer);
    }
}
function getLambdaLayerArn(region, version, runtime, isArm, isNode, accountId) {
    const baseLayerName = index_1.runtimeToLayerName[runtime];
    const layerName = isArm && !isNode ? `${baseLayerName}-ARM` : baseLayerName;
    const partition = getAWSPartitionFromRegion(region);
    // TODO: edge case where gov cloud is the region, but they are using a token so we can't resolve it.
    const isGovCloud = index_1.govCloudRegions.includes(region);
    // if this is a GovCloud region, use the GovCloud lambda layer
    if (isGovCloud) {
        loglevel_1.default.debug("GovCloud region detected, using the GovCloud lambda layer");
        return `arn:${partition}:lambda:${region}:${accountId !== null && accountId !== void 0 ? accountId : index_1.DD_GOV_ACCOUNT_ID}:layer:${layerName}:${version}`;
    }
    return `arn:${partition}:lambda:${region}:${accountId !== null && accountId !== void 0 ? accountId : index_1.DD_ACCOUNT_ID}:layer:${layerName}:${version}`;
}
exports.getLambdaLayerArn = getLambdaLayerArn;
function getExtensionLayerArn(region, version, isArm, accountId) {
    const baseLayerName = "Datadog-Extension";
    const layerName = isArm ? `${baseLayerName}-ARM` : baseLayerName;
    const partition = getAWSPartitionFromRegion(region);
    const isGovCloud = index_1.govCloudRegions.includes(region);
    if (isGovCloud) {
        loglevel_1.default.debug("GovCloud region detected, using the GovCloud extension layer");
        return `arn:${partition}:lambda:${region}:${accountId !== null && accountId !== void 0 ? accountId : index_1.DD_GOV_ACCOUNT_ID}:layer:${layerName}:${version}`;
    }
    return `arn:${partition}:lambda:${region}:${accountId !== null && accountId !== void 0 ? accountId : index_1.DD_ACCOUNT_ID}:layer:${layerName}:${version}`;
}
exports.getExtensionLayerArn = getExtensionLayerArn;
function getMissingLayerVersionErrorMsg(functionKey, formalRuntime, paramRuntime) {
    return (`Resource ${functionKey} has a ${formalRuntime} runtime, but no ${formalRuntime} Lambda Library version was provided. ` +
        `Please add the '${paramRuntime}LayerVersion' parameter for the Datadog serverless macro.`);
}
exports.getMissingLayerVersionErrorMsg = getMissingLayerVersionErrorMsg;
function generateLambdaLayerId(lambdaFunctionArn, runtime) {
    loglevel_1.default.debug("Generating construct Id for Datadog Lambda layer");
    const layerValue = crypto.createHash("sha256").update(lambdaFunctionArn).digest("hex");
    return `${index_1.LAYER_PREFIX}-${runtime}-${layerValue}`;
}
exports.generateLambdaLayerId = generateLambdaLayerId;
function generateExtensionLayerId(lambdaFunctionArn) {
    loglevel_1.default.debug("Generating construct Id for Datadog Extension layer");
    const layerValue = crypto.createHash("sha256").update(lambdaFunctionArn).digest("hex");
    return `${index_1.EXTENSION_LAYER_PREFIX}-${layerValue}`;
}
exports.generateExtensionLayerId = generateExtensionLayerId;
function generateLayerId(isExtensionLayer, functionArn, runtime) {
    if (isExtensionLayer) {
        return generateExtensionLayerId(functionArn);
    }
    return generateLambdaLayerId(functionArn, runtime);
}
exports.generateLayerId = generateLayerId;
function getAWSPartitionFromRegion(region) {
    if (region.startsWith("us-gov-")) {
        return "aws-us-gov";
    }
    if (region.startsWith("cn-")) {
        return "aws-cn";
    }
    return "aws";
}
//# sourceMappingURL=data:application/json;base64,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