"use strict";
/*
 * Unless explicitly stated otherwise all files in this repository are licensed
 * under the Apache License Version 2.0.
 *
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2021 Datadog, Inc.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.setDDEnvVariables = exports.applyEnvVariables = exports.setGitEnvironmentVariables = exports.DD_APM_FLUSH_DEADLINE_MILLISECONDS = exports.DD_DECODE_AUTHORIZER_CONTEXT = exports.DD_ENCODE_AUTHORIZER_CONTEXT = exports.DD_PROFILING_ENABLED = exports.DD_COLD_START_TRACE_SKIP_LIB = exports.DD_MIN_COLD_START_DURATION = exports.DD_COLD_START_TRACING = exports.DD_TAGS = exports.DD_VERSION_ENV_VAR = exports.DD_SERVICE_ENV_VAR = exports.DD_ENV_ENV_VAR = exports.CAPTURE_LAMBDA_PAYLOAD_ENV_VAR = exports.ENABLE_DD_LOGS_ENV_VAR = exports.LOG_LEVEL_ENV_VAR = exports.INJECT_LOG_CONTEXT_ENV_VAR = exports.ENABLE_XRAY_TRACE_MERGING_ENV_VAR = exports.ENABLE_DD_TRACING_ENV_VAR = void 0;
const loglevel_1 = require("loglevel");
exports.ENABLE_DD_TRACING_ENV_VAR = "DD_TRACE_ENABLED";
exports.ENABLE_XRAY_TRACE_MERGING_ENV_VAR = "DD_MERGE_XRAY_TRACES";
exports.INJECT_LOG_CONTEXT_ENV_VAR = "DD_LOGS_INJECTION";
exports.LOG_LEVEL_ENV_VAR = "DD_LOG_LEVEL";
exports.ENABLE_DD_LOGS_ENV_VAR = "DD_SERVERLESS_LOGS_ENABLED";
exports.CAPTURE_LAMBDA_PAYLOAD_ENV_VAR = "DD_CAPTURE_LAMBDA_PAYLOAD";
exports.DD_ENV_ENV_VAR = "DD_ENV";
exports.DD_SERVICE_ENV_VAR = "DD_SERVICE";
exports.DD_VERSION_ENV_VAR = "DD_VERSION";
exports.DD_TAGS = "DD_TAGS";
exports.DD_COLD_START_TRACING = "DD_COLD_START_TRACING";
exports.DD_MIN_COLD_START_DURATION = "DD_MIN_COLD_START_DURATION";
exports.DD_COLD_START_TRACE_SKIP_LIB = "DD_COLD_START_TRACE_SKIP_LIB";
exports.DD_PROFILING_ENABLED = "DD_PROFILING_ENABLED";
exports.DD_ENCODE_AUTHORIZER_CONTEXT = "DD_ENCODE_AUTHORIZER_CONTEXT";
exports.DD_DECODE_AUTHORIZER_CONTEXT = "DD_DECODE_AUTHORIZER_CONTEXT";
exports.DD_APM_FLUSH_DEADLINE_MILLISECONDS = "DD_APM_FLUSH_DEADLINE_MILLISECONDS";
const execSync = require("child_process").execSync;
const URL = require("url").URL;
function setGitEnvironmentVariables(lambdas) {
    loglevel_1.default.debug("Adding source code integration...");
    const { hash, gitRepoUrl } = getGitData();
    if (hash == "" || gitRepoUrl == "")
        return;
    // We're using an any type here because AWS does not expose the `environment` field in their type
    lambdas.forEach((lam) => {
        if (lam.environment[exports.DD_TAGS] !== undefined) {
            lam.environment[exports.DD_TAGS].value += `,git.commit.sha:${hash}`;
        }
        else {
            lam.addEnvironment(exports.DD_TAGS, `git.commit.sha:${hash}`);
        }
        lam.environment[exports.DD_TAGS].value += `,git.repository_url:${gitRepoUrl}`;
    });
}
exports.setGitEnvironmentVariables = setGitEnvironmentVariables;
function getGitData() {
    let hash;
    let gitRepoUrl;
    try {
        hash = execSync("git rev-parse HEAD").toString().trim();
        gitRepoUrl = execSync("git config --get remote.origin.url").toString().trim();
    }
    catch (e) {
        loglevel_1.default.debug(`Failed to add source code integration. Error: ${e}`);
        return { hash: "", gitRepoUrl: "" };
    }
    return { hash, gitRepoUrl: filterAndFormatGithubRemote(gitRepoUrl) };
}
// Removes sensitive info from the given git remote url and normalizes the url prefix.
// "git@github.com:" and "https://github.com/" prefixes will be normalized into "github.com/"
function filterAndFormatGithubRemote(rawRemote) {
    rawRemote = filterSensitiveInfoFromRepository(rawRemote);
    if (!rawRemote) {
        return rawRemote;
    }
    rawRemote = rawRemote.replace(/git@github\.com:|https:\/\/github\.com\//, "github.com/");
    return rawRemote;
}
function filterSensitiveInfoFromRepository(repositoryUrl) {
    try {
        if (!repositoryUrl) {
            return repositoryUrl;
        }
        if (repositoryUrl.startsWith("git@")) {
            return repositoryUrl;
        }
        const { protocol, hostname, pathname } = new URL(repositoryUrl);
        if (!protocol || !hostname) {
            return repositoryUrl;
        }
        return `${protocol}//${hostname}${pathname}`;
    }
    catch (e) {
        return repositoryUrl;
    }
}
function applyEnvVariables(lambdas, baseProps) {
    loglevel_1.default.debug(`Setting environment variables...`);
    lambdas.forEach((lam) => {
        lam.addEnvironment(exports.ENABLE_DD_TRACING_ENV_VAR, baseProps.enableDatadogTracing.toString().toLowerCase());
        lam.addEnvironment(exports.ENABLE_XRAY_TRACE_MERGING_ENV_VAR, baseProps.enableMergeXrayTraces.toString().toLowerCase());
        // Check for extensionLayerVersion and set INJECT_LOG_CONTEXT_ENV_VAR accordingly
        if (baseProps.extensionLayerVersion) {
            lam.addEnvironment(exports.INJECT_LOG_CONTEXT_ENV_VAR, "false");
        }
        else {
            lam.addEnvironment(exports.INJECT_LOG_CONTEXT_ENV_VAR, baseProps.injectLogContext.toString().toLowerCase());
        }
        lam.addEnvironment(exports.ENABLE_DD_LOGS_ENV_VAR, baseProps.enableDatadogLogs.toString().toLowerCase());
        lam.addEnvironment(exports.CAPTURE_LAMBDA_PAYLOAD_ENV_VAR, baseProps.captureLambdaPayload.toString().toLowerCase());
        if (baseProps.logLevel) {
            lam.addEnvironment(exports.LOG_LEVEL_ENV_VAR, baseProps.logLevel);
        }
    });
}
exports.applyEnvVariables = applyEnvVariables;
function setDDEnvVariables(lambdas, props) {
    lambdas.forEach((lam) => {
        if (props.extensionLayerVersion) {
            if (props.env) {
                lam.addEnvironment(exports.DD_ENV_ENV_VAR, props.env);
            }
            if (props.service) {
                lam.addEnvironment(exports.DD_SERVICE_ENV_VAR, props.service);
            }
            if (props.version) {
                lam.addEnvironment(exports.DD_VERSION_ENV_VAR, props.version);
            }
            if (props.tags) {
                lam.addEnvironment(exports.DD_TAGS, props.tags);
            }
        }
        if (props.enableColdStartTracing !== undefined) {
            lam.addEnvironment(exports.DD_COLD_START_TRACING, props.enableColdStartTracing.toString().toLowerCase());
        }
        if (props.minColdStartTraceDuration !== undefined) {
            lam.addEnvironment(exports.DD_MIN_COLD_START_DURATION, props.minColdStartTraceDuration.toString().toLowerCase());
        }
        if (props.coldStartTraceSkipLibs !== undefined) {
            lam.addEnvironment(exports.DD_COLD_START_TRACE_SKIP_LIB, props.coldStartTraceSkipLibs);
        }
        if (props.enableProfiling !== undefined) {
            lam.addEnvironment(exports.DD_PROFILING_ENABLED, props.enableProfiling.toString().toLowerCase());
        }
        if (props.encodeAuthorizerContext !== undefined) {
            lam.addEnvironment(exports.DD_ENCODE_AUTHORIZER_CONTEXT, props.encodeAuthorizerContext.toString().toLowerCase());
        }
        if (props.decodeAuthorizerContext !== undefined) {
            lam.addEnvironment(exports.DD_DECODE_AUTHORIZER_CONTEXT, props.decodeAuthorizerContext.toString().toLowerCase());
        }
        if (props.apmFlushDeadline !== undefined) {
            lam.addEnvironment(exports.DD_APM_FLUSH_DEADLINE_MILLISECONDS, props.apmFlushDeadline.toString().toLowerCase());
        }
    });
}
exports.setDDEnvVariables = setDDEnvVariables;
//# sourceMappingURL=data:application/json;base64,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