"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getAggregateAlarmState = exports.getAlarms = exports.getCloudwatchAlarms = exports.AlarmState = void 0;
const aws_sdk_1 = require("aws-sdk");
var AlarmState;
(function (AlarmState) {
    AlarmState["OK"] = "OK";
    AlarmState["ALARM"] = "ALARM";
    AlarmState["INSUFFICIENT_DATA"] = "INSUFFICIENT_DATA";
})(AlarmState = exports.AlarmState || (exports.AlarmState = {}));
exports.getCloudwatchAlarms = async (params) => {
    const cloudwatch = new aws_sdk_1.CloudWatch();
    let nextToken;
    let results = [];
    do {
        const result = await cloudwatch
            .describeAlarms({
            AlarmTypes: ['CompositeAlarm', 'MetricAlarm'],
            ...params,
            NextToken: nextToken,
        })
            .promise();
        results = result.CompositeAlarms
            ? results.concat(result.CompositeAlarms?.map((alarm) => {
                return {
                    alarmArn: alarm.AlarmArn,
                    state: alarm.StateValue,
                    reason: alarm.StateReason,
                    alarmDescription: alarm.AlarmDescription,
                };
            }))
            : results;
        results = result.MetricAlarms
            ? results.concat(result.MetricAlarms?.map((alarm) => {
                return {
                    alarmArn: alarm.AlarmArn,
                    state: alarm.StateValue,
                    reason: alarm.StateReason,
                    alarmDescription: alarm.AlarmDescription,
                };
            }))
            : results;
        nextToken = result.NextToken;
    } while (nextToken);
    return results;
};
/**
 * Returns all alarms starting with the provided prefix in the state of ALARM. In addition,
 * any exceptions caught will be logged and converted into an alarm state of options.defaultState
 * with an associated reason.
 */
exports.getAlarms = async (searchTerms) => {
    const states = [];
    try {
        const alarmsForAccountAndRegion = await exports.getCloudwatchAlarms();
        const alarms = hasAlarmMatching(alarmsForAccountAndRegion, searchTerms).map((alarm) => {
            alarm.reason = `${alarm.alarmArn} in ${alarm.state} due to ${alarm.reason}`;
            return alarm;
        });
        if (alarms.length === 0) {
            alarms.push({
                state: AlarmState.ALARM,
                reason: `No alarms were found for the provided search terms: ${searchTerms.join(', ')}.`,
            });
        }
    }
    catch (e) {
        const message = 'Unable to retrieve alarms.';
        states.push({ reason: message, state: AlarmState.ALARM });
        console.error(message, e);
    }
    return states;
};
/**
 * Calculate and return an aggregate alarm state across alarms returned from all
 * CloudWatch clients provided. If any client errors are thrown, the returned state
 * will be ALARM.
 *
 */
exports.getAggregateAlarmState = async (searchTerms) => {
    const alarmDetails = await exports.getAlarms(searchTerms);
    return {
        alarmDetails,
        state: alarmDetails.find((a) => a.state === AlarmState.ALARM)
            ?.state ?? AlarmState.OK,
        summary: `${alarmDetails
            .filter((a) => a.state === AlarmState.ALARM)
            .map((a) => a.reason)
            .join(', ')}`,
    };
};
const hasAlarmMatching = (alarms, searchTerms) => {
    return alarms
        .flatMap((alarm) => searchTerms
        .map((searchTerm) => {
        return alarm.alarmDescription?.includes(searchTerm)
            ? alarm
            : undefined;
    })
        .filter((sameAlarm) => sameAlarm))
        .filter((sameAlarm, index) => alarms.indexOf(sameAlarm) === index);
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYWxhcm0tc3RhdGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvY29tbW9uL2FsYXJtLXN0YXRlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHFDQUFxQztBQU1yQyxJQUFZLFVBSVg7QUFKRCxXQUFZLFVBQVU7SUFDcEIsdUJBQVMsQ0FBQTtJQUNULDZCQUFlLENBQUE7SUFDZixxREFBdUMsQ0FBQTtBQUN6QyxDQUFDLEVBSlcsVUFBVSxHQUFWLGtCQUFVLEtBQVYsa0JBQVUsUUFJckI7QUFzQlksUUFBQSxtQkFBbUIsR0FBRyxLQUFLLEVBQ3RDLE1BQTRCLEVBQ0gsRUFBRTtJQUMzQixNQUFNLFVBQVUsR0FBRyxJQUFJLG9CQUFVLEVBQUUsQ0FBQztJQUVwQyxJQUFJLFNBQTZCLENBQUM7SUFDbEMsSUFBSSxPQUFPLEdBQW1CLEVBQUUsQ0FBQztJQUVqQyxHQUFHO1FBQ0QsTUFBTSxNQUFNLEdBQXlCLE1BQU0sVUFBVTthQUNsRCxjQUFjLENBQUM7WUFDZCxVQUFVLEVBQUUsQ0FBQyxnQkFBZ0IsRUFBRSxhQUFhLENBQUM7WUFDN0MsR0FBRyxNQUFNO1lBQ1QsU0FBUyxFQUFFLFNBQVM7U0FDckIsQ0FBQzthQUNELE9BQU8sRUFBRSxDQUFDO1FBRWIsT0FBTyxHQUFHLE1BQU0sQ0FBQyxlQUFlO1lBQzlCLENBQUMsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUNaLE1BQU0sQ0FBQyxlQUFlLEVBQUUsR0FBRyxDQUFDLENBQUMsS0FBSyxFQUFFLEVBQUU7Z0JBQ3BDLE9BQU87b0JBQ0wsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRO29CQUN4QixLQUFLLEVBQUUsS0FBSyxDQUFDLFVBQVc7b0JBQ3hCLE1BQU0sRUFBRSxLQUFLLENBQUMsV0FBWTtvQkFDMUIsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLGdCQUFnQjtpQkFDekMsQ0FBQztZQUNKLENBQUMsQ0FBQyxDQUNIO1lBQ0gsQ0FBQyxDQUFDLE9BQU8sQ0FBQztRQUNaLE9BQU8sR0FBRyxNQUFNLENBQUMsWUFBWTtZQUMzQixDQUFDLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FDWixNQUFNLENBQUMsWUFBWSxFQUFFLEdBQUcsQ0FBQyxDQUFDLEtBQUssRUFBRSxFQUFFO2dCQUNqQyxPQUFPO29CQUNMLFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUTtvQkFDeEIsS0FBSyxFQUFFLEtBQUssQ0FBQyxVQUFXO29CQUN4QixNQUFNLEVBQUUsS0FBSyxDQUFDLFdBQVk7b0JBQzFCLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxnQkFBZ0I7aUJBQ3pDLENBQUM7WUFDSixDQUFDLENBQUMsQ0FDSDtZQUNILENBQUMsQ0FBQyxPQUFPLENBQUM7UUFFWixTQUFTLEdBQUcsTUFBTSxDQUFDLFNBQVMsQ0FBQztLQUM5QixRQUFRLFNBQVMsRUFBRTtJQUNwQixPQUFPLE9BQU8sQ0FBQztBQUNqQixDQUFDLENBQUM7QUFFRjs7OztHQUlHO0FBQ1UsUUFBQSxTQUFTLEdBQUcsS0FBSyxFQUM1QixXQUFxQixFQUNJLEVBQUU7SUFDM0IsTUFBTSxNQUFNLEdBQW1CLEVBQUUsQ0FBQztJQUNsQyxJQUFJO1FBQ0YsTUFBTSx5QkFBeUIsR0FBRyxNQUFNLDJCQUFtQixFQUFFLENBQUM7UUFDOUQsTUFBTSxNQUFNLEdBQUcsZ0JBQWdCLENBQUMseUJBQXlCLEVBQUUsV0FBVyxDQUFDLENBQUMsR0FBRyxDQUN6RSxDQUFDLEtBQUssRUFBRSxFQUFFO1lBQ1IsS0FBSyxDQUFDLE1BQU0sR0FBRyxHQUFHLEtBQUssQ0FBQyxRQUFRLE9BQU8sS0FBSyxDQUFDLEtBQUssV0FBVyxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUM7WUFDNUUsT0FBTyxLQUFLLENBQUM7UUFDZixDQUFDLENBQ0YsQ0FBQztRQUVGLElBQUksTUFBTSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDdkIsTUFBTSxDQUFDLElBQUksQ0FBQztnQkFDVixLQUFLLEVBQUUsVUFBVSxDQUFDLEtBQUs7Z0JBQ3ZCLE1BQU0sRUFBRSx1REFBdUQsV0FBVyxDQUFDLElBQUksQ0FDN0UsSUFBSSxDQUNMLEdBQUc7YUFDTCxDQUFDLENBQUM7U0FDSjtLQUNGO0lBQUMsT0FBTyxDQUFDLEVBQUU7UUFDVixNQUFNLE9BQU8sR0FBRyw0QkFBNEIsQ0FBQztRQUM3QyxNQUFNLENBQUMsSUFBSSxDQUFDLEVBQUUsTUFBTSxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUUsVUFBVSxDQUFDLEtBQUssRUFBRSxDQUFDLENBQUM7UUFDMUQsT0FBTyxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDLENBQUM7S0FDM0I7SUFDRCxPQUFPLE1BQU0sQ0FBQztBQUNoQixDQUFDLENBQUM7QUFFRjs7Ozs7R0FLRztBQUNVLFFBQUEsc0JBQXNCLEdBQUcsS0FBSyxFQUN6QyxXQUFxQixFQUNTLEVBQUU7SUFDaEMsTUFBTSxZQUFZLEdBQW1CLE1BQU0saUJBQVMsQ0FBQyxXQUFXLENBQUMsQ0FBQztJQUVsRSxPQUFPO1FBQ0wsWUFBWTtRQUNaLEtBQUssRUFDRixZQUFZLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxLQUFLLFVBQVUsQ0FBQyxLQUFLLENBQUM7WUFDckQsRUFBRSxLQUFvQixJQUFJLFVBQVUsQ0FBQyxFQUFFO1FBQzNDLE9BQU8sRUFBRSxHQUFHLFlBQVk7YUFDckIsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxLQUFLLFVBQVUsQ0FBQyxLQUFLLENBQUM7YUFDM0MsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDO2FBQ3BCLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRTtLQUNoQixDQUFDO0FBQ0osQ0FBQyxDQUFDO0FBRUYsTUFBTSxnQkFBZ0IsR0FBRyxDQUN2QixNQUFzQixFQUN0QixXQUFxQixFQUNMLEVBQUU7SUFDbEIsT0FBTyxNQUFNO1NBQ1YsT0FBTyxDQUFDLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FDakIsV0FBVztTQUNSLEdBQUcsQ0FBQyxDQUFDLFVBQVUsRUFBRSxFQUFFO1FBQ2xCLE9BQU8sS0FBSyxDQUFDLGdCQUFnQixFQUFFLFFBQVEsQ0FBQyxVQUFVLENBQUM7WUFDakQsQ0FBQyxDQUFDLEtBQUs7WUFDUCxDQUFDLENBQUMsU0FBUyxDQUFDO0lBQ2hCLENBQUMsQ0FBQztTQUNELE1BQU0sQ0FBQyxDQUFDLFNBQVMsRUFBRSxFQUFFLENBQUMsU0FBUyxDQUFDLENBQ3BDO1NBQ0EsTUFBTSxDQUNMLENBQUMsU0FBUyxFQUFFLEtBQUssRUFBRSxFQUFFLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxTQUFVLENBQUMsS0FBSyxLQUFLLENBQ3pDLENBQUM7QUFDeEIsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQ2xvdWRXYXRjaCB9IGZyb20gJ2F3cy1zZGsnO1xuaW1wb3J0IHtcbiAgRGVzY3JpYmVBbGFybXNJbnB1dCxcbiAgRGVzY3JpYmVBbGFybXNPdXRwdXQsXG59IGZyb20gJ2F3cy1zZGsvY2xpZW50cy9jbG91ZHdhdGNoJztcblxuZXhwb3J0IGVudW0gQWxhcm1TdGF0ZSB7XG4gIE9LID0gJ09LJyxcbiAgQUxBUk0gPSAnQUxBUk0nLFxuICBJTlNVRkZJQ0lFTlRfREFUQSA9ICdJTlNVRkZJQ0lFTlRfREFUQScsXG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgSUFsYXJtRGV0YWlsIHtcbiAgcmVhZG9ubHkgYWxhcm1Bcm4/OiBzdHJpbmc7XG4gIHJlYWRvbmx5IHN0YXRlOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGFsYXJtRGVzY3JpcHRpb24/OiBzdHJpbmc7XG4gIHJlYXNvbjogc3RyaW5nO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIEFnZ3JlZ2F0ZUFsYXJtU3RhdGUge1xuICByZWFkb25seSBhbGFybURldGFpbHM6IElBbGFybURldGFpbFtdO1xuICByZWFkb25seSBzdGF0ZTogQWxhcm1TdGF0ZTtcbiAgcmVhZG9ubHkgc3VtbWFyeTogc3RyaW5nO1xufVxuXG4vKipcbiAqIHNlYXJjaFRlcm1zOiBhIGxpc3Qgb2YgdGVybXMgdG8gbWF0Y2ggaW4gdGhlIGFsYXJtIGRlc2NyaXB0aW9uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgR2V0QWxhcm1TdGF0ZU9wdGlvbnMge1xuICByZWFkb25seSBzZWFyY2hUZXJtczogc3RyaW5nW107XG59XG5cbmV4cG9ydCBjb25zdCBnZXRDbG91ZHdhdGNoQWxhcm1zID0gYXN5bmMgKFxuICBwYXJhbXM/OiBEZXNjcmliZUFsYXJtc0lucHV0XG4pOiBQcm9taXNlPElBbGFybURldGFpbFtdPiA9PiB7XG4gIGNvbnN0IGNsb3Vkd2F0Y2ggPSBuZXcgQ2xvdWRXYXRjaCgpO1xuXG4gIGxldCBuZXh0VG9rZW46IHN0cmluZyB8IHVuZGVmaW5lZDtcbiAgbGV0IHJlc3VsdHM6IElBbGFybURldGFpbFtdID0gW107XG5cbiAgZG8ge1xuICAgIGNvbnN0IHJlc3VsdDogRGVzY3JpYmVBbGFybXNPdXRwdXQgPSBhd2FpdCBjbG91ZHdhdGNoXG4gICAgICAuZGVzY3JpYmVBbGFybXMoe1xuICAgICAgICBBbGFybVR5cGVzOiBbJ0NvbXBvc2l0ZUFsYXJtJywgJ01ldHJpY0FsYXJtJ10sXG4gICAgICAgIC4uLnBhcmFtcyxcbiAgICAgICAgTmV4dFRva2VuOiBuZXh0VG9rZW4sXG4gICAgICB9KVxuICAgICAgLnByb21pc2UoKTtcblxuICAgIHJlc3VsdHMgPSByZXN1bHQuQ29tcG9zaXRlQWxhcm1zXG4gICAgICA/IHJlc3VsdHMuY29uY2F0KFxuICAgICAgICAgIHJlc3VsdC5Db21wb3NpdGVBbGFybXM/Lm1hcCgoYWxhcm0pID0+IHtcbiAgICAgICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICAgIGFsYXJtQXJuOiBhbGFybS5BbGFybUFybixcbiAgICAgICAgICAgICAgc3RhdGU6IGFsYXJtLlN0YXRlVmFsdWUhLFxuICAgICAgICAgICAgICByZWFzb246IGFsYXJtLlN0YXRlUmVhc29uISxcbiAgICAgICAgICAgICAgYWxhcm1EZXNjcmlwdGlvbjogYWxhcm0uQWxhcm1EZXNjcmlwdGlvbixcbiAgICAgICAgICAgIH07XG4gICAgICAgICAgfSlcbiAgICAgICAgKVxuICAgICAgOiByZXN1bHRzO1xuICAgIHJlc3VsdHMgPSByZXN1bHQuTWV0cmljQWxhcm1zXG4gICAgICA/IHJlc3VsdHMuY29uY2F0KFxuICAgICAgICAgIHJlc3VsdC5NZXRyaWNBbGFybXM/Lm1hcCgoYWxhcm0pID0+IHtcbiAgICAgICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICAgIGFsYXJtQXJuOiBhbGFybS5BbGFybUFybixcbiAgICAgICAgICAgICAgc3RhdGU6IGFsYXJtLlN0YXRlVmFsdWUhLFxuICAgICAgICAgICAgICByZWFzb246IGFsYXJtLlN0YXRlUmVhc29uISxcbiAgICAgICAgICAgICAgYWxhcm1EZXNjcmlwdGlvbjogYWxhcm0uQWxhcm1EZXNjcmlwdGlvbixcbiAgICAgICAgICAgIH07XG4gICAgICAgICAgfSlcbiAgICAgICAgKVxuICAgICAgOiByZXN1bHRzO1xuXG4gICAgbmV4dFRva2VuID0gcmVzdWx0Lk5leHRUb2tlbjtcbiAgfSB3aGlsZSAobmV4dFRva2VuKTtcbiAgcmV0dXJuIHJlc3VsdHM7XG59O1xuXG4vKipcbiAqIFJldHVybnMgYWxsIGFsYXJtcyBzdGFydGluZyB3aXRoIHRoZSBwcm92aWRlZCBwcmVmaXggaW4gdGhlIHN0YXRlIG9mIEFMQVJNLiBJbiBhZGRpdGlvbixcbiAqIGFueSBleGNlcHRpb25zIGNhdWdodCB3aWxsIGJlIGxvZ2dlZCBhbmQgY29udmVydGVkIGludG8gYW4gYWxhcm0gc3RhdGUgb2Ygb3B0aW9ucy5kZWZhdWx0U3RhdGVcbiAqIHdpdGggYW4gYXNzb2NpYXRlZCByZWFzb24uXG4gKi9cbmV4cG9ydCBjb25zdCBnZXRBbGFybXMgPSBhc3luYyAoXG4gIHNlYXJjaFRlcm1zOiBzdHJpbmdbXVxuKTogUHJvbWlzZTxJQWxhcm1EZXRhaWxbXT4gPT4ge1xuICBjb25zdCBzdGF0ZXM6IElBbGFybURldGFpbFtdID0gW107XG4gIHRyeSB7XG4gICAgY29uc3QgYWxhcm1zRm9yQWNjb3VudEFuZFJlZ2lvbiA9IGF3YWl0IGdldENsb3Vkd2F0Y2hBbGFybXMoKTtcbiAgICBjb25zdCBhbGFybXMgPSBoYXNBbGFybU1hdGNoaW5nKGFsYXJtc0ZvckFjY291bnRBbmRSZWdpb24sIHNlYXJjaFRlcm1zKS5tYXAoXG4gICAgICAoYWxhcm0pID0+IHtcbiAgICAgICAgYWxhcm0ucmVhc29uID0gYCR7YWxhcm0uYWxhcm1Bcm59IGluICR7YWxhcm0uc3RhdGV9IGR1ZSB0byAke2FsYXJtLnJlYXNvbn1gO1xuICAgICAgICByZXR1cm4gYWxhcm07XG4gICAgICB9XG4gICAgKTtcblxuICAgIGlmIChhbGFybXMubGVuZ3RoID09PSAwKSB7XG4gICAgICBhbGFybXMucHVzaCh7XG4gICAgICAgIHN0YXRlOiBBbGFybVN0YXRlLkFMQVJNLFxuICAgICAgICByZWFzb246IGBObyBhbGFybXMgd2VyZSBmb3VuZCBmb3IgdGhlIHByb3ZpZGVkIHNlYXJjaCB0ZXJtczogJHtzZWFyY2hUZXJtcy5qb2luKFxuICAgICAgICAgICcsICdcbiAgICAgICAgKX0uYCxcbiAgICAgIH0pO1xuICAgIH1cbiAgfSBjYXRjaCAoZSkge1xuICAgIGNvbnN0IG1lc3NhZ2UgPSAnVW5hYmxlIHRvIHJldHJpZXZlIGFsYXJtcy4nO1xuICAgIHN0YXRlcy5wdXNoKHsgcmVhc29uOiBtZXNzYWdlLCBzdGF0ZTogQWxhcm1TdGF0ZS5BTEFSTSB9KTtcbiAgICBjb25zb2xlLmVycm9yKG1lc3NhZ2UsIGUpO1xuICB9XG4gIHJldHVybiBzdGF0ZXM7XG59O1xuXG4vKipcbiAqIENhbGN1bGF0ZSBhbmQgcmV0dXJuIGFuIGFnZ3JlZ2F0ZSBhbGFybSBzdGF0ZSBhY3Jvc3MgYWxhcm1zIHJldHVybmVkIGZyb20gYWxsXG4gKiBDbG91ZFdhdGNoIGNsaWVudHMgcHJvdmlkZWQuIElmIGFueSBjbGllbnQgZXJyb3JzIGFyZSB0aHJvd24sIHRoZSByZXR1cm5lZCBzdGF0ZVxuICogd2lsbCBiZSBBTEFSTS5cbiAqXG4gKi9cbmV4cG9ydCBjb25zdCBnZXRBZ2dyZWdhdGVBbGFybVN0YXRlID0gYXN5bmMgKFxuICBzZWFyY2hUZXJtczogc3RyaW5nW11cbik6IFByb21pc2U8QWdncmVnYXRlQWxhcm1TdGF0ZT4gPT4ge1xuICBjb25zdCBhbGFybURldGFpbHM6IElBbGFybURldGFpbFtdID0gYXdhaXQgZ2V0QWxhcm1zKHNlYXJjaFRlcm1zKTtcblxuICByZXR1cm4ge1xuICAgIGFsYXJtRGV0YWlscyxcbiAgICBzdGF0ZTpcbiAgICAgIChhbGFybURldGFpbHMuZmluZCgoYSkgPT4gYS5zdGF0ZSA9PT0gQWxhcm1TdGF0ZS5BTEFSTSlcbiAgICAgICAgPy5zdGF0ZSBhcyBBbGFybVN0YXRlKSA/PyBBbGFybVN0YXRlLk9LLFxuICAgIHN1bW1hcnk6IGAke2FsYXJtRGV0YWlsc1xuICAgICAgLmZpbHRlcigoYSkgPT4gYS5zdGF0ZSA9PT0gQWxhcm1TdGF0ZS5BTEFSTSlcbiAgICAgIC5tYXAoKGEpID0+IGEucmVhc29uKVxuICAgICAgLmpvaW4oJywgJyl9YCxcbiAgfTtcbn07XG5cbmNvbnN0IGhhc0FsYXJtTWF0Y2hpbmcgPSAoXG4gIGFsYXJtczogSUFsYXJtRGV0YWlsW10sXG4gIHNlYXJjaFRlcm1zOiBzdHJpbmdbXVxuKTogSUFsYXJtRGV0YWlsW10gPT4ge1xuICByZXR1cm4gYWxhcm1zXG4gICAgLmZsYXRNYXAoKGFsYXJtKSA9PlxuICAgICAgc2VhcmNoVGVybXNcbiAgICAgICAgLm1hcCgoc2VhcmNoVGVybSkgPT4ge1xuICAgICAgICAgIHJldHVybiBhbGFybS5hbGFybURlc2NyaXB0aW9uPy5pbmNsdWRlcyhzZWFyY2hUZXJtKVxuICAgICAgICAgICAgPyBhbGFybVxuICAgICAgICAgICAgOiB1bmRlZmluZWQ7XG4gICAgICAgIH0pXG4gICAgICAgIC5maWx0ZXIoKHNhbWVBbGFybSkgPT4gc2FtZUFsYXJtKVxuICAgIClcbiAgICAuZmlsdGVyKFxuICAgICAgKHNhbWVBbGFybSwgaW5kZXgpID0+IGFsYXJtcy5pbmRleE9mKHNhbWVBbGFybSEpID09PSBpbmRleFxuICAgICkgYXMgSUFsYXJtRGV0YWlsW107XG59O1xuIl19