# Copyright (C) 2023 Cochise Ruhulessin
#
# All rights reserved. No warranty, explicit or implicit, provided. In
# no event shall the author(s) be liable for any claim or damages.
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
import fastapi
from headless.ext.oauth2.models import TokenResponse

from cbra.types import Request
from ..types import OIDCProvider
from ..types import FatalAuthorizationException
from .localopenidprovider import LocalOpenIdProvider


__all__: list[str] = [
    'DownstreamTokenResponse'
]

async def get(
    request: Request,
    provider: OIDCProvider = LocalOpenIdProvider,
    expected_nonce: str = fastapi.Cookie(
        default=...,
        title="Nonce",
        description=(
            "The expected `nonce` claim in the ID Token, if requested."
        ),
        alias='oauth2.nonce'
    ),
    return_url: str = fastapi.Cookie(
        default=...,
        title="Return URL",
        alias='oauth2.return_url',
        description=(
            "The URL to redirect the user-agent to when an error occurs."
        )
    ),
    expected_issuer: str | None = fastapi.Cookie(
        default=None,
        title="Expected issuer",
        description=(
            "The expected value of the `iss` query parameter, if "
            "provided by the authorization server. This parameter "
            "is mandatory if the authorization server specifies "
            "`authorization_response_iss_parameter_supported=true` "
            "in its metadata."
        ),
        alias='oauth2.issuer'
    ),
    expected_state: str = fastapi.Cookie(
        default=...,
        title="Expected state",
        description=(
            "The expected value of the `state` query parameter."
        ),
        alias='oauth2.state'
    ),
    code: str | None = fastapi.Query(
        default=None,
        title='Authorization code',
        description=(
            'The authorization code generated by the authorization server. '
            'Required if using the Authorization Code Flow and the response '
            'mode is `query`, otherwise this parameter is ignored.'
        )
    ),
    state: str | None = fastapi.Query(
        default=None,
        title='State',
        description=(
            'An opaque value used by the client to maintain state '
            'between the request and callback. The authorization '
            'server includes this value when redirecting the '
            'user-agent back to the client. If supplied, this '
            'parameter **must** equal the `state` parameter used when '
            'creating the authorization request. The `state` parameter '
            'is ignored when using JARM because it is included in the '
            'JSON Web Token supplied using the `jwt` parameter per '
            'chosen response mode.'
        )
    ),
    iss: str | None = fastapi.Query(
        default=None,
        title='Issuer',
        description=(
            'Identifies the authorization server that redirected to '
            'this endpoint, as defined in RFC 9207. If the client supports '
            'OAuth 2.0 Authorization Server Issuer Identification, then '
            'this parameter is **required**, if the response mode is not '
            'JWT Secured Authorization Response Mode (JARM) per RFC 9101.\n\n'
            'The `iss` parameter value is the issuer identifier of '
            'the authorization server that created the authorization '
            'response, as defined in RFC 8414.  Its value **must** '
            'be a URL that uses the `https` scheme without any '
            'query or fragment components.'
        )
    ),
    error: str | None = fastapi.Query(
        default=None,
        title="Error code",
        description=(
            "The error code returned by the authorization server if "
            "the user cancelled the request, refused consent, or "
            "failed to authenticate."
        )
    )
) -> TokenResponse:
    if error:
        return None # type: ignore
    if code is None:
        raise FatalAuthorizationException(
            "The 'code' parameter is missing from the downstream "
            "response."
        )
    if state is None:
        raise FatalAuthorizationException(
            "The 'state' parameter is missing from the downstream "
            "response."
        )
    if state != expected_state:
        raise FatalAuthorizationException(
            "The 'state' parameter does not match the expected "
            "value."
        )
    if iss is not None and iss != expected_issuer:
        raise FatalAuthorizationException(
            "The 'iss' parameter mismatches the expected issuer."
        )
    token = await provider.obtain(
        code=code,
        state=state,
        redirect_uri=f'{request.url.scheme}://{request.url.netloc}{request.url.path}'
    )
    _, oidc = await provider.verify(token, nonce=expected_nonce)
    if oidc is None:
        raise FatalAuthorizationException(
            "The OIDC ID Token returned by the authorization server "
            "was malformed, invalid or otherwise unusable."
        )
    if oidc.email is None:
        raise FatalAuthorizationException(
            "The authorization server did not provide the 'email' "
            "claim."
        )
    return token


DownstreamTokenResponse: TokenResponse = fastapi.Depends(get)