import numpy as np

from mlhandmade.base.base import BaseEstimator
from .linear import LinearKernel
from .poly import PolynomialKernel
from .rbf import RBFKernel

kernels = {
    "linear" : LinearKernel,
    "poly" : PolynomialKernel,
    "rbf" : RBFKernel
}

class SupportVectorClassifier(BaseEstimator):
    """
    support vector classifier implemented with
    simplified SMO algorithm

    Parameters
    ----------
    kernel : Kernel
        kernel function to compute inner products
    C : float
        penalty of misclassification
    """
    def __init__(self, kernel="linear", C=np.Inf, **kernel_kwargs):
        self.kernel = kernels[kernel](**kernel_kwargs)
        self.C = C

    def _fit(self, X:np.ndarray, t:np.ndarray, tol:float=1e-8):
        """
        estimate support vectors and their parameters

        Parameters
        ----------
        X : (N, D) np.ndarray
            training independent variable
        t : (N,) np.ndarray
            training dependent variable
            binary -1 or 1
        tol : float, optional
            numerical tolerance (the default is 1e-8)
        """
        N = len(t)
        coef = np.zeros(N)
        grad = np.ones(N)
        Gram = self.kernel(X, X)

        while True:
            tg = t * grad
            mask_up = (t == 1) & (coef < self.C - tol)
            mask_up |= (t == -1) & (coef > tol)
            mask_down = (t == -1) & (coef < self.C - tol)
            mask_down |= (t == 1) & (coef > tol)
            i = np.where(mask_up)[0][np.argmax(tg[mask_up])]
            j = np.where(mask_down)[0][np.argmin(tg[mask_down])]
            if tg[i] < tg[j] + tol:
                self.b = 0.5 * (tg[i] + tg[j])
                break
            else:
                A = self.C - coef[i] if t[i] == 1 else coef[i]
                B = coef[j] if t[j] == 1 else self.C - coef[j]
                direction = (tg[i] - tg[j]) / (Gram[i, i] - 2 * Gram[i, j] + Gram[j, j])
                direction = min(A, B, direction)
                coef[i] += direction * t[i]
                coef[j] -= direction * t[j]
                grad -= direction * t * (Gram[i] - Gram[j])
        support_mask = coef > tol
        self.a = coef[support_mask]
        self.X = X[support_mask]
        self.t = t[support_mask]

    def _predict(self, x):
        """
        predict labels of the input

        Parameters
        ----------
        x : (sample_size, n_features) ndarray
            input

        Returns
        -------
        label : (sample_size,) ndarray
            predicted labels
        """
        y = self.decision_function(x)
        label = np.sign(y)
        return label

    def decision_function(self, x):
        """
        calculate distance from the decision boundary

        Parameters
        ----------
        x : (sample_size, n_features) ndarray
            input

        Returns
        -------
        distance : (sample_size,) ndarray
            distance from the boundary
        """
        distance = np.sum(
            self.a * self.t
            * self.kernel(x, self.X),
            axis=-1) + self.b
        return distance