# Lytspel, a Simple Phonetic Respelling for the English Language

The idea: Spelling should be fun, not a burden. The traditional English
spelling system is afflicted with exceptions and conflicting rules, making
writing and reading texts unnecessarily hard.

Lytspel is a proposal for reforming the English spelling in order to make
it strictly follow the alphabetic principle. The alphabetic principle means
that there is a predictable relationship between written letters and spoken
sounds. When you see a written word, you know how to pronounce it (even if
you don't know the word itself), and vice versa.

Traditional English spelling (tradspell) deviates from the alphabetic
principle in many ways. The mapping between written and spoken words is
very complex and riddled with irregularities and exceptions. Lytspel (for
«lytwait speling» or ‹lightweight spelling›) is a proposal to change this,
while keeping as much visual similarity to tradspell as reasonably
possible.

Lytspel not only creates an unambiguous mapping between the sounds used in
English and their written representations, it also indicates which syllable
in each word is the stressed one.

In addition to the reform proposal, there is a comprehensive dictionary and
a [converter](https://www.lytspel.org/) that translates traditional
spelling into Lytspel.

Here are two example paragraphs written in Lytspel. The first is [The North
Wind and the
Sun](https://en.wikipedia.org/wiki/The_North_Wind_and_the_Sun), one of
Aesop's fables that's frequently used as an example in phonetics, while the
second is made up of two nonsense sentences that contain most of the sounds
of English.

> The North Wind and the Sun wur dispueting which wos the strongger, when a
> traveler caim along rapd in a worm cloak. Thay agreed that the wun hu
> furst suxeeded in maiking the traveler taik his cloak of shood bee
> consíderd strongger than the uther. Then the North Wind blu as hard as hi
> cood, but the mor hi blu the mor cloassli did the traveler foald his
> cloak eround him; and at last the North Wind gaiv up the atempt. Then the
> Sun shynd out wormli, and imeediatli the traveler took of his cloak. And
> so the North Wind wos oblyjd tu conféss that the Sun wos the strongger ov
> the tuu.
>
> That quik baizh fox jumpd in the air oaver eech thin dog under a caam
> autum muun. Look out, y shout, for hi's foild ue yet agen, criáiting
> cayoss.

Here are the same paragraphs written in tradspell. Lytspel might look a
bit unusual at first, but should be easy to get used too. Just try reading
it out aloud and you'll soon get the hang of it.

> The North Wind and the Sun were disputing which was the stronger, when a
> traveler came along wrapped in a warm cloak. They agreed that the one who
> first succeeded in making the traveler take his cloak off should be
> considered stronger than the other. Then the North Wind blew as hard as
> he could, but the more he blew the more closely did the traveler fold his
> cloak around him; and at last the North Wind gave up the attempt. Then
> the Sun shined out warmly, and immediately the traveler took off his
> cloak. And so the North Wind was obliged to confess that the Sun was the
> stronger of the two.
>
> That quick beige fox jumped in the air over each thin dog under a calm
> autumn moon. Look out, I shout, for he's foiled you yet again, creating
> chaos.


## The Rules of Lytspel

The following formatting conventions are used in this document:

* «double guillemets» for Lytspel (and spellings that haven't changed)
* ‹single guillemets› for traditional spellings
* /slashes/ for phonetics, written in the
  [SAMPA](https://en.wikipedia.org/wiki/SAMPA_chart_for_English) alphabet
  (with some slight modifications to make it more suitable for English)

Abbreviations used:

* RP: [Received
  Pronunciation](https://en.wikipedia.org/wiki/Received_Pronunciation), the
  standard form of English spoken in the United Kingdom, especially in
  England and Wales
* GA: [General American](https://en.wikipedia.org/wiki/General_American),
  English as typically spoken in the United States


### Consonants

The consonants are written as follows:

* «b» /b/ as in «bed»
* «ch» /tS/ as in «much»
* «d» /d/ as in «desk»
* «f» /f/ as in «fat»
* «g» /g/ as in «big, garden, gess» ‹...guess›
* «h» /h/ as in «hot»
* «j» /dZ/ as in «joy, díjit» ‹...digit›
* «l» /l/ as in «leg»
* «m» /m/ as in «mad»
* «n» /n/ as in «now»
* «ng» /N/ as in «long»
* «p» /p/ as in «pop»
* «r» /r/ as in «run»
* «sh» /S/ as in «ship»
* «t» /t/ as in «test»
* «v» /v/ as in «ever»
* «w» /w/ as in «west»
* «wh» /hw/ as in «when». Many people speak «w» and «wh» [the same
  way](https://en.wikipedia.org/wiki/Pronunciation_of_English_%E2%9F%A8wh%E2%9F%A9#Wine.E2.80.93whine_merger),
  but not everybody does, and Lytspel preserves the distinction.
* «y» /j/ as in «yet»
* «zh» /Z/ as in «miraazh» ‹mirage›

The sound /k/ is written as «c» or «k», depending on context:

* «k» is used before «e, i, y», e.g. «keep, king, kynd, calcuelaition»
  ‹...kind, calculation›
* «k» is also used before «h» to prevent confusion with the digraph «ch»
  /tS/, e.g. «lunkhed» ‹lunkhead›
* «c» is used before any other letter, e.g. «cat, cost, crisp»
* «c» is also used at the end of words ending in unstressed /ak/ or /Ik/,
  e.g. «mainiac, majic, public» ‹maniac, magic...›
* «k» is used at the end of other words ending in /k/, e.g. «book, drink,
  quik, speek, naimsaik» ‹...quick, speak, namesake›

The sound /s/ is written as «c», «s», or «ss», depending on context:

* «c» is used between any vowel and a vowel starting with «e, i, y», e.g.
  «deecent, ixplicit, nececerri, dicyd» ‹decent, explicit, necessary,
  decide›
* «ss» is used between any vowel and a vowel starting with «a, o, u», e.g.
  «assault, épissoad, assumption» ‹assault, episode, assumption›
* «ss» is also used at the end of words except after any of the consonants
  /f, k, p, t, T/, e.g. «less, miss, évidenss» ‹...evidence›
* «s» is used anywhere else, e.g. «sun, desk, chips, upsets»
* «s» is also used in the prefixes «dis, mis» regardless of which letter
  follows, e.g. «disagree, disenchantment, misinturprit, misunderstand»
  ‹disagree, disenchantment, misinterpret, misunderstand›

The sound /z/ is written as «s» or «z», depending on context:

* «s» is used between two vowels, e.g. «visit, eesi, dusen, disyr, bisar,
  risult» ‹...easy, dozen, desire, bizarre, result›
* «s» is also used at the end of words except after any of the consonants
  /f, k, p, t, T/, e.g. «his, wishes, meens, paus, quis» ‹...means, pause,
  quiz›
* «z» is used anywhere else, e.g. «zeero, igzact, obzurv» ‹zero, exact,
  observe›

The two sounds traditionally written ‹th› are represented as follows:

* Voiceless /T/ is always written «th», e.g. «thin»
* Voiced /D/ is usually «dh», e.g. «smuudh, wurdhi» ‹smooth, worthy›
* However, it remains «th» in the frequent words «aulthó, tho, than, the,
  then, thay, them, thair, thiss, that, thuss» ‹although, though, than,
  the, then, they, them, their *or* there, this, that, thus› as well as in
  all words ending in /D@`/, such as «anuther, bother, muther, whether»
  ‹another, bother, mother, whether›
* This spelling is preserved in derivatives of all these words, such as
  «nevertheléss, nuntheléss» ‹nevertheless, nonetheless› from «the»,
  «themselvs» ‹themselves› from «them», «thairs, thairby, thairfor»
  ‹theirs, thereby, therefore› from «thair», «thees» ‹these› from «thiss»,
  «thoas» ‹those› from «that», «utherwys» ‹otherwise› from «uther» ‹other›.
  The adjectives «farthest, furthest» are considered derivatives of
  «farther, further» and thus also written with «th».
* To distinguish them, the small number of words ending in /T@`/ is written
  with «thur»: «Arthur, authur, eethur, panthur» ‹Arthur, author,
  ether/aether, panther›. Since no English words end in /T3:/, this
  spelling is unambiguous.


### Vowels

The short vowels are written as follows:

* «a» /a/ as in «cat»
* «e» /E/ as in «pen»
* «i» /I/ as in «big»
* «o» /Q/ as in «dog»
* «oo» /U/ as in «book»
* «u» /V/ as in «club»

If the vowels written with just one letter occur at the end of words, «h»
is appended, e.g. «Yaaweh, eh, huh» ‹Yahweh...›.

The long vowels and diphthongs are written as follows:

* «aa» /A/ as in «paam, faather» ‹palm, father›.

* «ai» /eI/ as in «aim, saint, faiss» ‹...face›. Written «ay» if it occurs
  at the end of words or before another vowel, e.g. «play, hóliday, layer,
  cayoss» ‹...holiday, layer, chaos›.

* «au» /O:/ as in «auther, paus» ‹author, pause›. Written «aw» if it occurs
  at the end of words or before another vowel, e.g. «law, drawing».

* «ee» /i:/ as in «tree, teem» ‹...team›. Written «i» if it occurs
  unstressed at the end of words or before another vowel, e.g. «fansi,
  nececiti, vidio, criaition» ‹fancy, necessity, video, creation›. This
  means that Lytspel cannot distinguish between unstressed /i:/ and
  unstressed /I/ (as in «big») in these positions, but that shouldn't hurt,
  as /I/ is rarely used in these positions and dictionaries often disagree
  on which of these sounds to use anyway.

    «i» is also used at the end of personal pronouns ending in /i:/ (as
    these are often less stressed than other words in a sentence): «hi,
    shi, wi, mi» ‹he, she, we, me›. And it's used at the end of prefixes
    such as «anti, semi» (spoken with /i:/ by many speakers, with /aI/ by
    others) as well as «di, pri, ri» ‹de, pre, re› (even if the prefix is
    stressed), e.g. «antisoashel; semifynel, semiautematic; digraid;
    primachóor, prirequisit; riact, riproduess» ‹antisocial; semifinal,
    semiautomatic; degrade; premature, prerequisite; react, reproduce›.

* «oa» /oU/ as in «boat, hoam» ‹...home›. Written «o» if it occurs at the
  end of words or before another vowel, e.g. «zeero, coóperait, sho,
  poetic» ‹zero, cooperate, show...›. This poses little risk of confusion,
  as /Q/ (as in «dog») occurs rarely or never before other vowels.

    «o» is also used at the end of prefixes such as «biyo, ethno, fyto,
    hetero, imueno, jio, keemo, macro, mycro, nio, nuero, palio, suudo,
    thurmo» ‹bio, ethno, phyto, hetero, immuno, geo, chemo, macro, micro,
    neo, neuro, paleo, pseudo, thermo›, e.g. «biyokemistri, ethnosentric,
    heterosecshual, imuenodifishenssi, jiofisicl, keemotherrepi, mycrowaiv,
    nioclassic, nuerosurjeri, suudosiyenss, thurmodynamics» ‹biochemistry,
    ethnocentric, heterosexual, immunodeficiency, geophysical,
    chemotherapy, microwave, neoclassic, neurosurgery, pseudoscience,
    thermodynamics›.

* «oi» /OI/ as in «oil, point, avoid». Written «oy» if it occurs at the end
  of words or before another vowel, e.g. «boy, enjoy, royel» ‹...royal›.

* «ou» /aU/ as in «mouth, doun» ‹...down›. Written «ow» if it occurs at the
  end of words or before another vowel, e.g. «now, power, alow» ‹...allow›.

* «uu» /u:/ as in «muun, ixcluud, gruup» ‹moon, exclude, group›. Written
  «u» if it occurs at the end of words or before another vowel, e.g. «clu,
  íshu, sichuaition» ‹clue, issue, situation›. This poses little risk of
  confusion, as /V/ (as in «bus») occurs rarely or never before other
  vowels.

* «ue» (/yU/ or /ju:/) as in «válue, cuet, compueter, móduelait» ‹value,
  cute, computer, modulate›. This spelling is also used in words that are
  typically spoken with /yU/ or /ju:/ in RP, with /u:/ or /U/ in GA, e.g.
  «due, nuetrel, tuen» ‹...neutral, tune›.

* «y» /aI/ as in «pryss, styl, dry» ‹price, style...›. Written «iy» if it
  occurs before another vowel (e.g. «diyámiter, quiyet, viyelenss, iyern»
  ‹diameter, quiet, violence, iron›) or at the end of words after another
  vowel (e.g. «raidiiy» ‹radii›), to prevent confusion with the consonant
  «y» /j/ (as in «yet»).


### The Schwa

The schwa is an unstressed and neutral vowel that occurs frequently in
English words. Lytspel generally writes it as «e» or «u». «u» is used
whenever the traditional spelling contains this vowel, e.g. «álbum,
nurvuss, suspénd» ‹album, nervous, suspend›. «e» is used otherwise, e.g.
«camel, problem, hiden, sentrel» ‹...hidden, central›.

This means that Lytspel doesn't distinguish between unstressed /V/ as in
«fundeméntel» ‹fundamental› or /E/ as in «índex» and the schwa /@/.
However, since the unstressed forms of these vowels sound quite similar to
the schwa and there is often no agreement between speakers whether to use
one or the other, this shouldn't hurt.

A schwa immediately preceding the primarily stressed syllable can also be
represented by one of the other three vowel letters («a, i, o», but not
«y»), e.g. «a» in «about», «i» in «imajinaition» ‹imagination›, «o» in
«compuet» ‹compute›. The reason for this is that the vowel preceding the
stressed syllable is often spoken so quickly that it's hard to decide
whether a schwa or one of the short vowels is pronounced and not all
speakers will agree on which one to use. For clarity, it's therefore better
to stick with the original representation of the vowel in such cases, even
if most speakers might reduce it to a schwa. Additionally, the reduced
vowel often becomes clearly audible in related words, e.g. while the second
«i» in «imajinaition» is often reduced to a schwa, in «imajin» ‹imagine›
it's clearly audible as /I/.

At the end of words, the schwa is always written «a», e.g. «extra, daita»
‹... data›. Before other vowels, it is always spelled «e» (never «u») to
prevent confusion with /u:/ (spelled «u» in this position). Schwa in this
position is very rare.

Between any of /b, p, k/ and word-final /l/, the schwa is omitted
altogether, e.g. «flexebl, simpl, articl» ‹flexible, simple, article›.
Likewise, final /z@m/ is written «sm» with the schwa omitted, e.g.
«criticism, sarcasm». (Why «sm» instead of "zm", you might ask? Because
this combination would be written «sem» if the schwa were not omitted --
see the notes on spelling /z/ above.)

Since all these consonant pairs never occur *without* an intervening vowel
at the end of words, there is no risk of confusion.


### Vowel Spellings before R

Before «r», the short vowels are spelled as follows:

* «arr» /ar/ as in «carri, embarress» ‹carry, embarrass›
* «ir» /Ir/ as in «mirer, íritait» ‹mirror, irritate›
* «er» /@`/ as in «number, modern» (r-colored schwa). Note: In RP and other
  [non-rhotic accents](https://en.wikipedia.org/wiki/Rhoticity_in_English),
  this usually sounds like a normal schwa; likewise, «r» after other vowels
  is often inaudible or reduced to a schwa.
* «err» /Er/ as in «merri, errer» ‹merry, error›
* «oor» /Ur/ as in «poor, toor, inshoorenss» ‹...tour, insurance›
* «orr» /Qr/ as in «sorri, tomórro» ‹sorry, tomorrow›. Note: Most GA
  speakers pronounce this in the same way as the «or» /O:r/ in «north», but
  the two sounds are distinguished in RP.
* «urr» /Vr/ as in «hurri, current» ‹hurry...›

The long vowels are spelled thus:

* «air» /e@/ as in «pair, shair, vairiaition» ‹...share, variation›
* «ar» /Ar/ as in «dark, arguement» ‹...argument›
* «eer» /i@/ as in «cheer, yeer» ‹...year›. Note: GA speakers often
  pronounce this in the same way as the «ir» /Ir/ in «mirer» ‹mirror›
  ([mirror–nearer
  merger](https://en.wikipedia.org/wiki/English-language_vowel_changes_before_historic_/r/#Mirror%E2%80%93nearer_merger)).
* «or» /O:r/ as in «north, order, port, ignor» ‹...ignore›. Note: Some
  speakers pronounce the vowel in «north» differently from the one in
  «port», but most do not ([horse–hoarse
  merger](https://en.wikipedia.org/wiki/English-language_vowel_changes_before_historic_%2Fr%2F#Horse%E2%80%93hoarse_merger)),
  and Lytspel does not express this distinction.
* «our» /aUr/ as in «flour, our» ‹flour, our *or* hour›
* «uer» (/ju:r/ or /jUr/) as in «Uerep, puer, obscuer» ‹Europe, pure,
  obscure›. This spelling is also used in words that are typically spoken
  with /ju:r/ or /jUr/ in RP, with /Ur/ in GA, e.g. «duering, nuerel»
  ‹during, neural›.
* «ur» /3:/ as in «turn, furst, consurn, obzurvaition» ‹...first, concern,
  observation›. This sound only occurs before «r» and can be considered a
  stronger (and often stressed) variant of the schwa.
* «yr» is spoken /aI@`/ at the end of words (e.g. «fyr, inspyr» ‹fire,
  inspire›), /aIr/ elsewhere (e.g. «vyruss, yreni, spyrel» ‹virus, irony,
  spiral›)


### Spellings Involving Several Sounds

* The vowel combination /i:@/ or /I@/ is written «ia», e.g. «misteeriass,
  mateerial, meediam, óbviass, vairiabl, airia, imeediat» ‹mysterious,
  material, medium, obvious, variable, area, immediate›. If the second
  vowel is stressed, this combination represents /i:a/ instead (e.g.
  «rialiti» ‹reality›), and in a few other words, it does so as well (e.g.
  «enthuesiasm, zoadiac» ‹enthusiasm, zodiac›) -- this distinction is not
  represented in writing.
* The r-colored vowel combination /i:@`/ is written «ier», e.g. «thieri»
  ‹theory›.
* The vowel combination /oU@/ is written «oe», e.g. «boe, coelition,
  yssócroess» ‹boa, coalition, isochroous›. If the second vowel is
  stressed, this combination represents /oUE/ instead (e.g. «poetic»), and
  in a few other words, it does so as well (e.g. «floem» ‹phloem›) -- this
  distinction is not represented in writing.
* The vowel combination /u:@/ or /U@/ is written «ua», e.g. «ácchual,
  indivijual, ínfluanss, soopurfluass, crual» ‹actual, individual,
  influence, superfluous, cruel›. If the second vowel is stressed, this
  combination represents /u:a/ instead (e.g. «secshualiti» ‹sexuality›),
  and in a few other words, it does so as well (e.g. «bivuac» ‹bivouac›) --
  this distinction is not represented in writing.
* The consonant combination /kw/ is written «qu», e.g. «quit, riquest»
  ‹...request›; the letter «q» only occurs in this combination.
* The consonant combination /ks/ is written «x», e.g. «mix, next, ixplicit,
  áxident» ‹...explicit, accident›.
* Generally, «ng» represents the single sound /N/ (as in «long»), but in
  words starting with any of «cong, eng, ing» not followed by another «g»
  or the end of the word, «ng» is usually pronounced /ng/ (e.g.
  «congrátuelait, engaij, ingreediant» ‹congratulate, engage, ingredient›).
  Elsewhere, /ng/ is written «nng» to distinguish it from /N/ (e.g.
  «cairnngorm, martenngail» ‹cairngorm, martingale›).
* Before /k/, «n» is pronounced /N/ rather then /n/ (e.g. «bank, tranquil,
  distinct, bronkytiss» ‹...bronchitis›), except if «n» and /k/ belong to
  different parts of a compound (e.g. «mankynd, raincoat» ‹mankind...›) or
  if a word starts with any of «con, en, in, non, un» followed by a /k/
  sound (e.g. «concluusion, encounter, incompitent, increess *or* íncreess,
  nonconformist, uncleer» ‹conclusion, encounter, incompetent, increase,
  nonconformist, unclear›). However, in some such words, /Nk/ is pronounced
  by some speakers (e.g. «conquest, íncrement, cóncreet, inquisition»
  ‹...increment, concrete, inquisition›) or by everyone (e.g. «conker, ink,
  íncling, uncl, unction» ‹conker *or* conquer, ink, inkling, uncle,
  unction›) -- this difference it not expressed in writing.
* /n/ is written «nn» if it occurs before /k/ in places where «n» would be
  spoken /N/, e.g. «canncan, melenncoli, noamennclaicher» ‹cancan,
  melancholy, nomenclature›.


### Spellings Used at the End of Words

Some sound combinations are written in a special way if they occur at the
end of words:

* Final /O:l/ is written «all», e.g. «ball, install». This spelling is
  preserved in derived words, e.g. «walls, calling, installment».
* Final /S@n/ is written «tion», e.g. «recognition, obzurvaition, section,
  permition, moation, oation, muesition» ‹...observation, section,
  permission, motion, ocean, musician›.
* Final /Z@n/ is written «sion», e.g. «vision, confuesion, iquaision»
  ‹...confusion, equation›.
* The «tion» and «sion» spellings aren't used in words that are derived by
  appending «n» or «en» to another word, e.g. «ashen, freshen, Rushan,
  Melaneezhan» ‹ashen, freshen, Russian, Melanesian› from «ash, fresh,
  Rusha, Melaneezha» ‹...Russia, Melanesia›.
* The «tion» and «sion» spellings are preserved in derived words (e.g.
  «rilaitionship, traditionel, ocaisionel» ‹relationship, traditional,
  occasional›), even if the derived form is irregular (e.g. «nationel»
  ‹national› from «naition» ‹nation›).


### Stress

Stress is marked using an acute accent in cases where it deviates from the
stress pattern predicted by a set of fairly reliable rules. The use of
accents is optional (they may be omitted), but recommended -- especially in
formal contexts such as books and newspapers. In a few cases, two words are
distinguished only by the presence of absence of an accent, e.g. «díscuss»
‹discus› versus «discuss».

The rules for predicting the default stress are as follows:

* If a word has just one syllable, no accent (stress marker) is used or
  needed.
* If a word ends in «ee», this final vowel is stressed (because unstressed
  /i:/ would be written «i» in this position) -- e.g. «agree».
* If a word ends in «a» (the schwa) or «i» (unstressed /i:/) and has just
  two syllables, the first vowel is stressed (because the second is
  unstressed by definition) -- e.g. «extra, hapi» ‹...happy›.
* If a word ends in any of «grefi, leji, ic, icl, ics, iti, sion, tion»,
  the vowel preceding this ending is stressed, e.g. «turminoleji,
  statistic, sicueriti, publicaition» ‹terminology, statistic, security,
  publication›.
* Otherwise the first long vowel in the word is considered stressed, e.g.
  «compleet, confurm, enjineer, ixplain, prisuem, sufyss» ‹complete,
  confirm, engineer, explain, presume, suffice›. Generally, all vowels
  except for «a, e, i, o, oo, u» and the schwa (as well as their r-colored
  variants) are considered long. However, with the exception of «y», if a
  long vowel is written with just one letter (because it precedes another
  vowel or the end of the word), it does not count for the purposes of this
  rule -- e.g. the «i» in «óbviass» ‹obvious› and the «o» in «windo»
  ‹window› do not count. If a word ends in any of «ait, y, ys, yt», the
  long vowel in this ending also does not count for the purposes of this
  rule.
* Otherwise, if a word starts with a short vowel or with «di» or «ri», the
  second vowel is considered stressed, e.g. «adopt, upon, difenss, riquest»
  ‹adopt, upon, defense, request›. If not, the first vowel is considered
  stressed, e.g. «current, quontiti» ‹...quantity›. But in either case, if
  the chosen vowel would be «er», the next vowel that is not «er» is chosen
  instead (because «er» is the r-colored schwa, which is never stressed) --
  e.g. «interject, perhaps». If there is no such next vowel, the first
  vowel is considered stressed, e.g. «errer» ‹error›.

If these rules fail to correctly predict the stress, an acute accent is
added on top of the stressed vowel. If this vowel has two letters, the
accent is added on top of the first one, e.g. «maintáin, uenéek» ‹maintain,
unique›.


### Homophones and Irregular Words

* The vowels in the articles «the, a, an» retain their traditional
  spelling.
* «ey» ‹eye› is written irregularly to distinguish it from «y» ‹I› and for
  easier recognition in compounds such as «fishey» ‹fisheye›.
* «four» is written irregularly to distinguish it from «for» ‹for *or*
  fore›.
* «noa» ‹know› is written somewhat irregularly to distinguish it from «no».
* «oa» ‹owe› is written somewhat irregularly since a verb with just one
  letter could be confusing (and also to distinguish it from the
  interjection «o» ‹oh›).
* «tuu» ‹too *or* two› is written somewhat irregularly to distinguish it
  from «tu» ‹to›.


### Rules for Derived and Related Words

* In front of suffixes starting with «e, i, y», final «c» is changed to
  «k», e.g. «trafiking» ‹trafficking› from «trafic» ‹traffic›.
* Final «c» remains unchanged in front of such suffixes if its
  pronunciation changes from /k/ to /s/, e.g. «publicist, publicys»
  ‹...publicize› from «public».
* In front of suffixes consisting in a single consonant («d, n, s»), final
  «i» (unstressed /i:/) becomes «ie» to mark it as long, e.g. «studied»
  from «studi» ‹study›, «vairies» ‹varies› from «vairi» ‹vary›. Otherwise
  forms such as "studid" would look like «splendid», but be spoken quite
  differently.
* To prevent misreadings, final «o» /oU/ is changed back to «oa» and final
  «u» /u:/ to «uu» in front of the same suffixes, e.g. «foload» ‹followed›
  from «folo» ‹follow›, «chuud» ‹chewed› from «chu» ‹chew›, «throan»
  ‹thrown› from «thro» ‹throw›, «struun» ‹strewn› from «stru» ‹strew›,
  «potaitoas» ‹potatoes› from «potaito» ‹potato›, «shuus» ‹shoes› from
  «shu» ‹shoe›.
* A schwa /@/ at the start of suffixes is written as «a» if the base word
  ends in /i:/ or /u:/, e.g. «eesiast» ‹easiest› from «eesi» ‹easy›,
  «vairiabl» ‹variable› from «vairi» ‹vary›, «duabl» ‹doable› from «du»
  ‹do›, «agreeabl» ‹agreeable› from «agree» ‹agree›. This ensures
  consistency with the combined spellings «ia» and «ua» used elsewhere (see
  above) and prevents confusion with «ie» in words such as «studied». But
  note that r-colored schwa remains «er», e.g. «eesier» ‹easier›.
* As noted above, the schwa is omitted between a few consonants and final
  «l». To prevent misreadings, the omitted schwa is restored as «e» in
  front of the suffixes «ait/et, erri, ism, ist, oid, uss, ys» ‹ate,
  ary..., ous, ise/ize› as well as before stressed «ee», e.g. «puepelerri»
  ‹pupillary› from «puepl» ‹pupil›, «voakelist» ‹vocalist› from «voacl»
  ‹vocal›, «lybeluss» ‹libelous› and «lybelee» ‹libelee› from «lybl»
  ‹libel›. The obvious exception are derived words where no schwa is
  spoken, e.g. «syclys» ‹cyclize› from «sycl» ‹cycle›, «simplism» from
  «simpl» ‹simple›.
* Otherwise, end-of-word spellings are preserved in all derived forms as
  long as the base word doesn't change (except for possible shifts of
  stress), e.g.

  * «c» representing /k/: «publicli, publicaition» ‹publicly, publication›
    from «public»; «mecanics, mecanicl» ‹mechanics, mechanical› from
    «mecanic» ‹mechanic›
  * «k» representing /k/: «weeks, weekli» ‹...weekly› from «week»
  * «ss» representing /s/: «classic» from «class»; «convínssd, convínssing»
    ‹convinced, convincing› from «convínss» ‹convince›; «anounssment»
    ‹announcement› from «anounss» ‹announce›; «prinssess» ‹princess› from
    «prinss» ‹prince›
  * «a» representing /@/: «extras» from «extra», «Amerrican» ‹American›
    from «Amerrica» ‹America›
  * «aw» representing /O:/: «draws, drawn, drawing» from «draw»
  * «i» representing unstressed /i:/: «fansiful, fansiing» ‹fanciful,
    fancying› from «fansi» ‹fancy›; «hapiness» ‹happiness› from «hapi»
    ‹happy›
  * «oy» representing /OI/: «distroys, distroyd, distroyer» ‹destroys,
    destroyed, destroyer› from «distroy» ‹destroy›
  * «y» representing /aI/ (unchanged even before vowels): «hyer» ‹higher›
    from «hy» ‹high›; «suplýer» ‹supplier› from «suplý» ‹supply›
  * «yr» representing /aI@`/: «inspyrd» ‹inspired› from «inspyr» ‹inspire›

* They are also preserved in compounds, e.g. «bakground, extraterestrial,
  fyrwurks» ‹background, extraterrestrial, fireworks›.
* If the pronunciation of the base word itself changes (beyond mere shifts
  of stress), the spelling reflects this, e.g. «creecher» ‹creature› from
  «criáit» ‹create›; «discution» ‹discussion› from «discuss»; «concluusion»
  ‹conclusion› from «concluud» ‹conclude›; «ácchual» ‹actual› from «act»;
  «assumption» from «assuum» ‹assume›; «mentaliti» ‹mentality› from
  «mentel» ‹mental›.
* The past tense of regular verbs is formed by appending «ed» if /@d/ is
  spoken (e.g. «silécted, waisted» ‹selected, wasted›), «d» if /d/ or /t/
  is spoken (e.g. «riveeld, consíderd, payd, fixd, convínss» ‹revealed,
  considered, paid, fixed, convinced›). The past tense of irregular verbs
  is spelled as it's spoken, e.g. «kept» from «keep».
* While in tradspell a noun and a related verb are often written the same
  even if spoken differently, in Lytspel the spelling reflects the actual
  pronunciation, e.g. «a suspect, tu suspéct; a récord, tu ricord; a houss,
  tu hous» ‹a suspect, to suspect; a record, to record; a house, to house›.
* Normally Lytspel doesn't use double consonants, except in those cases
  where a double consonant is spoken differently than a single one (e.g.
  «ss» spoken /s/ in certain contexts where «s» is spoken /z/). However,
  double consonants can also occur if they belong to different parts of a
  compound or if one belongs to a prefix or suffix and the other to the
  base word, e.g. «toatelli, suuperreejenel, unnececerri» ‹totally,
  superregional, unnecessary›.
* Unstressed final /Iks/ traditionally written ‹ics› is treated as if it is
  a plural (and hence written «ics») even if the corresponding singular
  form (ending in /Ik/) is unknown or rarely used, e.g. «astrofisics,
  diyanetics» ‹astrophysics, dianetics›. This is for consistency with the
  many cases where «ics» indeed marks a plural or quasi-plural, e.g.
  «ilectronics» ‹electronics› from «ilectronic» ‹electronic›.
* For consistency with forms derived by appending «li» ‹ly› to words ending
  in «cl» ‹cal› (e.g. «practiclli» ‹practically› from «practicl»
  ‹practical›), all forms derived by appending ‹ally› (pronounced /@li:/ or
  /li:/) to words ending in «c« are written «clli», e.g. «baissiclli»
  ‹basically› from «baissic» ‹basic›, «specificlli» ‹specifically› from
  «specific».

Stress marking in derived words:

* If a word is formed by joining two words into one, no accent is added or
  removed (e.g. «lyftym» ‹lifetime› from «lyf» + «tym», «élboruum»
  ‹elbowroom› from «élbo» + «ruum»). Suffixed forms likewise don't add or
  remove accents, except when the stressed syllable shifts, e.g.
  «ixperriméntel» ‹experimental› from «ixperriment», «uenivúrssel»
  ‹universal› from «uenivurss» ‹universe›.
* If a prefix is added in front of a word, no accent is added or removed if
  the prefix has two or more syllables, e.g. «antisoashel, oaverpryss»
  ‹antisocial, overprice›. In such cases -- just as in the case of
  compounds formed of two separate words -- usually both parts carry a
  certain amount of stress, so it's hardly necessary to indicate which has
  more.
* If a prefix has just one syllable, on the other hand, usually either it
  or the main word is audibly stressed -- more often than not the latter.
  If that is the case, no accent is added or removed, e.g. «riact,
  unlimited» ‹react...›. But if it's the prefix that is stressed, it is
  marked with an (acute) accent and any accents from the main word are
  removed, e.g. «nónsenss, prívue» ‹nonsense, preview›.


### Disambiguating Multigraphs

**Note:** This section is of interest mainly to dictionary makers. Users
only need to know that they should add a diaeresis if a vowel sequence
could otherwise be misread, e.g. «coöperaition» ‹cooperation› (pronounced
with two adjacent vowel sounds, not one as in «book»). If a consonant
sequence could be misread, a hyphen may be added, e.g. «sheeps-hed»
‹sheepshead› (could otherwise be read as «sheep-shed»).

* Spellings involving digraphs (such as «ch, sh, ss, ai, oa, oy, ur») and
  trigraphs (such as «arr, eer») are generally read from left to right: the
  first letter sequence that *can* be read as a multigraph *should* be read
  as such; e.g. «dueel» ‹dual *or* duel› is read as «d-ue-e-l» (/dj'u:@l/
  or /d'u:@l/), «layer» is read as «l-ay-er» (/l'eI@`/).
* But if any of «i, o, u» is followed by «aa» or «ai», the first vowel is
  read separately from the rest of the sequence, e.g. «coaala, apreeshiait,
  oaiciss, sichuaition» ‹koala, appreciate, oasis, situation›. The rare
  sequences «iay« and «uay» are read the same way, e.g. «dóciay» ‹dossier›.
* Since final /OI/ is spelled «oy», final «oi» unambiguously represents the
  two vowels /oUi:/. Except for a handful of words (e.g. «joi» ‹joey›),
  this combination occurs mostly in derived words, (e.g. «shoi» ‹showy›
  from «sho» ‹show›).
* «oic» at the end of words is always spoken /oUIk/ (two separate vowels),
  e.g. «hiroic, stoic» ‹heroic...›.
* Double «rr» always forms a trigraph with the vowel to its left, e.g.
  «Februerri» ‹February› is read as «F-e-b-r-u-err-i».
* In cases where the previous rules would lead to a misreading, Lytspel
  recommends adding a diaeresis on top of the first vowel that would
  otherwise be misinterpreted, e.g. «coöperaition, poït, suër»
  ‹cooperation, poet, sewer›. The recommended way to avoid a possible
  misreading in a consonant sequence is the insertion of a hyphen, e.g.
  «sheeps-hed» ‹sheepshead›. Note that if a word begins with «adh» (e.g.
  «adheer» ‹adhere›), the consonants are always read as /dh/ (two
  consonants), not as a single /D/, therefore no hyphen is needed in such
  cases.
* If «y» or «w» occurs between two vowels, it is in very rare cases useful
  to add a hyphen before this letter to make it clear that it is read
  standalone (as a consonant) instead of as second part of one of the long
  vowels «ay, oy, uy, aw, ow», e.g. «Ma-yorca, sooki-yaaki, Ha-wiyi»
  ‹Majorca, sukiyaki, Hawaii›. In words starting with «aw« followed by a
  vowel (e.g. «awaik» ‹awake›) this is not necessary, as they are always
  read as «a-w» (two sounds) rather than «aw» (one sound).
* If preferred, a hyphen instead of a diaeresis can also be used in the
  case of vowel sequences, e.g. «co-operaition, po-it».
* Diaereses are generally not inserted between a base word and a suffix,
  e.g. «truer» from «tru» ‹true›.
* After a prefix or the first part of a compound, a diaeresis is only
  inserted if the last letter of the first part and the first letter of the
  second part would otherwise form one of the combinations «oa, oi, oo, ou»
  (each of which represents a single vowel), e.g. «throäway, coïgzist,
  bloöut» ‹throwaway, coexist, blowout›. In other cases, no diaereses or
  hyphens are inserted, e.g. «shorthand».
* Writers may, of course, insert a diaeresis or hyphen between the parts of
  an affixed or compound word whenever they consider such a visual
  separator helpful to prevent misreadings or confusion.


### Spelling of Proper Nouns

* Proper nouns that are in widespread and general use should preferably be
  respelled, e.g. «Lunden, Juupiter, Micicípi» ‹London, Jupiter,
  Mississippi›. This might also apply to the names of well-known dead
  persons, e.g. «Wilyem Shaixpeer, Juulyuss Seeser» ‹William Shakespeare,
  Julius Caesar›.
* Personal, family, and company names are spelled as their owners prefer
  it.
* If Irish names starting with «O'» are respelled, the capitalized «O»
  followed by an apostrophe and a capital letter represents /oU/ followed
  by a stressed syllable, hence «O'Coner, O'Neel» ‹O'Connor, O'Neil› would
  be regular respellings.
* Scottish and Irish names traditionally starting with ‹Mc› or ‹Mac› /m@k/
  followed by a capitalized and stressed second syllable may be respelled
  by writing «Mc» followed by a capital letter instead of a stress marker,
  e.g. «McAdem, McDonel» ‹McAdam, McDonnell›. If the stressed syllable
  starts with a /k/ sound, that letter is capitalized (and repeated after
  the initial «Mc»), e.g. «McCarti, McKee, McQueen» ‹McCarty...›. If any
  other syllable is stressed, Lytspel recommend writing the word regularly
  without inner capitalization, e.g. «Makintosh» ‹Macintosh *or* McIntosh›.


## International Pronunciation Differences

As an international language, English is currently written and spoken
somewhat differently in various parts of the world.

On the one hand, by using a phonetic spelling, Lytspel eliminates most
differences that traditionally exist between written British (BE) and
American English (AE), e.g. «culer, senter, orgenys, ánelys, lycenss,
catalog, aijing, program, chek, gray» ‹colour/color, centre/center,
organise/organize, analyse/analyze, licence/license, catalogue/catalog,
ageing/aging, programme/program, cheque/check, grey/gray›.

On the other hand, tailoring Lytspel to faithfully represent just one
regional variety would cause many deviations from the alphabetic principle
for everyone else. And creating different variants of Lytspel each fitting
just one regional variety would introduce even larger differences between
different regional varieties of written English than traditionally exist
between BE and AE.

Instead of going down either of theses routes, Lytspel had been designed as
some kind of "global compromise." It aims to keep the phonetic principle
reasonably intact for all varieties of spoken English, and particularly for
RP (since the English language originates in England) and GA (since the
United States are the country with the highest number of native English
speakers).

Several details of the Lytspel spellings have already been motivated as
representing sound distinctions that exist either in RP or in GA. But such
compromises are not almost possible without making the written language
extremely complex and unwieldy. In other cases, Lytspel therefore resolves
differences between RP and GA by choosing one spelling -- often the one
that keeps the written language more similar to tradspell or else the
shorter one:

* /a/ («a») is preferred if a vowel is commonly spoken as either /a/ or
  /A/, e.g. «last, guulash» ‹...goulash›.
* /a/ is also preferred if a vowel is commonly spoken as either /a/ or
  /eI/, e.g. «patent, patriat» ‹...patriot›.
* /a/ is also preferred if a vowel is commonly spoken as either /a/ or /Q/,
  e.g. «rath» ‹wrath›.
* /A/ («aa)» is preferred if a vowel is commonly spoken as either /A/ or
  /eI/, e.g. «tomaato, swaadh» ‹tomato, swathe›.
* /aI/ («y», «iy» before vowels) is preferred if a vowel is commonly spoken
  as either /aI/ or /i:/, e.g. «nydher, fílistyn» ‹neither, philistine›.
* /E/ («e)» is preferred if a vowel is commonly spoken as either /E/ or
  /eI/, e.g. «agen» ‹again›.
* /E/ is also preferred if a vowel is commonly spoken as either /E/ or
  /i:/, e.g. «lezher, predececer» ‹leisure, predecessor›.
* /I/ («i») is preferred if a vowel is commonly spoken as either /I/ or
  /aI/, e.g. «cristelin, direct, orgenisaition» ‹crystalline, direct,
  organization›.
* /I/ is also preferred if a vowel is commonly spoken as either /I/ or /i:/
  (e.g. «afrodísiac» ‹aphrodisiac›), unless its original representation is
  ‹ee› (e.g. «been»).
* /i:/ («ee», «i» before vowels) is preferred if a vowel is commonly spoken
  as either /i:/ or /eI/, e.g. «beeta, gaseebo, Ízrial» ‹beta, gazebo,
  Israel›.
* /Q/ («o») is preferred if a vowel is commonly spoken as either /Q/ or
  /O:/, e.g. «álkehol, cloth» ‹alcohol...›.
* /Q/ is also preferred if a vowel is commonly spoken as either /Q/ or
  /oU/, e.g. «process, shon» ‹...shone›.
* /Q/ is also preferred if a vowel is commonly spoken as either /Q/ or /V/,
  e.g. «from, whot, acomplish» ‹...what, accomplish›.
* /oU/ («oa») is preferred if a vowel is commonly spoken as either /oU/ or
  /O:/, e.g. «sloath» ‹sloth›.
* /U/ («oo») is preferred if a vowel is commonly spoken as either /U/ or
  /u:/, e.g. «coocu, hoof» ‹cuckoo...›.
* /jU/ «ue» is preferred if a vowel is commonly spoken as either /jU/ or
  /j@/, e.g. «dócuement, ócuepy» ‹document, occupy›.
* /Ir/ («ir») is preferred if an r-colored vowel is commonly spoken as
  either /Ir/ or /3:/, e.g. «squirel» ‹squirrel›.
* /3:/ («ur») is preferred if an r-colored vowel is commonly spoken as
  either /3:/ or /A(r)/, e.g. «clurk» ‹clerk›.
* /w/ («w») is preferred if a sound preceding a vowel is commonly spoken as
  either /w/ or one of /ju:/ or /jU/, e.g. «jágwar» ‹jaguar›.
* If a vowel is spoken as a schwa or omitted altogether in some regions, as
  /aI/ elsewhere, Lytspel generally uses «i» /I/ as a compromise, e.g.
  «ájil, vursetil», ‹agile, versatile›.
* Otherwise, if a vowel is spoken with a full vowel sound in some regions,
  while reduced to a schwa or omitted altogether elsewhere, Lytspel tends
  to write the full version, e.g. «cátegori, ínevaitiv, militerri,
  portrait, prymerri, python, roazmairi, téstimoani» ‹category, innovative,
  military, portrait, primary, python, rosemary, testimony›.
* If some people pronounce /I/ and others a schwa, Lytspel uses «i» if
  that's the original representation, e.g. «simplify».
* /z/ («s» or «z», depending on position) is preferred if a consonant is
  commonly spoken as either /z/ or /s/, e.g. «irais, venisen, talizmen»
  ‹erase, venison, talisman›.
* If a consonant is commonly spoken as either /Z/ or /S/, final «sion»
  /Z@n/ is preserved if it's already the traditional ending (e.g.
  «convursion» ‹conversion›), otherwise /S/ (usually «sh») is preferred,
  e.g. «Aisha, iraisher» ‹Asia, erasure›.
* If some people pronounce /D/ and others /T/, «th» (or «ther») is
  preferred, e.g. «with, zither».
* /si:/ (depending on position, but often «ci») is preferred if a sequence
  is commonly spoken as either /si:/ or /S/, e.g. «hecian, omniciant»
  ‹hessian, omniscient›.
* /zi:/ (depending on position, but often «si») is preferred if a sequence
  is commonly spoken as either /zi:/ or /Z/, e.g. «hoasieri, Malaisia»
  ‹hosiery, Malaysia›.
* /ti:/ («ti» before vowels) is preferred if a sequence is commonly spoken
  as either /ti:/ or as one of /S, Si:, tS, tSi:/, e.g. «sentiant,
  consortiam, bestial, bestierri» ‹sentient, consortium, bestial,
  bestiary›.

If a letter is commonly either spoken or silent, Lytspel tends to preserve
it, e.g. «platinum, trait» (not "platnum, tray").

If speakers in different regions tend to stress different syllables,
Lytspel usually places the stress early rather than late, e.g. «an ádress,
décaid, truncait» ‹an address, decade, truncate›. But in the case of
originally foreign words where a late stress placement corresponds to the
pronunciation in the original language, it is often preserved, e.g.
«cafeen, expoasáy» ‹caffeine, exposé›.

Other differences are usually unique to just one or very few words. In such
cases, Lytspel generally prefers the pronunciation that is more similar to
the traditional written form, e.g. «cordial, figer, shéduel» ‹...figure,
schedule› (rather than "corjel, figyer, skejuul").

In a few cases, one pronunciation is preferred because it is more similar
to that of a closely related word, e.g. «pryveci» ‹privacy› from «pryvet»
‹private› (rather than "priveci").


## Limitations of the Current Dictionary

The Lytspel dictionary, as currently published, can distinguish between
words traditionally written the same but pronounced differently
(heteronyms) as long as their grammatical role in a sentence is different
-- one is a noun, while the other is a verb, for example. This allows
distinguishing the verb «cloas» ‹close› from the adjective «cloass», the
noun «óbject» from the verb «object», etc. («Thay wur tuu cloass tu the dor
tu cloas it.» ‹They were too close to the door to close it.› / «Y did not
object tu the óbject.» ‹I did not object to the object.›)

However, heteronyms are currently only disambiguated by their grammatical
role (commonly known as *part of speech,* or *POS*), hence if two
heteronyms share the same grammatical role, the dictionary will list only
one of them. The other one should, of course, nevertheless be used when
appropriate. Words where this is the case include (the missing spelling is
listed in parentheses):

* ‹bass›: «baiss» (or «bass» for various fish species)
* ‹drawer›: «dror» (or «drawer» for a person who draws)
* ‹logos›: «loagoas» (plural of «loago» ‹logo›; or «logoss» for the Greek
  word)
* ‹micrometer›: «mycrómiter» (a measuring device; or «mycromeeter» for the
  millionth part of a meter)
* ‹prayer›: «prair» (or «prayer» for a person who prays)
* ‹read›: «reed» (or «red» for the past tense)
* ‹routed, router, routing›: «ruuted, ruuter, ruuting» (if derived from
  «ruut» ‹route›; or «routed, router, routing» if derived from «rout»)
* ‹subsequence›: «subsiquenss» (for some subsequent occurrence; or
  «súbseequenss» for a sequence derived from another sequence)
* ‹taxis›: «taxies» (multiple taxis, or taxicabs; or «taxiss», a scientific
  term)
* ‹tier›: «teer» (or «tyer» for a person who ties)


## Other Documentation

**docs/lytspel-on-two-pages.md|odt|pdf**: A compact, two-page description
of the most important rules. The PDF is exported from the ODT
(OpenDocument/LibreOffice) document, while the Markdown (md) version is
maintained independently.

**LICENSE.txt** contains the license used for the Lytspel program suite (a
permissive license known as "ISC license"). The proposed spellings do not
fall under copyright (you cannot copyright a single word, let alone a way
to spell it) and may be considered a part of the public domain.

**CHANGELOG.md** contains a compact list of the changes made between the
various released versions of the program (and dictionary).

If you want to install and use the program locally (rather then using the
web frontend), see **INSTALL-USE.md**.

If you want to experiment with the program suite locally, in order to add
some functionality, fix bugs, or modify spellings, see **devnotes.md**.

**files.md** quickly describes the various files in the Lytspel repository.
This too is mostly interesting if you want to modify the dictionary or some
of the programs used to generate it.
