# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['fakeredis']

package_data = \
{'': ['*']}

install_requires = \
['redis<4.4', 'sortedcontainers>=2.4.0,<3.0.0']

extras_require = \
{'aioredis': ['aioredis>=2.0.1,<3.0.0'], 'lua': ['lupa>=1.13,<2.0']}

setup_kwargs = {
    'name': 'fakeredis',
    'version': '1.9.3',
    'description': 'Fake implementation of redis API for testing purposes.',
    'long_description': 'fakeredis: A fake version of a redis-py\n=======================================\n\n![badge](https://img.shields.io/endpoint?url=https://gist.githubusercontent.com/cunla/b756396efb895f0e34558c980f1ca0c7/raw/fakeredis-py.json)\n\n- [fakeredis: A fake version of a redis-py](#fakeredis--a-fake-version-of-a-redis-py)\n- [How to Use](#how-to-use)\n  - [Use to test django-rq](#use-to-test-django-rq)\n- [Other limitations](#other-limitations)\n- [Support for redis-py <4.2 with aioredis](#support-for-redis-py--42-with-aioredis)\n    + [aioredis 1.x](#aioredis-1x)\n    + [aioredis 2.x](#aioredis-2x)\n- [Running the Tests](#running-the-tests)\n- [Contributing](#contributing)\n- [Alternatives](#alternatives)\n\nfakeredis is a pure-Python implementation of the redis-py python client\nthat simulates talking to a redis server. This was created for a single\npurpose: **to write unittests**. Setting up redis is not hard, but\nmany times you want to write unittests that do not talk to an external server\n(such as redis). This module now allows tests to simply use this\nmodule as a reasonable substitute for redis.\n\nAlthough fakeredis is pure Python, you will need [lupa](https://pypi.org/project/lupa/) if you want to run Lua\nscripts (this includes features like ``redis.lock.Lock``, which are implemented\nin Lua). If you install fakeredis with ``pip install fakeredis[lua]`` it will\nbe automatically installed.\n\nFor a list of supported/unsupported redis commands, see [REDIS_COMMANDS.md](REDIS_COMMANDS.md)\n\n# How to Use\nFakeRedis can imitate Redis server version 6.x or 7.x - There are a few minor behavior differences. \nIf you do not specify the version, version 7 is used by default.\n\nThe intent is for fakeredis to act as though you\'re talking to a real\nredis server. It does this by storing state internally.\nFor example:\n\n```\n>>> import fakeredis\n>>> r = fakeredis.FakeStrictRedis(version=6)\n>>> r.set(\'foo\', \'bar\')\nTrue\n>>> r.get(\'foo\')\n\'bar\'\n>>> r.lpush(\'bar\', 1)\n1\n>>> r.lpush(\'bar\', 2)\n2\n>>> r.lrange(\'bar\', 0, -1)\n[2, 1]\n```\n\nThe state is stored in an instance of `FakeServer`. If one is not provided at\nconstruction, a new instance is automatically created for you, but you can\nexplicitly create one to share state:\n\n```\n>>> import fakeredis\n>>> server = fakeredis.FakeServer()\n>>> r1 = fakeredis.FakeStrictRedis(server=server)\n>>> r1.set(\'foo\', \'bar\')\nTrue\n>>> r2 = fakeredis.FakeStrictRedis(server=server)\n>>> r2.get(\'foo\')\n\'bar\'\n>>> r2.set(\'bar\', \'baz\')\nTrue\n>>> r1.get(\'bar\')\n\'baz\'\n>>> r2.get(\'bar\')\n\'baz\'\n```\n\nIt is also possible to mock connection errors so you can effectively test\nyour error handling. Simply set the connected attribute of the server to\n`False` after initialization.\n\n```\n>>> import fakeredis\n>>> server = fakeredis.FakeServer()\n>>> server.connected = False\n>>> r = fakeredis.FakeStrictRedis(server=server)\n>>> r.set(\'foo\', \'bar\')\nConnectionError: FakeRedis is emulating a connection error.\n>>> server.connected = True\n>>> r.set(\'foo\', \'bar\')\nTrue\n```\n\nFakeredis implements the same interface as `redis-py`, the\npopular redis client for python, and models the responses\nof redis 6.2 (although most new features are not supported).\n\n## Use to test django-rq\n\nThere is a need to override `django_rq.queues.get_redis_connection` with\na method returning the same connection.\n\n```python\nfrom fakeredis import FakeRedisConnSingleton\n\ndjango_rq.queues.get_redis_connection = FakeRedisConnSingleton()\n```\n\n# Other limitations\n\nApart from unimplemented commands, there are a number of cases where fakeredis\nwon\'t give identical results to real redis. The following are differences that\nare unlikely to ever be fixed; there are also differences that are fixable\n(such as commands that do not support all features) which should be filed as\nbugs in Github.\n\n1. Hyperloglogs are implemented using sets underneath. This means that the\n   `type` command will return the wrong answer, you can\'t use `get` to retrieve\n   the encoded value, and counts will be slightly different (they will in fact be\n   exact).\n\n2. When a command has multiple error conditions, such as operating on a key of\n   the wrong type and an integer argument is not well-formed, the choice of\n   error to return may not match redis.\n\n3. The `incrbyfloat` and `hincrbyfloat` commands in redis use the C `long\n   double` type, which typically has more precision than Python\'s `float`\n   type.\n\n4. Redis makes guarantees about the order in which clients blocked on blocking\n   commands are woken up. Fakeredis does not honour these guarantees.\n\n5. Where redis contains bugs, fakeredis generally does not try to provide exact\n   bug-compatibility. It\'s not practical for fakeredis to try to match the set\n   of bugs in your specific version of redis.\n\n6. There are a number of cases where the behaviour of redis is undefined, such\n   as the order of elements returned by set and hash commands. Fakeredis will\n   generally not produce the same results, and in Python versions before 3.6\n   may produce different results each time the process is re-run.\n\n7. SCAN/ZSCAN/HSCAN/SSCAN will not necessarily iterate all items if items are\n   deleted or renamed during iteration. They also won\'t necessarily iterate in\n   the same chunk sizes or the same order as redis.\n\n8. DUMP/RESTORE will not return or expect data in the RDB format. Instead the\n   `pickle` module is used to mimic an opaque and non-standard format.\n   **WARNING**: Do not use RESTORE with untrusted data, as a malicious pickle\n   can execute arbitrary code.\n\n# Support for redis-py <4.2 with aioredis\n\nAioredis is now in redis-py 4.2.0. But support is maintained until fakeredis 2 for older version of redis-py.\n\nYou can also use fakeredis to mock out [aioredis](https://aioredis.readthedocs.io/). This is a much newer\naddition to fakeredis (added in 1.4.0) with less testing, so your mileage may\nvary. Both version 1 and version 2 (which have very different APIs) are\nsupported. The API provided by fakeredis depends on the version of aioredis that is\ninstalled.\n\n### aioredis 1.x\n\nExample:\n\n```\n>>> import fakeredis.aioredis\n>>> r = await fakeredis.aioredis.create_redis_pool()\n>>> await r.set(\'foo\', \'bar\')\nTrue\n>>> await r.get(\'foo\')\nb\'bar\'\n```\n\nYou can pass a `FakeServer` as the first argument to `create_redis` or\n`create_redis_pool` to share state (you can even share state with a\n`fakeredis.FakeRedis`). It should even be safe to do this state sharing between\nthreads (as long as each connection/pool is only used in one thread).\n\nIt is highly recommended that you only use the aioredis support with\nPython 3.5.3 or higher. Earlier versions will not work correctly with\nnon-default event loops.\n\n### aioredis 2.x\n\nExample:\n\n```\n>>> import fakeredis.aioredis\n>>> r = fakeredis.aioredis.FakeRedis()\n>>> await r.set(\'foo\', \'bar\')\nTrue\n>>> await r.get(\'foo\')\nb\'bar\'\n```\n\nThe support is essentially the same as for redis-py e.g., you can pass a\n`server` keyword argument to the `FakeRedis` constructor.\n\n# Running the Tests\n\nTo ensure parity with the real redis, there are a set of integration tests\nthat mirror the unittests. For every unittest that is written, the same\ntest is run against a real redis instance using a real redis-py client\ninstance. In order to run these tests you must have a redis server running\non localhost, port 6379 (the default settings). **WARNING**: the tests will\ncompletely wipe your database!\n\nFirst install poetry if you don\'t have it, and then install all the dependencies:\n\n```   \npip install poetry\npoetry install\n``` \n\nTo run all the tests:\n\n```\npoetry run pytest -v\n```\n\nIf you only want to run tests against fake redis, without a real redis::\n\n```\npoetry run pytest -m fake\n```\n\nBecause this module is attempting to provide the same interface as `redis-py`,\nthe python bindings to redis, a reasonable way to test this to to take each\nunittest and run it against a real redis server. fakeredis and the real redis\nserver should give the same result. To run tests against a real redis instance\ninstead::\n\n```\npoetry run pytest -m real\n```\n\nIf redis is not running and you try to run tests against a real redis server,\nthese tests will have a result of \'s\' for skipped.\n\nThere are some tests that test redis blocking operations that are somewhat\nslow. If you want to skip these tests during day to day development,\nthey have all been tagged as \'slow\' so you can skip them by running::\n\n```\npoetry run pytest -m "not slow"\n```\n\n# Contributing\n\nContributions are welcome. Please see the\n[contributing guide](.github/CONTRIBUTING.md) for more details.\nThe maintainer generally has very little time to work on fakeredis, so the\nbest way to get a bug fixed is to contribute a pull request.\n\nIf you\'d like to help out, you can start with any of the issues\nlabeled with `Help wanted`.\n\n# Alternatives\n\nConsider using [redislite](https://redislite.readthedocs.io/en/latest/) instead of fakeredis.\nIt runs a real redis server and connects to it over a UNIX domain socket, so it will behave just like a real\nserver. Another alternative is [birdisle](https://birdisle.readthedocs.io/en/latest/), which\nruns the redis code as a Python extension (no separate process), but which is currently unmaintained.\n',
    'author': 'James Saryerwinnie',
    'author_email': 'js@jamesls.com',
    'maintainer': 'Daniel Moran',
    'maintainer_email': 'daniel.maruani@gmail.com',
    'url': 'https://github.com/cunla/fakeredis-py',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
