import pytest

from sokoenginepy.game import (
    BaseTessellation,
    Direction,
    OctobanTessellation,
    Tessellation,
    TriobanTessellation,
    index_1d,
    is_on_board_1d,
)
from sokoenginepy.io import CellOrientation


def triangle_points_down(position, board_width, board_height):
    return (
        TriobanTessellation().cell_orientation(position, board_width, board_height)
        == CellOrientation.TRIANGLE_DOWN
    )


def is_octagon(position, board_width, board_height):
    return (
        OctobanTessellation().cell_orientation(position, board_width, board_height)
        == CellOrientation.OCTAGON
    )


class SokobanTessellationAutogeneratedSpecMixin:
    class Describe_neighbor_position:
        def test_autogenerated_topLeft(self):
            width = 10
            height = 10
            row = 0
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.SOKOBAN)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(1,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.UP, width, height), width, height)
            assert t.neighbor_position(index, Direction.DOWN, width, height) == index_1d(0,1, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_EAST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_EAST, width, height)

        def test_autogenerated_topRight(self):
            width = 10
            height = 10
            row = 0
            column = 9
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.SOKOBAN)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(8,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.UP, width, height), width, height)
            assert t.neighbor_position(index, Direction.DOWN, width, height) == index_1d(9,1, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_EAST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_EAST, width, height)

        def test_autogenerated_bottomLeft(self):
            width = 10
            height = 10
            row = 9
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.SOKOBAN)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(1,9, width)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(0,8, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.DOWN, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_EAST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_EAST, width, height)

        def test_autogenerated_bottomRight(self):
            width = 10
            height = 10
            row = 9
            column = 9
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.SOKOBAN)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(8,9, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(9,8, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.DOWN, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_EAST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_EAST, width, height)

        def test_autogenerated_midleTop(self):
            width = 10
            height = 10
            row = 0
            column = 5
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.SOKOBAN)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(4,0, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(6,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.UP, width, height), width, height)
            assert t.neighbor_position(index, Direction.DOWN, width, height) == index_1d(5,1, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_EAST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_EAST, width, height)

        def test_autogenerated_midleBottom(self):
            width = 10
            height = 10
            row = 9
            column = 5
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.SOKOBAN)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(4,9, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(6,9, width)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(5,8, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.DOWN, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_EAST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_EAST, width, height)

        def test_autogenerated_midleLeft(self):
            width = 10
            height = 10
            row = 5
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.SOKOBAN)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(1,5, width)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(0,4, width)
            assert t.neighbor_position(index, Direction.DOWN, width, height) == index_1d(0,6, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_EAST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_EAST, width, height)

        def test_autogenerated_midleRight(self):
            width = 10
            height = 10
            row = 5
            column = 9
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.SOKOBAN)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(8,5, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(9,4, width)
            assert t.neighbor_position(index, Direction.DOWN, width, height) == index_1d(9,6, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_EAST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_EAST, width, height)

        def test_autogenerated_board_1x1(self):
            width = 1
            height = 1
            row = 0
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.SOKOBAN)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.UP, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.DOWN, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_EAST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_EAST, width, height)

        def test_autogenerated_board_2x1_left(self):
            width = 2
            height = 1
            row = 0
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.SOKOBAN)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(1,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.UP, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.DOWN, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_EAST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_EAST, width, height)

        def test_autogenerated_boarD_2x1_right(self):
            width = 2
            height = 1
            row = 0
            column = 1
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.SOKOBAN)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(0,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.UP, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.DOWN, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_EAST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_EAST, width, height)

        def test_autogenerated_1x2_top(self):
            width = 1
            height = 2
            row = 0
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.SOKOBAN)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.UP, width, height), width, height)
            assert t.neighbor_position(index, Direction.DOWN, width, height) == index_1d(0,1, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_EAST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_EAST, width, height)

        def test_autogenerated_1x2_bottom(self):
            width = 1
            height = 2
            row = 1
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.SOKOBAN)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(0,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.DOWN, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_WEST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.NORTH_EAST, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.SOUTH_EAST, width, height)


class TriobanTessellationAutogeneratedSpecMixin:
    class Describe_neighbor_position:
        def test_autogenerated_topLeft(self):
            width = 5
            height = 5
            row = 0
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert triangle_points_down(index, width, height)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(1,0, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(1,0, width)

        def test_autogenerated_topRight_columnOdd(self):
            width = 4
            height = 4
            row = 0
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert column % 2 == 1 and not triangle_points_down(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(2,0, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(3,1, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(3,1, width)

        def test_autogenerated_topRight_columnEven(self):
            width = 5
            height = 5
            row = 0
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert column % 2 == 0 and triangle_points_down(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(3,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_bottomLeft_rowOdd(self):
            width = 4
            height = 4
            row = 3
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert row % 2 == 1 and not triangle_points_down(index, width, height)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(1,3, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(1,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_bottomLeft_rowEven(self):
            width = 5
            height = 5
            row = 4
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert row % 2 == 0 and triangle_points_down(index, width, height)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(1,4, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(0,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(0,3, width)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(1,4, width)

        def test_autogenerated_bottomRight_rowEven_columnEven(self):
            width = 5
            height = 5
            row = 4
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert row % 2 == 0 and column % 2 == 0 and triangle_points_down(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,4, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(4,3, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(3,4, width)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(4,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_bottomRight_rowEven_columnOdd(self):
            width = 4
            height = 5
            row = 4
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert row % 2 == 0 and column % 2 == 1 and not triangle_points_down(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,4, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(2,4, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_bottomRight_rowOdd_columnEven(self):
            width = 5
            height = 4
            row = 3
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert row % 2 == 1 and column % 2 == 0 and not triangle_points_down(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(3,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_bottomRight_rowOdd_columnOdd(self):
            width = 4
            height = 4
            row = 3
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert row % 2 == 1 and column % 2 == 1 and triangle_points_down(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(3,2, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(2,3, width)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(3,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_midleTop_columnOdd(self):
            width = 10
            height = 10
            row = 0
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert column % 2 == 1 and not triangle_points_down(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,0, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(4,0, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(2,0, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(3,1, width)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(4,0, width)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(3,1, width)

        def test_autogenerated_midleTop_columnEven(self):
            width = 10
            height = 10
            row = 0
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert column % 2 == 0 and triangle_points_down(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,0, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(5,0, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(3,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(5,0, width)

        def test_autogenerated_midleBottom_columnOdd_rowEven(self):
            width = 10
            height = 5
            row = 4
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert column % 2 == 1 and row % 2 == 0 and not triangle_points_down(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,4, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(4,4, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(2,4, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(4,4, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_middleBottom_columnEven_rowEven(self):
            width = 10
            height = 5
            row = 4
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert column % 2 == 0 and row % 2 == 0 and triangle_points_down(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,4, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(5,4, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(4,3, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(3,4, width)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(4,3, width)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(5,4, width)

        def test_autogenerated_midleBottom_columnOdd_rowOdd(self):
            width = 10
            height = 4
            row = 3
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert column % 2 == 1 and row % 2 == 1 and triangle_points_down(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,3, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(4,3, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(3,2, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(2,3, width)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(3,2, width)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(4,3, width)

        def test_autogenerated_middleBottom_columnEven_rowOdd(self):
            width = 10
            height = 4
            row = 3
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert column % 2 == 0 and row % 2 == 1 and not triangle_points_down(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,3, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(5,3, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(3,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(5,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_midleLeft_rowOdd(self):
            width = 10
            height = 10
            row = 3
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert row % 2 == 1 and not triangle_points_down(index, width, height)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(1,3, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(0,4, width)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(1,3, width)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(0,4, width)

        def test_autogenerated_midleLeft_rowEven(self):
            width = 10
            height = 10
            row = 2
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert row % 2 == 0 and triangle_points_down(index, width, height)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(1,2, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(0,1, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(0,1, width)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(1,2, width)

        def test_autogenerated_midleRight_rowOdd_columnOdd(self):
            width = 4
            height = 10
            row = 3
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert row % 2 == 1 and column % 2 == 1 and triangle_points_down(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(3,2, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(2,3, width)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(3,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_midleRight_rowEven_columnOdd(self):
            width = 4
            height = 10
            row = 2
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert row % 2 == 0 and column % 2 == 1 and not triangle_points_down(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(2,2, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(3,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(3,3, width)

        def test_autogenerated_midleRight_rowOdd_columnEven(self):
            width = 5
            height = 10
            row = 3
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert row % 2 == 1 and column % 2 == 0 and not triangle_points_down(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(3,3, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(4,4, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(4,4, width)

        def test_autogenerated_midleRight_rowEven_columnEven(self):
            width = 5
            height = 10
            row = 2
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert row % 2 == 0 and column % 2 == 0 and triangle_points_down(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(4,1, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(3,2, width)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(4,1, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_midle_rowOdd_columnOdd(self):
            width = 10
            height = 10
            row = 3
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert row % 2 == 1 and column % 2 == 1 and triangle_points_down(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,3, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(4,3, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(3,2, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(2,3, width)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(3,2, width)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(4,3, width)

        def test_autogenerated_midle_rowOdd_columnEven(self):
            width = 10
            height = 10
            row = 3
            column = 2
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert row % 2 == 1 and column % 2 == 0 and not triangle_points_down(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(1,3, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(3,3, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(1,3, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(2,4, width)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(3,3, width)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(2,4, width)

        def test_autogenerated_midle_rowEven_columnOdd(self):
            width = 10
            height = 10
            row = 2
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert row % 2 == 0 and column % 2 == 1 and not triangle_points_down(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,2, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(4,2, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(2,2, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(3,3, width)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(4,2, width)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(3,3, width)

        def test_autogenerated_midle_rowEven_columnEven(self):
            width = 10
            height = 10
            row = 2
            column = 2
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert row % 2 == 0 and column % 2 == 0 and triangle_points_down(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(1,2, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(3,2, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(2,1, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(1,2, width)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(2,1, width)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(3,2, width)

        def test_autogenerated_board_1x1(self):
            width = 1
            height = 1
            row = 0
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_board_2x1_left(self):
            width = 2
            height = 1
            row = 0
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(1,0, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(1,0, width)

        def test_autogenerated_board_2x1_right(self):
            width = 2
            height = 1
            row = 0
            column = 1
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(0,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(0,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_board_3x1_right(self):
            width = 3
            height = 1
            row = 0
            column = 2
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(1,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(1,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_1x2_top(self):
            width = 1
            height = 2
            row = 0
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_1x2_bottom(self):
            width = 1
            height = 2
            row = 1
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_1x3_bottom(self):
            width = 1
            height = 3
            row = 2
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.TRIOBAN)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(0,1, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(0,1, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)


class HexobanTessellationAutogeneratedSpecMixin:
    class Describe_neighbor_position:
        def test_autogenerated_topLeft(self):
            width = 5
            height = 5
            row = 0
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(1,0, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(0,1, width)

        def test_autogenerated_topRight_columnOdd(self):
            width = 4
            height = 4
            row = 0
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert column % 2 == 1

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(2,1, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(3,1, width)

        def test_autogenerated_topRight_columnEven(self):
            width = 5
            height = 5
            row = 0
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert column % 2 == 0

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(3,1, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(4,1, width)

        def test_autogenerated_bottomLeft_rowOdd(self):
            width = 4
            height = 4
            row = 3
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert row % 2 == 1

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(1,3, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(0,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(1,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_bottomLeft_rowEven(self):
            width = 5
            height = 5
            row = 4
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert row % 2 == 0

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(1,4, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(0,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_bottomRight_rowEven_columnEven(self):
            width = 5
            height = 5
            row = 4
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert row % 2 == 0 and column % 2 == 0

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,4, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(3,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(4,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_bottomRight_rowEven_columnOdd(self):
            width = 4
            height = 5
            row = 4
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert row % 2 == 0 and column % 2 == 1

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,4, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(2,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(3,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_bottomRight_rowOdd_columnEven(self):
            width = 5
            height = 4
            row = 3
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert row % 2 == 1 and column % 2 == 0

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(4,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_bottomRight_rowOdd_columnOdd(self):
            width = 4
            height = 4
            row = 3
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert row % 2 == 1 and column % 2 == 1

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(3,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_midleTop_columnOdd(self):
            width = 10
            height = 10
            row = 0
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert column % 2 == 1

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,0, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(4,0, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(2,1, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(3,1, width)

        def test_autogenerated_midleTop_columnEven(self):
            width = 10
            height = 10
            row = 0
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert column % 2 == 0

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,0, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(5,0, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(3,1, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(4,1, width)

        def test_autogenerated_midleBottom_columnOdd_rowEven(self):
            width = 10
            height = 5
            row = 4
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert column % 2 == 1 and row % 2 == 0

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,4, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(4,4, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(2,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(3,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_middleBottom_columnEven_rowEven(self):
            width = 10
            height = 5
            row = 4
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert column % 2 == 0 and row % 2 == 0

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,4, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(5,4, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(3,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(4,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_midleBottom_columnOdd_rowOdd(self):
            width = 10
            height = 4
            row = 3
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert column % 2 == 1 and row % 2 == 1

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,3, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(4,3, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(3,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(4,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_middleBottom_columnEven_rowOdd(self):
            width = 10
            height = 4
            row = 3
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert column % 2 == 0 and row % 2 == 1

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,3, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(5,3, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(4,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(5,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_midleLeft_rowOdd(self):
            width = 10
            height = 10
            row = 3
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert row % 2 == 1

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(1,3, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(0,2, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(0,4, width)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(1,2, width)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(1,4, width)

        def test_autogenerated_midleLeft_rowEven(self):
            width = 10
            height = 10
            row = 2
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert row % 2 == 0

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(1,2, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(0,1, width)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(0,3, width)

        def test_autogenerated_midleRight_rowOdd_columnOdd(self):
            width = 4
            height = 10
            row = 3
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert row % 2 == 1 and column % 2 == 1

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(3,2, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(3,4, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_midleRight_rowEven_columnOdd(self):
            width = 4
            height = 10
            row = 2
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert row % 2 == 0 and column % 2 == 1

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(2,1, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(2,3, width)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(3,1, width)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(3,3, width)

        def test_autogenerated_midleRight_rowOdd_columnEven(self):
            width = 5
            height = 10
            row = 3
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert row % 2 == 1 and column % 2 == 0

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(4,2, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(4,4, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_midleRight_rowEven_columnEven(self):
            width = 5
            height = 10
            row = 2
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert row % 2 == 0 and column % 2 == 0

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(3,1, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(3,3, width)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(4,1, width)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(4,3, width)

        def test_autogenerated_midle_rowOdd_columnOdd(self):
            width = 10
            height = 10
            row = 3
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert row % 2 == 1 and column % 2 == 1

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,3, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(4,3, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(3,2, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(3,4, width)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(4,2, width)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(4,4, width)

        def test_autogenerated_midle_rowOdd_columnEven(self):
            width = 10
            height = 10
            row = 3
            column = 2
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert row % 2 == 1 and column % 2 == 0

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(1,3, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(3,3, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(2,2, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(2,4, width)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(3,2, width)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(3,4, width)

        def test_autogenerated_midle_rowEven_columnOdd(self):
            width = 10
            height = 10
            row = 2
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert row % 2 == 0 and column % 2 == 1

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,2, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(4,2, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(2,1, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(2,3, width)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(3,1, width)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(3,3, width)

        def test_autogenerated_midle_rowEven_columnEven(self):
            width = 10
            height = 10
            row = 2
            column = 2
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert row % 2 == 0 and column % 2 == 0

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(1,2, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(3,2, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(1,1, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(1,3, width)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(2,1, width)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(2,3, width)

        def test_autogenerated_board_1x1(self):
            width = 1
            height = 1
            row = 0
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_board_2x1_left(self):
            width = 2
            height = 1
            row = 0
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(1,0, width)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_board_2x1_right(self):
            width = 2
            height = 1
            row = 0
            column = 1
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(0,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_board_3x1_right(self):
            width = 3
            height = 1
            row = 0
            column = 2
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(1,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_1x2_top(self):
            width = 1
            height = 2
            row = 0
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(0,1, width)

        def test_autogenerated_1x2_bottom(self):
            width = 1
            height = 2
            row = 1
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(0,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_1x3_bottom(self):
            width = 1
            height = 3
            row = 2
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.HEXOBAN)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.UP, width, height)
            with pytest.raises(ValueError):
                t.neighbor_position(index, Direction.DOWN, width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(0,1, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)


class OctobanTessellationAutogeneratedSpecMixin:
    class Describe_neighbor_position:
        def test_autogenerated_topLeft(self):
            width = 5
            height = 5
            row = 0
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert is_octagon(index, width, height)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(1,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.UP, width, height), width, height)
            assert t.neighbor_position(index, Direction.DOWN, width, height) == index_1d(0,1, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(1,1, width)

        def test_autogenerated_topRight_columnOdd(self):
            width = 4
            height = 4
            row = 0
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert column % 2 == 1 and not is_octagon(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.UP, width, height), width, height)
            assert t.neighbor_position(index, Direction.DOWN, width, height) == index_1d(3,1, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_topRight_columnEven(self):
            width = 5
            height = 5
            row = 0
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert column % 2 == 0 and is_octagon(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.UP, width, height), width, height)
            assert t.neighbor_position(index, Direction.DOWN, width, height) == index_1d(4,1, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(3,1, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_bottomLeft_rowOdd(self):
            width = 4
            height = 4
            row = 3
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert row % 2 == 1 and not is_octagon(index, width, height)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(1,3, width)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(0,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.DOWN, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_bottomLeft_rowEven(self):
            width = 5
            height = 5
            row = 4
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert row % 2 == 0 and is_octagon(index, width, height)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(1,4, width)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(0,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.DOWN, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(1,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_bottomRight_rowEven_columnEven(self):
            width = 5
            height = 5
            row = 4
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert row % 2 == 0 and column % 2 == 0 and is_octagon(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,4, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(4,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.DOWN, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(3,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_bottomRight_rowEven_columnOdd(self):
            width = 4
            height = 5
            row = 4
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert row % 2 == 0 and column % 2 == 1 and not is_octagon(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,4, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(3,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.DOWN, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_bottomRight_rowOdd_columnEven(self):
            width = 5
            height = 4
            row = 3
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert row % 2 == 1 and column % 2 == 0 and not is_octagon(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(4,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.DOWN, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_bottomRight_rowOdd_columnOdd(self):
            width = 4
            height = 4
            row = 3
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert row % 2 == 1 and column % 2 == 1 and is_octagon(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(3,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.DOWN, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(2,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_midleTop_columnOdd(self):
            width = 10
            height = 10
            row = 0
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert column % 2 == 1 and not is_octagon(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,0, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(4,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.UP, width, height), width, height)
            assert t.neighbor_position(index, Direction.DOWN, width, height) == index_1d(3,1, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_midleTop_columnEven(self):
            width = 10
            height = 10
            row = 0
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert column % 2 == 0 and is_octagon(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,0, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(5,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.UP, width, height), width, height)
            assert t.neighbor_position(index, Direction.DOWN, width, height) == index_1d(4,1, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(3,1, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(5,1, width)

        def test_autogenerated_midleBottom_columnOdd_rowEven(self):
            width = 10
            height = 5
            row = 4
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert column % 2 == 1 and row % 2 == 0 and not is_octagon(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,4, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(4,4, width)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(3,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.DOWN, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_middleBottom_columnEven_rowEven(self):
            width = 10
            height = 5
            row = 4
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert column % 2 == 0 and row % 2 == 0 and is_octagon(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,4, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(5,4, width)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(4,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.DOWN, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(3,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(5,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_midleBottom_columnOdd_rowOdd(self):
            width = 10
            height = 4
            row = 3
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert column % 2 == 1 and row % 2 == 1 and is_octagon(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,3, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(4,3, width)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(3,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.DOWN, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(2,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(4,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_middleBottom_columnEven_rowOdd(self):
            width = 10
            height = 4
            row = 3
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert column % 2 == 0 and row % 2 == 1 and not is_octagon(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,3, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(5,3, width)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(4,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.DOWN, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_midleLeft_rowOdd(self):
            width = 10
            height = 10
            row = 3
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert row % 2 == 1 and not is_octagon(index, width, height)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(1,3, width)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(0,2, width)
            assert t.neighbor_position(index, Direction.DOWN, width, height) == index_1d(0,4, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_midleLeft_rowEven(self):
            width = 10
            height = 10
            row = 2
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert row % 2 == 0 and is_octagon(index, width, height)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(1,2, width)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(0,1, width)
            assert t.neighbor_position(index, Direction.DOWN, width, height) == index_1d(0,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(1,1, width)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(1,3, width)

        def test_autogenerated_midleRight_rowOdd_columnOdd(self):
            width = 4
            height = 10
            row = 3
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert row % 2 == 1 and column % 2 == 1 and is_octagon(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(3,2, width)
            assert t.neighbor_position(index, Direction.DOWN, width, height) == index_1d(3,4, width)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(2,2, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(2,4, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_midleRight_rowEven_columnOdd(self):
            width = 4
            height = 10
            row = 2
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert row % 2 == 0 and column % 2 == 1 and not is_octagon(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(3,1, width)
            assert t.neighbor_position(index, Direction.DOWN, width, height) == index_1d(3,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_midleRight_rowOdd_columnEven(self):
            width = 5
            height = 10
            row = 3
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert row % 2 == 1 and column % 2 == 0 and not is_octagon(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(4,2, width)
            assert t.neighbor_position(index, Direction.DOWN, width, height) == index_1d(4,4, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_midleRight_rowEven_columnEven(self):
            width = 5
            height = 10
            row = 2
            column = 4
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert row % 2 == 0 and column % 2 == 0 and is_octagon(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(3,2, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(4,1, width)
            assert t.neighbor_position(index, Direction.DOWN, width, height) == index_1d(4,3, width)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(3,1, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(3,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_midle_rowOdd_columnOdd(self):
            width = 10
            height = 10
            row = 3
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert row % 2 == 1 and column % 2 == 1 and is_octagon(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,3, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(4,3, width)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(3,2, width)
            assert t.neighbor_position(index, Direction.DOWN, width, height) == index_1d(3,4, width)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(2,2, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(2,4, width)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(4,2, width)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(4,4, width)

        def test_autogenerated_midle_rowOdd_columnEven(self):
            width = 10
            height = 10
            row = 3
            column = 2
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert row % 2 == 1 and column % 2 == 0 and not is_octagon(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(1,3, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(3,3, width)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(2,2, width)
            assert t.neighbor_position(index, Direction.DOWN, width, height) == index_1d(2,4, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_midle_rowEven_columnOdd(self):
            width = 10
            height = 10
            row = 2
            column = 3
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert row % 2 == 0 and column % 2 == 1 and not is_octagon(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(2,2, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(4,2, width)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(3,1, width)
            assert t.neighbor_position(index, Direction.DOWN, width, height) == index_1d(3,3, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_midle_rowEven_columnEven(self):
            width = 10
            height = 10
            row = 2
            column = 2
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert row % 2 == 0 and column % 2 == 0 and is_octagon(index, width, height)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(1,2, width)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(3,2, width)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(2,1, width)
            assert t.neighbor_position(index, Direction.DOWN, width, height) == index_1d(2,3, width)
            assert t.neighbor_position(index, Direction.NORTH_WEST, width, height) == index_1d(1,1, width)
            assert t.neighbor_position(index, Direction.SOUTH_WEST, width, height) == index_1d(1,3, width)
            assert t.neighbor_position(index, Direction.NORTH_EAST, width, height) == index_1d(3,1, width)
            assert t.neighbor_position(index, Direction.SOUTH_EAST, width, height) == index_1d(3,3, width)

        def test_autogenerated_board_1x1(self):
            width = 1
            height = 1
            row = 0
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.UP, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.DOWN, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_board_2x1_left(self):
            width = 2
            height = 1
            row = 0
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert t.neighbor_position(index, Direction.RIGHT, width, height) == index_1d(1,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.UP, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.DOWN, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_board_2x1_right(self):
            width = 2
            height = 1
            row = 0
            column = 1
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(0,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.UP, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.DOWN, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_board_3x1_right(self):
            width = 3
            height = 1
            row = 0
            column = 2
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert t.neighbor_position(index, Direction.LEFT, width, height) == index_1d(1,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.UP, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.DOWN, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_1x2_top(self):
            width = 1
            height = 2
            row = 0
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.UP, width, height), width, height)
            assert t.neighbor_position(index, Direction.DOWN, width, height) == index_1d(0,1, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_1x2_bottom(self):
            width = 1
            height = 2
            row = 1
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(0,0, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.DOWN, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)

        def test_autogenerated_1x3_bottom(self):
            width = 1
            height = 3
            row = 2
            column = 0
            index = index_1d(column, row, width)
            t = BaseTessellation.instance(Tessellation.OCTOBAN)

            assert not is_on_board_1d(t.neighbor_position(index, Direction.LEFT, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.RIGHT, width, height), width, height)
            assert t.neighbor_position(index, Direction.UP, width, height) == index_1d(0,1, width)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.DOWN, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_WEST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.NORTH_EAST, width, height), width, height)
            assert not is_on_board_1d(t.neighbor_position(index, Direction.SOUTH_EAST, width, height), width, height)


