"""Unit tests for layout functions."""
import networkx as nx
import pytest

np = pytest.importorskip("numpy")
pytest.importorskip("scipy")


class TestLayout:
    @classmethod
    def setup_class(cls):
        cls.Gi = nx.grid_2d_graph(5, 5)
        cls.Gs = nx.Graph()
        nx.add_path(cls.Gs, "abcdef")
        cls.bigG = nx.grid_2d_graph(25, 25)  # > 500 nodes for sparse

    def test_smoke_empty_graph(self):
        G = []
        nx.force_directed_hyper_graphs_using_social_and_gravity_scaling(G)

    def test_smoke_int(self):
        G = self.Gi
        nx.force_directed_hyper_graphs_using_social_and_gravity_scaling(G)

    def test_smoke_string(self):
        G = self.Gs
        nx.force_directed_hyper_graphs_using_social_and_gravity_scaling(G)

    def test_force_directed_hyper_graphs_using_social_and_gravity_scaling_layout(self):
        import math

        G = self.Gs
        pytest.raises(ValueError, nx.force_directed_hyper_graphs_using_social_and_gravity_scaling, "hello")
        pos = nx.force_directed_hyper_graphs_using_social_and_gravity_scaling(G)
        has_nan = any(math.isnan(c) for coords in pos.values() for c in coords)
        assert not has_nan, "values should not be nan"

    def test_forest_pos(self):
        # random forest
        g: nx.Graph = nx.random_tree(70, 1)
        f: list[nx.Graph] = []
        for i in range(1, 5):
            f.append(nx.random_tree(70, i))
        for i in range(1, 5):
            for j in f[i - 1].edges:
                g.add_edge(j[0] + 70 * (i + 3) + 1, j[1] + 70 * (i + 3) + 1)
        pos = nx.force_directed(g, seed=1, iterations=1000)
        print(pos)
        expected_pos = np.array([[-3.68566218e-02, 7.86416559e-01]
            , [1.27516605e-01, 1.19084672e+00]
            , [-1.35318486e-01, 6.75176347e-01]
            , [5.78499100e-02, 1.04766063e+00]
            , [2.94534533e-01, 6.13809730e-01]
            , [-6.41512530e-01, 3.67453415e-01]
            , [7.19277828e-02, 4.67120821e-01]
            , [-6.97703121e-01, 7.74501318e-01]
            , [1.13467555e-01, 5.25971569e-01]
            , [1.89029501e-01, 7.45918091e-01]
            , [8.17096702e-03, 7.30509414e-01]
            , [-3.34542015e-01, 5.77065387e-01]
            , [1.50426523e-02, 7.95072680e-01]
            , [-4.88829000e-01, 8.03672529e-01]
            , [-4.63012123e-01, 1.30283807e-01]
            , [1.34457264e-01, 6.07590097e-01]
            , [1.71575424e-01, 1.14336251e+00]
            , [-5.96470033e-01, 4.17012632e-01]
            , [5.97055937e-03, 8.74278373e-01]
            , [-2.79709332e-01, 7.19291417e-01]
            , [-4.93946984e-01, 9.01601697e-01]
            , [-8.33858674e-03, 1.08854212e+00]
            , [-1.05083233e-01, 5.61216639e-01]
            , [4.32616449e-02, 6.85829707e-01]
            , [-1.05493718e-01, 9.87882095e-01]
            , [-4.20867096e-02, 1.20431775e+00]
            , [-5.36551684e-01, 4.74241137e-01]
            , [2.02037976e-01, 9.63685704e-01]
            , [-1.50115260e-01, 4.15890534e-01]
            , [-2.64023814e-01, 3.38756068e-01]
            , [-1.09442178e-01, 8.28184376e-01]
            , [1.60447272e-01, 5.51488874e-01]
            , [-6.33656061e-01, 7.85997683e-01]
            , [-3.17352464e-01, 3.01341988e-01]
            , [-5.66841348e-01, 8.00014209e-01]
            , [-2.38335149e-01, 8.18726672e-01]
            , [8.24370355e-02, 8.57952444e-01]
            , [4.36308550e-02, 5.89885914e-01]
            , [-1.64542573e-02, 9.32999312e-01]
            , [6.79542383e-02, 1.10687716e+00]
            , [-1.92111202e-01, 7.55691076e-01]
            , [1.17197798e-03, 1.03519111e+00]
            , [2.14192522e-01, 6.62284036e-01]
            , [-5.26837154e-01, 1.05645580e+00]
            , [-3.44363392e-01, 2.52463849e-01]
            , [9.77765278e-02, 1.26646983e+00]
            , [-1.78008825e-01, 5.45729482e-01]
            , [-2.61396945e-01, 5.52049075e-01]
            , [1.71091311e-01, 1.07999637e+00]
            , [8.13372331e-04, 1.14221876e+00]
            , [2.95019985e-01, 6.78382089e-01]
            , [5.46367588e-02, 5.58924995e-01]
            , [-2.09828794e-01, 4.09981732e-01]
            , [-4.96794948e-02, 6.02384436e-01]
            , [2.15085114e-01, 8.58627676e-01]
            , [-4.80639036e-01, 5.39983579e-01]
            , [-4.23876908e-01, 6.11193338e-01]
            , [5.95629547e-02, 7.64431416e-01]
            , [-4.47905061e-02, 4.93766401e-01]
            , [-3.36285639e-01, 3.58518850e-01]
            , [-4.08623164e-01, 1.84544969e-01]
            , [2.35902287e-01, 5.75602068e-01]
            , [-2.19745029e-01, 6.92671340e-01]
            , [1.51683626e-01, 7.76465641e-01]
            , [2.37692268e-01, 7.45687985e-01]
            , [-4.50576387e-01, 7.07906241e-01]
            , [-4.19345136e-02, 8.52015610e-01]
            , [1.37562392e-01, 9.49712263e-01]
            , [-1.72958825e-01, 1.04070370e+00]
            , [-5.14831986e-01, 9.87740591e-01]
            , [3.98468781e-01, 1.05277350e+00]
            , [3.65578076e-01, 1.11543279e+00]
            , [4.40298035e-01, 9.69878841e-01]
            , [6.48926258e-01, 1.25544203e+00]
            , [6.49751824e-01, 1.33551831e+00]
            , [6.37214963e-01, 1.14631716e+00]
            , [1.00269201e+00, 9.47729650e-01]
            , [1.03718857e+00, 1.02185189e+00]
            , [1.08474864e+00, 9.71327035e-01]
            , [9.36568728e-01, 8.84428592e-01]
            , [5.12905407e-01, 1.21881531e+00]
            , [4.60993308e-01, 1.24767631e+00]
            , [5.27945899e-01, 1.27947815e+00]
            , [5.62469524e-01, 1.23336149e+00]
            , [4.85374967e-01, 1.32177538e+00]
            , [5.39041534e-01, 1.08649426e+00]
            , [6.07415819e-01, 7.57970566e-01]
            , [5.62773574e-01, 7.90437107e-01]
            , [6.69331708e-01, 7.71659148e-01]
            , [1.43366647e+00, 1.16567527e+00]
            , [1.40782880e+00, 1.10119044e+00]
            , [8.06548804e-01, 9.62223819e-01]
            , [7.52352076e-01, 9.33710372e-01]
            , [1.67825259e+00, 8.26288511e-01]
            , [1.62625937e+00, 7.98518573e-01]
            , [6.99195674e-01, 9.44226484e-01]
            , [4.95034350e-01, 8.60203188e-01]
            , [4.95151050e-01, 9.26400247e-01]
            , [3.94721964e-01, 9.58623297e-01]
            , [1.25728925e+00, 8.36585665e-01]
            , [1.17225500e+00, 8.23802533e-01]
            , [1.34121226e+00, 8.52051137e-01]
            , [7.81299500e-01, 1.24414072e+00]
            , [7.59417729e-01, 1.31769709e+00]
            , [8.10591513e-01, 1.30583973e+00]
            , [7.67005263e-01, 1.13122364e+00]
            , [1.49094522e+00, 7.29730662e-01]
            , [1.55344540e+00, 7.65442038e-01]
            , [1.41048549e+00, 7.86566838e-01]
            , [1.53714694e+00, 6.51504151e-01]
            , [1.01652499e+00, 1.47640557e+00]
            , [1.01457904e+00, 1.41260953e+00]
            , [7.59608581e-01, 1.03517037e+00]
            , [6.32941818e-01, 8.78071693e-01]
            , [6.80840882e-01, 1.17580941e+00]
            , [1.38640286e+00, 1.02535833e+00]
            , [1.14645687e+00, 1.00097467e+00]
            , [1.59743254e+00, 5.93618619e-01]
            , [1.01260195e+00, 8.48355517e-01]
            , [1.09468607e+00, 8.29816802e-01]
            , [3.73415157e-01, 1.25418517e+00]
            , [3.44490523e-01, 1.33638471e+00]
            , [4.21003245e-01, 1.14824368e+00]
            , [4.71039891e-01, 1.40546533e+00]
            , [1.36089357e+00, 9.41418725e-01]
            , [6.12610431e-01, 1.05060229e+00]
            , [5.69630834e-01, 9.81029472e-01]
            , [9.12991431e-01, 1.15990103e+00]
            , [9.66924261e-01, 1.14637470e+00]
            , [9.72906181e-01, 1.25481275e+00]
            , [8.41849764e-01, 1.08921119e+00]
            , [1.03933905e+00, 1.25205119e+00]
            , [9.37776295e-01, 1.30260491e+00]
            , [9.98571326e-01, 1.33339372e+00]
            , [7.40544154e-01, 8.07133047e-01]
            , [8.30468341e-01, 8.49359918e-01]
            , [1.08127403e+00, 1.07712920e+00]
            , [4.65315804e-01, 1.03676347e+00]
            , [5.62211933e-01, 8.74285753e-01]
            , [1.64834126e+00, 5.48524204e-01]
            , [8.86887842e-01, 5.23283079e-01]
            , [8.30056050e-01, 5.43999353e-01]
            , [1.06151231e-01, 2.57806745e-01]
            , [1.31187196e-01, 3.16344127e-01]
            , [1.09898478e+00, 5.83066201e-01]
            , [1.02420712e+00, 5.92209602e-01]
            , [8.37848694e-01, 6.14837145e-01]
            , [9.25707429e-01, 5.90748164e-01]
            , [9.27075789e-01, 6.47091311e-01]
            , [7.31441198e-01, 6.02408982e-01]
            , [9.74647657e-01, 5.64654456e-01]
            , [-6.26366527e-02, 2.02754035e-01]
            , [-4.15726085e-02, 1.60525890e-01]
            , [3.00233157e-01, 8.72726671e-01]
            , [3.48275586e-01, 8.75509170e-01]
            , [7.72408895e-01, 5.68126496e-01]
            , [1.30614714e+00, 6.39173141e-01]
            , [1.23131784e+00, 6.51761409e-01]
            , [2.87492754e-01, 4.15417037e-01]
            , [3.13046491e-01, 4.57286872e-01]
            , [1.54774040e-01, 3.53206449e-01]
            , [4.34165552e-01, 4.92170037e-01]
            , [4.63492786e-01, 5.22850766e-01]
            , [-1.65177546e-01, 2.64396626e-01]
            , [-1.01164212e-01, 2.87577076e-01]
            , [5.10686120e-01, 6.40114090e-01]
            , [5.48439171e-01, 6.01133630e-01]
            , [5.16344574e-01, 5.13454909e-01]
            , [5.33626475e-01, 5.53826485e-01]
            , [2.08690563e-01, 1.34021251e+00]
            , [2.40148023e-01, 1.24447406e+00]
            , [6.37980655e-01, 6.02553459e-01]
            , [-2.09263926e-01, 1.16740771e-01]
            , [-1.30668043e-01, 1.48429703e-01]
            , [5.47498199e-01, 7.04475002e-01]
            , [5.82876361e-01, 6.62551978e-01]
            , [5.01681263e-02, 2.71835282e-01]
            , [2.44634375e-01, 4.28375597e-01]
            , [3.88678716e-01, 8.57243351e-01]
            , [3.93870023e-01, 7.79403275e-01]
            , [3.05528023e-01, 9.99745232e-01]
            , [7.15056070e-01, 6.44031312e-01]
            , [-1.71235648e-02, 2.99283032e-01]
            , [2.71667867e-01, 1.05812176e+00]
            , [6.06396181e-03, 1.29705773e-01]
            , [3.26596999e-01, 4.92552737e-01]
            , [4.29482552e-01, 7.03121496e-01]
            , [4.36366187e-01, 6.32928972e-01]
            , [1.03899136e+00, 6.67184273e-01]
            , [1.14296050e+00, 6.65647885e-01]
            , [3.91330823e-01, 6.41247795e-01]
            , [3.64401367e-01, 6.86747971e-01]
            , [-9.52253120e-02, 4.22188194e-03]
            , [-5.07477818e-02, 7.29436406e-02]
            , [2.70791116e-01, 1.13376695e+00]
            , [7.32868582e-01, 7.15253220e-01]
            , [8.18011004e-01, 7.41578549e-01]
            , [6.53034203e-01, 6.88253881e-01]
            , [-1.77373895e-01, 3.48652598e-02]
            , [-1.09959060e-01, 9.25327281e-02]
            , [8.33765906e-01, 7.94169753e-01]
            , [4.42939359e-01, 7.78041046e-01]
            , [3.91411664e-01, 5.31022860e-01]
            , [4.36460138e-01, 5.71146082e-01]
            , [4.86794370e-01, 5.66269370e-01]
            , [5.83478302e-01, 6.00614325e-01]
            , [9.74845363e-01, 7.94310640e-01]
            , [9.02870268e-01, 7.68575454e-01]
            , [3.28308419e-01, 5.40363651e-01]
            , [8.67043805e-01, 7.20689259e-01]
            , [1.41935961e+00, -4.21606674e-02]
            , [1.35690300e+00, -1.68923422e-03]
            , [6.20681796e-01, 5.00429490e-01]
            , [6.73447696e-01, 5.29633647e-01]
            , [6.18970384e-01, 5.44071682e-01]
            , [5.98289683e-01, 4.42682721e-01]
            , [7.77128819e-01, 2.99219433e-01]
            , [7.87702097e-01, 3.50081775e-01]
            , [1.12528256e+00, 1.38701007e-01]
            , [1.12665876e+00, 8.48868759e-02]
            , [1.20384961e+00, 9.64795987e-02]
            , [1.03905068e+00, 1.96032651e-01]
            , [1.51208521e-01, 2.85465586e-01]
            , [1.19777045e-01, 3.88749569e-01]
            , [1.54451785e-01, 1.92753921e-01]
            , [9.79006640e-01, 4.53771988e-01]
            , [1.04825882e+00, 4.63994173e-01]
            , [9.22449207e-01, 4.32985397e-01]
            , [1.40492070e+00, 3.04129348e-01]
            , [1.32882564e+00, 3.26675717e-01]
            , [8.29367171e-01, 5.36593450e-02]
            , [7.98641588e-01, 1.22972065e-01]
            , [5.57955552e-01, 4.75982670e-01]
            , [6.59345236e-01, 3.63664057e-01]
            , [5.25746563e-01, 4.33886193e-01]
            , [6.59891887e-01, 4.52716953e-01]
            , [1.27831637e+00, 2.91232710e-01]
            , [1.24836619e+00, 3.54107715e-01]
            , [4.48157536e-01, 4.31045335e-01]
            , [3.74417327e-01, 3.85986530e-01]
            , [1.31448453e-01, 4.80930032e-01]
            , [9.62576519e-01, 4.34136178e-02]
            , [1.02294888e+00, -1.78853242e-02]
            , [8.77746154e-01, 1.20575307e-01]
            , [7.16172393e-01, 2.71403951e-01]
            , [8.32720649e-02, 7.94788819e-02]
            , [2.30501407e-01, 2.64176717e-01]
            , [1.16099949e+00, 4.22778892e-01]
            , [1.12816814e+00, 3.78397405e-01]
            , [7.13420833e-01, 4.05897640e-01]
            , [7.46765962e-01, 4.35426211e-01]
            , [7.16209890e-01, 3.60056154e-01]
            , [9.53729229e-01, 2.41143881e-01]
            , [9.65821550e-01, 1.95901816e-01]
            , [8.59481430e-01, 2.89956867e-01]
            , [1.28448742e+00, -2.48784257e-02]
            , [1.29266164e+00, 4.04181415e-02]
            , [9.08422887e-01, 3.43076681e-01]
            , [8.49757780e-01, 3.56694583e-01]
            , [1.31171022e+00, 3.88117570e-01]
            , [1.22304494e+00, 2.97265458e-01]
            , [-4.28073084e-02, -1.05061750e-01]
            , [-9.81156307e-03, -3.40161955e-02]
            , [9.15958442e-01, 6.68598880e-02]
            , [7.90863994e-01, 1.87984319e-01]
            , [1.34719840e+00, 7.60566787e-02]
            , [1.08343233e+00, -7.44543930e-02]
            , [1.01762142e+00, 3.95264679e-01]
            , [6.23463114e-01, 3.47856299e-01]
            , [5.78556307e-01, 3.77979397e-01]
            , [8.13209405e-01, 4.58107351e-01]
            , [1.44666425e-01, 5.85649721e-01]
            , [1.27040150e+00, 4.19213776e-01]
            , [5.46117542e-01, 3.58994380e-01]
            , [6.11444098e-01, 3.09565934e-01]
            , [-9.98428959e-02, -1.17377308e-01]
            , [7.24630082e-01, 4.78836962e-01]
            , [-1.67390796e-01, -1.87811592e-01]
            , [3.62009427e-01, 4.23030291e-01]
            , [2.96600224e-01, 3.31935827e-01]
            , [2.50041381e-01, -4.25119735e-02]
            , [1.92760787e-01, -1.14852698e-01]
            , [3.15618739e-01, 2.71055594e-02]
            , [3.21243692e-01, -2.18083546e-01]
            , [3.28174004e-01, -1.41100982e-01]
            , [5.81185871e-01, -6.56802586e-01]
            , [6.21294888e-01, -6.99357563e-01]
            , [5.39856265e-01, -5.97822467e-01]
            , [4.84261168e-01, 1.04207605e-01]
            , [5.05720017e-01, 1.47277287e-02]
            , [4.46496780e-01, 1.78420193e-01]
            , [1.39044889e-01, -1.04754918e-01]
            , [5.01044608e-01, 2.29972886e-01]
            , [5.52178238e-01, 1.57595125e-01]
            , [4.89188996e-01, 2.92783668e-01]
            , [3.32438320e-01, 2.50940718e-01]
            , [3.94105963e-01, 2.48156154e-01]
            , [3.44285246e-01, 1.75827851e-01]
            , [2.63192404e-01, 1.57497331e-01]
            , [4.41568282e-01, 2.02367878e-01]
            , [6.29667359e-01, 1.21895907e-01]
            , [6.84109056e-01, 1.17840913e-01]
            , [5.49170073e-01, 1.98848442e-01]
            , [6.77090942e-01, 1.38258006e-02]
            , [8.22021987e-01, -3.69644206e-01]
            , [8.35864030e-01, -3.00428263e-01]
            , [5.80090692e-01, 7.22368763e-02]
            , [1.42914431e-01, -2.13644567e-01]
            , [8.44349521e-02, -2.06686704e-01]
            , [8.27663416e-02, -2.74239919e-01]
            , [1.75516867e-01, -2.81968607e-01]
            , [1.77294589e-01, -3.54698938e-01]
            , [3.63390417e-01, 2.92568230e-01]
            , [1.83130668e-01, 3.40539417e-02]
            , [2.43744359e-01, 3.90549430e-02]
            , [8.05770669e-01, -1.31428278e-01]
            , [8.61293737e-01, -2.21227951e-01]
            , [7.51439124e-01, -5.44849808e-02]
            , [4.36225699e-01, -5.73300542e-02]
            , [4.52058526e-01, -1.43610770e-01]
            , [4.22362946e-01, 3.58977593e-02]
            , [3.40410324e-01, 1.21576274e-01]
            , [2.86425058e-01, 1.10453039e-01]
            , [3.89765903e-01, 1.01804217e-01]
            , [1.05537084e+00, -3.95392460e-01]
            , [1.03834967e+00, -4.62462125e-01]
            , [1.12042210e+00, -4.25630825e-01]
            , [1.02198542e+00, -3.17217870e-01]
            , [6.65395300e-01, -9.84652039e-02]
            , [6.46811176e-01, -1.91715652e-01]
            , [6.91322529e-01, -1.52397624e-01]
            , [8.08069267e-01, -6.01843021e-02]
            , [6.18544690e-01, 2.38159060e-01]
            , [5.56449596e-01, 2.70144933e-01]
            , [5.25074338e-01, -5.42771782e-02]
            , [3.27145198e-01, -5.88601912e-02]
            , [1.08936996e+00, -3.03833840e-01]
            , [9.47990964e-01, -2.52575363e-01]
            , [4.35966936e-01, 3.38212009e-01]
            , [4.95573097e-01, 3.31930952e-01]
            , [6.49006858e-01, -2.73450156e-01]
            , [4.72330618e-01, 3.79920819e-01]
            , [4.55911774e-01, -3.13836703e-01]
            , [4.57533199e-01, -3.92832177e-01]
            , [4.57281143e-01, -2.30624326e-01]
            , [3.18953130e-02, -3.30387828e-01]
            , [5.00985009e-01, -5.35069902e-01]
            , [4.71840700e-01, -4.66233866e-01]
            , [1.00235428e+00, -2.30731189e-01]
            , [6.01229333e-01, 7.29795997e-03]])
        expected_pos = np.round(expected_pos, 3)
        pos = np.round(pos, 3)
        assert np.array_equal(pos, expected_pos)

    def test_tree_pos(self):
        # random tree
        g: nx.Graph = nx.random_tree(70, 1)
        pos = nx.force_directed(g, seed=1, iterations=1000)
        print(pos)
        expected_pos = np.array([[0.29736633,0.63480194]
,[0.22681043,0.37510787]
,[0.40212277,0.3312742,]
,[0.25974642,0.51895173]
,[0.52069446,0.50224947]
,[0.47352115,0.10714621]
,[0.31879629,0.69826861]
,[0.5851787,0.94745479]
,[0.38113126,0.64796596]
,[0.58218931,0.54017084]
,[0.31434904,0.56810926]
,[0.6392644,0.36593043]
,[0.33531096,0.76750511]
,[0.78287099,0.6074936,]
,[0.81953485,0.68414165]
,[0.43561931,0.60710369]
,[0.28812055,0.33903729]
,[0.55733422,0.14905953]
,[0.23926388,0.77976418]
,[0.38406491,0.20973905]
,[0.85012432,0.53255001]
,[0.24404243,0.45043798]
,[0.49870312,0.37106506]
,[0.41144232,0.71911198]
,[0.67340197,0.52766416]
,[0.13423489,0.62571773]
,[0.63222212,0.20159345]
,[0.38297599,0.44266139]
,[0.4877126,0.75474561]
,[0.56211373,0.77868405]
,[0.21615144,0.69054084]
,[0.53467846,0.43869035]
,[0.68057376,0.8799594,]
,[0.4994492,0.85538634]
,[0.76009928,0.77053384]
,[0.25462868,0.25421459]
,[0.56483745,0.60203424]
,[0.46891478,0.66072044]
,[0.64049051,0.59276557]
,[0.22405809,0.60029235]
,[0.33805387,0.28333972]
,[0.17411716,0.48207917]
,[0.49246132,0.59751012]
,[0.8456204,0.32164017]
,[0.64774544,0.77187046]
,[0.14727295,0.38572115]
,[0.52141348,0.30667629]
,[0.5838381,0.32963697]
,[0.3166799,0.40417372]
,[0.17165679,0.55273609]
,[0.46533648,0.49568567]
,[0.47304277,0.43841244]
,[0.42231348,0.82315062]
,[0.4337512,0.39652444]
,[0.40677577,0.49899813]
,[0.68925071,0.27405237]
,[0.71486056,0.37350978]
,[0.38193898,0.57255518]
,[0.53398696,0.67807706]
,[0.63006807,0.70744133]
,[0.72785481,0.69549327]
,[0.57197277,0.47824503]
,[0.44474713,0.26340353]
,[0.50644667,0.55391936]
,[0.44674829,0.54377425]
,[0.76307536,0.48670877]
,[0.32511892,0.84341103]
,[0.34662513,0.50107046]
,[0.68819608,0.45091517]
,[0.86190814,0.42512409]])
        expected_pos = np.round(expected_pos, 3)
        pos = np.round(pos, 3)
        assert np.array_equal(pos, expected_pos)

    def test_graph_pos(self):
        # random graph
        g: nx.Graph = nx.random_regular_graph(3,70,1)
        pos = nx.force_directed(g, seed=1, iterations=1000)
        print(pos)
        expected_pos = np.array([[0.29487569,0.54793363]
,[0.25936477,0.50697479]
,[0.26721351,0.64120238]
,[0.35095122,0.70345289]
,[0.41056274,0.41843575]
,[0.45547612,0.49279781]
,[0.32517505,0.64111801]
,[0.44466642,0.65990752]
,[0.52920145,0.44504326]
,[0.53948418,0.53681293]
,[0.63823465,0.66061808]
,[0.58815324,0.67351904]
,[0.43325261,0.53061651]
,[0.41329515,0.4826041]
,[0.3106868,0.42792989]
,[0.37455752,0.37992962]
,[0.49058185,0.33503471]
,[0.51783873,0.39186841]
,[0.51123767,0.49641318]
,[0.39474001,0.74797708]
,[0.49783845,0.73770676]
,[0.58542755,0.50597937]
,[0.53070636,0.3417669]
,[0.44460386,0.44921663]
,[0.55192003,0.47851399]
,[0.58704149,0.56016038]
,[0.63316088,0.53201911]
,[0.38098183,0.50556403]
,[0.2622334,0.58345949]
,[0.3472142,0.47445703]
,[0.36349348,0.43086505]
,[0.55391005,0.63248059]
,[0.47623836,0.5289497]
,[0.3810434,0.64106975]
,[0.30295688,0.69448903]
,[0.68242398,0.4880281]
,[0.68677557,0.54519107]
,[0.30762139,0.38470651]
,[0.59881548,0.61906293]
,[0.67388912,0.60915348]
,[0.62619315,0.58804182]
,[0.35504202,0.54242074]
,[0.63625457,0.47404475]
,[0.34462461,0.34638152]
,[0.50354399,0.55735443]
,[0.53153143,0.58619836]
,[0.56391721,0.40780679]
,[0.47021987,0.40460445]
,[0.6224424,0.37869353]
,[0.39546789,0.55566477]
,[0.31012121,0.4994453]
,[0.48053279,0.59980906]
,[0.48370596,0.68285249]
,[0.54827761,0.72723147]
,[0.26363307,0.4515309]
,[0.48779587,0.45651853]
,[0.45254093,0.31696055]
,[0.57785986,0.35213231]
,[0.66725288,0.4229507]
,[0.44103403,0.5757737]
,[0.61171979,0.43070393]
,[0.40277656,0.69292547]
,[0.38347827,0.59613876]
,[0.3320348,0.58843611]
,[0.50006433,0.63434473]
,[0.44846359,0.74089007]
,[0.39755878,0.33409645]
,[0.5337679,0.67234487]
,[0.43183937,0.6191751]
,[0.43679015,0.37046018]])
        expected_pos = np.round(expected_pos, 3)
        pos = np.round(pos, 3)
        assert np.array_equal(pos, expected_pos)