"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CreateCloudfrontSite = exports.CreateBasicSite = void 0;
const acm = require("@aws-cdk/aws-certificatemanager");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const origins = require("@aws-cdk/aws-cloudfront-origins");
const route53 = require("@aws-cdk/aws-route53");
const targets = require("@aws-cdk/aws-route53-targets");
const s3 = require("@aws-cdk/aws-s3");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const cdk = require("@aws-cdk/core");
/**
 * @stability stable
 */
class CreateBasicSite extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        if (props.websiteDomain && props.websiteSubDomain) {
            new s3.Bucket(scope, 'WebsiteRedirectBucket', {
                bucketName: props.websiteSubDomain,
                websiteRedirect: {
                    hostName: props.websiteDomain,
                    protocol: s3.RedirectProtocol.HTTP,
                },
            });
        }
        const websiteBucket = new s3.Bucket(scope, 'WebsiteBucket', {
            bucketName: props.websiteDomain,
            websiteIndexDocument: props.indexDoc,
            websiteErrorDocument: props.errorDoc,
            publicReadAccess: true,
            encryption: props.encryptBucket
                ? s3.BucketEncryption.S3_MANAGED
                : s3.BucketEncryption.UNENCRYPTED,
        });
        new s3deploy.BucketDeployment(scope, 'WebsiteDeploy', {
            sources: [s3deploy.Source.asset(props.websiteFolder)],
            destinationBucket: websiteBucket,
        });
    }
}
exports.CreateBasicSite = CreateBasicSite;
/**
 * @stability stable
 */
class CreateCloudfrontSite extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        const hostedZone = route53.HostedZone.fromLookup(this, 'WebsiteHostedZone', {
            domainName: props.hostedZoneDomain,
        });
        const websiteCert = new acm.DnsValidatedCertificate(this, 'WebsiteCert', {
            domainName: props.websiteDomain,
            hostedZone: hostedZone,
            region: 'us-east-1',
        });
        const websiteBucket = new s3.Bucket(scope, 'WebsiteBucket', {
            publicReadAccess: false,
            encryption: props.encryptBucket
                ? s3.BucketEncryption.S3_MANAGED
                : s3.BucketEncryption.UNENCRYPTED,
        });
        const websiteDist = new cloudfront.Distribution(this, 'WebsiteDist', {
            defaultBehavior: {
                origin: new origins.S3Origin(websiteBucket),
                allowedMethods: cloudfront.AllowedMethods.ALLOW_ALL,
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            },
            priceClass: props.priceClass
                ? props.priceClass
                : cloudfront.PriceClass.PRICE_CLASS_100,
            errorResponses: [
                {
                    httpStatus: 404,
                    responsePagePath: `/${props.errorDoc ? props.errorDoc : props.indexDoc}`,
                },
                {
                    httpStatus: 500,
                    ttl: cdk.Duration.seconds(2),
                },
                {
                    httpStatus: 403,
                    responseHttpStatus: 200,
                    responsePagePath: `/${props.indexDoc}`,
                },
            ],
            defaultRootObject: props.indexDoc,
            domainNames: [props.websiteDomain],
            certificate: websiteCert,
        });
        new s3deploy.BucketDeployment(scope, 'WebsiteDeploy', {
            sources: [s3deploy.Source.asset(props.websiteFolder)],
            destinationBucket: websiteBucket,
            distribution: websiteDist,
            distributionPaths: ['/', `/${props.indexDoc}`],
        });
        new route53.ARecord(this, 'WebisteAlias', {
            zone: hostedZone,
            recordName: props.websiteDomain,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(websiteDist)),
        });
    }
}
exports.CreateCloudfrontSite = CreateCloudfrontSite;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2RrLXNpbXBsZXdlYnNpdGUtZGVwbG95LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vc3JjL2Nkay1zaW1wbGV3ZWJzaXRlLWRlcGxveS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx1REFBdUQ7QUFDdkQsc0RBQXNEO0FBQ3RELDJEQUEyRDtBQUMzRCxnREFBZ0Q7QUFDaEQsd0RBQXdEO0FBQ3hELHNDQUFzQztBQUN0Qyx1REFBdUQ7QUFDdkQscUNBQXFDOzs7O0FBOERyQyxNQUFhLGVBQWdCLFNBQVEsR0FBRyxDQUFDLFNBQVM7Ozs7SUFDaEQsWUFBWSxLQUFvQixFQUFFLEVBQVUsRUFBRSxLQUE2QjtRQUN6RSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLElBQUksS0FBSyxDQUFDLGFBQWEsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEVBQUU7WUFDakQsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSx1QkFBdUIsRUFBRTtnQkFDNUMsVUFBVSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0I7Z0JBQ2xDLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUUsS0FBSyxDQUFDLGFBQWE7b0JBQzdCLFFBQVEsRUFBRSxFQUFFLENBQUMsZ0JBQWdCLENBQUMsSUFBSTtpQkFDbkM7YUFDRixDQUFDLENBQUM7U0FDSjtRQUVELE1BQU0sYUFBYSxHQUFHLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsZUFBZSxFQUFFO1lBQzFELFVBQVUsRUFBRSxLQUFLLENBQUMsYUFBYTtZQUMvQixvQkFBb0IsRUFBRSxLQUFLLENBQUMsUUFBUTtZQUNwQyxvQkFBb0IsRUFBRSxLQUFLLENBQUMsUUFBUTtZQUNwQyxnQkFBZ0IsRUFBRSxJQUFJO1lBQ3RCLFVBQVUsRUFBRSxLQUFLLENBQUMsYUFBYTtnQkFDN0IsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVO2dCQUNoQyxDQUFDLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLFdBQVc7U0FDcEMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxRQUFRLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxFQUFFLGVBQWUsRUFBRTtZQUNwRCxPQUFPLEVBQUUsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUM7WUFDckQsaUJBQWlCLEVBQUUsYUFBYTtTQUNqQyxDQUFDLENBQUM7SUFDTCxDQUFDO0NBQ0Y7QUE3QkQsMENBNkJDOzs7O0FBRUQsTUFBYSxvQkFBcUIsU0FBUSxHQUFHLENBQUMsU0FBUzs7OztJQUNyRCxZQUNFLEtBQW9CLEVBQ3BCLEVBQVUsRUFDVixLQUFrQztRQUVsQyxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLE1BQU0sVUFBVSxHQUFHLE9BQU8sQ0FBQyxVQUFVLENBQUMsVUFBVSxDQUM5QyxJQUFJLEVBQ0osbUJBQW1CLEVBQ25CO1lBQ0UsVUFBVSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0I7U0FDbkMsQ0FDRixDQUFDO1FBRUYsTUFBTSxXQUFXLEdBQUcsSUFBSSxHQUFHLENBQUMsdUJBQXVCLENBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRTtZQUN2RSxVQUFVLEVBQUUsS0FBSyxDQUFDLGFBQWE7WUFDL0IsVUFBVSxFQUFFLFVBQVU7WUFDdEIsTUFBTSxFQUFFLFdBQVc7U0FDcEIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxhQUFhLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUU7WUFDMUQsZ0JBQWdCLEVBQUUsS0FBSztZQUN2QixVQUFVLEVBQUUsS0FBSyxDQUFDLGFBQWE7Z0JBQzdCLENBQUMsQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsVUFBVTtnQkFDaEMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxXQUFXO1NBQ3BDLENBQUMsQ0FBQztRQUVILE1BQU0sV0FBVyxHQUFHLElBQUksVUFBVSxDQUFDLFlBQVksQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFO1lBQ25FLGVBQWUsRUFBRTtnQkFDZixNQUFNLEVBQUUsSUFBSSxPQUFPLENBQUMsUUFBUSxDQUFDLGFBQWEsQ0FBQztnQkFDM0MsY0FBYyxFQUFFLFVBQVUsQ0FBQyxjQUFjLENBQUMsU0FBUztnQkFDbkQsb0JBQW9CLEVBQUUsVUFBVSxDQUFDLG9CQUFvQixDQUFDLGlCQUFpQjthQUN4RTtZQUNELFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtnQkFDMUIsQ0FBQyxDQUFDLEtBQUssQ0FBQyxVQUFVO2dCQUNsQixDQUFDLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQyxlQUFlO1lBQ3pDLGNBQWMsRUFBRTtnQkFDZDtvQkFDRSxVQUFVLEVBQUUsR0FBRztvQkFDZixnQkFBZ0IsRUFBRSxJQUNoQixLQUFLLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsUUFDMUMsRUFBRTtpQkFDSDtnQkFDRDtvQkFDRSxVQUFVLEVBQUUsR0FBRztvQkFDZixHQUFHLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO2lCQUM3QjtnQkFDRDtvQkFDRSxVQUFVLEVBQUUsR0FBRztvQkFDZixrQkFBa0IsRUFBRSxHQUFHO29CQUN2QixnQkFBZ0IsRUFBRSxJQUFJLEtBQUssQ0FBQyxRQUFRLEVBQUU7aUJBQ3ZDO2FBQ0Y7WUFDRCxpQkFBaUIsRUFBRSxLQUFLLENBQUMsUUFBUTtZQUNqQyxXQUFXLEVBQUUsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDO1lBQ2xDLFdBQVcsRUFBRSxXQUFXO1NBQ3pCLENBQUMsQ0FBQztRQUVILElBQUksUUFBUSxDQUFDLGdCQUFnQixDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUU7WUFDcEQsT0FBTyxFQUFFLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDO1lBQ3JELGlCQUFpQixFQUFFLGFBQWE7WUFDaEMsWUFBWSxFQUFFLFdBQVc7WUFDekIsaUJBQWlCLEVBQUUsQ0FBQyxHQUFHLEVBQUUsSUFBSSxLQUFLLENBQUMsUUFBUSxFQUFFLENBQUM7U0FDL0MsQ0FBQyxDQUFDO1FBRUgsSUFBSSxPQUFPLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxjQUFjLEVBQUU7WUFDeEMsSUFBSSxFQUFFLFVBQVU7WUFDaEIsVUFBVSxFQUFFLEtBQUssQ0FBQyxhQUFhO1lBQy9CLE1BQU0sRUFBRSxPQUFPLENBQUMsWUFBWSxDQUFDLFNBQVMsQ0FDcEMsSUFBSSxPQUFPLENBQUMsZ0JBQWdCLENBQUMsV0FBVyxDQUFDLENBQzFDO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztDQUNGO0FBM0VELG9EQTJFQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGFjbSBmcm9tICdAYXdzLWNkay9hd3MtY2VydGlmaWNhdGVtYW5hZ2VyJztcbmltcG9ydCAqIGFzIGNsb3VkZnJvbnQgZnJvbSAnQGF3cy1jZGsvYXdzLWNsb3VkZnJvbnQnO1xuaW1wb3J0ICogYXMgb3JpZ2lucyBmcm9tICdAYXdzLWNkay9hd3MtY2xvdWRmcm9udC1vcmlnaW5zJztcbmltcG9ydCAqIGFzIHJvdXRlNTMgZnJvbSAnQGF3cy1jZGsvYXdzLXJvdXRlNTMnO1xuaW1wb3J0ICogYXMgdGFyZ2V0cyBmcm9tICdAYXdzLWNkay9hd3Mtcm91dGU1My10YXJnZXRzJztcbmltcG9ydCAqIGFzIHMzIGZyb20gJ0Bhd3MtY2RrL2F3cy1zMyc7XG5pbXBvcnQgKiBhcyBzM2RlcGxveSBmcm9tICdAYXdzLWNkay9hd3MtczMtZGVwbG95bWVudCc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgQmFzaWNTaXRlQ29uZmlndXJhdGlvbiB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB3ZWJzaXRlRm9sZGVyOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaW5kZXhEb2M6IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGVycm9yRG9jPzogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZW5jcnlwdEJ1Y2tldD86IGJvb2xlYW47XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHdlYnNpdGVEb21haW4/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHdlYnNpdGVTdWJEb21haW4/OiBzdHJpbmc7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgQ2xvdWRmcm9udFNpdGVDb25maWd1cmF0aW9uIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHdlYnNpdGVGb2xkZXI6IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGluZGV4RG9jOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGVycm9yRG9jPzogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZW5jcnlwdEJ1Y2tldD86IGJvb2xlYW47XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaG9zdGVkWm9uZURvbWFpbjogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB3ZWJzaXRlRG9tYWluOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBwcmljZUNsYXNzPzogY2xvdWRmcm9udC5QcmljZUNsYXNzO1xufVxuXG5leHBvcnQgY2xhc3MgQ3JlYXRlQmFzaWNTaXRlIGV4dGVuZHMgY2RrLkNvbnN0cnVjdCB7XG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQmFzaWNTaXRlQ29uZmlndXJhdGlvbikge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBpZiAocHJvcHMud2Vic2l0ZURvbWFpbiAmJiBwcm9wcy53ZWJzaXRlU3ViRG9tYWluKSB7XG4gICAgICBuZXcgczMuQnVja2V0KHNjb3BlLCAnV2Vic2l0ZVJlZGlyZWN0QnVja2V0Jywge1xuICAgICAgICBidWNrZXROYW1lOiBwcm9wcy53ZWJzaXRlU3ViRG9tYWluLFxuICAgICAgICB3ZWJzaXRlUmVkaXJlY3Q6IHtcbiAgICAgICAgICBob3N0TmFtZTogcHJvcHMud2Vic2l0ZURvbWFpbixcbiAgICAgICAgICBwcm90b2NvbDogczMuUmVkaXJlY3RQcm90b2NvbC5IVFRQLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgY29uc3Qgd2Vic2l0ZUJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQoc2NvcGUsICdXZWJzaXRlQnVja2V0Jywge1xuICAgICAgYnVja2V0TmFtZTogcHJvcHMud2Vic2l0ZURvbWFpbixcbiAgICAgIHdlYnNpdGVJbmRleERvY3VtZW50OiBwcm9wcy5pbmRleERvYyxcbiAgICAgIHdlYnNpdGVFcnJvckRvY3VtZW50OiBwcm9wcy5lcnJvckRvYyxcbiAgICAgIHB1YmxpY1JlYWRBY2Nlc3M6IHRydWUsXG4gICAgICBlbmNyeXB0aW9uOiBwcm9wcy5lbmNyeXB0QnVja2V0XG4gICAgICAgID8gczMuQnVja2V0RW5jcnlwdGlvbi5TM19NQU5BR0VEXG4gICAgICAgIDogczMuQnVja2V0RW5jcnlwdGlvbi5VTkVOQ1JZUFRFRCxcbiAgICB9KTtcblxuICAgIG5ldyBzM2RlcGxveS5CdWNrZXREZXBsb3ltZW50KHNjb3BlLCAnV2Vic2l0ZURlcGxveScsIHtcbiAgICAgIHNvdXJjZXM6IFtzM2RlcGxveS5Tb3VyY2UuYXNzZXQocHJvcHMud2Vic2l0ZUZvbGRlcildLFxuICAgICAgZGVzdGluYXRpb25CdWNrZXQ6IHdlYnNpdGVCdWNrZXQsXG4gICAgfSk7XG4gIH1cbn1cblxuZXhwb3J0IGNsYXNzIENyZWF0ZUNsb3VkZnJvbnRTaXRlIGV4dGVuZHMgY2RrLkNvbnN0cnVjdCB7XG4gIGNvbnN0cnVjdG9yKFxuICAgIHNjb3BlOiBjZGsuQ29uc3RydWN0LFxuICAgIGlkOiBzdHJpbmcsXG4gICAgcHJvcHM6IENsb3VkZnJvbnRTaXRlQ29uZmlndXJhdGlvbixcbiAgKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGNvbnN0IGhvc3RlZFpvbmUgPSByb3V0ZTUzLkhvc3RlZFpvbmUuZnJvbUxvb2t1cChcbiAgICAgIHRoaXMsXG4gICAgICAnV2Vic2l0ZUhvc3RlZFpvbmUnLFxuICAgICAge1xuICAgICAgICBkb21haW5OYW1lOiBwcm9wcy5ob3N0ZWRab25lRG9tYWluLFxuICAgICAgfSxcbiAgICApO1xuXG4gICAgY29uc3Qgd2Vic2l0ZUNlcnQgPSBuZXcgYWNtLkRuc1ZhbGlkYXRlZENlcnRpZmljYXRlKHRoaXMsICdXZWJzaXRlQ2VydCcsIHtcbiAgICAgIGRvbWFpbk5hbWU6IHByb3BzLndlYnNpdGVEb21haW4sXG4gICAgICBob3N0ZWRab25lOiBob3N0ZWRab25lLFxuICAgICAgcmVnaW9uOiAndXMtZWFzdC0xJyxcbiAgICB9KTtcblxuICAgIGNvbnN0IHdlYnNpdGVCdWNrZXQgPSBuZXcgczMuQnVja2V0KHNjb3BlLCAnV2Vic2l0ZUJ1Y2tldCcsIHtcbiAgICAgIHB1YmxpY1JlYWRBY2Nlc3M6IGZhbHNlLFxuICAgICAgZW5jcnlwdGlvbjogcHJvcHMuZW5jcnlwdEJ1Y2tldFxuICAgICAgICA/IHMzLkJ1Y2tldEVuY3J5cHRpb24uUzNfTUFOQUdFRFxuICAgICAgICA6IHMzLkJ1Y2tldEVuY3J5cHRpb24uVU5FTkNSWVBURUQsXG4gICAgfSk7XG5cbiAgICBjb25zdCB3ZWJzaXRlRGlzdCA9IG5ldyBjbG91ZGZyb250LkRpc3RyaWJ1dGlvbih0aGlzLCAnV2Vic2l0ZURpc3QnLCB7XG4gICAgICBkZWZhdWx0QmVoYXZpb3I6IHtcbiAgICAgICAgb3JpZ2luOiBuZXcgb3JpZ2lucy5TM09yaWdpbih3ZWJzaXRlQnVja2V0KSxcbiAgICAgICAgYWxsb3dlZE1ldGhvZHM6IGNsb3VkZnJvbnQuQWxsb3dlZE1ldGhvZHMuQUxMT1dfQUxMLFxuICAgICAgICB2aWV3ZXJQcm90b2NvbFBvbGljeTogY2xvdWRmcm9udC5WaWV3ZXJQcm90b2NvbFBvbGljeS5SRURJUkVDVF9UT19IVFRQUyxcbiAgICAgIH0sXG4gICAgICBwcmljZUNsYXNzOiBwcm9wcy5wcmljZUNsYXNzXG4gICAgICAgID8gcHJvcHMucHJpY2VDbGFzc1xuICAgICAgICA6IGNsb3VkZnJvbnQuUHJpY2VDbGFzcy5QUklDRV9DTEFTU18xMDAsXG4gICAgICBlcnJvclJlc3BvbnNlczogW1xuICAgICAgICB7XG4gICAgICAgICAgaHR0cFN0YXR1czogNDA0LFxuICAgICAgICAgIHJlc3BvbnNlUGFnZVBhdGg6IGAvJHtcbiAgICAgICAgICAgIHByb3BzLmVycm9yRG9jID8gcHJvcHMuZXJyb3JEb2MgOiBwcm9wcy5pbmRleERvY1xuICAgICAgICAgIH1gLFxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgaHR0cFN0YXR1czogNTAwLFxuICAgICAgICAgIHR0bDogY2RrLkR1cmF0aW9uLnNlY29uZHMoMiksXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBodHRwU3RhdHVzOiA0MDMsXG4gICAgICAgICAgcmVzcG9uc2VIdHRwU3RhdHVzOiAyMDAsXG4gICAgICAgICAgcmVzcG9uc2VQYWdlUGF0aDogYC8ke3Byb3BzLmluZGV4RG9jfWAsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgICAgZGVmYXVsdFJvb3RPYmplY3Q6IHByb3BzLmluZGV4RG9jLFxuICAgICAgZG9tYWluTmFtZXM6IFtwcm9wcy53ZWJzaXRlRG9tYWluXSxcbiAgICAgIGNlcnRpZmljYXRlOiB3ZWJzaXRlQ2VydCxcbiAgICB9KTtcblxuICAgIG5ldyBzM2RlcGxveS5CdWNrZXREZXBsb3ltZW50KHNjb3BlLCAnV2Vic2l0ZURlcGxveScsIHtcbiAgICAgIHNvdXJjZXM6IFtzM2RlcGxveS5Tb3VyY2UuYXNzZXQocHJvcHMud2Vic2l0ZUZvbGRlcildLFxuICAgICAgZGVzdGluYXRpb25CdWNrZXQ6IHdlYnNpdGVCdWNrZXQsXG4gICAgICBkaXN0cmlidXRpb246IHdlYnNpdGVEaXN0LFxuICAgICAgZGlzdHJpYnV0aW9uUGF0aHM6IFsnLycsIGAvJHtwcm9wcy5pbmRleERvY31gXSxcbiAgICB9KTtcblxuICAgIG5ldyByb3V0ZTUzLkFSZWNvcmQodGhpcywgJ1dlYmlzdGVBbGlhcycsIHtcbiAgICAgIHpvbmU6IGhvc3RlZFpvbmUsXG4gICAgICByZWNvcmROYW1lOiBwcm9wcy53ZWJzaXRlRG9tYWluLFxuICAgICAgdGFyZ2V0OiByb3V0ZTUzLlJlY29yZFRhcmdldC5mcm9tQWxpYXMoXG4gICAgICAgIG5ldyB0YXJnZXRzLkNsb3VkRnJvbnRUYXJnZXQod2Vic2l0ZURpc3QpLFxuICAgICAgKSxcbiAgICB9KTtcbiAgfVxufVxuIl19