# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['funcparserlib']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'funcparserlib',
    'version': '1.0.0a0',
    'description': 'Recursive descent parsing library based on functional combinators',
    'long_description': 'funcparserlib\n=============\n\nRecursive descent parsing library for Python based on functional combinators.\n\n[![PyPI](https://img.shields.io/pypi/v/funcparserlib)](https://pypi.org/project/funcparserlib/)\n[![PyPI - Downloads](https://img.shields.io/pypi/dm/funcparserlib)](https://pypi.org/project/funcparserlib/)\n\n\nDescription\n-----------\n\n**Parser combinators** are just higher-order functions that take parsers as\ntheir arguments and return them as result values. Parser combinators are:\n\n  * First-class values\n  * Extremely composable\n  * Tend to make the code quite compact\n  * Resemble the readable notation of xBNF grammars\n\nParsers made with `funcparserlib` are pure-Python LL(\\*) parsers. It means that\nit\'s **very easy to write them** without thinking about lookaheads and all\nthat hardcore parsing stuff. However, the recursive descent parsing is a rather\nslow method compared to LL(k) or LR(k) algorithms.\n\nSo the primary domain for `funcparserlib` is **parsing little languages** or\n**external DSLs** (domain specific languages).\n\nThe library itself is very small. Its source code is only 600 lines of code,\nwith lots of comments included. It features the longest parsed prefix error\nreporting, as well as a tiny lexer generator for token position tracking.\n\n\nShow Me the Code\n----------------\n\nThis is an excerpt from a JSON parser\n([RFC 4627](https://tools.ietf.org/html/rfc4627)) written using\n`funcparserlib`. This full example as well as others can be found\n[here](tests/json.py).\n\n```python\ndef parse(seq):\n    """Sequence(Token) -> object"""\n    ...\n    n = lambda s: tok("Name", s)\n    def make_array(values):\n        if values is None:\n            return []\n        else:\n            return [values[0]] + values[1]\n    ...\n    null = n("null") >> const(None)\n    true = n("true") >> const(True)\n    false = n("false") >> const(False)\n    number = tok("Number") >> make_number\n    string = tok("String") >> make_string\n    value = forward_decl()\n    member = string + -op(":") + value >> tuple\n    object = (\n        -op("{") +\n        maybe(member + many(-op(",") + member)) +\n        -op("}")\n        >> make_object)\n    array = (\n        -op("[") +\n        maybe(value + many(-op(",") + value)) +\n        -op("]")\n        >> make_array)\n    value.define(\n          null\n        | true\n        | false\n        | object\n        | array\n        | number\n        | string)\n    json_text = object | array\n    json_file = json_text + -finished\n\n    return json_file.parse(seq)\n```\n\n\nInstallation\n------------\n\nYou can install the `funcparserlib` library from\n[PyPI](https://pypi.python.org/pypi/funcparserlib) via `pip`:\n\n    $ pip install funcparserlib\n\nThere are no dependencies on other libraries.\n\n\nDocumentation\n-------------\n\n* [Nested Brackets Mini-HOWTO](doc/Brackets.md)\n    * A short intro to `funcparserlib`\n* [Tutorial](doc/Tutorial.md)\n    * The comprehensive `funcparserlib` tutorial\n\nSee also comments inside the modules `funcparserlib.parser` and\n`funcparserlib.lexer` or generate the API docs from the modules using `pydoc`.\n\nThere a couple of examples available in the tests/ directory:\n\n* [GraphViz DOT parser](tests/dot.py)\n* [JSON parser](tests/json.py)\n\nSee also [the changelog](docs/changes.md) and [FAQ](doc/FAQ.md).\n\n\nPerformance and Code Size\n-------------------------\n\nDespite being an LL(`*`) parser, `funcparserlib` has a reasonable performance.\nFor example, a JSON parser written using `funcparserlib` is 3 times faster\nthan a parser using the popular `pyparsing` library and only 5 times slower\nthan the specialized JSON library `simplejson` that uses _ad hoc_ parsing.\nHere are some stats:\n\n| **File Size** | **cjson** | **simplejson** | **funcparserlib** | **json-ply** | **pyparsing** |\n|:--------------|:----------|:---------------|:------------------|:-------------|:--------------|\n| 6 KB        | 0 ms    | 45 ms        | 228 ms          | n/a     | 802 ms      |\n| 11 KB       | 0 ms    | 80 ms        | 395 ms          | 367 ms  | 1355 ms     |\n| 100 KB      | 4 ms    | 148 ms       | 855 ms          | 1071 ms | 2611 ms     |\n| 134 KB      | 11 ms   | 957 ms       | 4775 ms         | n/a     | 16534 ms    |\n| 1009 KB     | 87 ms   | 6904 ms      | 36826 ms        | n/a     | 116510 ms   |\n| **User Code**    | 0.9 KLOC | 0.8 KLOC | 0.1 KLOC | 0.5 KLOC | 0.1 KLOC |\n| **Library Code** | 0 KLOC   | 0 KLOC   | 0.5 KLOC | 5.3 KLOC | 3.7 KLOC |\n\nBoth `funcparserlib` and `pyparsing` have the smallest user code size (that is\na common feature of parsing libraries compared to _ad hoc_ parsers). The\nlibrary code of `funcparserlib` is 7 times smaller (and much cleaner) than\n`pyparsing`. The `json-ply` uses a LALR parser `ply` (similar to Yacc) and\nperforms like `funcparserlib`. `cjson` is a C library, hence the incredible\nperformance :)\n\n\nUsed By\n-------\n\nSome open-source projects that use `funcparserlib` as an explicit dependency:\n\n* https://github.com/hylang/hy\n    * 3.8K stars, version `>= 0.3.6`, Python 3.6+\n* https://github.com/scrapinghub/splash\n    * 3.3K stars, version `*`. Python 3 in Docker\n* https://github.com/klen/graphite-beacon\n    * 460 stars, version `==0.3.6`, Python 2 and 3\n* https://github.com/blockdiag/blockdiag\n    * 118 stars, version `*`, Python 3.5+\n* https://github.com/pyta-uoft/pyta\n    * 48 stars, version `*`, Python 3.8+\n\n\nUsages in tests / secondary dependencies:\n\n* https://github.com/buildbot/buildbot\n    * 4.6K stars, version `== 0.3.6`\n* https://github.com/quay/quay\n    * 1.7K stars, version `==0.3.6`\n\n\n\nSimilar Projects\n----------------\n\n* [LEPL](https://code.google.com/p/lepl/). A recursive descent parsing\n  library that uses two-way generators for backtracking. Its source code is\n  rather large: 17 KLOC.\n* [pyparsing](https://github.com/pyparsing/pyparsing/). A recursive descent\n  parsing library. Probably the most popular Python parsing library.\n  Nevertheless, its source code is quite dirty (though 4 KLOC only).\n* [Monadic Parsing in Python](https://web.archive.org/web/20120507001413/http://sandersn.com/blog/?tag=/monads).\n  A series of blog entries on monadic parsing.\n* [Pysec (aka Parsec in Python)](http://www.valuedlessons.com/2008/02/pysec-monadic-combinatoric-parsing-in.html).\n  A blog entry on monadic parsing, with nice syntax for Python.\n',
    'author': 'Andrey Vlasovskikh',
    'author_email': 'andrey.vlasovskikh@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/vlasovskikh/funcparserlib',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=2.7, !=3.0.*, !=3.1.*, !=3.2.*, !=3.3.*, !=3.4.*, !=3.5.*',
}


setup(**setup_kwargs)
