import React from "react";
import { classNames } from "../../utils/classNames";
import { isBoolDtype, isDatetimeDtype, isFloatDtype, isIntDtype, isTimedeltaDtype } from "../../utils/dtypes";
import { getWidthData } from "./widthUtils";
export const isNumberInRangeInclusive = (num, start, end) => {
    return start <= num && num <= end;
};
/*
    A helper function for getting the first non-null or undefined
    value from a list of arguments.

    NOTE: May cause issues with all null or undefined values
*/
export function firstNonNullOrUndefined(...args) {
    for (let i = 0; i < args.length; i++) {
        const arg = args[i];
        if (arg !== null && arg !== undefined) {
            return arg;
        }
    }
    // NOTE: make sure one of the options is null or undefined
    return args[0];
}
/**
 * Get the grid state of a sheet that was just rendered/switched to.
 */
export const getDefaultGridState = (sheetDataArray, selectedSheetIndex) => {
    return {
        sheetIndex: selectedSheetIndex,
        viewport: {
            width: 0,
            height: 0,
        },
        scrollPosition: {
            scrollLeft: 0,
            scrollTop: 0
        },
        selections: [{
                startingColumnIndex: 0,
                endingColumnIndex: 0,
                startingRowIndex: -1,
                endingRowIndex: -1
            }],
        // When sheetDataArray is empty, we create a default widthDataArray so that we avoid 
        // indexing into undefined variables across the codebase.
        widthDataArray: (sheetDataArray.length === 0) ? [getWidthData(undefined)] : sheetDataArray.map(sheetData => getWidthData(sheetData)),
        columnIDsArray: getColumnIDsArrayFromSheetDataArray(sheetDataArray),
        searchString: ''
    };
};
// Returns an JSX Element with the type identifier for that type of column
export const getTypeIdentifier = (columnDtype, purpleOrDark) => {
    // Default to identifying the column as a string if we can't figure out what it is
    let typeText = 'str';
    if (isFloatDtype(columnDtype)) {
        typeText = 'float';
    }
    else if (isIntDtype(columnDtype)) {
        typeText = 'int';
    }
    else if (isDatetimeDtype(columnDtype)) {
        typeText = 'date';
    }
    else if (isTimedeltaDtype(columnDtype)) {
        typeText = 'time';
    }
    else if (isBoolDtype(columnDtype)) {
        typeText = 'bool';
    }
    return React.createElement("p", { className: classNames('text-subtext-1', { 'text-color-mito-purple-important': purpleOrDark === 'purple' }, { 'text-color-gray-important': purpleOrDark === 'dark' }) }, typeText);
};
/**
 * A helper function to get data describing a cell from
 * indexes, in a type safe way.
 */
export const getCellDataFromCellIndexes = (sheetData, rowIndex, columnIndex) => {
    var _a, _b;
    const columnID = (_a = sheetData === null || sheetData === void 0 ? void 0 : sheetData.data[columnIndex]) === null || _a === void 0 ? void 0 : _a.columnID;
    const columnHeader = (_b = sheetData === null || sheetData === void 0 ? void 0 : sheetData.data[columnIndex]) === null || _b === void 0 ? void 0 : _b.columnHeader;
    const columnFormula = columnID !== undefined ? sheetData === null || sheetData === void 0 ? void 0 : sheetData.columnSpreadsheetCodeMap[columnID] : undefined;
    const columnDtype = columnID !== undefined ? sheetData === null || sheetData === void 0 ? void 0 : sheetData.data[columnIndex].columnDtype : undefined;
    const columnFilters = columnID !== undefined ? sheetData === null || sheetData === void 0 ? void 0 : sheetData.columnFiltersMap[columnID] : undefined;
    const cellValue = columnID !== undefined ? sheetData === null || sheetData === void 0 ? void 0 : sheetData.data[columnIndex].columnData[rowIndex] : undefined;
    return {
        columnID: columnID,
        columnHeader: columnHeader,
        columnFormula: columnFormula,
        columnDtype: columnDtype,
        columnFilters: columnFilters,
        cellValue: cellValue,
    };
};
/*
    Helper function for creating the ColumnIDsMapping: sheetIndex -> columnIndex -> columnID
    from the Sheet Data Array
*/
export const getColumnIDsArrayFromSheetDataArray = (sheetDataArray) => {
    return sheetDataArray.map(sheetData => sheetData.data.map(c => c.columnID)) || [];
};
export const cellInSearch = (cellValue, searchString) => {
    if (searchString === '') {
        return false;
    }
    return ('' + cellValue).toLowerCase().search(searchString.toLowerCase()) > -1;
};
/*
    Determines if any sheet exists. Returns True if a sheet exists.
*/
export const doesAnySheetExist = (sheetDataArray) => {
    return sheetDataArray.length !== 0;
};
/*
    Determines if a columnID exists in a specific sheet. Returns True
*/
export const doesColumnExist = (columnID, sheetIndex, sheetDataArray) => {
    var _a;
    return columnID !== undefined && ((_a = sheetDataArray[sheetIndex]) === null || _a === void 0 ? void 0 : _a.columnDtypeMap[columnID]) !== undefined;
};
/*
    Determines if the sheet contains data
*/
export const doesSheetContainData = (sheetIndex, sheetDataArray) => {
    const sheetData = sheetDataArray[sheetIndex];
    return sheetData !== undefined && sheetData.numRows > 0 && sheetData.numColumns > 0;
};
//# sourceMappingURL=utils.js.map