# Copyright 2018-2021 Streamlit Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import hashlib
from typing import MutableMapping, TYPE_CHECKING
from weakref import WeakKeyDictionary

from streamlit import config
from streamlit.logger import get_logger
from streamlit.proto.ForwardMsg_pb2 import ForwardMsg

if TYPE_CHECKING:
    from streamlit.report_session import ReportSession

LOGGER = get_logger(__name__)


def populate_hash_if_needed(msg):
    """Computes and assigns the unique hash for a ForwardMsg.

    If the ForwardMsg already has a hash, this is a no-op.

    Parameters
    ----------
    msg : ForwardMsg

    Returns
    -------
    string
        The message's hash, returned here for convenience. (The hash
        will also be assigned to the ForwardMsg; callers do not need
        to do this.)

    """
    if msg.hash == "":
        # Move the message's metadata aside. It's not part of the
        # hash calculation.
        metadata = msg.metadata
        msg.ClearField("metadata")

        # MD5 is good enough for what we need, which is uniqueness.
        hasher = hashlib.md5()
        hasher.update(msg.SerializeToString())
        msg.hash = hasher.hexdigest()

        # Restore metadata.
        msg.metadata.CopyFrom(metadata)

    return msg.hash


def create_reference_msg(msg):
    """Create a ForwardMsg that refers to the given message via its hash.

    The reference message will also get a copy of the source message's
    metadata.

    Parameters
    ----------
    msg : ForwardMsg
        The ForwardMsg to create the reference to.

    Returns
    -------
    ForwardMsg
        A new ForwardMsg that "points" to the original message via the
        ref_hash field.

    """
    ref_msg = ForwardMsg()
    ref_msg.ref_hash = populate_hash_if_needed(msg)
    ref_msg.metadata.CopyFrom(msg.metadata)
    return ref_msg


class ForwardMsgCache(object):
    """A cache of ForwardMsgs.

    Large ForwardMsgs (e.g. those containing big DataFrame payloads) are
    stored in this cache. The server can choose to send a ForwardMsg's hash,
    rather than the message itself, to a client. Clients can then
    request messages from this cache via another endpoint.

    This cache is *not* thread safe. It's intended to only be accessed by
    the server thread.

    """

    class Entry(object):
        """Cache entry.

        Stores the cached message, and the set of ReportSessions
        that we've sent the cached message to.

        """

        def __init__(self, msg):
            self.msg = msg
            self._session_report_run_counts = (
                WeakKeyDictionary()
            )  # type: MutableMapping[ReportSession, int]

        def add_session_ref(self, session, report_run_count):
            """Adds a reference to a ReportSession that has referenced
            this Entry's message.

            Parameters
            ----------
            session : ReportSession
            report_run_count : int
                The session's run count at the time of the call

            """
            prev_run_count = self._session_report_run_counts.get(session, 0)
            if report_run_count < prev_run_count:
                LOGGER.error(
                    "New report_run_count (%s) is < prev_run_count (%s). "
                    "This should never happen!" % (report_run_count, prev_run_count)
                )
                report_run_count = prev_run_count
            self._session_report_run_counts[session] = report_run_count

        def has_session_ref(self, session):
            return session in self._session_report_run_counts

        def get_session_ref_age(self, session, report_run_count):
            """The age of the given session's reference to the Entry,
            given a new report_run_count.

            """
            return report_run_count - self._session_report_run_counts[session]

        def remove_session_ref(self, session):
            del self._session_report_run_counts[session]

        def has_refs(self):
            """True if this Entry has references from any ReportSession.

            If not, it can be removed from the cache.
            """
            return len(self._session_report_run_counts) > 0

    def __init__(self):
        self._entries = {}  # Map: hash -> Entry

    def add_message(self, msg, session, report_run_count):
        """Add a ForwardMsg to the cache.

        The cache will also record a reference to the given ReportSession,
        so that it can track which sessions have already received
        each given ForwardMsg.

        Parameters
        ----------
        msg : ForwardMsg
        session : ReportSession
        report_run_count : int
            The number of times the session's report has run

        """
        populate_hash_if_needed(msg)
        entry = self._entries.get(msg.hash, None)
        if entry is None:
            entry = ForwardMsgCache.Entry(msg)
            self._entries[msg.hash] = entry
        entry.add_session_ref(session, report_run_count)

    def get_message(self, hash):
        """Return the message with the given ID if it exists in the cache.

        Parameters
        ----------
        hash : string
            The id of the message to retrieve.

        Returns
        -------
        ForwardMsg | None

        """
        entry = self._entries.get(hash, None)
        return entry.msg if entry else None

    def has_message_reference(self, msg, session, report_run_count):
        """Return True if a session has a reference to a message.

        Parameters
        ----------
        msg : ForwardMsg
        session : ReportSession
        report_run_count : int
            The number of times the session's report has run

        Returns
        -------
        bool

        """
        populate_hash_if_needed(msg)

        entry = self._entries.get(msg.hash, None)
        if entry is None or not entry.has_session_ref(session):
            return False

        # Ensure we're not expired
        age = entry.get_session_ref_age(session, report_run_count)
        return age <= config.get_option("global.maxCachedMessageAge")

    def remove_expired_session_entries(self, session, report_run_count):
        """Remove any cached messages that have expired from the given session.

        This should be called each time a ReportSession finishes executing.

        Parameters
        ----------
        session : ReportSession
        report_run_count : int
            The number of times the session's report has run

        """
        max_age = config.get_option("global.maxCachedMessageAge")

        # Operate on a copy of our entries dict.
        # We may be deleting from it.
        for msg_hash, entry in self._entries.copy().items():
            if not entry.has_session_ref(session):
                continue

            age = entry.get_session_ref_age(session, report_run_count)
            if age > max_age:
                LOGGER.debug(
                    "Removing expired entry [session=%s, hash=%s, age=%s]",
                    id(session),
                    msg_hash,
                    age,
                )
                entry.remove_session_ref(session)
                if not entry.has_refs():
                    # The entry has no more references. Remove it from
                    # the cache completely.
                    del self._entries[msg_hash]

    def clear(self):
        """Remove all entries from the cache"""
        self._entries.clear()
