"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Schema = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const core_1 = require("@aws-cdk/core");
const appsync_generated_1 = require("./appsync.generated");
const private_1 = require("./private");
const schema_intermediate_1 = require("./schema-intermediate");
;
/**
 * The Schema for a GraphQL Api
 *
 * If no options are configured, schema will be generated
 * code-first.
 */
class Schema {
    constructor(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_SchemaOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Schema);
            }
            throw error;
        }
        if (options?.filePath) {
            this.mode = private_1.SchemaMode.FILE;
            this.definition = fs_1.readFileSync(options.filePath).toString('utf-8');
        }
        else {
            this.mode = private_1.SchemaMode.CODE;
            this.definition = '';
        }
        this.types = [];
    }
    /**
     * Generate a Schema from file
     *
     * @returns `SchemaAsset` with immutable schema defintion
     * @param filePath the file path of the schema file
     */
    static fromAsset(filePath) {
        return new Schema({ filePath });
    }
    /**
     * Called when the GraphQL Api is initialized to allow this object to bind
     * to the stack.
     *
     * @param api The binding GraphQL Api
     */
    bind(api) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_GraphqlApi(api);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        if (!this.schema) {
            this.schema = new appsync_generated_1.CfnGraphQLSchema(api, 'Schema', {
                apiId: api.apiId,
                definition: this.mode === private_1.SchemaMode.CODE ?
                    core_1.Lazy.string({
                        produce: () => this.types.reduce((acc, type) => `${acc}${type._bindToGraphqlApi(api).toString()}\n`, `${this.declareSchema()}${this.definition}`),
                    })
                    : this.definition,
            });
        }
        return this.schema;
    }
    /**
     * Escape hatch to add to Schema as desired. Will always result
     * in a newline.
     *
     * @param addition the addition to add to schema
     * @param delimiter the delimiter between schema and addition
     * @default - ''
     *
     */
    addToSchema(addition, delimiter) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error('API cannot append to schema because schema definition mode is not configured as CODE.');
        }
        const sep = delimiter ?? '';
        this.definition = `${this.definition}${sep}${addition}\n`;
    }
    /**
     * Add a query field to the schema's Query. CDK will create an
     * Object Type called 'Query'. For example,
     *
     * type Query {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the query
     * @param field the resolvable field to for this query
     */
    addQuery(fieldName, field) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_ResolvableField(field);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addQuery);
            }
            throw error;
        }
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error(`Unable to add query. Schema definition mode must be ${private_1.SchemaMode.CODE}. Received: ${this.mode}`);
        }
        if (!this.query) {
            this.query = new schema_intermediate_1.ObjectType('Query', { definition: {} });
            this.addType(this.query);
        }
        ;
        this.query.addField({ fieldName, field });
        return this.query;
    }
    /**
     * Add a mutation field to the schema's Mutation. CDK will create an
     * Object Type called 'Mutation'. For example,
     *
     * type Mutation {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Mutation
     * @param field the resolvable field to for this Mutation
     */
    addMutation(fieldName, field) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_ResolvableField(field);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addMutation);
            }
            throw error;
        }
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error(`Unable to add mutation. Schema definition mode must be ${private_1.SchemaMode.CODE}. Received: ${this.mode}`);
        }
        if (!this.mutation) {
            this.mutation = new schema_intermediate_1.ObjectType('Mutation', { definition: {} });
            this.addType(this.mutation);
        }
        ;
        this.mutation.addField({ fieldName, field });
        return this.mutation;
    }
    /**
     * Add a subscription field to the schema's Subscription. CDK will create an
     * Object Type called 'Subscription'. For example,
     *
     * type Subscription {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Subscription
     * @param field the resolvable field to for this Subscription
     */
    addSubscription(fieldName, field) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_Field(field);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addSubscription);
            }
            throw error;
        }
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error(`Unable to add subscription. Schema definition mode must be ${private_1.SchemaMode.CODE}. Received: ${this.mode}`);
        }
        if (!this.subscription) {
            this.subscription = new schema_intermediate_1.ObjectType('Subscription', { definition: {} });
            this.addType(this.subscription);
        }
        const directives = field.fieldOptions?.directives?.filter((directive) => directive.mutationFields);
        if (directives && directives.length > 1) {
            throw new Error(`Subscription fields must not have more than one @aws_subscribe directives. Received: ${directives.length}`);
        }
        this.subscription.addField({ fieldName, field });
        return this.subscription;
    }
    /**
     * Add type to the schema
     *
     * @param type the intermediate type to add to the schema
     *
     */
    addType(type) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_IIntermediateType(type);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addType);
            }
            throw error;
        }
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error('API cannot add type because schema definition mode is not configured as CODE.');
        }
        this.types.push(type);
        return type;
    }
    /**
     * Set the root types of this schema if they are defined.
     *
     * For example:
     * schema {
     *   query: Query
     *   mutation: Mutation
     *   subscription: Subscription
     * }
     */
    declareSchema() {
        if (!this.query && !this.mutation && !this.subscription) {
            return '';
        }
        const list = ['query', 'mutation', 'subscription'];
        return private_1.shapeAddition({
            prefix: 'schema',
            fields: list.map((key) => this[key] ? `${key}: ${this[key]?.name}` : '')
                .filter((field) => field != ''),
        }) + '\n';
    }
}
exports.Schema = Schema;
_a = JSII_RTTI_SYMBOL_1;
Schema[_a] = { fqn: "@aws-cdk/aws-appsync.Schema", version: "1.188.0" };
//# sourceMappingURL=data:application/json;base64,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