# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['dbd',
 'dbd.cli',
 'dbd.config',
 'dbd.db',
 'dbd.executors',
 'dbd.generator',
 'dbd.log',
 'dbd.tasks',
 'dbd.utils']

package_data = \
{'': ['*'],
 'dbd': ['resources/template/*', 'resources/template/model/*'],
 'dbd.generator': ['generator_templates/*']}

install_requires = \
['PyMySQL>=1.0.2,<2.0.0',
 'cerberus>=1.3.4,<2.0.0',
 'click>=8.0.3,<9.0.0',
 'fsspec==2022.1.0',
 'google-cloud-bigquery-storage>=2.10.1,<3.0.0',
 'google-cloud-storage>=2.0.0,<3.0.0',
 'greenlet>=1.1.2,<2.0.0',
 'jinja2>=3.0.3,<4.0.0',
 'networkx>=2.6.3,<3.0.0',
 'openpyxl>=3.0.9,<4.0.0',
 'pandas-gbq>=0.16.0,<0.17.0',
 'pandas>=1.3.5,<2.0.0',
 'psycopg2>=2.9.3,<3.0.0',
 'pyarrow>=6.0.0,<7.0.0',
 'pyyaml>=6.0,<7.0',
 'requests>=2.27.1,<3.0.0',
 's3fs==2022.1.0',
 'snowflake-connector-python>=2.7.2,<3.0.0',
 'snowflake-sqlalchemy>=1.3.3,<2.0.0',
 'sql-metadata>=2.3.0,<3.0.0',
 'sqlalchemy-bigquery>=1.3.0,<2.0.0',
 'sqlalchemy-redshift>=0.8.9,<0.9.0',
 'sqlalchemy>=1.4.25,<2.0.0',
 'sqlparse>=0.4.2,<0.5.0']

entry_points = \
{'console_scripts': ['dbd = dbd.cli.dbdcli:cli']}

setup_kwargs = {
    'name': 'dbd',
    'version': '0.7.5',
    'description': 'dbd is a data loading and transformation tool that enables data analysts and engineers to load and transform data in SQL databases.',
    'long_description': '# dbd: database loading and transformation tool\ndbd is a data loading and transformation tool that enables data analysts and engineers to load and transform data in SQL databases.\n\ndbd helps you with following tasks:\n- Loading CSV, JSON, Excel, and Parquet data to database. It supports both local and online files (HTTP URLs). Data can be loaded incrementally or in full. \n- Transforming data in existing database tables using insert-from-sql statements.\n- Executing DDL (Data Definition Language) SQL scripts (stetements like `CREATE SCHEMA`, etc.).    \n\n## How dbd works\ndbd processes a model directory that contains following elements:\n\n- **Directories** create new database schemas.\n- **Files** create new database table or view. The new table\'s or view\'s name is the same as the data file name.\n  - `.csv`, `.json`, `.xlsx`, and `.parquet` data files are introspected and loaded to database as  tables.   \n  - `.sql` files that contain SQL SELECT statements are executed and the result is loaded to database as table or view.\n  - `.ref` files contain one or more local paths or URLs pointing to supported data files. The referenced files are loaded to database as tables.  \n  - `.yaml` files contain metadata for the files above. The `.yaml` file has the same name as a data, `.sql`, or `.ref` file and specifies details of target table\'s columns (data types, constraints, indexes, etc.). `.yaml` files are optional. If not specified, dbd uses defaults (e.g. `TEXT` data types for CSV columns)\n  - `.ddl` files contain multiple SQL statements separated by semicolon that are executed against the database.\n\ndbd knows the correct order in which to process files in the model directory to respect mutual dependencies between created objects.\n\n![How dbd works](https://raw.githubusercontent.com/zsvoboda/dbd/master/img/dbd.infographic.png)\n\ndbd currently supports Postgres, MySQL/MariaDB, SQLite, Snowflake, BigQuery, and Redshift databases. \n\n## Getting started\nA short 5-minute getting started tutorial is available [here](https://zsvoboda.medium.com/analyze-covid-data-in-less-than-5-minutes-9176f440dd1a).\n\n## Examples\nCheck out dbd\'s [model directory examples](https://github.com/zsvoboda/dbd/tree/master/examples). The easiest way how to execute them is to either clone or download dbd\'s github repository and start with the SQLite examples.\n\n```shell\npip3 install dbd\ngit clone https://github.com/zsvoboda/dbd.git\ncd dbd/examples/sqlite/basic\ndbd --profile=../dbd.profile run . \n```\n\nThese commands should create a new `basic.db` SQLite database with `area`, `population`, and `state` tables that are created and loaded from the corresponding files in the `model` directory.\n\n## Installing dbd\ndbd requires Python 3.7.1 or higher. \n\n### PyPI\n\n```shell\npip3 install dbd\n```\n\nOR\n\n```shell\ngit clone https://github.com/zsvoboda/dbd.git\ncd dbd\npip3 install .\n```\n\n### Running dbd\nTry to execute `dbd` after the installation. If the command cannot be found, try to execute\n\n```shell\nexport PATH=~/.local/bin:$PATH\n```\n\nand run `dbd` again.\n\ndbd installs with SqLite driver only by default. To install dbd with other drivers, use the steps described in the \nchapters below.\n\n#### Postgres\nFirst, you\'ll need to install the Postgres development libraries:\n\n* For Ubuntu/Debian systems: `sudo apt-get install libpq-dev`\n* On Red Hat Linux (RHEL) systems: `yum install postgresql-devel`\n* For Mac Homebrew: `brew install postgresql`\n* For Mac MacPorts: `gem install pg -- --with-pg-config=/opt/local/lib/postgresql[version number]/bin/pg_config`\n* For OpenSuse: `zypper in postgresql-devel`\n* For ArchLinux: `pacman -S postgresql-libs`\n\nThen, you can install the Postgres driver:\n\n```shell\npip3 install psycopg2\n```\n\n#### MySQL/MariaDB\n\n```shell\npip3 install greenlet\npip3 install PyMySQL\n````\n\n#### Snowflake\nFollow the [steps in this document](https://docs.snowflake.com/en/user-guide/python-connector-install.html) \nto install the Snowflake driver.\n\nEssentially, you need to install the Snowflake Python connector with the following command:\n\n```shell\npip3 install snowflake-connector-python\npip3 snowflake-sqlalchemy\n```\n\nThe document above install the connector prerequisites.\n\n#### AWS Redshift\n\n`pip3 install sqlalchemy-redshift`\n\n#### Google BigQuery \n\n```shell\npip3 install google-cloud-storage\npip3 install google-cloud-bigquery-storage\npip3 install sqlalchemy-bigquery\npip3 install pandas-gbq\n````\n\n### Developers who want to contribute \n\n```shell\ngit clone https://github.com/zsvoboda/dbd.git\ncd dbd\npip3 install -e .\n``` \n\n## Starting new dbd project\nYou can generate dbd project initial layout by executing `init` command:\n\n```shell\ndbd init <new-project-name>\n```\n\nThe `init` command generates a new dbd project directory with the following content: \n\n- `model` directory that contains the content files.   \n- `dbd.profile` configuration file that defines database connections. The profile file is usually shared by more dbd projects. \n- `dbd.project` project configuration file references one of the connections from the profile file and define the `model` directory location.  \n\n## dbd profile configuration file\ndbd stores database connections in the `dbd.profile` configuration file. dbd searches for it in the current directory or in your home directory. You can use `--profile` option to point it to a profile file in different location.   \n\nThe profile file is YAML file with the following structure:\n\n```yaml\ndatabases:\n  db1:\n    db.url: <sql-alchemy-database-url>\n  db2:\n    db.url: <sql-alchemy-database-url>\n  db3:\n    db.url: <sql-alchemy-database-url>\n```\n\nRead [this document](https://docs.sqlalchemy.org/en/14/core/engines.html) for more details about  specific SQLAlchemy database URL formats.  \n\n## dbd project configuration file\ndbd stores project configuration in project configuration file that is usually stored in your dbd project directory. dbd searches for `dbd.project` file in your project\'s directory root. You can also use the `--project` option of the `dbd` command to specify a custom project configuration file. \n\nThe project configuration file also uses YAML format and references dbd model directory and databse connection from a profile config file. All paths in project file are either absolute or relative to the directory where the profile file is located. \n\nFor example:\n\n```yaml\nmodel: ./model\ndatabase: db2\n```\n\n## Model directory\nModel directory contains directories and files. Directories represent database schemas. Files, in  most cases, represent database tables. \n\nFor example, this `model` directory layout\n\n```text\ndbd-model-directory\n+- schema1\n +-- us_states.csv\n+- schema2\n +-- us_counties.csv\n```\n\ncreates two database schemas: `schema1` and `schema2` and two database tables: `us_states` in `schema1` and `us_counties` in `schema2`. Both tables are populated with the data from the CSV files.  \n\ndbd supports following files located in the `model` directory:\n\n* __DATA files:__ `.csv`, `.json`, `.xls`, `.xlsx`, `.parquet` files are loaded to database as tables\n* __REF files:__ `.ref` files contain one or more absolute or relative paths to local files or URLs of online data files that are loaded to database as tables. All referenced files must have the same structure as they are loaded to the same table.  \n* __SQL files:__ `.sql` with SQL SELECT statements are executed using insert-from-select SQL construct. The INSERT command is generated (the SQL file only contains a SQL SELECT statement)\n* __DDL files:__ contain a sequence of SQL statements separated by semicolon. The DDL files can be named `prolog.ddl` and `epilog.ddl`. The `prolog.ddl` is executed before all other files in a specific schema. The `epilog.ddl` is executed last. The `prolog.ddl` and `epilog.ddl` in the top-level model directory are executed as the very first or the very last files in the model. \n* __YAML files:__ specify additional configuration for the __DATA__, __SQL__, and __REF__ files.\n\n## REF files\n`.ref` file contains one or more references to files that dbd loads to the database as tables. The references can be URLs, absolute file paths or paths relative to the `.ref` file. All referenced data files must have the same structure as they are loaded to the same database table.\n\nHere is an example of a `.ref` file: \n\n```\nhttps://raw.githubusercontent.com/CSSEGISandData/COVID-19/master/csse_covid_19_data/csse_covid_19_daily_reports/01-03-2022.csv\nhttps://raw.githubusercontent.com/CSSEGISandData/COVID-19/master/csse_covid_19_data/csse_covid_19_daily_reports/01-04-2022.csv\n../data/01-05-2022.csv\n../data/01-06-2022.csv\n```\n\nThe paths and URLs can point to data files with different formats (e.g. CSV or JSON) as long as the files have the same structure (number of columns and column types).\n\n## SQL files \n`.sql` file performs SQL data transformation in the target database. It contains a SQL SELECT statement that dbd wraps in insert-from-select statement, executes it, and stores the result into a table or view that inherits its name from the SQL file name.\n\nHere is an example of `us_states.sql` file that creates a new `us_states` database table:\n\n```sqlite\nSELECT\n        state.abbrev AS state_code,\n        state.state AS state_name,\n        population.population AS state_population,\n        area.area_sq_mi  AS state_area_sq_mi\n    FROM state\n        JOIN population ON population.state = state.abbrev\n        JOIN area on area.state_name = state.state\n```\n\n## YAML files\n`.yaml` file specifies additional configuration for a corresponding __DATA__, __REF__ or __SQL__ file with the same base file name. Here is a YAML configuration example for the `us_states.sql` file above:\n\n```yaml\ntable:\n  columns:\n    state_code:\n      nullable: false\n      primary_key: true\n      type: CHAR(2)\n    state_name:\n      nullable: false\n      index: true\n      type: VARCHAR(50)\n    state_population:\n      nullable: false\n      type: INTEGER\n    state_area_sq_mi:\n      nullable: false\n      type: INTEGER\nprocess:\n  materialization: table\n  mode: drop\n```\n\nThis `.yaml` file re-types the `state_population` and the `state_area_sq_mi` columns to INTEGER, disallows NULL values in all columns, and makes the `state_code` column table\'s primary key. \n\nYou don\'t have to describe all table\'s columns. The columns that you leave out will have their types\nset to the default TEXT datatype in case of DATA files and is defined by the insert-from-select in case of SQL files.    \n\nThe `us_states.sql` table is dropped and data are re-loaded in full everytime the dbd executes this model. \n\n### Table section\n`.yaml` file\'s columns are mapped to a columns of the table that dbd creates from a corresponding __DATA__, __REF__ or __SQL__ file. For example, a CSV header columns or SQL SELECT column `AS` column clauses. \n\ndbd supports following column\'s parameters:\n\n* __type:__ column\'s SQL type.\n* __primary_key:__ is the column part of table\'s primary key (true|false)?\n* __foreign_keys:__ all other database table columns that are referenced from a column in table (in format `foreign-table`.`referenced-column`).\n* __nullable:__ does column allow null values (true|false)?\n* __index:__ is column indexed (true|false)?\n* __unique:__ does column store unique values (true|false)?\n\n### Process section\nThe `process` section defines following processing options:\n\n* __materialization:__ specifies whether dbd creates a physical `table` or a `view` when processing  SQL file. The __REF__ and __DATA__ files always yield physical table. \n* __mode:__ specifies what dbd does with table\'s data. You can specify values `drop`, `truncate`, or `keep`. The  __mode__ option is ignored for views.\n\n## Jinja templates\nMost of model files support [Jinja2 templates](https://jinja.palletsprojects.com/en/3.0.x/). For example, this __REF__ file loads 6 CSV files to database (4 online files from a URL and 2 from a local filesystem):\n\n```jinja\n{% for n in range(4) %}\nhttps://raw.githubusercontent.com/CSSEGISandData/COVID-19/master/csse_covid_19_data/csse_covid_19_daily_reports/01-0{{ n+1 }}-2022.csv\n{% endfor %}\n../data/01-05-2022.csv\n../data/01-06-2022.csv\n```\nProfile an project configuration files also us Jinja2 templates. You can expend any environment variable with the `{{ environment-variable-name }}` syntax.\nFor example, you can define your database connection parameters like username or password in environment variables and use this profile configuration file:\n\n```yaml\ndatabases:\n  states_snowflake:\n    db.url: "snowflake://{{ SNOWFLAKE_USER }}:{{ SNOWFLAKE_PASSWORD }}@{{ SNOWFLAKE_ACCOUNT_IDENTIFIER }}/{{ SNOWFLAKE_DB }}/{{ SNOWFLAKE_SCHEMA }}?warehouse={{SNOWFLAKE_WAREHOUSE }}"\n  covid:\n    db.url: "snowflake://{{ SNOWFLAKE_USER }}:{{ SNOWFLAKE_PASSWORD }}@{{ SNOWFLAKE_ACCOUNT_IDENTIFIER }}/{{ SNOWFLAKE_DB }}/{{ SNOWFLAKE_SCHEMA }}?warehouse={{SNOWFLAKE_WAREHOUSE }}"\n```\n\n## Fast data loading mode\nAll supported database engines except SQLite support fast data loading mode. In this mode, data are loaded to a \ndatabase table using bulk load (SQL COPY) command instead of individual INSERT statements.\n\nMySQL and Redshift require additional configuration to enable fast data loading mode. \nWithout this extra configuration dbd reverts to slow inserting mode via INSERT statements.\n\n### MySQL \nTo enable fast loading mode, you need specify `local_infile=1` query parameter in the MySQL connection url.\nYou also must enable the LOCAL INFILE mode on your MySQL server. You can for example do this by executing this \nSQL statement:\n\n```mysql\nSET GLOBAL local_infile = true;\n```\n\n# Redsift\nTo enable fast loading mode, you need specify `copy_stage` parameter in the `dbd.project` configuration file. \nThe `copy_stage` parameter must reference a storage definition in your `dbd.profile` configuration file.\nCheck the example configuration files in the `examples/redshift/covid_cz` directory. Here are the example definitions of the \nenvironment variables that these configuration files use:\n\n```shell\nexport AWS_COVID_STAGE_S3_URL="s3://covid/stage"\nexport AWS_COVID_STAGE_S3_ACCESS_KEY="AKIA43SWERQGXMUYFIGMA"\nexport AWS_COVID_STAGE_S3_S3_SECRET_KEY="iujI78eDuFFGJF6PSjY/4CIhEJdMNkuS3g4t0BRwX"\n```\n\n\n\n## License\ndbd code is open-sourced under [BSD 3-clause license](LICENSE). \n\n## Resources and References\n- [dbd getting started](https://zsvoboda.medium.com/analyze-covid-data-in-less-than-5-minutes-9176f440dd1a)\n- [dbd github repo](https://github.com/zsvoboda/dbd)\n- [dbd PyPi](https://pypi.org/project/dbd/)\n- [Submit issue](https://github.com/zsvoboda/dbd/issues)\n',
    'author': 'zsvoboda',
    'author_email': 'zsvoboda@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/zsvoboda/dbd/blob/master/README.md',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7.1,<3.11',
}


setup(**setup_kwargs)
