import copy
from json import JSONDecodeError
from typing import List, Type

import requests
from pydantic import BaseModel, parse_obj_as

from tktl.core import loggers as sdk_logger
from tktl.core import utils
from tktl.core.exceptions.exceptions import APIClientException
from tktl.core.managers.auth import AuthConfigManager


class API(object):
    DEFAULT_HEADERS = {
        "Accept": "application/json",
    }

    def __init__(self, api_url, headers=None, logger=sdk_logger.MuteLogger()):
        """

        Parameters
        ----------
        api_url
        headers
        logger
        """

        self.api_url = api_url
        headers = headers or self.DEFAULT_HEADERS
        self.headers = headers.copy()
        self.logger = logger

    @property
    def default_headers(self):
        api_key = AuthConfigManager.get_api_key()
        self.DEFAULT_HEADERS.update({"X-Api-Key": api_key})
        return self.DEFAULT_HEADERS

    def get_path(self, url=None):
        if not url:
            return self.api_url
        full_path = utils.concatenate_urls(self.api_url, url)
        return full_path

    def post(self, url=None, json=None, params=None, files=None, data=None):
        path = self.get_path(url)
        headers = copy.deepcopy(self.default_headers)

        self.logger.debug(
            "POST request sent to: {} \n\theaders: {}\n\tjson: {}\n\tparams: {}\n\tfiles: {}\n\tdata: {}".format(
                path, headers, json, params, files, data
            )
        )
        response = requests.post(
            path, json=json, params=params, headers=headers, files=files, data=data
        )
        self.logger.debug("Response status code: {}".format(response.status_code))
        self.logger.debug("Response content: {}".format(response.content))
        return response

    def put(self, url=None, json=None, params=None, data=None):
        path = self.get_path(url)
        self.logger.debug(
            "PUT request sent to: {} \n\theaders: {}\n\tjson: {}\n\tparams: {}".format(
                path, self.headers, json, params
            )
        )
        response = requests.put(
            path, json=json, params=params, headers=self.default_headers, data=data
        )
        self.logger.debug("Response status code: {}".format(response.status_code))
        self.logger.debug("Response content: {}".format(response.content))
        return response

    def patch(self, url=None, json=None, params=None, data=None):
        path = self.get_path(url)
        self.logger.debug(
            "PATCH request sent to: {} \n\theaders: {}\n\tjson: {}\n\tparams: {}".format(
                path, self.headers, json, params
            )
        )
        response = requests.patch(
            path, json=json, params=params, headers=self.default_headers, data=data
        )
        self.logger.debug("Response status code: {}".format(response.status_code))
        self.logger.debug("Response content: {}".format(response.content))
        return response

    def get(self, url=None, json=None, params=None):
        path = self.get_path(url)

        self.logger.debug(
            "GET request sent to: {} \n\theaders: {}\n\tjson: {}\n\tparams: {}".format(
                path, self.headers, json, params
            )
        )
        response = requests.get(
            path, params=params, headers=self.default_headers, json=json
        )
        self.logger.debug("Response status code: {}".format(response.status_code))
        self.logger.debug("Response content: {}".format(response.content))
        return response

    def delete(self, url=None, json=None, params=None):
        path = self.get_path(url)
        response = requests.delete(path, params=params, headers=self.headers, json=json)
        self.logger.debug(
            "DELETE request sent to: {} \n\theaders: {}\n\tjson: {}\n\tparams: {}".format(
                response.url, self.headers, json, params
            )
        )
        self.logger.debug("Response status code: {}".format(response.status_code))
        self.logger.debug("Response content: {}".format(response.content))
        return response


def interpret_response(response: requests.Response, model: Type[BaseModel]):
    """
    Parameters
    ----------
    response

    model
    """
    try:
        response.raise_for_status()
    except requests.exceptions.RequestException as e:
        if e.response.status_code != 500:
            try:
                response = response.json()
                raise APIClientException(
                    status_code=e.response.status_code, detail=response["detail"]
                )
            except JSONDecodeError:
                raise APIClientException(
                    status_code=e.response.status_code, detail=repr(response)
                )
        else:
            raise APIClientException(
                status_code=e.response.status_code,
                detail=f"Something went wrong: {repr(response.content)}",
            )
    if model:
        json = response.json()
        if isinstance(json, list):
            if not json:
                return []

            if model.schema()["type"] == "array":
                return model.parse_obj(json)

            elif model.schema()["type"] == "object":
                return parse_obj_as(List[model], json)  # noqa
        return model(**response.json())
    else:
        if response.content:
            return response.json()
