from typing import Any
from pathlib import Path

import numpy as np
from numpy.typing import NDArray as NDArray
from sl_shared_assets import SessionData

from .tools import (
    MesoscopeData as MesoscopeData,
    get_system_configuration as get_system_configuration,
)
from ..shared_components import (
    WaterSheet as WaterSheet,
    SurgerySheet as SurgerySheet,
)

def _delete_directory(directory_path: Path) -> None:
    """Removes the input directory and all its subdirectories using parallel processing.

    This function outperforms default approaches like subprocess call with rm -rf and shutil rmtree for directories with
    a comparably small number of large files. For example, this is the case for the mesoscope frame directories, which
    are deleted ~6 times faster with this method over sh.rmtree. Potentially, it may also outperform these approaches
    for all comparatively shallow directories.

    Args:
        directory_path: The path to the directory to delete.
    """

def _check_stack_size(file: Path) -> int:
    """Reads the header of the input TIFF file, and if the file is a stack, extracts its size.

    This function is used to both determine the stack size of the processed TIFF files and to exclude non-mesoscope
    TIFFs from processing.

    Notes:
        This function only works with monochrome TIFF stacks generated by the mesoscope. It expects each TIFF file to
        be a stack of 2D frames.

    Args:
        file: The path to the TIFF file to evaluate.

    Returns:
        If the file is a stack, returns the number of frames (pages) in the stack. Otherwise, returns 0 to indicate that
        the file is not a stack.
    """

def _process_stack(
    tiff_path: Path, first_frame_number: int, output_directory: Path, verify_integrity: bool, batch_size: int = 250
) -> dict[str, Any]:
    """Reads a TIFF stack, extracts its frame-variant ScanImage data, and saves it as a LERC-compressed stacked TIFF
    file.

    This is a worker function called by the process_mesoscope_directory in-parallel for each stack inside each
    processed directory. It re-compresses the input TIFF stack using LERC-compression and extracts the frame-variant
    ScanImage metadata for each frame inside the stack. Optionally, the function can be configured to verify data
    integrity after compression.

    Notes:
        This function can reserve up to double the processed stack size of RAM bytes to hold the data in memory. If the
        host-computer does not have enough RAM, reduce the number of concurrent processes, reduce the batch size, or
        disable verification.

    Raises:
        RuntimeError: If any extracted frame does not match the original frame stored inside the TIFF stack.
        NotImplementedError: If extracted frame-variant metadata contains unexpected keys or expected keys for which
            we do not have a custom extraction implementation.

    Args:
        tiff_path: The path to the TIFF stack to process.
        first_frame_number: The position (number) of the first frame stored in the stack, relative to the overall
            sequence of frames acquired during the experiment. This is used to configure the output file name to include
            the range of frames stored in the stack.
        output_directory: The path to the directory where to save the processed stacks.
        verify_integrity: Determines whether to verify the integrity of compressed data against the source data.
            The conversion does not alter the source data, so it is usually safe to disable this option, as the chance
            of compromising the data is negligible. Note, enabling this function doubles the RAM usage for each worker
            process.
        batch_size: The number of frames to process at the same time. This directly determines the RAM footprint of
            this function, as frames are kept in RAM during compression. Note, verification doubles the RAM footprint,
            as it requires both compressed and uncompressed data to be kept in RAM for comparison.

    Returns:
        A dictionary containing the extracted frame-variant ScanImage metadata for the processed stack.
    """

def _generate_ops(metadata: dict[str, Any], frame_data: NDArray[np.int16], ops_path: Path) -> None:
    """Uses frame-invariant ScanImage metadata to create an ops.json at the specified ops_path.

    This function is an implementation of the mesoscope data extraction helper from the original suite2p library. The
    helper function has been reworked to use the metadata parsed by tifffile and reimplemented in Python. Primarily,
    this function generates the 'fs', 'dx', 'dy', 'lines', 'nroi', 'nplanes' and 'mesoscan' fields of the 'ops'
    configuration file. The Sun lab suite2p implementation is statically configured to find and use these files when
    working with mesoscope data to configure the single-day cell activity extraction pipeline.

    Notes:
        The generated ops.json file will be saved at the location and filename specified by the ops_path.

    Args:
        metadata: The dictionary containing ScanImage metadata extracted from a mesoscope tiff stack file.
        frame_data: A numpy array containing the extracted pixel data for the first frame of the stack.
        ops_path: The path to the output ops.json file. This is generated by the ProjectData class and passed down to
            this method via the main directory processing function.
    """

def _process_invariant_metadata(file: Path, ops_path: Path, metadata_path: Path) -> None:
    """Extracts frame-invariant ScanImage metadata from the target tiff file and uses it to generate metadata.json and
    ops.json files.

    This function only needs to be called for one raw ScanImage TIFF stack acquired as part of the same experimental
    session. It extracts the ScanImage metadata that is common for all frames across all stacks and outputs it as a
    metadata.json file. This function also calls the _generate_ops() function that generates a suite2p ops.json file
    from the parsed metadata.

    Notes:
        This function is primarily designed to preserve the metadata before compressing raw TIFF stacks with the
        Limited Error Raster Compression (LERC) scheme.

    Args:
        file: The path to the mesoscope TIFF stack file. This can be any file in the directory as the
            frame-invariant metadata is the same for all stacks.
        ops_path: The path to the ops.json file that should be created by this function. This is resolved by the
            ProjectData class to match the processed project, animal, and session combination.
        metadata_path: The path to the metadata.json file that should be created by this function. This is resolved
            by the ProjectData class to match the processed project, animal, and session combination.
    """

def _preprocess_video_names(session_data: SessionData) -> None:
    """Renames the video (camera) files generated during runtime to use human-friendly camera names, rather than
    ID-codes.

    This is a minor preprocessing function primarily designed to make further data processing steps more human-friendly.
    Since version 2.0.0, this method is also essential for further data processing, as it ensures that video names are
    unique by including session names (timestamps). This is necessary to support DeepLabCut tracking, as DLC is not
    able to resolve unique videos based on their storage paths.

    Args:
        session_data: The SessionData instance for the processed session.
    """

def _pull_mesoscope_data(
    session_data: SessionData,
    num_threads: int = 30,
    remove_sources: bool = True,
    verify_transfer_integrity: bool = False,
) -> None:
    """Pulls the data acquired by the Mesoscope from the ScanImagePC to the VRPC.

    This function should be called after the data acquisition runtime to aggregate all recorded data on the VRPC
    before running the preprocessing pipeline. The function expects that the mesoscope frames source directory
    contains only the frames acquired during the current session runtime alongside additional data, such as
    MotionEstimation .csv files.

    Notes:
        It is safe to call this function for sessions that did not acquire mesoscope frames. It is designed to
        abort early if it cannot discover the cached mesoscope frames data for the target session on the ScanImagePC.

        This function expects that the data acquisition runtime has renamed the mesoscope_frames source directory for
        the session to include the session name. Manual intervention may be necessary if the runtime fails before the
        mesoscope_frames source directory is renamed.

        This function is configured to parallelize data transfer and verification to optimize runtime speeds where
        possible.

    Args:
        session_data: The SessionData instance for the processed session.
        remove_sources: Determines whether to remove the transferred mesoscope frame data from the ScanImagePC.
            Generally, it is recommended to remove source data to keep ScanImagePC disk usage low. Note, setting
            this to True will only mark the data for removal. The removal is carried out by the dedicated data purging
            function that runs at the end of the session data preprocessing sequence.
        verify_transfer_integrity: Determines whether to verify the integrity of the transferred data. This is
            performed before the source folder is marked for removal from the ScanImagePC if remove_sources is True.
    """

def _preprocess_mesoscope_directory(
    session_data: SessionData,
    num_processes: int,
    remove_sources: bool = True,
    batch: bool = False,
    verify_integrity: bool = False,
    batch_size: int = 250,
) -> None:
    """Loops over all multi-frame Mesoscope TIFF stacks acquired for the session, recompresses them using the Limited
    Error Raster Compression (LERC) scheme, and extracts ScanImage metadata.

    This function is used as a preprocessing step for mesoscope-acquired data that optimizes the size of raw images for
    long-term storage and streaming over the network. To do so, all stacks are re-encoded using LERC scheme, which
    achieves ~70% compression ratio, compared to the original frame stacks obtained from the mesoscope. Additionally,
    this function also extracts frame-variant and frame-invariant ScanImage metadata from raw stacks and saves it as
    efficiently encoded JSON (.json) and compressed numpy archive (.npz) files to minimize disk space usage.

    Notes:
        This function is specifically calibrated to work with TIFF stacks produced by the ScanImage matlab software.
        Critically, these stacks are named using '_' to separate acquisition and stack number from the rest of the
        file name, and the stack number is always found last, e.g.: 'Tyche-A7_2022_01_25_1_00001_00067.tif'. If the
        input TIFF files do not follow this naming convention, the function will not process them. Similarly, if the
        stacks do not contain ScanImage metadata, they will be excluded from processing.

        To optimize runtime efficiency, this function employs multiple processes to work with multiple TIFFs at the
        same time. Given the overall size of each image dataset, this function can run out of RAM if it is allowed to
        operate on the entire folder at the same time. To prevent this, disable verification, use fewer processes, or
        change the batch_size to load fewer frames in memory at the same time.

        In addition to frame compression and data extraction, this function also generates the ops.json configuration
        file. This file is used during suite2p cell registration, performed as part of our standard data processing
        pipeline.

        This function is purposefully designed to collapse data from multiple acquisitions stored inside the same
        directory into the same frame volume. This implementation was chosen based on the specific patterns of data
        acquisition in the Sun lab, where all data acquired for a single session is necessarily expected to belong to
        the same acquisition.

    Args:
        session_data: The SessionData instance for the processed session.
        num_processes: The maximum number of processes to use while processing the directory. Each process is used to
            compress a stack of TIFF files in parallel.
        remove_sources: Determines whether to remove the original TIFF files after they have been processed.
        batch: Determines whether the function is called as part of batch-processing multiple directories. This is used
            to optimize progress reporting to avoid cluttering the terminal window.
        verify_integrity: Determines whether to verify the integrity of compressed data against the source data.
            The conversion does not alter the source data, so it is usually safe to disable this option, as the chance
            of compromising the data is negligible. Note, enabling this function doubles the RAM used by each parallel
            worker spawned by this function.
        batch_size: Determines how many frames are loaded into memory at the same time during processing. Note, the same
            number of frames will be loaded from each stack processed in parallel.
    """

def _preprocess_log_directory(
    session_data: SessionData, num_processes: int, remove_sources: bool = True, verify_integrity: bool = False
) -> None:
    """Compresses all .npy (uncompressed) log entries stored in the behavior log directory into one or more .npz
    archives.

    This service function is used during data preprocessing to optimize the size and format used to store all log
    entries. Primarily, this is necessary to facilitate data transfer over the network and log processing on the
    BioHPC server.

    Args:
        session_data: The SessionData instance for the processed session.
        num_processes: The maximum number of processes to use while processing the directory.
        remove_sources: Determines whether to remove the original .npy files after they are compressed into .npz
            archives. It is recommended to have this option enabled.
        verify_integrity: Determines whether to verify the integrity of compressed data against the source data.
            It is advised to have this disabled for most runtimes, as data corruption is highly unlikely, but enabling
            this option adds a significant overhead to the processing time.

    Raises:
        RuntimeError: If the target log directory contains both compressed and uncompressed log entries.
    """

def _resolve_telomere_marker(session_data: SessionData) -> None:
    """Reads the value of the 'incomplete' flag from the session's descriptor file and, if necessary, creates the
    telomere.bin marker.

    The telomere marker file is used by our data processing pipelines to determine whether to process the session.
    Incomplete sessions lacking telomere.bin are excluded from all further automated processing.

    Args:
        session_data: The SessionData instance for the processed session.
    """

def _preprocess_google_sheet_data(session_data: SessionData) -> None:
    """Updates the water restriction log and the surgery_data.yaml file.

    This internal method is called as part of preprocessing. Primarily, it is used to ensure that each session folder
    contains the up-to-date information about the surgical intervention(s) performed on the animal before running the
    session. It also updates the water restriction log for the managed animal to reflect the water received before and
    after runtime.

    Args:
        session_data: The SessionData instance for the processed session.

    Raises:
        ValueError: If the session_type attribute of the input SessionData instance is not one of the supported options.
    """

def _push_data(session_data: SessionData, parallel: bool = True, num_threads: int = 15) -> None:
    """Copies the raw_data directory from the VRPC to the NAS and the BioHPC server.

    This internal method is called as part of preprocessing to move the preprocessed data to the NAS and the server.
    This method generates the xxHash3-128 checksum for the source folder that the server processing pipeline uses to
    verify the integrity of the transferred data.

    Args:
        session_data: The SessionData instance for the processed session.
        parallel: Determines whether to parallelize the data transfer. When enabled, the method will transfer the
            data to all destinations at the same time (in parallel). Note, this argument does not affect the number
            of parallel threads used by each transfer process or the number of threads used to compute the
            xxHash3-128 checksum. This is determined by the 'num_threads' argument (see below). Note; each parallel
            process can use as many threads as specified by 'num_threads' at the same time.
        num_threads: Determines the number of threads used by each transfer process to copy the files and calculate
            the xxHash3-128 checksums. Since each process uses the same number of threads, it is highly
            advised to set this value so that num_threads * 2 (number of destinations) does not exceed the total
            number of CPU cores - 4.
    """

def _verify_remote_data_integrity(session_data: SessionData) -> None:
    """Verifies that the data was moved to the BioHPC server intact and creates the session's processed data directories
    on the server.

    This service function runs at the end of the data preprocessing pipeline, after the data has been transferred to
    the BioHPC server. Primarily, it is used to verify that the data was moved intact, and it is safe to delete the
    local copy of the data stored on the VRPC. Additionally, it is the only function allowed to create processed
    data directories on the BioHPC server, which is a prerequisite for all further data processing steps.

    Notes:
        To optimize runtime speed, the server executes the verification check submitted as a remote job. Depending on
        the overall server utilization, it is possible for the job execution to be delayed if the server does not have
        enough spare resources to run the job.

        If integrity verification fails, this job will delete the telomere.bin marker file stored in the session's
        raw_data folder on the server. If it succeeds, the job will generate the 'ubiquitin.bin' marker file in the
        local (VRPC) session's raw_data folder, marking it for deletion. Generating the deletion marker file will not
        itself delete the data, that step is performed at a later time point by a dedicated 'purging' function.

    Args:
        session_data: The SessionData instance for the processed session.

    """

def rename_mesoscope_directory(session_data: SessionData) -> None:
    """This function renames the 'shared' mesoscope_data directory to use the name specific to the target session.

    Since this is an essential step for the preprocessing pipeline to discover and pull the mesoscope data to VRPC
    during preprocessing, it has to be done before running the mesoscope data preprocessing. Ideally, this function
    should be called by the MesoscopeVRSystem stop() method, but it is also called by the preprocessing pipeline's
    main function.
    """

def preprocess_session_data(session_data: SessionData) -> None:
    """Aggregates all data on VRPC, compresses it for efficient network transmission, safely transfers the data to the
    BioHPC server and the Synology NAS for long-term storage, and removes all local data copies.

    This method should be called at the end of each training and experiment runtime to preprocess the data. Primarily,
    it prepares the data for further processing, moves it to appropriate long-term storage destinations, and keeps the
    VRPC and ScanImagePC filesystem free from clutter by removing redundant local data copies.

    Args:
        session_data: The SessionData instance for the processed session.
    """

def purge_redundant_data() -> None:
    """Loops over ScanImagePC and VRPC directories that store training and experiment data and removes no longer
    necessary data caches.

    This function searches the ScanImagePC and VRPC for no longer necessary directories and removes them from the
    respective systems. ScanImagePC directories are marked for deletion once they are safely copied to the VRPC. VRPC
    directories are marked for deletion once the data is safely copied to the BioHPC server. Both copying steps include
    verifying the integrity of the transferred data using xxHash-128 checksums.

    Notes:
        This is a service function intended to maintain the ScanImagePC and VRPC disk space. Once the data is moved to
        the BioHPC server and the NAS, it is generally safe to remove the copies stored on the ScanImagePC and VRPC.

        Currently, this function does not discriminate between projects or animals. It will remove all data marked for
        deletion via the ubiquitin.bin markers.
    """

def purge_failed_session(session_data: SessionData) -> None:
    """Removes all data and directories associated with the input session.

    This function is extremely dangerous and should be used with caution. It is designed to remove all data from failed
    or no longer necessary sessions. Never use this function on sessions that contain valid scientific data.

    Args:
        session_data: The SessionData instance for the session whose data needs to be removed.
    """
