# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Exceptions with bundle logic reason"""
from adcm_pytest_plugin.exceptions import BuiltinLikeAssertionError


class BundleError(BuiltinLikeAssertionError):
    """Generic bundle error"""

    def __init__(self, msg=""):
        super().__init__(msg or self.__doc__)

    @classmethod
    def raise_if_suitable(cls, message):
        """Raise suitable bundle error if possible"""
        __tracebackhide__ = True  # pylint: disable=unused-variable

        if any(map(lambda x: x in message, ["Wrong context", "AnsibleFilterError", "ERROR!"])):
            raise AnsibleError(message)
        if "control process exited with error" in message:
            raise SystemctlError(message)
        if "Cannot allocate memory" in message:
            raise AllocateMemoryError(message)
        if "Timeout when waiting for" in message:
            raise TimeoutWaitingForError(message)


class AnsibleError(BundleError):
    """Error in ansible logic"""


class SystemctlError(BundleError):
    """Systemctl job not working"""


class AllocateMemoryError(BundleError):
    """Cannot allocate memory"""


class TimeoutWaitingForError(BundleError):
    """Timeout when waiting for"""
