"""Tests for `pystv` package."""
import pytest
from click.testing import CliRunner
from numpy.testing import assert_equal

import pystv
from pystv import RaceInfo, RaceMetadata, RaceResult
from pystv import RoundResult as RR
from pystv import cli

from .utils import assert_race_equal

FAIRVOTE_BALLOTS = [
    ([1, 2, 3], 625),  # R4
    ([1, 2, 4], 125),  # R4
    ([1, 2, 5], 250),  # R4; Last is any unviable candidate
    ([1, 2, 6], 250),  # R4; Last is any unviable candidate
    ([1, 5, 3], 500),  # R3
    ([1, 5, 4], 500),  # R3
    ([1, 3], 250),
    ([2, 3, 0], 875),  # R4
    ([2, 4], 175),  # R4
    ([2, 5, 0], 350),  # R4; Last is any unviable candidate
    ([2, 6, 0], 350),  # R4; Last is any unviable candidate
    ([3], 1300),
    ([4, 0, 0], 1300),
    ([5, 2, 3], 625),  # R3, R4
    ([5, 2, 4], 125),  # R3, R4
    ([5, 2, 6], 500),  # R3, R4; Last is any unviable candidate
    ([5, 3], 100),  # R3
    ([6, 3, 0], 580),
    ([6, 4], 300),
    ([6, 2, 3], 50),  # R4
    ([6, 2, 4], 10),  # R4
    ([6, 2, 5], 40),  # R4; Last is any unviable candidate
    ([6, 5, 3], 10),  # R3
    ([6, 5, 4], 10),  # R3
]


def test_2cands_1seat():
    metadata = RaceMetadata("race", num_seats=1, names=["A", "B"])
    ballots = [[2, 1], [1, 2]]
    votes = [2, 1]
    actual = pystv.run_stv(RaceInfo(metadata, ballots, votes))
    desired = RaceResult(metadata, [RR([0, 1, 2], [2], [], {})])
    assert actual == desired


def test_2cands_1seat_undervote():
    metadata = RaceMetadata("race", num_seats=1, names=["A", "B"])
    ballots = [[2, 0], [2, 1], [1, 2]]
    votes = [1, 1, 1]
    actual = pystv.run_stv(RaceInfo(metadata, ballots, votes))
    desired = RaceResult(metadata, [RR([0, 1, 2], [2], [], {})])
    assert actual == desired


def test_4cands_2seat_undervote():
    metadata = RaceMetadata("race", num_seats=2, names=["A", "B", "C", "D"])
    ballots = [[4, 0], [3, 0], [2, 4], [2, 0], [1, 4], [1, 0]]
    votes = [3, 3, 1, 1, 1, 1]
    actual = pystv.run_stv(RaceInfo(metadata, ballots, votes))
    desired = RaceResult(
        metadata,
        [
            RR([0, 2, 2, 3, 3], [], [1], {1: {0: 1, 4: 1}}),
            RR([1, 0, 2, 3, 4], [4], [], {}),
            RR([1, 0, 2, 3, 4], [], [2], {2: {0: 2}}),
            RR([3, 0, 0, 3, 4], [3], [], {}),
        ],
    )
    assert_race_equal(actual, desired)


def test_3cands_2seats_1round():
    metadata = RaceMetadata("race", num_seats=2, names=["A", "B", "C"])
    ballots = [[2, 1, 3], [1, 2, 3]]
    votes = [3, 2]
    actual = pystv.run_stv(RaceInfo(metadata, ballots, votes))
    desired = RaceResult(metadata, [RR([0, 2, 3, 0], [1, 2], [], {2: {3: 1}})])
    assert actual == desired


def test_3cands_1seat_multiround():
    metadata = RaceMetadata("race", num_seats=1, names=["A", "B", "C"])
    ballots = [[1, 2, 3], [2, 1, 3], [3, 1, 2]]
    votes = [2, 2, 1]
    actual = pystv.run_stv(RaceInfo(metadata, ballots, votes))
    desired = RaceResult(
        metadata,
        [
            RR([0, 2, 2, 1], [], [3], {3: {1: 1}}),
            RR([0, 3, 2, 0], [1], [], {}),
        ],
    )
    assert actual == desired


def test_3cands_2seats_multiround():
    metadata = RaceMetadata("race", num_seats=2, names=["A", "B", "C"])
    ballots = [[1, 3, 2], [2, 1, 3], [3, 1, 2], [3, 2, 1]]
    votes = [2, 4, 1, 2]
    actual = pystv.run_stv(RaceInfo(metadata, ballots, votes))
    desired = RaceResult(metadata, [RR([0, 2, 4, 3], [2, 3], [], {2: {1: 1}})])
    assert actual == desired


def test_3cands_2seats_multiround_with_adjust():
    metadata = RaceMetadata("race", num_seats=2, names=["A", "B", "C"])
    ballots = [[1, 3, 2], [2, 1, 3], [3, 2, 1], [3, 2, 1]]
    votes = [2, 5, 1, 2]
    actual = pystv.run_stv(RaceInfo(metadata, ballots, votes))
    desired = RaceResult(
        metadata,
        [
            RR([0, 2, 5, 3], [2], [], {2: {1: 1}}),
            RR([0, 3, 4, 3], [], [1], {1: {3: 3}}),
            RR([0, 0, 4, 6], [3], [], {}),
        ],
    )
    assert_race_equal(actual, desired)


def test_fairvote_example():
    """Example from FairVote's website.

    https://fairvote.org/archives/multi_winner_rcv_example/
    """
    metadata = RaceMetadata("race", num_seats=3, names=["A", "B", "C", "D", "E", "F"])
    ballots, votes = zip(*FAIRVOTE_BALLOTS)
    actual = pystv.run_stv(RaceInfo(metadata, ballots, votes))
    desired = RaceResult(
        metadata,
        [
            RR(
                [0, 2500, 1750, 1300, 1300, 1350, 1000],
                [1],
                [],
                {1: {2: 100, 3: 20, 5: 80}},
            ),
            RR(
                [0, 2300, 1850, 1320, 1300, 1430, 1000],
                [],
                [6],
                {6: {2: 100, 3: 580, 4: 300, 5: 20}},
            ),
            RR(
                [0, 2300, 1950, 1900, 1600, 1450, 0],
                [],
                [5],
                {5: {2: 1250, 3: 150, 4: 50}},
            ),
            RR(
                [0, 2300, 3200, 2050, 1650, 0, 0], [2], [], {2: {0: 360, 3: 450, 4: 90}}
            ),
            RR([360, 2300, 2300, 2500, 1740, 0, 0], [3], [], {3: {0: 200}}),
        ],
    )
    assert_race_equal(actual, desired)


def test_validate_and_standardize_ballots_ok():
    ballots = [[1, 0, 0], [1, 2, 0], [1, 2, 3]]
    result = pystv.validate_and_standardize_ballots(ballots, num_cands=3)
    assert_equal(result, [[1, 0, 0, 0], [1, 2, 0, 0], [1, 2, 3, 0]])


def test_validate_and_standardize_ballots_ragged():
    ballots = [[1, 0, 0], [1, 2], [1, 2, 3]]
    result = pystv.validate_and_standardize_ballots(ballots, num_cands=3)
    assert_equal(result, [[1, 0, 0, 0], [1, 2, 0, 0], [1, 2, 3, 0]])


def test_validate_and_standardize_ballots_negative():
    ballots = [[1, 0, 0], [1, 2, -1], [1, 2, 3]]
    with pytest.raises(
        ValueError, match=r"Ranking out of allowed range on ballots indices: \[1, 2\]"
    ):
        pystv.validate_and_standardize_ballots(ballots, num_cands=2)


def test_command_line_interface():
    """Test the CLI."""
    runner = CliRunner()
    result = runner.invoke(cli.main)
    assert result.exit_code == 0
    assert "pystv.cli.main" in result.output
    help_result = runner.invoke(cli.main, ["--help"])
    assert help_result.exit_code == 0
    assert "--help  Show this message and exit." in help_result.output
