![openapy Logo](https://raw.githubusercontent.com/edge-minato/openapy/main/docs/img/logo.jpg)

[![pypi version](https://img.shields.io/pypi/v/openapy.svg?style=flat)](https://pypi.org/pypi/openapy/)
[![python versions](https://img.shields.io/pypi/pyversions/openapy.svg?style=flat)](https://pypi.org/pypi/openapy/)
[![license](https://img.shields.io/pypi/l/openapy.svg?style=flat)](https://github.com/edge-minato/openapy/blob/master/LICENSE)
[![Unittest](https://github.com/edge-minato/openapy/actions/workflows/unittest.yml/badge.svg)](https://github.com/edge-minato/openapy/actions/workflows/unittest.yml)
[![codecov](https://codecov.io/gh/edge-minato/openapy/branch/main/graph/badge.svg?token=YDZAMKUNS0)](https://codecov.io/gh/edge-minato/openapy)
[![Code style](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black")
[![Downloads](https://pepy.tech/badge/openapy)](https://pepy.tech/project/openapy)
[![Downloads](https://pepy.tech/badge/openapy/week)](https://pepy.tech/project/openapy)

`Openapy` simplifies continuous development with [OpenAPI generator](https://github.com/OpenAPITools/openapi-generator).
What this tool does is reading python source files and copying functions into individual files.
This will prevent the openapi generator from overwriting the code you have written.

- **Documentation**: https://openapy.readthedocs.io
- **Dockerhub**: https://hub.docker.com/r/edgem/openapy
- **Source Code**: https://github.com/edge-minato/openapy

## Quick start

```sh
docker run --rm -v "$PWD:/src" edgem/openapy \
openapy generate --src /src/openapi-server/apis
```

```sh
pip install openapy
openapy generate --src ./openapi-server/apis
```

## What openapy does

`Openapy` just splits each of the functions into a single file under `processor` directory.

```sh
# processor directory and the files will be generated
 .
 ├── api
 │   └── source.py
#└── processor
#    ├── __init__.py
#    ├── function_a.py
#    └── function_b.py
```

```python
# api/source.py
def function_a(name: str, age: int)->None:
    ...

def function_b(height: int, weight: int)-> int:
    ...
```

This command generates following files

```sh
openapy generate --src ./api
```

```python
# processor/__init__.py
from .function_a import function_a # noqa: F401
from .function_b import function_b # noqa: F401
```

```python
# processor/function_a.py
def function_a(name: str, age: int)->None:
    ...
```

```python
# processor/function_b.py
def function_b(height: int, weight: int)-> int:
    ...
```

## Working with `OpenAPI generator`

The expected usage is using the file generated with `OpenAPI Generator` as interfaces, and using the file generated with `Openapy` as the implementation.

```python
# apis/pet_api.py
import .processor
from fastapi import APIRouter
router = APIRouter()

@router.get("/pet/{petId}")
async def get_pet_by_id(petId: int = Path(None, description="ID of pet to return")) -> Pet:
    return processor.get_pet_by_id(petId)
```

```python
# processor/get_pet_by_id.py
def get_pet_by_id(petId: int) -> Pet:
    """Returns a single pet"""
    # implement me
    ...
```

In this use case, `api.mustache` file should be customized. It is possible to generate an example of mustache file with following command.

```sh
openapy example mustache > ./mustache/api.mustache
```

**NOTE**: Without this structure, which means writing the implementation of apis on the files generated by `OpenAPI generator`, a regeneration of `OpenAPI generator` will overwrite any existing code you have written, even if only one api has been updated. This is because the `OpenAPI generator` aggregates apis into a file with a `tag`.

## Features

### Custom Template

It is possible to define the format of generated code with `Openapy` just like the mustache for `OpenAPI generator`. For more details, see the [documentation](https://openapy.readthedocs.io).

### Difference Detection

TBD

### Move to docs

Following variables with `{}` brackets are available.

- **IMPORTS**: All of imports of the source file like `import X`, `from X import Y`
- **ASSIGNS**: All assigns of the source file like `var = "string"`
- **DEF**: `async def` or `def` of the function
- **NAME**: The function name
- **ARGS**: Arguments of the function with type annotations
- **RETURN_TYPE**: A type annotation for the return of the function
- **COMMENT**: A comment inside of the function
- **BODY**: A body of the function, like assign statement
- **RETURN**: A return statement of the function
