# Repository: https://gitlab.com/quantify-os/quantify-scheduler
# Licensed according to the LICENCE file on the main branch
"""Enums for quantify-scheduler."""

from enum import Enum, unique


class StrEnum(Enum):
    """
    This class functions to include explicit string serialization without adding `str`
    as a base class.
    """

    def __str__(self):
        return self.value


@unique
class SignalModeType(str, Enum):

    """
    The signal output enum type.

    Used to set the output signal type to a
    modulated or real respectively.
    """

    COMPLEX = "complex"
    REAL = "real"


@unique
class ReferenceSourceType(str, Enum):

    """
    The reference source enum type.

    Used to set the source trigger type to
    internal or external respectively.
    """

    NONE = "none"
    INTERNAL = "int"
    EXTERNAL = "ext"


@unique
class ModulationModeType(str, Enum):

    """
    The modulation mode enum type.

    Used to set the modulation type to

    1. no modulation. ('none')
    2. Software premodulation applied in the numerical waveforms. ('premod')
    3. Hardware real-time modulation. ('modulate')

    See also :class:`~quantify_scheduler.backends.types.common.Modulation` for the use.
    """

    NONE = "none"
    PREMODULATE = "premod"
    MODULATE = "modulate"


@unique
class BinMode(StrEnum):

    """
    The acquisition protocol bin mode enum type.

    Used to set the bin type to
    append or average respectively.

    BinMode `APPEND` uses a list where every new
    result will be appended to the list.

    BinMode `AVERAGE` incrementally stores the weighted
    average result.
    """

    APPEND = "append"
    AVERAGE = "average"


@unique
class InstrumentOperationMode(str, Enum):
    """
    The InstrumentOperationMode enum defines in what operational mode an instrument is
    in.

    OPERATING mode sets the Instrument in its default operation mode.
    CALIBRATING mode sets the Instrument in calibration mode in which for example the
    numeric pulses generated by a backend for an AWG are set to np.ones.
    """

    OPERATING = "operate"
    CALIBRATING = "calibrate"
