"""
A dict with the exact element masses retrieved from the
CRC Handbook of Chemistry and Physics 2015

There are entries for each element containing the masses
and relative abundances of several abundant isotopes and
a separate entry for undefined isotope with zero key,
mass of the most abundant isotope and 1.0 abundance.
"""

crc_mass = {
    'Ac': {
        0: (None, 100.000000),
        227: (227.027752, 0.000000),
    },
    'Ag': {
        0: (106.905097, 100.000000),
        107: (106.905097, 0.518390),
        109: (108.904752, 0.481610),
    },
    'Al': {
        0: (26.981539, 100.000000),
        27: (26.981539, 1.000000),
    },
    'Am': {
        0: (None, 100.000000),
        241: (241.056829, 0.000000),
        243: (243.061381, 0.000000),
    },
    'Ar': {
        0: (39.962383, 100.000000),
        36: (35.967545, 0.003365),
        38: (37.962732, 0.000632),
        40: (39.962383, 0.996003),
    },
    'As': {
        0: (74.921597, 100.000000),
        75: (74.921597, 1.000000),
    },
    'At': {
        0: (None, 100.000000),
        210: (209.987148, 0.000000),
        211: (210.987496, 0.000000),
    },
    'Au': {
        0: (196.966569, 100.000000),
        197: (196.966569, 1.000000),
        198: (197.968242, 0.000000),
    },
    'B': {
        0: (11.009305, 100.000000),
        10: (10.012937, 0.199000),
        11: (11.009305, 0.801000),
    },
    'Ba': {
        0: (137.905247, 100.000000),
        130: (129.906321, 0.001060),
        132: (131.905061, 0.001010),
        133: (132.906007, 0.000000),
        134: (133.904508, 0.024170),
        135: (134.905689, 0.065920),
        136: (135.904576, 0.078540),
        137: (136.905827, 0.112320),
        138: (137.905247, 0.716980),
        140: (139.910605, 0.000000),
    },
    'Be': {
        0: (9.012182, 100.000000),
        9: (9.012182, 1.000000),
    },
    'Bh': {
        0: (None, 100.000000),
        264: (264.124600, 0.000000),
    },
    'Bi': {
        0: (208.980399, 100.000000),
        207: (206.978471, 0.000000),
        209: (208.980399, 1.000000),
    },
    'Bk': {
        0: (None, 100.000000),
        247: (247.070307, 0.000000),
        249: (249.074987, 0.000000),
    },
    'Br': {
        0: (78.918337, 100.000000),
        79: (78.918337, 0.506900),
        81: (80.916291, 0.493100),
    },
    'C': {
        0: (12.000000, 100.000000),
        11: (11.011434, 0.000000),
        12: (12.000000, 0.989300),
        13: (13.003355, 0.010700),
        14: (14.003242, 0.000000),
    },
    'Ca': {
        0: (39.962591, 100.000000),
        40: (39.962591, 0.969410),
        42: (41.958618, 0.006470),
        43: (42.958767, 0.001350),
        44: (43.955482, 0.020860),
        45: (44.956187, 0.000000),
        46: (45.953693, 0.000040),
        47: (46.954546, 0.000000),
        48: (47.952534, 0.001870),
    },
    'Cd': {
        0: (113.903358, 100.000000),
        106: (105.906459, 0.012500),
        108: (107.904184, 0.008900),
        110: (109.903002, 0.124900),
        111: (110.904178, 0.128000),
        112: (111.902758, 0.241300),
        113: (112.904402, 0.122200),
        114: (113.903358, 0.287300),
        116: (115.904756, 0.074900),
    },
    'Ce': {
        0: (139.905439, 100.000000),
        136: (135.907172, 0.001850),
        138: (137.905991, 0.002510),
        140: (139.905439, 0.884500),
        141: (140.908276, 0.000000),
        142: (141.909244, 0.111140),
        144: (143.913647, 0.000000),
    },
    'Cf': {
        0: (None, 100.000000),
        249: (249.074853, 0.000000),
        250: (250.076406, 0.000000),
        251: (251.079587, 0.000000),
        252: (252.081626, 0.000000),
    },
    'Cl': {
        0: (34.968853, 100.000000),
        35: (34.968853, 0.757600),
        37: (36.965903, 0.242400),
    },
    'Cm': {
        0: (None, 100.000000),
        243: (243.061389, 0.000000),
        244: (244.062753, 0.000000),
        245: (245.065491, 0.000000),
        246: (246.067224, 0.000000),
        247: (247.070354, 0.000000),
        248: (248.072349, 0.000000),
    },
    'Co': {
        0: (58.933195, 100.000000),
        57: (56.936291, 0.000000),
        58: (57.935753, 0.000000),
        59: (58.933195, 1.000000),
        60: (59.933817, 0.000000),
    },
    'Cr': {
        0: (51.940508, 100.000000),
        50: (49.946044, 0.043450),
        51: (50.944767, 0.000000),
        52: (51.940508, 0.837890),
        53: (52.940649, 0.095010),
        54: (53.938880, 0.023650),
    },
    'Cs': {
        0: (132.905452, 100.000000),
        129: (128.906064, 0.000000),
        133: (132.905452, 1.000000),
        134: (133.906718, 0.000000),
        136: (135.907312, 0.000000),
        137: (136.907090, 0.000000),
    },
    'Cu': {
        0: (62.929597, 100.000000),
        63: (62.929597, 0.691500),
        64: (63.929764, 0.000000),
        65: (64.927790, 0.308500),
    },
    'Db': {
        0: (None, 100.000000),
        262: (262.114080, 0.000000),
    },
    'Ds': {
        0: (None, 100.000000),
        281: (281.162060, 0.000000),
    },
    'Dy': {
        0: (163.929175, 100.000000),
        156: (155.924283, 0.000560),
        158: (157.924409, 0.000950),
        160: (159.925197, 0.023290),
        161: (160.926933, 0.188890),
        162: (161.926798, 0.254750),
        163: (162.928731, 0.248960),
        164: (163.929175, 0.282600),
    },
    'Er': {
        0: (165.930293, 100.000000),
        162: (161.928778, 0.001390),
        164: (163.929200, 0.016010),
        166: (165.930293, 0.335030),
        167: (166.932048, 0.228690),
        168: (167.932370, 0.269780),
        170: (169.935464, 0.149100),
    },
    'Es': {
        0: (None, 100.000000),
        252: (252.082980, 0.000000),
    },
    'Eu': {
        0: (152.921230, 100.000000),
        151: (150.919850, 0.478100),
        153: (152.921230, 0.521900),
    },
    'F': {
        0: (18.998403, 100.000000),
        18: (18.000938, 0.000000),
        19: (18.998403, 1.000000),
    },
    'Fe': {
        0: (55.934937, 100.000000),
        52: (51.948114, 0.000000),
        54: (53.939611, 0.058450),
        55: (54.938293, 0.000000),
        56: (55.934937, 0.917540),
        57: (56.935394, 0.021190),
        58: (57.933276, 0.002820),
        59: (58.934875, 0.000000),
    },
    'Fm': {
        0: (None, 100.000000),
        257: (257.095105, 0.000000),
    },
    'Fr': {
        0: (None, 100.000000),
        223: (223.019736, 0.000000),
    },
    'Ga': {
        0: (68.925574, 100.000000),
        67: (66.928202, 0.000000),
        68: (67.927980, 0.000000),
        69: (68.925574, 0.601080),
        71: (70.924701, 0.398920),
    },
    'Gd': {
        0: (157.924104, 100.000000),
        152: (151.919791, 0.002000),
        154: (153.920866, 0.021800),
        155: (154.922622, 0.148000),
        156: (155.922123, 0.204700),
        157: (156.923960, 0.156500),
        158: (157.924104, 0.248400),
        160: (159.927054, 0.218600),
    },
    'Ge': {
        0: (73.921178, 100.000000),
        68: (67.928094, 0.000000),
        70: (69.924247, 0.203800),
        72: (71.922076, 0.273100),
        73: (72.923459, 0.077600),
        74: (73.921178, 0.367200),
        76: (75.921403, 0.078300),
    },
    'H': {
        0: (1.007825, 100.000000),
        1: (1.007825, 0.999885),
        2: (2.014102, 0.000115),
        3: (3.016049, 0.000000),
    },
    'He': {
        0: (4.002603, 100.000000),
        3: (3.016029, 0.000001),
        4: (4.002603, 0.999999),
    },
    'Hf': {
        0: (179.946550, 100.000000),
        174: (173.940046, 0.001600),
        176: (175.941409, 0.052600),
        177: (176.943221, 0.186000),
        178: (177.943699, 0.272800),
        179: (178.945816, 0.136200),
        180: (179.946550, 0.350800),
    },
    'Hg': {
        0: (201.970643, 100.000000),
        196: (195.965833, 0.001500),
        197: (196.967213, 0.000000),
        198: (197.966769, 0.099700),
        199: (198.968280, 0.168700),
        200: (199.968326, 0.231000),
        201: (200.970302, 0.131800),
        202: (201.970643, 0.298600),
        203: (202.972872, 0.000000),
        204: (203.973494, 0.068700),
    },
    'Ho': {
        0: (164.930322, 100.000000),
        165: (164.930322, 1.000000),
    },
    'Hs': {
        0: (None, 100.000000),
        265: (265.130090, 0.000000),
    },
    'I': {
        0: (126.904473, 100.000000),
        123: (122.905589, 0.000000),
        125: (124.904630, 0.000000),
        127: (126.904473, 1.000000),
        129: (128.904988, 0.000000),
        131: (130.906125, 0.000000),
    },
    'In': {
        0: (114.903878, 100.000000),
        111: (110.905103, 0.000000),
        113: (112.904058, 0.042900),
        115: (114.903878, 0.957100),
    },
    'Ir': {
        0: (192.962926, 100.000000),
        191: (190.960594, 0.373000),
        193: (192.962926, 0.627000),
    },
    'K': {
        0: (38.963707, 100.000000),
        39: (38.963707, 0.932581),
        40: (39.963998, 0.000117),
        41: (40.961826, 0.067302),
        42: (41.962403, 0.000000),
        43: (42.960716, 0.000000),
    },
    'Kr': {
        0: (83.911507, 100.000000),
        78: (77.920365, 0.003550),
        80: (79.916379, 0.022860),
        82: (81.913484, 0.115930),
        83: (82.914136, 0.115000),
        84: (83.911507, 0.569870),
        86: (85.910611, 0.172790),
    },
    'La': {
        0: (138.906353, 100.000000),
        138: (137.907112, 0.000900),
        139: (138.906353, 0.999100),
    },
    'Li': {
        0: (7.016005, 100.000000),
        6: (6.015123, 0.075900),
        7: (7.016005, 0.924100),
    },
    'Lr': {
        0: (None, 100.000000),
        262: (262.109630, 0.000000),
    },
    'Lu': {
        0: (174.940772, 100.000000),
        175: (174.940772, 0.974100),
        176: (175.942686, 0.025900),
    },
    'Md': {
        0: (None, 100.000000),
        256: (256.094060, 0.000000),
        258: (258.098431, 0.000000),
    },
    'Mg': {
        0: (23.985042, 100.000000),
        24: (23.985042, 0.789900),
        25: (24.985837, 0.100000),
        26: (25.982593, 0.110100),
    },
    'Mn': {
        0: (54.938045, 100.000000),
        54: (53.940359, 0.000000),
        55: (54.938045, 1.000000),
    },
    'Mo': {
        0: (97.905408, 100.000000),
        92: (91.906811, 0.147700),
        94: (93.905088, 0.092300),
        95: (94.905842, 0.159000),
        96: (95.904680, 0.166800),
        97: (96.906021, 0.095600),
        98: (97.905408, 0.241900),
        99: (98.907712, 0.000000),
        100: (99.907477, 0.096700),
    },
    'Mt': {
        0: (None, 100.000000),
        268: (268.138730, 0.000000),
    },
    'N': {
        0: (14.003074, 100.000000),
        14: (14.003074, 0.996360),
        15: (15.000109, 0.003640),
    },
    'Na': {
        0: (22.989769, 100.000000),
        22: (21.994436, 0.000000),
        23: (22.989769, 1.000000),
        24: (23.990963, 0.000000),
    },
    'Nb': {
        0: (92.906378, 100.000000),
        93: (92.906378, 1.000000),
    },
    'Nd': {
        0: (141.907723, 100.000000),
        142: (141.907723, 0.272000),
        143: (142.909814, 0.122000),
        144: (143.910087, 0.238000),
        145: (144.912574, 0.083000),
        146: (145.913117, 0.172000),
        148: (147.916893, 0.057000),
        150: (149.920891, 0.056000),
    },
    'Ne': {
        0: (19.992440, 100.000000),
        20: (19.992440, 0.904800),
        21: (20.993847, 0.002700),
        22: (21.991385, 0.092500),
    },
    'Ni': {
        0: (57.935343, 100.000000),
        58: (57.935343, 0.680769),
        59: (58.934347, 0.000000),
        60: (59.930786, 0.262231),
        61: (60.931056, 0.011399),
        62: (61.928345, 0.036345),
        63: (62.929669, 0.000000),
        64: (63.927966, 0.009256),
    },
    'No': {
        0: (None, 100.000000),
        259: (259.101030, 0.000000),
    },
    'Np': {
        0: (None, 100.000000),
        237: (237.048173, 0.000000),
        239: (239.052939, 0.000000),
    },
    'O': {
        0: (15.994915, 100.000000),
        16: (15.994915, 0.997570),
        17: (16.999132, 0.000380),
        18: (17.999161, 0.002050),
    },
    'Os': {
        0: (191.961481, 100.000000),
        184: (183.952489, 0.000200),
        186: (185.953838, 0.015900),
        187: (186.955750, 0.019600),
        188: (187.955838, 0.132400),
        189: (188.958147, 0.161500),
        190: (189.958447, 0.262600),
        192: (191.961481, 0.407800),
    },
    'P': {
        0: (30.973762, 100.000000),
        31: (30.973762, 1.000000),
        32: (31.973907, 0.000000),
    },
    'Pa': {
        0: (231.035884, 100.000000),
        231: (231.035884, 1.000000),
    },
    'Pb': {
        0: (207.976652, 100.000000),
        204: (203.973044, 0.014000),
        206: (205.974465, 0.241000),
        207: (206.975897, 0.221000),
        208: (207.976652, 0.524000),
        210: (209.984188, 0.000000),
    },
    'Pd': {
        0: (105.903486, 100.000000),
        102: (101.905609, 0.010200),
        104: (103.904036, 0.111400),
        105: (104.905085, 0.223300),
        106: (105.903486, 0.273300),
        108: (107.903892, 0.264600),
        110: (109.905153, 0.117200),
    },
    'Pm': {
        0: (None, 100.000000),
        145: (144.912749, 0.000000),
        147: (146.915139, 0.000000),
    },
    'Po': {
        0: (None, 100.000000),
        209: (208.982430, 0.000000),
        210: (209.982874, 0.000000),
    },
    'Pr': {
        0: (140.907653, 100.000000),
        141: (140.907653, 1.000000),
    },
    'Pt': {
        0: (194.964791, 100.000000),
        190: (189.959932, 0.000140),
        192: (191.961038, 0.007820),
        194: (193.962680, 0.329670),
        195: (194.964791, 0.338320),
        196: (195.964952, 0.252420),
        198: (197.967893, 0.071630),
    },
    'Pu': {
        0: (None, 100.000000),
        238: (238.049560, 0.000000),
        239: (239.052163, 0.000000),
        240: (240.053813, 0.000000),
        241: (241.056851, 0.000000),
        242: (242.058743, 0.000000),
        244: (244.064204, 0.000000),
    },
    'Ra': {
        0: (None, 100.000000),
        223: (223.018502, 0.000000),
        224: (224.020212, 0.000000),
        226: (226.025410, 0.000000),
        228: (228.031070, 0.000000),
    },
    'Rb': {
        0: (84.911790, 100.000000),
        85: (84.911790, 0.721700),
        86: (85.911167, 0.000000),
        87: (86.909181, 0.278300),
    },
    'Re': {
        0: (186.955753, 100.000000),
        185: (184.952955, 0.374000),
        187: (186.955753, 0.626000),
    },
    'Rf': {
        0: (None, 100.000000),
        261: (261.108770, 0.000000),
    },
    'Rg': {
        0: (None, 100.000000),
        272: (273.153620, 0.000000),
    },
    'Rh': {
        0: (102.905504, 100.000000),
        103: (102.905504, 1.000000),
    },
    'Rn': {
        0: (None, 100.000000),
        211: (210.990601, 0.000000),
        220: (220.011394, 0.000000),
        222: (222.017578, 0.000000),
    },
    'Ru': {
        0: (101.904349, 100.000000),
        96: (95.907598, 0.055400),
        98: (97.905287, 0.018700),
        99: (98.905939, 0.127600),
        100: (99.904219, 0.126000),
        101: (100.905582, 0.170600),
        102: (101.904349, 0.315500),
        104: (103.905433, 0.186200),
        106: (105.907329, 0.000000),
    },
    'S': {
        0: (31.972071, 100.000000),
        32: (31.972071, 0.949900),
        33: (32.971459, 0.007500),
        34: (33.967867, 0.042500),
        35: (34.969032, 0.000000),
        36: (35.967081, 0.000100),
    },
    'Sb': {
        0: (120.903816, 100.000000),
        121: (120.903816, 0.572100),
        123: (122.904214, 0.427900),
    },
    'Sc': {
        0: (44.955912, 100.000000),
        45: (44.955912, 1.000000),
    },
    'Se': {
        0: (79.916521, 100.000000),
        74: (73.922476, 0.008900),
        75: (74.922523, 0.000000),
        76: (75.919214, 0.093700),
        77: (76.919914, 0.076300),
        78: (77.917309, 0.237700),
        79: (78.918499, 0.000000),
        80: (79.916521, 0.496100),
        82: (81.916699, 0.087300),
    },
    'Sg': {
        0: (None, 100.000000),
        263: (263.118320, 0.000000),
    },
    'Si': {
        0: (27.976927, 100.000000),
        28: (27.976927, 0.922230),
        29: (28.976495, 0.046850),
        30: (29.973770, 0.030920),
    },
    'Sm': {
        0: (151.919732, 100.000000),
        144: (143.911999, 0.030700),
        147: (146.914898, 0.149900),
        148: (147.914823, 0.112400),
        149: (148.917185, 0.138200),
        150: (149.917275, 0.073800),
        152: (151.919732, 0.267500),
        154: (153.922209, 0.227500),
    },
    'Sn': {
        0: (119.902195, 100.000000),
        112: (111.904818, 0.009700),
        113: (112.905171, 0.000000),
        114: (113.902779, 0.006600),
        115: (114.903342, 0.003400),
        116: (115.901741, 0.145400),
        117: (116.902952, 0.076800),
        118: (117.901603, 0.242200),
        119: (118.903308, 0.085900),
        120: (119.902195, 0.325800),
        122: (121.903439, 0.046300),
        124: (123.905274, 0.057900),
    },
    'Sr': {
        0: (87.905612, 100.000000),
        84: (83.913425, 0.005600),
        85: (84.912933, 0.000000),
        86: (85.909260, 0.098600),
        87: (86.908877, 0.070000),
        88: (87.905612, 0.825800),
        89: (88.907451, 0.000000),
        90: (89.907738, 0.000000),
    },
    'Ta': {
        0: (180.947996, 100.000000),
        180: (179.947465, 0.000120),
        181: (180.947996, 0.999880),
    },
    'Tb': {
        0: (158.925347, 100.000000),
        159: (158.925347, 1.000000),
    },
    'Tc': {
        0: (None, 100.000000),
        97: (96.906365, 0.000000),
        98: (97.907216, 0.000000),
        99: (98.906255, 0.000000),
    },
    'Te': {
        0: (129.906224, 100.000000),
        120: (119.904020, 0.000900),
        122: (121.903044, 0.025500),
        123: (122.904270, 0.008900),
        124: (123.902818, 0.047400),
        125: (124.904431, 0.070700),
        126: (125.903312, 0.188400),
        128: (127.904463, 0.317400),
        130: (129.906224, 0.340800),
    },
    'Th': {
        0: (232.038055, 100.000000),
        228: (228.028741, 0.000000),
        230: (230.033134, 0.000000),
        232: (232.038055, 1.000000),
    },
    'Ti': {
        0: (47.947946, 100.000000),
        46: (45.952632, 0.082500),
        47: (46.951763, 0.074400),
        48: (47.947946, 0.737200),
        49: (48.947870, 0.054100),
        50: (49.944791, 0.051800),
    },
    'Tl': {
        0: (204.974427, 100.000000),
        201: (200.970819, 0.000000),
        203: (202.972344, 0.295200),
        205: (204.974427, 0.704800),
    },
    'Tm': {
        0: (168.934213, 100.000000),
        169: (168.934213, 1.000000),
    },
    'U': {
        0: (238.050788, 100.000000),
        233: (233.039635, 0.000000),
        234: (234.040952, 0.000054),
        235: (235.043930, 0.007204),
        236: (236.045568, 0.000000),
        238: (238.050788, 0.992742),
    },
    'V': {
        0: (50.943959, 100.000000),
        50: (49.947159, 0.002500),
        51: (50.943959, 0.997500),
    },
    'W': {
        0: (183.950931, 100.000000),
        180: (179.946704, 0.001200),
        182: (181.948204, 0.265000),
        183: (182.950223, 0.143100),
        184: (183.950931, 0.306400),
        186: (185.954364, 0.284300),
    },
    'Xe': {
        0: (131.904154, 100.000000),
        124: (123.905893, 0.000952),
        126: (125.904274, 0.000890),
        128: (127.903531, 0.019102),
        129: (128.904779, 0.264006),
        130: (129.903508, 0.040710),
        131: (130.905082, 0.212324),
        132: (131.904154, 0.269086),
        134: (133.905394, 0.104357),
        136: (135.907219, 0.088573),
    },
    'Y': {
        0: (88.905848, 100.000000),
        89: (88.905848, 1.000000),
    },
    'Yb': {
        0: (173.938862, 100.000000),
        168: (167.933897, 0.001300),
        169: (168.935190, 0.000000),
        170: (169.934762, 0.030400),
        171: (170.936326, 0.142800),
        172: (171.936382, 0.218300),
        173: (172.938211, 0.161300),
        174: (173.938862, 0.318300),
        176: (175.942572, 0.127600),
    },
    'Zn': {
        0: (63.929142, 100.000000),
        64: (63.929142, 0.482680),
        65: (64.929241, 0.000000),
        66: (65.926033, 0.279750),
        67: (66.927127, 0.041020),
        68: (67.924844, 0.190240),
        70: (69.925319, 0.006310),
    },
    'Zr': {
        0: (89.904704, 100.000000),
        90: (89.904704, 0.514500),
        91: (90.905646, 0.112200),
        92: (91.905041, 0.171500),
        94: (93.906315, 0.173800),
        96: (95.908273, 0.028000),
    },
}


"""
A dict with the exact element masses downloaded from the NIST website:
http://www.nist.gov/pml/data/comp.cfm . There are entries for each
element containing the masses and relative abundances of several
abundant isotopes and a separate entry for undefined isotope with zero
key, mass of the most abundant isotope and 1.0 abundance.

snagged from pyteomics
"""

nist_mass = {
    'Ac': {
        0: (227.000000, 1.000000),
        206: (206.014500, 0.0),
        207: (207.011950, 0.0),
        208: (208.011550, 0.0),
        209: (209.009490, 0.0),
        210: (210.009440, 0.0),
        211: (211.007730, 0.0),
        212: (212.007810, 0.0),
        213: (213.006610, 0.0),
        214: (214.006902, 0.0),
        215: (215.006454, 0.0),
        216: (216.008720, 0.0),
        217: (217.009347, 0.0),
        218: (218.011640, 0.0),
        219: (219.012420, 0.0),
        220: (220.014763, 0.0),
        221: (221.015590, 0.0),
        222: (222.017844, 0.0),
        223: (223.019137, 0.0),
        224: (224.021723, 0.0),
        225: (225.023230, 0.0),
        226: (226.026098, 0.0),
        227: (227.027752, 0.0),
        228: (228.031021, 0.0),
        229: (229.033020, 0.0),
        230: (230.036290, 0.0),
        231: (231.038560, 0.0),
        232: (232.042030, 0.0),
        233: (233.044550, 0.0),
        234: (234.048420, 0.0),
        235: (235.051230, 0.0),
        236: (236.055300, 0.0),
    },
    'Ag': {
        0: (106.905097, 1.000000),
        93: (92.949780, 0.0),
        94: (93.942780, 0.0),
        95: (94.935480, 0.0),
        96: (95.930680, 0.0),
        97: (96.923970, 0.0),
        98: (97.921570, 0.0),
        99: (98.917600, 0.0),
        100: (99.916100, 0.0),
        101: (100.912800, 0.0),
        102: (101.911690, 0.0),
        103: (102.908973, 0.0),
        104: (103.908629, 0.0),
        105: (104.906529, 0.0),
        106: (105.906669, 0.0),
        107: (106.905097, 0.518390),
        108: (107.905956, 0.0),
        109: (108.904752, 0.481610),
        110: (109.906107, 0.0),
        111: (110.905291, 0.0),
        112: (111.907005, 0.0),
        113: (112.906567, 0.0),
        114: (113.908804, 0.0),
        115: (114.908760, 0.0),
        116: (115.911360, 0.0),
        117: (116.911680, 0.0),
        118: (117.914580, 0.0),
        119: (118.915670, 0.0),
        120: (119.918790, 0.0),
        121: (120.919850, 0.0),
        122: (121.923530, 0.0),
        123: (122.924900, 0.0),
        124: (123.928640, 0.0),
        125: (124.930430, 0.0),
        126: (125.934500, 0.0),
        127: (126.936770, 0.0),
        128: (127.941170, 0.0),
        129: (128.943690, 0.0),
        130: (129.950450, 0.0),
    },
    'Al': {
        0: (26.981539, 1.000000),
        21: (21.028040, 0.0),
        22: (22.019520, 0.0),
        23: (23.007267, 0.0),
        24: (23.999939, 0.0),
        25: (24.990428, 0.0),
        26: (25.986892, 0.0),
        27: (26.981539, 1.000000),
        28: (27.981910, 0.0),
        29: (28.980445, 0.0),
        30: (29.982960, 0.0),
        31: (30.983947, 0.0),
        32: (31.988120, 0.0),
        33: (32.990840, 0.0),
        34: (33.996850, 0.0),
        35: (34.999860, 0.0),
        36: (36.006210, 0.0),
        37: (37.010680, 0.0),
        38: (38.017230, 0.0),
        39: (39.022970, 0.0),
        40: (40.031450, 0.0),
        41: (41.038330, 0.0),
        42: (42.046890, 0.0),
    },
    'Am': {
        0: (243.000000, 1.000000),
        231: (231.045560, 0.0),
        232: (232.046590, 0.0),
        233: (233.046350, 0.0),
        234: (234.047810, 0.0),
        235: (235.047950, 0.0),
        236: (236.049580, 0.0),
        237: (237.050000, 0.0),
        238: (238.051980, 0.0),
        239: (239.053024, 0.0),
        240: (240.055300, 0.0),
        241: (241.056829, 0.0),
        242: (242.059549, 0.0),
        243: (243.061381, 0.0),
        244: (244.064285, 0.0),
        245: (245.066452, 0.0),
        246: (246.069775, 0.0),
        247: (247.072090, 0.0),
        248: (248.075750, 0.0),
        249: (249.078480, 0.0),
    },
    'Ar': {
        0: (39.962383, 1.000000),
        30: (30.021560, 0.0),
        31: (31.012120, 0.0),
        32: (31.997638, 0.0),
        33: (32.989926, 0.0),
        34: (33.980271, 0.0),
        35: (34.975258, 0.0),
        36: (35.967545, 0.003365),
        37: (36.966776, 0.0),
        38: (37.962732, 0.000632),
        39: (38.964313, 0.0),
        40: (39.962383, 0.996003),
        41: (40.964501, 0.0),
        42: (41.963046, 0.0),
        43: (42.965636, 0.0),
        44: (43.964924, 0.0),
        45: (44.968040, 0.0),
        46: (45.968090, 0.0),
        47: (46.972190, 0.0),
        48: (47.974540, 0.0),
        49: (48.980520, 0.0),
        50: (49.984430, 0.0),
        51: (50.991630, 0.0),
        52: (51.996780, 0.0),
        53: (53.004940, 0.0),
    },
    'As': {
        0: (74.921597, 1.000000),
        60: (59.993130, 0.0),
        61: (60.980620, 0.0),
        62: (61.973200, 0.0),
        63: (62.963690, 0.0),
        64: (63.957570, 0.0),
        65: (64.949560, 0.0),
        66: (65.944710, 0.0),
        67: (66.939190, 0.0),
        68: (67.936770, 0.0),
        69: (68.932270, 0.0),
        70: (69.930920, 0.0),
        71: (70.927112, 0.0),
        72: (71.926752, 0.0),
        73: (72.923825, 0.0),
        74: (73.923929, 0.0),
        75: (74.921597, 1.000000),
        76: (75.922394, 0.0),
        77: (76.920647, 0.0),
        78: (77.921827, 0.0),
        79: (78.920948, 0.0),
        80: (79.922534, 0.0),
        81: (80.922132, 0.0),
        82: (81.924500, 0.0),
        83: (82.924980, 0.0),
        84: (83.929060, 0.0),
        85: (84.932020, 0.0),
        86: (85.936500, 0.0),
        87: (86.939900, 0.0),
        88: (87.944940, 0.0),
        89: (88.949390, 0.0),
        90: (89.955500, 0.0),
        91: (90.960430, 0.0),
        92: (91.966800, 0.0),
    },
    'At': {
        0: (210.000000, 1.000000),
        193: (192.999840, 0.0),
        194: (193.998730, 0.0),
        195: (194.996268, 0.0),
        196: (195.995790, 0.0),
        197: (196.993190, 0.0),
        198: (197.992840, 0.0),
        199: (198.990530, 0.0),
        200: (199.990351, 0.0),
        201: (200.988417, 0.0),
        202: (201.988630, 0.0),
        203: (202.986942, 0.0),
        204: (203.987251, 0.0),
        205: (204.986074, 0.0),
        206: (205.986667, 0.0),
        207: (206.985784, 0.0),
        208: (207.986590, 0.0),
        209: (208.986173, 0.0),
        210: (209.987148, 0.0),
        211: (210.987496, 0.0),
        212: (211.990745, 0.0),
        213: (212.992937, 0.0),
        214: (213.996372, 0.0),
        215: (214.998653, 0.0),
        216: (216.002423, 0.0),
        217: (217.004719, 0.0),
        218: (218.008694, 0.0),
        219: (219.011162, 0.0),
        220: (220.015410, 0.0),
        221: (221.018050, 0.0),
        222: (222.022330, 0.0),
        223: (223.025190, 0.0),
    },
    'Au': {
        0: (196.966569, 1.000000),
        169: (168.998080, 0.0),
        170: (169.996120, 0.0),
        171: (170.991879, 0.0),
        172: (171.990040, 0.0),
        173: (172.986237, 0.0),
        174: (173.984760, 0.0),
        175: (174.981270, 0.0),
        176: (175.980100, 0.0),
        177: (176.976865, 0.0),
        178: (177.976030, 0.0),
        179: (178.973213, 0.0),
        180: (179.972521, 0.0),
        181: (180.970079, 0.0),
        182: (181.969618, 0.0),
        183: (182.967593, 0.0),
        184: (183.967452, 0.0),
        185: (184.965789, 0.0),
        186: (185.965953, 0.0),
        187: (186.964568, 0.0),
        188: (187.965324, 0.0),
        189: (188.963948, 0.0),
        190: (189.964700, 0.0),
        191: (190.963700, 0.0),
        192: (191.964813, 0.0),
        193: (192.964150, 0.0),
        194: (193.965365, 0.0),
        195: (194.965035, 0.0),
        196: (195.966570, 0.0),
        197: (196.966569, 1.000000),
        198: (197.968242, 0.0),
        199: (198.968765, 0.0),
        200: (199.970730, 0.0),
        201: (200.971657, 0.0),
        202: (201.973810, 0.0),
        203: (202.975155, 0.0),
        204: (203.977720, 0.0),
        205: (204.979870, 0.0),
    },
    'B': {
        0: (11.009305, 1.000000),
        6: (6.046810, 0.0),
        7: (7.029920, 0.0),
        8: (8.024607, 0.0),
        9: (9.013329, 0.0),
        10: (10.012937, 0.199000),
        11: (11.009305, 0.801000),
        12: (12.014352, 0.0),
        13: (13.017780, 0.0),
        14: (14.025404, 0.0),
        15: (15.031103, 0.0),
        16: (16.039810, 0.0),
        17: (17.046990, 0.0),
        18: (18.056170, 0.0),
        19: (19.063730, 0.0),
    },
    'Ba': {
        0: (137.905247, 1.000000),
        114: (113.950680, 0.0),
        115: (114.947370, 0.0),
        116: (115.941380, 0.0),
        117: (116.938500, 0.0),
        118: (117.933040, 0.0),
        119: (118.930660, 0.0),
        120: (119.926040, 0.0),
        121: (120.924050, 0.0),
        122: (121.919900, 0.0),
        123: (122.918781, 0.0),
        124: (123.915094, 0.0),
        125: (124.914473, 0.0),
        126: (125.911250, 0.0),
        127: (126.911094, 0.0),
        128: (127.908318, 0.0),
        129: (128.908679, 0.0),
        130: (129.906321, 0.001060),
        131: (130.906941, 0.0),
        132: (131.905061, 0.001010),
        133: (132.906007, 0.0),
        134: (133.904508, 0.024170),
        135: (134.905689, 0.065920),
        136: (135.904576, 0.078540),
        137: (136.905827, 0.112320),
        138: (137.905247, 0.716980),
        139: (138.908841, 0.0),
        140: (139.910605, 0.0),
        141: (140.914411, 0.0),
        142: (141.916453, 0.0),
        143: (142.920627, 0.0),
        144: (143.922953, 0.0),
        145: (144.927630, 0.0),
        146: (145.930220, 0.0),
        147: (146.934950, 0.0),
        148: (147.937720, 0.0),
        149: (148.942580, 0.0),
        150: (149.945680, 0.0),
        151: (150.950810, 0.0),
        152: (151.954270, 0.0),
        153: (152.959610, 0.0),
    },
    'Be': {
        0: (9.012182, 1.000000),
        5: (5.040790, 0.0),
        6: (6.019726, 0.0),
        7: (7.016930, 0.0),
        8: (8.005305, 0.0),
        9: (9.012182, 1.000000),
        10: (10.013534, 0.0),
        11: (11.021658, 0.0),
        12: (12.026921, 0.0),
        13: (13.035690, 0.0),
        14: (14.042890, 0.0),
        15: (15.053460, 0.0),
        16: (16.061920, 0.0),
    },
    'Bh': {
        0: (272.000000, 1.000000),
        260: (260.121970, 0.0),
        261: (261.121660, 0.0),
        262: (262.122890, 0.0),
        263: (263.123040, 0.0),
        264: (264.124600, 0.0),
        265: (265.125150, 0.0),
        266: (266.126940, 0.0),
        267: (267.127650, 0.0),
        268: (268.129760, 0.0),
        269: (269.130690, 0.0),
        270: (270.133620, 0.0),
        271: (271.135180, 0.0),
        272: (272.138030, 0.0),
        273: (273.139620, 0.0),
        274: (274.142440, 0.0),
        275: (275.144250, 0.0),
    },
    'Bi': {
        0: (208.980399, 1.000000),
        184: (184.001120, 0.0),
        185: (184.997630, 0.0),
        186: (185.996600, 0.0),
        187: (186.993158, 0.0),
        188: (187.992270, 0.0),
        189: (188.989200, 0.0),
        190: (189.988300, 0.0),
        191: (190.985786, 0.0),
        192: (191.985460, 0.0),
        193: (192.982960, 0.0),
        194: (193.982830, 0.0),
        195: (194.980651, 0.0),
        196: (195.980667, 0.0),
        197: (196.978864, 0.0),
        198: (197.979210, 0.0),
        199: (198.977672, 0.0),
        200: (199.978132, 0.0),
        201: (200.977009, 0.0),
        202: (201.977742, 0.0),
        203: (202.976876, 0.0),
        204: (203.977813, 0.0),
        205: (204.977389, 0.0),
        206: (205.978499, 0.0),
        207: (206.978471, 0.0),
        208: (207.979742, 0.0),
        209: (208.980399, 1.000000),
        210: (209.984120, 0.0),
        211: (210.987269, 0.0),
        212: (211.991286, 0.0),
        213: (212.994385, 0.0),
        214: (213.998712, 0.0),
        215: (215.001770, 0.0),
        216: (216.006306, 0.0),
        217: (217.009470, 0.0),
        218: (218.014320, 0.0),
    },
    'Bk': {
        0: (247.000000, 1.000000),
        235: (235.056580, 0.0),
        236: (236.057330, 0.0),
        237: (237.057000, 0.0),
        238: (238.058280, 0.0),
        239: (239.058280, 0.0),
        240: (240.059760, 0.0),
        241: (241.060230, 0.0),
        242: (242.061980, 0.0),
        243: (243.063008, 0.0),
        244: (244.065181, 0.0),
        245: (245.066362, 0.0),
        246: (246.068670, 0.0),
        247: (247.070307, 0.0),
        248: (248.073090, 0.0),
        249: (249.074987, 0.0),
        250: (250.078317, 0.0),
        251: (251.080760, 0.0),
        252: (252.084310, 0.0),
        253: (253.086880, 0.0),
        254: (254.090600, 0.0),
    },
    'Br': {
        0: (78.918337, 1.000000),
        67: (66.964790, 0.0),
        68: (67.958520, 0.0),
        69: (68.950110, 0.0),
        70: (69.944790, 0.0),
        71: (70.938740, 0.0),
        72: (71.936640, 0.0),
        73: (72.931690, 0.0),
        74: (73.929891, 0.0),
        75: (74.925776, 0.0),
        76: (75.924541, 0.0),
        77: (76.921379, 0.0),
        78: (77.921146, 0.0),
        79: (78.918337, 0.506900),
        80: (79.918529, 0.0),
        81: (80.916291, 0.493100),
        82: (81.916804, 0.0),
        83: (82.915180, 0.0),
        84: (83.916479, 0.0),
        85: (84.915608, 0.0),
        86: (85.918798, 0.0),
        87: (86.920711, 0.0),
        88: (87.924070, 0.0),
        89: (88.926390, 0.0),
        90: (89.930630, 0.0),
        91: (90.933970, 0.0),
        92: (91.939260, 0.0),
        93: (92.943050, 0.0),
        94: (93.948680, 0.0),
        95: (94.952870, 0.0),
        96: (95.958530, 0.0),
        97: (96.962800, 0.0),
    },
    'C': {
        0: (12.000000, 1.000000),
        8: (8.037675, 0.0),
        9: (9.031037, 0.0),
        10: (10.016853, 0.0),
        11: (11.011434, 0.0),
        12: (12.000000, 0.989300),
        13: (13.003355, 0.010700),
        14: (14.003242, 0.0),
        15: (15.010599, 0.0),
        16: (16.014701, 0.0),
        17: (17.022586, 0.0),
        18: (18.026760, 0.0),
        19: (19.034810, 0.0),
        20: (20.040320, 0.0),
        21: (21.049340, 0.0),
        22: (22.057200, 0.0),
    },
    'Ca': {
        0: (39.962591, 1.000000),
        34: (34.014120, 0.0),
        35: (35.004940, 0.0),
        36: (35.993090, 0.0),
        37: (36.985870, 0.0),
        38: (37.976318, 0.0),
        39: (38.970720, 0.0),
        40: (39.962591, 0.969410),
        41: (40.962278, 0.0),
        42: (41.958618, 0.006470),
        43: (42.958767, 0.001350),
        44: (43.955482, 0.020860),
        45: (44.956187, 0.0),
        46: (45.953693, 0.000040),
        47: (46.954546, 0.0),
        48: (47.952534, 0.001870),
        49: (48.955674, 0.0),
        50: (49.957519, 0.0),
        51: (50.961500, 0.0),
        52: (51.965100, 0.0),
        53: (52.970050, 0.0),
        54: (53.974350, 0.0),
        55: (54.980550, 0.0),
        56: (55.985570, 0.0),
        57: (56.992360, 0.0),
    },
    'Cd': {
        0: (113.903358, 1.000000),
        95: (94.949870, 0.0),
        96: (95.939770, 0.0),
        97: (96.934940, 0.0),
        98: (97.927400, 0.0),
        99: (98.925010, 0.0),
        100: (99.920290, 0.0),
        101: (100.918680, 0.0),
        102: (101.914460, 0.0),
        103: (102.913419, 0.0),
        104: (103.909849, 0.0),
        105: (104.909468, 0.0),
        106: (105.906459, 0.012500),
        107: (106.906618, 0.0),
        108: (107.904184, 0.008900),
        109: (108.904982, 0.0),
        110: (109.903002, 0.124900),
        111: (110.904178, 0.128000),
        112: (111.902758, 0.241300),
        113: (112.904402, 0.122200),
        114: (113.903358, 0.287300),
        115: (114.905431, 0.0),
        116: (115.904756, 0.074900),
        117: (116.907219, 0.0),
        118: (117.906915, 0.0),
        119: (118.909920, 0.0),
        120: (119.909850, 0.0),
        121: (120.912980, 0.0),
        122: (121.913330, 0.0),
        123: (122.917000, 0.0),
        124: (123.917650, 0.0),
        125: (124.921250, 0.0),
        126: (125.922350, 0.0),
        127: (126.926440, 0.0),
        128: (127.927760, 0.0),
        129: (128.932150, 0.0),
        130: (129.933900, 0.0),
        131: (130.940670, 0.0),
        132: (131.945550, 0.0),
    },
    'Ce': {
        0: (139.905439, 1.000000),
        119: (118.952760, 0.0),
        120: (119.946640, 0.0),
        121: (120.943420, 0.0),
        122: (121.937910, 0.0),
        123: (122.935400, 0.0),
        124: (123.930410, 0.0),
        125: (124.928440, 0.0),
        126: (125.923970, 0.0),
        127: (126.922730, 0.0),
        128: (127.918910, 0.0),
        129: (128.918100, 0.0),
        130: (129.914740, 0.0),
        131: (130.914420, 0.0),
        132: (131.911460, 0.0),
        133: (132.911515, 0.0),
        134: (133.908925, 0.0),
        135: (134.909151, 0.0),
        136: (135.907172, 0.001850),
        137: (136.907806, 0.0),
        138: (137.905991, 0.002510),
        139: (138.906653, 0.0),
        140: (139.905439, 0.884500),
        141: (140.908276, 0.0),
        142: (141.909244, 0.111140),
        143: (142.912386, 0.0),
        144: (143.913647, 0.0),
        145: (144.917230, 0.0),
        146: (145.918760, 0.0),
        147: (146.922670, 0.0),
        148: (147.924430, 0.0),
        149: (148.928400, 0.0),
        150: (149.930410, 0.0),
        151: (150.933980, 0.0),
        152: (151.936540, 0.0),
        153: (152.940580, 0.0),
        154: (153.943420, 0.0),
        155: (154.948040, 0.0),
        156: (155.951260, 0.0),
        157: (156.956340, 0.0),
    },
    'Cf': {
        0: (251.000000, 1.000000),
        237: (237.062070, 0.0),
        238: (238.061410, 0.0),
        239: (239.062420, 0.0),
        240: (240.062300, 0.0),
        241: (241.063730, 0.0),
        242: (242.063700, 0.0),
        243: (243.065430, 0.0),
        244: (244.066001, 0.0),
        245: (245.068049, 0.0),
        246: (246.068805, 0.0),
        247: (247.071001, 0.0),
        248: (248.072185, 0.0),
        249: (249.074853, 0.0),
        250: (250.076406, 0.0),
        251: (251.079587, 0.0),
        252: (252.081626, 0.0),
        253: (253.085133, 0.0),
        254: (254.087323, 0.0),
        255: (255.091050, 0.0),
        256: (256.093440, 0.0),
    },
    'Cl': {
        0: (34.968853, 1.000000),
        28: (28.028510, 0.0),
        29: (29.014110, 0.0),
        30: (30.004770, 0.0),
        31: (30.992410, 0.0),
        32: (31.985690, 0.0),
        33: (32.977452, 0.0),
        34: (33.973763, 0.0),
        35: (34.968853, 0.757600),
        36: (35.968307, 0.0),
        37: (36.965903, 0.242400),
        38: (37.968010, 0.0),
        39: (38.968008, 0.0),
        40: (39.970420, 0.0),
        41: (40.970680, 0.0),
        42: (41.973250, 0.0),
        43: (42.974050, 0.0),
        44: (43.978280, 0.0),
        45: (44.980290, 0.0),
        46: (45.984210, 0.0),
        47: (46.988710, 0.0),
        48: (47.994950, 0.0),
        49: (49.000320, 0.0),
        50: (50.007840, 0.0),
        51: (51.014490, 0.0),
    },
    'Cm': {
        0: (247.000000, 1.000000),
        233: (233.050770, 0.0),
        234: (234.050160, 0.0),
        235: (235.051430, 0.0),
        236: (236.051410, 0.0),
        237: (237.052900, 0.0),
        238: (238.053030, 0.0),
        239: (239.054960, 0.0),
        240: (240.055530, 0.0),
        241: (241.057653, 0.0),
        242: (242.058836, 0.0),
        243: (243.061389, 0.0),
        244: (244.062753, 0.0),
        245: (245.065491, 0.0),
        246: (246.067224, 0.0),
        247: (247.070354, 0.0),
        248: (248.072349, 0.0),
        249: (249.075953, 0.0),
        250: (250.078357, 0.0),
        251: (251.082285, 0.0),
        252: (252.084870, 0.0),
    },
    'Cn': {
        0: (285.000000, 1.000000),
        277: (277.163940, 0.0),
        278: (278.164310, 0.0),
        279: (279.166550, 0.0),
        280: (280.167040, 0.0),
        281: (281.169290, 0.0),
        282: (282.169770, 0.0),
        283: (283.171790, 0.0),
        284: (284.172380, 0.0),
        285: (285.174110, 0.0),
    },
    'Co': {
        0: (58.933195, 1.000000),
        47: (47.011490, 0.0),
        48: (48.001760, 0.0),
        49: (48.989720, 0.0),
        50: (49.981540, 0.0),
        51: (50.970720, 0.0),
        52: (51.963590, 0.0),
        53: (52.954219, 0.0),
        54: (53.948460, 0.0),
        55: (54.941999, 0.0),
        56: (55.939839, 0.0),
        57: (56.936291, 0.0),
        58: (57.935753, 0.0),
        59: (58.933195, 1.000000),
        60: (59.933817, 0.0),
        61: (60.932476, 0.0),
        62: (61.934051, 0.0),
        63: (62.933612, 0.0),
        64: (63.935810, 0.0),
        65: (64.936478, 0.0),
        66: (65.939760, 0.0),
        67: (66.940890, 0.0),
        68: (67.944870, 0.0),
        69: (68.946320, 0.0),
        70: (69.951000, 0.0),
        71: (70.952900, 0.0),
        72: (71.957810, 0.0),
        73: (72.960240, 0.0),
        74: (73.965380, 0.0),
        75: (74.968330, 0.0),
    },
    'Cr': {
        0: (51.940508, 1.000000),
        42: (42.006430, 0.0),
        43: (42.997710, 0.0),
        44: (43.985550, 0.0),
        45: (44.979640, 0.0),
        46: (45.968359, 0.0),
        47: (46.962900, 0.0),
        48: (47.954032, 0.0),
        49: (48.951336, 0.0),
        50: (49.946044, 0.043450),
        51: (50.944767, 0.0),
        52: (51.940508, 0.837890),
        53: (52.940649, 0.095010),
        54: (53.938880, 0.023650),
        55: (54.940840, 0.0),
        56: (55.940653, 0.0),
        57: (56.943613, 0.0),
        58: (57.944350, 0.0),
        59: (58.948590, 0.0),
        60: (59.950080, 0.0),
        61: (60.954720, 0.0),
        62: (61.956610, 0.0),
        63: (62.961860, 0.0),
        64: (63.964410, 0.0),
        65: (64.970160, 0.0),
        66: (65.973380, 0.0),
        67: (66.979550, 0.0),
    },
    'Cs': {
        0: (132.905452, 1.000000),
        112: (111.950300, 0.0),
        113: (112.944490, 0.0),
        114: (113.941450, 0.0),
        115: (114.935910, 0.0),
        116: (115.933370, 0.0),
        117: (116.928670, 0.0),
        118: (117.926559, 0.0),
        119: (118.922377, 0.0),
        120: (119.920677, 0.0),
        121: (120.917229, 0.0),
        122: (121.916110, 0.0),
        123: (122.912996, 0.0),
        124: (123.912258, 0.0),
        125: (124.909728, 0.0),
        126: (125.909452, 0.0),
        127: (126.907418, 0.0),
        128: (127.907749, 0.0),
        129: (128.906064, 0.0),
        130: (129.906709, 0.0),
        131: (130.905464, 0.0),
        132: (131.906434, 0.0),
        133: (132.905452, 1.000000),
        134: (133.906718, 0.0),
        135: (134.905977, 0.0),
        136: (135.907312, 0.0),
        137: (136.907090, 0.0),
        138: (137.911017, 0.0),
        139: (138.913364, 0.0),
        140: (139.917282, 0.0),
        141: (140.920046, 0.0),
        142: (141.924299, 0.0),
        143: (142.927352, 0.0),
        144: (143.932077, 0.0),
        145: (144.935526, 0.0),
        146: (145.940290, 0.0),
        147: (146.944160, 0.0),
        148: (147.949220, 0.0),
        149: (148.952930, 0.0),
        150: (149.958170, 0.0),
        151: (150.962190, 0.0),
    },
    'Cu': {
        0: (62.929597, 1.000000),
        52: (51.997180, 0.0),
        53: (52.985550, 0.0),
        54: (53.976710, 0.0),
        55: (54.966050, 0.0),
        56: (55.958560, 0.0),
        57: (56.949211, 0.0),
        58: (57.944539, 0.0),
        59: (58.939498, 0.0),
        60: (59.937365, 0.0),
        61: (60.933458, 0.0),
        62: (61.932584, 0.0),
        63: (62.929597, 0.691500),
        64: (63.929764, 0.0),
        65: (64.927790, 0.308500),
        66: (65.928869, 0.0),
        67: (66.927730, 0.0),
        68: (67.929611, 0.0),
        69: (68.929429, 0.0),
        70: (69.932392, 0.0),
        71: (70.932677, 0.0),
        72: (71.935820, 0.0),
        73: (72.936675, 0.0),
        74: (73.939875, 0.0),
        75: (74.941900, 0.0),
        76: (75.945275, 0.0),
        77: (76.947850, 0.0),
        78: (77.951960, 0.0),
        79: (78.954560, 0.0),
        80: (79.960870, 0.0),
    },
    'Db': {
        0: (268.000000, 1.000000),
        255: (255.107400, 0.0),
        256: (256.108130, 0.0),
        257: (257.107720, 0.0),
        258: (258.109230, 0.0),
        259: (259.109610, 0.0),
        260: (260.111300, 0.0),
        261: (261.112060, 0.0),
        262: (262.114080, 0.0),
        263: (263.114990, 0.0),
        264: (264.117400, 0.0),
        265: (265.118600, 0.0),
        266: (266.121030, 0.0),
        267: (267.122380, 0.0),
        268: (268.125450, 0.0),
        269: (269.127460, 0.0),
        270: (270.130710, 0.0),
    },
    'Ds': {
        0: (281.000000, 1.000000),
        267: (267.144340, 0.0),
        268: (268.143800, 0.0),
        269: (269.145120, 0.0),
        270: (270.144720, 0.0),
        271: (271.146060, 0.0),
        272: (272.146320, 0.0),
        273: (273.148860, 0.0),
        274: (274.149490, 0.0),
        275: (275.152180, 0.0),
        276: (276.153030, 0.0),
        277: (277.155650, 0.0),
        278: (278.156470, 0.0),
        279: (279.158860, 0.0),
        280: (280.159800, 0.0),
        281: (281.162060, 0.0),
    },
    'Dy': {
        0: (163.929175, 1.000000),
        138: (137.962490, 0.0),
        139: (138.959540, 0.0),
        140: (139.954010, 0.0),
        141: (140.951350, 0.0),
        142: (141.946370, 0.0),
        143: (142.943830, 0.0),
        144: (143.939250, 0.0),
        145: (144.937430, 0.0),
        146: (145.932845, 0.0),
        147: (146.931092, 0.0),
        148: (147.927150, 0.0),
        149: (148.927305, 0.0),
        150: (149.925585, 0.0),
        151: (150.926185, 0.0),
        152: (151.924718, 0.0),
        153: (152.925765, 0.0),
        154: (153.924424, 0.0),
        155: (154.925754, 0.0),
        156: (155.924283, 0.000560),
        157: (156.925466, 0.0),
        158: (157.924409, 0.000950),
        159: (158.925739, 0.0),
        160: (159.925197, 0.023290),
        161: (160.926933, 0.188890),
        162: (161.926798, 0.254750),
        163: (162.928731, 0.248960),
        164: (163.929175, 0.282600),
        165: (164.931703, 0.0),
        166: (165.932807, 0.0),
        167: (166.935660, 0.0),
        168: (167.937130, 0.0),
        169: (168.940310, 0.0),
        170: (169.942390, 0.0),
        171: (170.946200, 0.0),
        172: (171.948760, 0.0),
        173: (172.953000, 0.0),
    },
    'Er': {
        0: (165.930293, 1.000000),
        143: (142.966340, 0.0),
        144: (143.960380, 0.0),
        145: (144.957390, 0.0),
        146: (145.952000, 0.0),
        147: (146.949490, 0.0),
        148: (147.944550, 0.0),
        149: (148.942310, 0.0),
        150: (149.937914, 0.0),
        151: (150.937449, 0.0),
        152: (151.935050, 0.0),
        153: (152.935063, 0.0),
        154: (153.932783, 0.0),
        155: (154.933209, 0.0),
        156: (155.931065, 0.0),
        157: (156.931920, 0.0),
        158: (157.929893, 0.0),
        159: (158.930684, 0.0),
        160: (159.929083, 0.0),
        161: (160.929995, 0.0),
        162: (161.928778, 0.001390),
        163: (162.930033, 0.0),
        164: (163.929200, 0.016010),
        165: (164.930726, 0.0),
        166: (165.930293, 0.335030),
        167: (166.932048, 0.228690),
        168: (167.932370, 0.269780),
        169: (168.934590, 0.0),
        170: (169.935464, 0.149100),
        171: (170.938030, 0.0),
        172: (171.939356, 0.0),
        173: (172.942400, 0.0),
        174: (173.944230, 0.0),
        175: (174.947770, 0.0),
        176: (175.950080, 0.0),
        177: (176.954050, 0.0),
    },
    'Es': {
        0: (252.000000, 1.000000),
        240: (240.068920, 0.0),
        241: (241.068540, 0.0),
        242: (242.069750, 0.0),
        243: (243.069550, 0.0),
        244: (244.070880, 0.0),
        245: (245.071320, 0.0),
        246: (246.072900, 0.0),
        247: (247.073660, 0.0),
        248: (248.075470, 0.0),
        249: (249.076410, 0.0),
        250: (250.078610, 0.0),
        251: (251.079992, 0.0),
        252: (252.082980, 0.0),
        253: (253.084825, 0.0),
        254: (254.088022, 0.0),
        255: (255.090273, 0.0),
        256: (256.093600, 0.0),
        257: (257.095980, 0.0),
        258: (258.099520, 0.0),
    },
    'Eu': {
        0: (152.921230, 1.000000),
        130: (129.963570, 0.0),
        131: (130.957750, 0.0),
        132: (131.954370, 0.0),
        133: (132.949240, 0.0),
        134: (133.946510, 0.0),
        135: (134.941820, 0.0),
        136: (135.939600, 0.0),
        137: (136.935570, 0.0),
        138: (137.933710, 0.0),
        139: (138.929792, 0.0),
        140: (139.928090, 0.0),
        141: (140.924931, 0.0),
        142: (141.923430, 0.0),
        143: (142.920298, 0.0),
        144: (143.918817, 0.0),
        145: (144.916265, 0.0),
        146: (145.917206, 0.0),
        147: (146.916746, 0.0),
        148: (147.918086, 0.0),
        149: (148.917931, 0.0),
        150: (149.919702, 0.0),
        151: (150.919850, 0.478100),
        152: (151.921744, 0.0),
        153: (152.921230, 0.521900),
        154: (153.922979, 0.0),
        155: (154.922893, 0.0),
        156: (155.924752, 0.0),
        157: (156.925424, 0.0),
        158: (157.927850, 0.0),
        159: (158.929089, 0.0),
        160: (159.931970, 0.0),
        161: (160.933680, 0.0),
        162: (161.937040, 0.0),
        163: (162.939210, 0.0),
        164: (163.942990, 0.0),
        165: (164.945720, 0.0),
        166: (165.949970, 0.0),
        167: (166.953210, 0.0),
    },
    'F': {
        0: (18.998403, 1.000000),
        14: (14.035060, 0.0),
        15: (15.018010, 0.0),
        16: (16.011466, 0.0),
        17: (17.002095, 0.0),
        18: (18.000938, 0.0),
        19: (18.998403, 1.000000),
        20: (19.999981, 0.0),
        21: (20.999949, 0.0),
        22: (22.002999, 0.0),
        23: (23.003570, 0.0),
        24: (24.008120, 0.0),
        25: (25.012100, 0.0),
        26: (26.019620, 0.0),
        27: (27.026760, 0.0),
        28: (28.035670, 0.0),
        29: (29.043260, 0.0),
        30: (30.052500, 0.0),
        31: (31.060430, 0.0),
    },
    'Fe': {
        0: (55.934937, 1.000000),
        45: (45.014580, 0.0),
        46: (46.000810, 0.0),
        47: (46.992890, 0.0),
        48: (47.980500, 0.0),
        49: (48.973610, 0.0),
        50: (49.962990, 0.0),
        51: (50.956820, 0.0),
        52: (51.948114, 0.0),
        53: (52.945308, 0.0),
        54: (53.939611, 0.058450),
        55: (54.938293, 0.0),
        56: (55.934937, 0.917540),
        57: (56.935394, 0.021190),
        58: (57.933276, 0.002820),
        59: (58.934875, 0.0),
        60: (59.934072, 0.0),
        61: (60.936745, 0.0),
        62: (61.936767, 0.0),
        63: (62.940370, 0.0),
        64: (63.941200, 0.0),
        65: (64.945380, 0.0),
        66: (65.946780, 0.0),
        67: (66.950950, 0.0),
        68: (67.953700, 0.0),
        69: (68.958780, 0.0),
        70: (69.961460, 0.0),
        71: (70.966720, 0.0),
        72: (71.969620, 0.0),
    },
    'Fm': {
        0: (257.000000, 1.000000),
        242: (242.073430, 0.0),
        243: (243.074350, 0.0),
        244: (244.074080, 0.0),
        245: (245.075390, 0.0),
        246: (246.075300, 0.0),
        247: (247.076850, 0.0),
        248: (248.077195, 0.0),
        249: (249.079030, 0.0),
        250: (250.079521, 0.0),
        251: (251.081575, 0.0),
        252: (252.082467, 0.0),
        253: (253.085185, 0.0),
        254: (254.086854, 0.0),
        255: (255.089962, 0.0),
        256: (256.091773, 0.0),
        257: (257.095105, 0.0),
        258: (258.097080, 0.0),
        259: (259.100600, 0.0),
        260: (260.102680, 0.0),
    },
    'Fr': {
        0: (223.000000, 1.000000),
        199: (199.007260, 0.0),
        200: (200.006570, 0.0),
        201: (201.003860, 0.0),
        202: (202.003370, 0.0),
        203: (203.000925, 0.0),
        204: (204.000653, 0.0),
        205: (204.998594, 0.0),
        206: (205.998670, 0.0),
        207: (206.996950, 0.0),
        208: (207.997140, 0.0),
        209: (208.995954, 0.0),
        210: (209.996408, 0.0),
        211: (210.995537, 0.0),
        212: (211.996202, 0.0),
        213: (212.996189, 0.0),
        214: (213.998971, 0.0),
        215: (215.000341, 0.0),
        216: (216.003198, 0.0),
        217: (217.004632, 0.0),
        218: (218.007578, 0.0),
        219: (219.009252, 0.0),
        220: (220.012327, 0.0),
        221: (221.014255, 0.0),
        222: (222.017552, 0.0),
        223: (223.019736, 0.0),
        224: (224.023250, 0.0),
        225: (225.025570, 0.0),
        226: (226.029390, 0.0),
        227: (227.031840, 0.0),
        228: (228.035730, 0.0),
        229: (229.038450, 0.0),
        230: (230.042510, 0.0),
        231: (231.045440, 0.0),
        232: (232.049770, 0.0),
    },
    'Ga': {
        0: (68.925574, 1.000000),
        56: (55.994910, 0.0),
        57: (56.982930, 0.0),
        58: (57.974250, 0.0),
        59: (58.963370, 0.0),
        60: (59.957060, 0.0),
        61: (60.949450, 0.0),
        62: (61.944175, 0.0),
        63: (62.939294, 0.0),
        64: (63.936839, 0.0),
        65: (64.932735, 0.0),
        66: (65.931589, 0.0),
        67: (66.928202, 0.0),
        68: (67.927980, 0.0),
        69: (68.925574, 0.601080),
        70: (69.926022, 0.0),
        71: (70.924701, 0.398920),
        72: (71.926366, 0.0),
        73: (72.925175, 0.0),
        74: (73.926946, 0.0),
        75: (74.926500, 0.0),
        76: (75.928828, 0.0),
        77: (76.929154, 0.0),
        78: (77.931608, 0.0),
        79: (78.932890, 0.0),
        80: (79.936520, 0.0),
        81: (80.937750, 0.0),
        82: (81.942990, 0.0),
        83: (82.946980, 0.0),
        84: (83.952650, 0.0),
        85: (84.957000, 0.0),
        86: (85.963120, 0.0),
    },
    'Gd': {
        0: (157.924104, 1.000000),
        134: (133.955370, 0.0),
        135: (134.952570, 0.0),
        136: (135.947340, 0.0),
        137: (136.945020, 0.0),
        138: (137.940120, 0.0),
        139: (138.938240, 0.0),
        140: (139.933670, 0.0),
        141: (140.932126, 0.0),
        142: (141.928120, 0.0),
        143: (142.926750, 0.0),
        144: (143.922960, 0.0),
        145: (144.921709, 0.0),
        146: (145.918311, 0.0),
        147: (146.919094, 0.0),
        148: (147.918115, 0.0),
        149: (148.919341, 0.0),
        150: (149.918659, 0.0),
        151: (150.920348, 0.0),
        152: (151.919791, 0.002000),
        153: (152.921750, 0.0),
        154: (153.920866, 0.021800),
        155: (154.922622, 0.148000),
        156: (155.922123, 0.204700),
        157: (156.923960, 0.156500),
        158: (157.924104, 0.248400),
        159: (158.926389, 0.0),
        160: (159.927054, 0.218600),
        161: (160.929669, 0.0),
        162: (161.930985, 0.0),
        163: (162.933990, 0.0),
        164: (163.935860, 0.0),
        165: (164.939380, 0.0),
        166: (165.941600, 0.0),
        167: (166.945570, 0.0),
        168: (167.948360, 0.0),
        169: (168.952870, 0.0),
    },
    'Ge': {
        0: (73.921178, 1.000000),
        58: (57.991010, 0.0),
        59: (58.981750, 0.0),
        60: (59.970190, 0.0),
        61: (60.963790, 0.0),
        62: (61.954650, 0.0),
        63: (62.949640, 0.0),
        64: (63.941650, 0.0),
        65: (64.939440, 0.0),
        66: (65.933840, 0.0),
        67: (66.932734, 0.0),
        68: (67.928094, 0.0),
        69: (68.927965, 0.0),
        70: (69.924247, 0.203800),
        71: (70.924951, 0.0),
        72: (71.922076, 0.273100),
        73: (72.923459, 0.077600),
        74: (73.921178, 0.367200),
        75: (74.922859, 0.0),
        76: (75.921403, 0.078300),
        77: (76.923549, 0.0),
        78: (77.922853, 0.0),
        79: (78.925400, 0.0),
        80: (79.925370, 0.0),
        81: (80.928820, 0.0),
        82: (81.929550, 0.0),
        83: (82.934620, 0.0),
        84: (83.937470, 0.0),
        85: (84.943030, 0.0),
        86: (85.946490, 0.0),
        87: (86.952510, 0.0),
        88: (87.956910, 0.0),
        89: (88.963830, 0.0),
    },
    'H': {
        0: (1.007825, 1.000000),
        1: (1.007825, 0.999885),
        2: (2.014102, 0.000115),
        3: (3.016049, 0.0),
        4: (4.027810, 0.0),
        5: (5.035310, 0.0),
        6: (6.044940, 0.0),
        7: (7.052750, 0.0),
    },
    'H+': {
        0: (1.007276, 1.000000),
        1: (1.007276, 1.000000),
    },
    'He': {
        0: (4.002603, 1.000000),
        3: (3.016029, 0.000001),
        4: (4.002603, 0.999999),
        5: (5.012220, 0.0),
        6: (6.018889, 0.0),
        7: (7.028021, 0.0),
        8: (8.033922, 0.0),
        9: (9.043950, 0.0),
        10: (10.052400, 0.0),
    },
    'Hf': {
        0: (179.946550, 1.000000),
        153: (152.970690, 0.0),
        154: (153.964860, 0.0),
        155: (154.963390, 0.0),
        156: (155.959360, 0.0),
        157: (156.958400, 0.0),
        158: (157.954799, 0.0),
        159: (158.953995, 0.0),
        160: (159.950684, 0.0),
        161: (160.950275, 0.0),
        162: (161.947210, 0.0),
        163: (162.947090, 0.0),
        164: (163.944367, 0.0),
        165: (164.944570, 0.0),
        166: (165.942180, 0.0),
        167: (166.942600, 0.0),
        168: (167.940570, 0.0),
        169: (168.941260, 0.0),
        170: (169.939610, 0.0),
        171: (170.940490, 0.0),
        172: (171.939448, 0.0),
        173: (172.940510, 0.0),
        174: (173.940046, 0.001600),
        175: (174.941509, 0.0),
        176: (175.941409, 0.052600),
        177: (176.943221, 0.186000),
        178: (177.943699, 0.272800),
        179: (178.945816, 0.136200),
        180: (179.946550, 0.350800),
        181: (180.949101, 0.0),
        182: (181.950554, 0.0),
        183: (182.953530, 0.0),
        184: (183.955450, 0.0),
        185: (184.958820, 0.0),
        186: (185.960890, 0.0),
        187: (186.964590, 0.0),
        188: (187.966850, 0.0),
    },
    'Hg': {
        0: (201.970643, 1.000000),
        171: (171.003760, 0.0),
        172: (171.998830, 0.0),
        173: (172.997240, 0.0),
        174: (173.992864, 0.0),
        175: (174.991420, 0.0),
        176: (175.987355, 0.0),
        177: (176.986280, 0.0),
        178: (177.982483, 0.0),
        179: (178.981834, 0.0),
        180: (179.978266, 0.0),
        181: (180.977819, 0.0),
        182: (181.974690, 0.0),
        183: (182.974450, 0.0),
        184: (183.971713, 0.0),
        185: (184.971899, 0.0),
        186: (185.969362, 0.0),
        187: (186.969814, 0.0),
        188: (187.967577, 0.0),
        189: (188.968190, 0.0),
        190: (189.966322, 0.0),
        191: (190.967157, 0.0),
        192: (191.965634, 0.0),
        193: (192.966665, 0.0),
        194: (193.965439, 0.0),
        195: (194.966720, 0.0),
        196: (195.965833, 0.001500),
        197: (196.967213, 0.0),
        198: (197.966769, 0.099700),
        199: (198.968280, 0.168700),
        200: (199.968326, 0.231000),
        201: (200.970302, 0.131800),
        202: (201.970643, 0.298600),
        203: (202.972872, 0.0),
        204: (203.973494, 0.068700),
        205: (204.976073, 0.0),
        206: (205.977514, 0.0),
        207: (206.982590, 0.0),
        208: (207.985940, 0.0),
        209: (208.991040, 0.0),
        210: (209.994510, 0.0),
    },
    'Ho': {
        0: (164.930322, 1.000000),
        140: (139.968540, 0.0),
        141: (140.963100, 0.0),
        142: (141.959770, 0.0),
        143: (142.954610, 0.0),
        144: (143.951480, 0.0),
        145: (144.947200, 0.0),
        146: (145.944640, 0.0),
        147: (146.940060, 0.0),
        148: (147.937720, 0.0),
        149: (148.933775, 0.0),
        150: (149.933496, 0.0),
        151: (150.931688, 0.0),
        152: (151.931714, 0.0),
        153: (152.930199, 0.0),
        154: (153.930602, 0.0),
        155: (154.929103, 0.0),
        156: (155.929840, 0.0),
        157: (156.928256, 0.0),
        158: (157.928941, 0.0),
        159: (158.927712, 0.0),
        160: (159.928729, 0.0),
        161: (160.927855, 0.0),
        162: (161.929096, 0.0),
        163: (162.928734, 0.0),
        164: (163.930234, 0.0),
        165: (164.930322, 1.000000),
        166: (165.932284, 0.0),
        167: (166.933133, 0.0),
        168: (167.935520, 0.0),
        169: (168.936872, 0.0),
        170: (169.939620, 0.0),
        171: (170.941470, 0.0),
        172: (171.944820, 0.0),
        173: (172.947290, 0.0),
        174: (173.951150, 0.0),
        175: (174.954050, 0.0),
    },
    'Hs': {
        0: (270.000000, 1.000000),
        263: (263.128560, 0.0),
        264: (264.128390, 0.0),
        265: (265.130090, 0.0),
        266: (266.130100, 0.0),
        267: (267.131790, 0.0),
        268: (268.132160, 0.0),
        269: (269.134060, 0.0),
        270: (270.134650, 0.0),
        271: (271.137660, 0.0),
        272: (272.139050, 0.0),
        273: (273.141990, 0.0),
        274: (274.143130, 0.0),
        275: (275.145950, 0.0),
        276: (276.147210, 0.0),
        277: (277.149840, 0.0),
    },
    'I': {
        0: (126.904473, 1.000000),
        108: (107.943480, 0.0),
        109: (108.938150, 0.0),
        110: (109.935240, 0.0),
        111: (110.930280, 0.0),
        112: (111.927970, 0.0),
        113: (112.923640, 0.0),
        114: (113.921850, 0.0),
        115: (114.918050, 0.0),
        116: (115.916810, 0.0),
        117: (116.913650, 0.0),
        118: (117.913074, 0.0),
        119: (118.910070, 0.0),
        120: (119.910048, 0.0),
        121: (120.907367, 0.0),
        122: (121.907589, 0.0),
        123: (122.905589, 0.0),
        124: (123.906210, 0.0),
        125: (124.904630, 0.0),
        126: (125.905624, 0.0),
        127: (126.904473, 1.000000),
        128: (127.905809, 0.0),
        129: (128.904988, 0.0),
        130: (129.906674, 0.0),
        131: (130.906125, 0.0),
        132: (131.907997, 0.0),
        133: (132.907797, 0.0),
        134: (133.909744, 0.0),
        135: (134.910048, 0.0),
        136: (135.914650, 0.0),
        137: (136.917871, 0.0),
        138: (137.922350, 0.0),
        139: (138.926100, 0.0),
        140: (139.931000, 0.0),
        141: (140.935030, 0.0),
        142: (141.940180, 0.0),
        143: (142.944560, 0.0),
        144: (143.949990, 0.0),
    },
    'In': {
        0: (114.903878, 1.000000),
        97: (96.949540, 0.0),
        98: (97.942140, 0.0),
        99: (98.934220, 0.0),
        100: (99.931110, 0.0),
        101: (100.926340, 0.0),
        102: (101.924090, 0.0),
        103: (102.919914, 0.0),
        104: (103.918300, 0.0),
        105: (104.914674, 0.0),
        106: (105.913465, 0.0),
        107: (106.910295, 0.0),
        108: (107.909698, 0.0),
        109: (108.907151, 0.0),
        110: (109.907165, 0.0),
        111: (110.905103, 0.0),
        112: (111.905532, 0.0),
        113: (112.904058, 0.042900),
        114: (113.904914, 0.0),
        115: (114.903878, 0.957100),
        116: (115.905260, 0.0),
        117: (116.904514, 0.0),
        118: (117.906354, 0.0),
        119: (118.905845, 0.0),
        120: (119.907960, 0.0),
        121: (120.907846, 0.0),
        122: (121.910280, 0.0),
        123: (122.910438, 0.0),
        124: (123.913180, 0.0),
        125: (124.913600, 0.0),
        126: (125.916460, 0.0),
        127: (126.917350, 0.0),
        128: (127.920170, 0.0),
        129: (128.921700, 0.0),
        130: (129.924970, 0.0),
        131: (130.926850, 0.0),
        132: (131.932990, 0.0),
        133: (132.937810, 0.0),
        134: (133.944150, 0.0),
        135: (134.949330, 0.0),
    },
    'Ir': {
        0: (192.962926, 1.000000),
        164: (163.992200, 0.0),
        165: (164.987520, 0.0),
        166: (165.985820, 0.0),
        167: (166.981665, 0.0),
        168: (167.979880, 0.0),
        169: (168.976295, 0.0),
        170: (169.974970, 0.0),
        171: (170.971630, 0.0),
        172: (171.970460, 0.0),
        173: (172.967502, 0.0),
        174: (173.966861, 0.0),
        175: (174.964113, 0.0),
        176: (175.963649, 0.0),
        177: (176.961302, 0.0),
        178: (177.961082, 0.0),
        179: (178.959122, 0.0),
        180: (179.959229, 0.0),
        181: (180.957625, 0.0),
        182: (181.958076, 0.0),
        183: (182.956846, 0.0),
        184: (183.957480, 0.0),
        185: (184.956700, 0.0),
        186: (185.957946, 0.0),
        187: (186.957363, 0.0),
        188: (187.958853, 0.0),
        189: (188.958719, 0.0),
        190: (189.960546, 0.0),
        191: (190.960594, 0.373000),
        192: (191.962605, 0.0),
        193: (192.962926, 0.627000),
        194: (193.965078, 0.0),
        195: (194.965980, 0.0),
        196: (195.968400, 0.0),
        197: (196.969653, 0.0),
        198: (197.972280, 0.0),
        199: (198.973800, 0.0),
    },
    'K': {
        0: (38.963707, 1.000000),
        32: (32.021920, 0.0),
        33: (33.007260, 0.0),
        34: (33.998410, 0.0),
        35: (34.988010, 0.0),
        36: (35.981292, 0.0),
        37: (36.973376, 0.0),
        38: (37.969081, 0.0),
        39: (38.963707, 0.932581),
        40: (39.963998, 0.000117),
        41: (40.961826, 0.067302),
        42: (41.962403, 0.0),
        43: (42.960716, 0.0),
        44: (43.961560, 0.0),
        45: (44.960699, 0.0),
        46: (45.961977, 0.0),
        47: (46.961678, 0.0),
        48: (47.965514, 0.0),
        49: (48.967450, 0.0),
        50: (49.972780, 0.0),
        51: (50.976380, 0.0),
        52: (51.982610, 0.0),
        53: (52.987120, 0.0),
        54: (53.994200, 0.0),
        55: (54.999710, 0.0),
    },
    'Kr': {
        0: (83.911507, 1.000000),
        69: (68.965180, 0.0),
        70: (69.955260, 0.0),
        71: (70.949630, 0.0),
        72: (71.942092, 0.0),
        73: (72.939289, 0.0),
        74: (73.933084, 0.0),
        75: (74.930946, 0.0),
        76: (75.925910, 0.0),
        77: (76.924670, 0.0),
        78: (77.920365, 0.003550),
        79: (78.920082, 0.0),
        80: (79.916379, 0.022860),
        81: (80.916592, 0.0),
        82: (81.913484, 0.115930),
        83: (82.914136, 0.115000),
        84: (83.911507, 0.569870),
        85: (84.912527, 0.0),
        86: (85.910611, 0.172790),
        87: (86.913355, 0.0),
        88: (87.914447, 0.0),
        89: (88.917630, 0.0),
        90: (89.919517, 0.0),
        91: (90.923450, 0.0),
        92: (91.926156, 0.0),
        93: (92.931270, 0.0),
        94: (93.934360, 0.0),
        95: (94.939840, 0.0),
        96: (95.943070, 0.0),
        97: (96.948560, 0.0),
        98: (97.951910, 0.0),
        99: (98.957600, 0.0),
        100: (99.961140, 0.0),
    },
    'La': {
        0: (138.906353, 1.000000),
        117: (116.950070, 0.0),
        118: (117.946730, 0.0),
        119: (118.940990, 0.0),
        120: (119.938070, 0.0),
        121: (120.933010, 0.0),
        122: (121.930710, 0.0),
        123: (122.926240, 0.0),
        124: (123.924570, 0.0),
        125: (124.920816, 0.0),
        126: (125.919510, 0.0),
        127: (126.916375, 0.0),
        128: (127.915590, 0.0),
        129: (128.912693, 0.0),
        130: (129.912369, 0.0),
        131: (130.910070, 0.0),
        132: (131.910100, 0.0),
        133: (132.908220, 0.0),
        134: (133.908514, 0.0),
        135: (134.906977, 0.0),
        136: (135.907640, 0.0),
        137: (136.906494, 0.0),
        138: (137.907112, 0.000900),
        139: (138.906353, 0.999100),
        140: (139.909478, 0.0),
        141: (140.910962, 0.0),
        142: (141.914079, 0.0),
        143: (142.916063, 0.0),
        144: (143.919600, 0.0),
        145: (144.921650, 0.0),
        146: (145.925790, 0.0),
        147: (146.928240, 0.0),
        148: (147.932230, 0.0),
        149: (148.934730, 0.0),
        150: (149.938770, 0.0),
        151: (150.941720, 0.0),
        152: (151.946250, 0.0),
        153: (152.949620, 0.0),
        154: (153.954500, 0.0),
        155: (154.958350, 0.0),
    },
    'Li': {
        0: (7.016005, 1.000000),
        3: (3.030780, 0.0),
        4: (4.027190, 0.0),
        5: (5.012540, 0.0),
        6: (6.015123, 0.075900),
        7: (7.016005, 0.924100),
        8: (8.022487, 0.0),
        9: (9.026789, 0.0),
        10: (10.035481, 0.0),
        11: (11.043798, 0.0),
        12: (12.053780, 0.0),
    },
    'Lr': {
        0: (262.000000, 1.000000),
        251: (251.094360, 0.0),
        252: (252.095370, 0.0),
        253: (253.095210, 0.0),
        254: (254.096450, 0.0),
        255: (255.096680, 0.0),
        256: (256.098630, 0.0),
        257: (257.099560, 0.0),
        258: (258.101810, 0.0),
        259: (259.102900, 0.0),
        260: (260.105500, 0.0),
        261: (261.106880, 0.0),
        262: (262.109630, 0.0),
        263: (263.111290, 0.0),
        264: (264.114040, 0.0),
        265: (265.115840, 0.0),
        266: (266.119310, 0.0),
    },
    'Lu': {
        0: (174.940772, 1.000000),
        150: (149.973230, 0.0),
        151: (150.967580, 0.0),
        152: (151.964120, 0.0),
        153: (152.958770, 0.0),
        154: (153.957520, 0.0),
        155: (154.954316, 0.0),
        156: (155.953030, 0.0),
        157: (156.950098, 0.0),
        158: (157.949313, 0.0),
        159: (158.946630, 0.0),
        160: (159.946030, 0.0),
        161: (160.943570, 0.0),
        162: (161.943280, 0.0),
        163: (162.941180, 0.0),
        164: (163.941340, 0.0),
        165: (164.939407, 0.0),
        166: (165.939860, 0.0),
        167: (166.938270, 0.0),
        168: (167.938740, 0.0),
        169: (168.937651, 0.0),
        170: (169.938475, 0.0),
        171: (170.937913, 0.0),
        172: (171.939086, 0.0),
        173: (172.938931, 0.0),
        174: (173.940337, 0.0),
        175: (174.940772, 0.974100),
        176: (175.942686, 0.025900),
        177: (176.943758, 0.0),
        178: (177.945955, 0.0),
        179: (178.947327, 0.0),
        180: (179.949880, 0.0),
        181: (180.951970, 0.0),
        182: (181.955040, 0.0),
        183: (182.957570, 0.0),
        184: (183.960910, 0.0),
    },
    'Md': {
        0: (258.000000, 1.000000),
        245: (245.080830, 0.0),
        246: (246.081890, 0.0),
        247: (247.081640, 0.0),
        248: (248.082820, 0.0),
        249: (249.083010, 0.0),
        250: (250.084420, 0.0),
        251: (251.084840, 0.0),
        252: (252.086560, 0.0),
        253: (253.087280, 0.0),
        254: (254.089660, 0.0),
        255: (255.091083, 0.0),
        256: (256.094060, 0.0),
        257: (257.095541, 0.0),
        258: (258.098431, 0.0),
        259: (259.100510, 0.0),
        260: (260.103650, 0.0),
        261: (261.105720, 0.0),
        262: (262.108870, 0.0),
    },
    'Mg': {
        0: (23.985042, 1.000000),
        19: (19.035470, 0.0),
        20: (20.018863, 0.0),
        21: (21.011713, 0.0),
        22: (21.999574, 0.0),
        23: (22.994124, 0.0),
        24: (23.985042, 0.789900),
        25: (24.985837, 0.100000),
        26: (25.982593, 0.110100),
        27: (26.984341, 0.0),
        28: (27.983877, 0.0),
        29: (28.988600, 0.0),
        30: (29.990434, 0.0),
        31: (30.996546, 0.0),
        32: (31.998975, 0.0),
        33: (33.005254, 0.0),
        34: (34.009460, 0.0),
        35: (35.017340, 0.0),
        36: (36.023000, 0.0),
        37: (37.031400, 0.0),
        38: (38.037570, 0.0),
        39: (39.046770, 0.0),
        40: (40.053930, 0.0),
    },
    'Mn': {
        0: (54.938045, 1.000000),
        44: (44.006870, 0.0),
        45: (44.994510, 0.0),
        46: (45.986720, 0.0),
        47: (46.976100, 0.0),
        48: (47.968520, 0.0),
        49: (48.959618, 0.0),
        50: (49.954238, 0.0),
        51: (50.948211, 0.0),
        52: (51.945566, 0.0),
        53: (52.941290, 0.0),
        54: (53.940359, 0.0),
        55: (54.938045, 1.000000),
        56: (55.938905, 0.0),
        57: (56.938285, 0.0),
        58: (57.939980, 0.0),
        59: (58.940440, 0.0),
        60: (59.942910, 0.0),
        61: (60.944650, 0.0),
        62: (61.948430, 0.0),
        63: (62.950240, 0.0),
        64: (63.954250, 0.0),
        65: (64.956340, 0.0),
        66: (65.961080, 0.0),
        67: (66.964140, 0.0),
        68: (67.969300, 0.0),
        69: (68.972840, 0.0),
    },
    'Mo': {
        0: (97.905408, 1.000000),
        83: (82.948740, 0.0),
        84: (83.940090, 0.0),
        85: (84.936550, 0.0),
        86: (85.930700, 0.0),
        87: (86.927330, 0.0),
        88: (87.921953, 0.0),
        89: (88.919480, 0.0),
        90: (89.913937, 0.0),
        91: (90.911750, 0.0),
        92: (91.906811, 0.147700),
        93: (92.906813, 0.0),
        94: (93.905088, 0.092300),
        95: (94.905842, 0.159000),
        96: (95.904680, 0.166800),
        97: (96.906021, 0.095600),
        98: (97.905408, 0.241900),
        99: (98.907712, 0.0),
        100: (99.907477, 0.096700),
        101: (100.910347, 0.0),
        102: (101.910297, 0.0),
        103: (102.913210, 0.0),
        104: (103.913760, 0.0),
        105: (104.916970, 0.0),
        106: (105.918137, 0.0),
        107: (106.921690, 0.0),
        108: (107.923450, 0.0),
        109: (108.927810, 0.0),
        110: (109.929730, 0.0),
        111: (110.934410, 0.0),
        112: (111.936840, 0.0),
        113: (112.941880, 0.0),
        114: (113.944920, 0.0),
        115: (114.950290, 0.0),
    },
    'Mt': {
        0: (276.000000, 1.000000),
        265: (265.136150, 0.0),
        266: (266.137300, 0.0),
        267: (267.137310, 0.0),
        268: (268.138730, 0.0),
        269: (269.139060, 0.0),
        270: (270.140660, 0.0),
        271: (271.141140, 0.0),
        272: (272.143740, 0.0),
        273: (273.144910, 0.0),
        274: (274.147490, 0.0),
        275: (275.148650, 0.0),
        276: (276.151160, 0.0),
        277: (277.152420, 0.0),
        278: (278.154810, 0.0),
        279: (279.156190, 0.0),
    },
    'N': {
        0: (14.003074, 1.000000),
        10: (10.041650, 0.0),
        11: (11.026090, 0.0),
        12: (12.018613, 0.0),
        13: (13.005739, 0.0),
        14: (14.003074, 0.996360),
        15: (15.000109, 0.003640),
        16: (16.006102, 0.0),
        17: (17.008450, 0.0),
        18: (18.014079, 0.0),
        19: (19.017029, 0.0),
        20: (20.023370, 0.0),
        21: (21.027110, 0.0),
        22: (22.034390, 0.0),
        23: (23.041220, 0.0),
        24: (24.051040, 0.0),
        25: (25.060660, 0.0),
    },
    'Na': {
        0: (22.989769, 1.000000),
        18: (18.025970, 0.0),
        19: (19.013877, 0.0),
        20: (20.007351, 0.0),
        21: (20.997655, 0.0),
        22: (21.994436, 0.0),
        23: (22.989769, 1.000000),
        24: (23.990963, 0.0),
        25: (24.989954, 0.0),
        26: (25.992633, 0.0),
        27: (26.994077, 0.0),
        28: (27.998938, 0.0),
        29: (29.002861, 0.0),
        30: (30.008976, 0.0),
        31: (31.013590, 0.0),
        32: (32.020470, 0.0),
        33: (33.026720, 0.0),
        34: (34.035170, 0.0),
        35: (35.042490, 0.0),
        36: (36.051480, 0.0),
        37: (37.059340, 0.0),
    },
    'Nb': {
        0: (92.906378, 1.000000),
        81: (80.949030, 0.0),
        82: (81.943130, 0.0),
        83: (82.936710, 0.0),
        84: (83.933570, 0.0),
        85: (84.927910, 0.0),
        86: (85.925040, 0.0),
        87: (86.920360, 0.0),
        88: (87.918330, 0.0),
        89: (88.913418, 0.0),
        90: (89.911265, 0.0),
        91: (90.906996, 0.0),
        92: (91.907194, 0.0),
        93: (92.906378, 1.000000),
        94: (93.907284, 0.0),
        95: (94.906836, 0.0),
        96: (95.908101, 0.0),
        97: (96.908099, 0.0),
        98: (97.910328, 0.0),
        99: (98.911618, 0.0),
        100: (99.914182, 0.0),
        101: (100.915252, 0.0),
        102: (101.918040, 0.0),
        103: (102.919140, 0.0),
        104: (103.922460, 0.0),
        105: (104.923940, 0.0),
        106: (105.927970, 0.0),
        107: (106.930310, 0.0),
        108: (107.934840, 0.0),
        109: (108.937630, 0.0),
        110: (109.942440, 0.0),
        111: (110.945650, 0.0),
        112: (111.950830, 0.0),
        113: (112.954700, 0.0),
    },
    'Nd': {
        0: (141.907723, 1.000000),
        124: (123.952230, 0.0),
        125: (124.948880, 0.0),
        126: (125.943220, 0.0),
        127: (126.940500, 0.0),
        128: (127.935390, 0.0),
        129: (128.933190, 0.0),
        130: (129.928510, 0.0),
        131: (130.927250, 0.0),
        132: (131.923321, 0.0),
        133: (132.922350, 0.0),
        134: (133.918790, 0.0),
        135: (134.918181, 0.0),
        136: (135.914976, 0.0),
        137: (136.914567, 0.0),
        138: (137.911950, 0.0),
        139: (138.911978, 0.0),
        140: (139.909550, 0.0),
        141: (140.909610, 0.0),
        142: (141.907723, 0.272000),
        143: (142.909814, 0.122000),
        144: (143.910087, 0.238000),
        145: (144.912574, 0.083000),
        146: (145.913117, 0.172000),
        147: (146.916100, 0.0),
        148: (147.916893, 0.057000),
        149: (148.920149, 0.0),
        150: (149.920891, 0.056000),
        151: (150.923829, 0.0),
        152: (151.924682, 0.0),
        153: (152.927698, 0.0),
        154: (153.929480, 0.0),
        155: (154.932930, 0.0),
        156: (155.935020, 0.0),
        157: (156.939030, 0.0),
        158: (157.941600, 0.0),
        159: (158.946090, 0.0),
        160: (159.949090, 0.0),
        161: (160.953880, 0.0),
    },
    'Ne': {
        0: (19.992440, 1.000000),
        16: (16.025761, 0.0),
        17: (17.017672, 0.0),
        18: (18.005708, 0.0),
        19: (19.001880, 0.0),
        20: (19.992440, 0.904800),
        21: (20.993847, 0.002700),
        22: (21.991385, 0.092500),
        23: (22.994467, 0.0),
        24: (23.993611, 0.0),
        25: (24.997737, 0.0),
        26: (26.000461, 0.0),
        27: (27.007590, 0.0),
        28: (28.012070, 0.0),
        29: (29.019390, 0.0),
        30: (30.024800, 0.0),
        31: (31.033110, 0.0),
        32: (32.040020, 0.0),
        33: (33.049380, 0.0),
        34: (34.057030, 0.0),
    },
    'Ni': {
        0: (57.935343, 1.000000),
        48: (48.019750, 0.0),
        49: (49.009660, 0.0),
        50: (49.995930, 0.0),
        51: (50.987720, 0.0),
        52: (51.975680, 0.0),
        53: (52.968470, 0.0),
        54: (53.957910, 0.0),
        55: (54.951330, 0.0),
        56: (55.942132, 0.0),
        57: (56.939794, 0.0),
        58: (57.935343, 0.680769),
        59: (58.934347, 0.0),
        60: (59.930786, 0.262231),
        61: (60.931056, 0.011399),
        62: (61.928345, 0.036345),
        63: (62.929669, 0.0),
        64: (63.927966, 0.009256),
        65: (64.930084, 0.0),
        66: (65.929139, 0.0),
        67: (66.931569, 0.0),
        68: (67.931869, 0.0),
        69: (68.935610, 0.0),
        70: (69.936500, 0.0),
        71: (70.940740, 0.0),
        72: (71.942090, 0.0),
        73: (72.946470, 0.0),
        74: (73.948070, 0.0),
        75: (74.952870, 0.0),
        76: (75.955330, 0.0),
        77: (76.960550, 0.0),
        78: (77.963180, 0.0),
    },
    'No': {
        0: (259.000000, 1.000000),
        248: (248.086600, 0.0),
        249: (249.087830, 0.0),
        250: (250.087510, 0.0),
        251: (251.089010, 0.0),
        252: (252.088977, 0.0),
        253: (253.090680, 0.0),
        254: (254.090955, 0.0),
        255: (255.093241, 0.0),
        256: (256.094283, 0.0),
        257: (257.096877, 0.0),
        258: (258.098210, 0.0),
        259: (259.101030, 0.0),
        260: (260.102640, 0.0),
        261: (261.105750, 0.0),
        262: (262.107300, 0.0),
        263: (263.110550, 0.0),
        264: (264.112350, 0.0),
    },
    'Np': {
        0: (237.000000, 1.000000),
        225: (225.033910, 0.0),
        226: (226.035150, 0.0),
        227: (227.034960, 0.0),
        228: (228.036180, 0.0),
        229: (229.036260, 0.0),
        230: (230.037830, 0.0),
        231: (231.038250, 0.0),
        232: (232.040110, 0.0),
        233: (233.040740, 0.0),
        234: (234.042895, 0.0),
        235: (235.044063, 0.0),
        236: (236.046570, 0.0),
        237: (237.048173, 0.0),
        238: (238.050946, 0.0),
        239: (239.052939, 0.0),
        240: (240.056162, 0.0),
        241: (241.058250, 0.0),
        242: (242.061640, 0.0),
        243: (243.064280, 0.0),
        244: (244.067850, 0.0),
    },
    'O': {
        0: (15.994915, 1.000000),
        12: (12.034405, 0.0),
        13: (13.024812, 0.0),
        14: (14.008596, 0.0),
        15: (15.003066, 0.0),
        16: (15.994915, 0.997570),
        17: (16.999132, 0.000380),
        18: (17.999161, 0.002050),
        19: (19.003580, 0.0),
        20: (20.004077, 0.0),
        21: (21.008656, 0.0),
        22: (22.009970, 0.0),
        23: (23.015690, 0.0),
        24: (24.020470, 0.0),
        25: (25.029460, 0.0),
        26: (26.038340, 0.0),
        27: (27.048260, 0.0),
        28: (28.057810, 0.0),
    },
    'Os': {
        0: (191.961481, 1.000000),
        162: (161.984430, 0.0),
        163: (162.982690, 0.0),
        164: (163.978040, 0.0),
        165: (164.976760, 0.0),
        166: (165.972691, 0.0),
        167: (166.971550, 0.0),
        168: (167.967804, 0.0),
        169: (168.967019, 0.0),
        170: (169.963577, 0.0),
        171: (170.963185, 0.0),
        172: (171.960023, 0.0),
        173: (172.959808, 0.0),
        174: (173.957062, 0.0),
        175: (174.956946, 0.0),
        176: (175.954810, 0.0),
        177: (176.954965, 0.0),
        178: (177.953251, 0.0),
        179: (178.953816, 0.0),
        180: (179.952379, 0.0),
        181: (180.953240, 0.0),
        182: (181.952110, 0.0),
        183: (182.953130, 0.0),
        184: (183.952489, 0.000200),
        185: (184.954042, 0.0),
        186: (185.953838, 0.015900),
        187: (186.955750, 0.019600),
        188: (187.955838, 0.132400),
        189: (188.958147, 0.161500),
        190: (189.958447, 0.262600),
        191: (190.960930, 0.0),
        192: (191.961481, 0.407800),
        193: (192.964152, 0.0),
        194: (193.965182, 0.0),
        195: (194.968130, 0.0),
        196: (195.969640, 0.0),
    },
    'P': {
        0: (30.973762, 1.000000),
        24: (24.034350, 0.0),
        25: (25.020260, 0.0),
        26: (26.011780, 0.0),
        27: (26.999230, 0.0),
        28: (27.992315, 0.0),
        29: (28.981801, 0.0),
        30: (29.978314, 0.0),
        31: (30.973762, 1.000000),
        32: (31.973907, 0.0),
        33: (32.971725, 0.0),
        34: (33.973636, 0.0),
        35: (34.973314, 0.0),
        36: (35.978260, 0.0),
        37: (36.979610, 0.0),
        38: (37.984160, 0.0),
        39: (38.986180, 0.0),
        40: (39.991300, 0.0),
        41: (40.994340, 0.0),
        42: (42.001010, 0.0),
        43: (43.006190, 0.0),
        44: (44.012990, 0.0),
        45: (45.019220, 0.0),
        46: (46.027380, 0.0),
    },
    'Pa': {
        0: (231.035884, 1.000000),
        212: (212.023200, 0.0),
        213: (213.021110, 0.0),
        214: (214.020920, 0.0),
        215: (215.019190, 0.0),
        216: (216.019110, 0.0),
        217: (217.018320, 0.0),
        218: (218.020042, 0.0),
        219: (219.019880, 0.0),
        220: (220.021880, 0.0),
        221: (221.021880, 0.0),
        222: (222.023740, 0.0),
        223: (223.023960, 0.0),
        224: (224.025626, 0.0),
        225: (225.026130, 0.0),
        226: (226.027948, 0.0),
        227: (227.028805, 0.0),
        228: (228.031051, 0.0),
        229: (229.032097, 0.0),
        230: (230.034541, 0.0),
        231: (231.035884, 1.000000),
        232: (232.038592, 0.0),
        233: (233.040247, 0.0),
        234: (234.043308, 0.0),
        235: (235.045440, 0.0),
        236: (236.048680, 0.0),
        237: (237.051150, 0.0),
        238: (238.054500, 0.0),
        239: (239.057260, 0.0),
        240: (240.060980, 0.0),
    },
    'Pb': {
        0: (207.976652, 1.000000),
        178: (178.003830, 0.0),
        179: (179.002150, 0.0),
        180: (179.997918, 0.0),
        181: (180.996620, 0.0),
        182: (181.992672, 0.0),
        183: (182.991870, 0.0),
        184: (183.988142, 0.0),
        185: (184.987610, 0.0),
        186: (185.984239, 0.0),
        187: (186.983918, 0.0),
        188: (187.980874, 0.0),
        189: (188.980810, 0.0),
        190: (189.978082, 0.0),
        191: (190.978270, 0.0),
        192: (191.975785, 0.0),
        193: (192.976170, 0.0),
        194: (193.974012, 0.0),
        195: (194.974542, 0.0),
        196: (195.972774, 0.0),
        197: (196.973431, 0.0),
        198: (197.972034, 0.0),
        199: (198.972917, 0.0),
        200: (199.971827, 0.0),
        201: (200.972885, 0.0),
        202: (201.972159, 0.0),
        203: (202.973391, 0.0),
        204: (203.973044, 0.014000),
        205: (204.974482, 0.0),
        206: (205.974465, 0.241000),
        207: (206.975897, 0.221000),
        208: (207.976652, 0.524000),
        209: (208.981090, 0.0),
        210: (209.984188, 0.0),
        211: (210.988737, 0.0),
        212: (211.991897, 0.0),
        213: (212.996581, 0.0),
        214: (213.999805, 0.0),
        215: (215.004810, 0.0),
    },
    'Pd': {
        0: (105.903486, 1.000000),
        91: (90.949110, 0.0),
        92: (91.940420, 0.0),
        93: (92.935910, 0.0),
        94: (93.928770, 0.0),
        95: (94.924690, 0.0),
        96: (95.918160, 0.0),
        97: (96.916480, 0.0),
        98: (97.912721, 0.0),
        99: (98.911768, 0.0),
        100: (99.908506, 0.0),
        101: (100.908289, 0.0),
        102: (101.905609, 0.010200),
        103: (102.906087, 0.0),
        104: (103.904036, 0.111400),
        105: (104.905085, 0.223300),
        106: (105.903486, 0.273300),
        107: (106.905133, 0.0),
        108: (107.903892, 0.264600),
        109: (108.905950, 0.0),
        110: (109.905153, 0.117200),
        111: (110.907671, 0.0),
        112: (111.907314, 0.0),
        113: (112.910150, 0.0),
        114: (113.910363, 0.0),
        115: (114.913680, 0.0),
        116: (115.914160, 0.0),
        117: (116.917840, 0.0),
        118: (117.918980, 0.0),
        119: (118.923110, 0.0),
        120: (119.924690, 0.0),
        121: (120.928870, 0.0),
        122: (121.930550, 0.0),
        123: (122.934930, 0.0),
        124: (123.936880, 0.0),
    },
    'Pm': {
        0: (145.000000, 1.000000),
        126: (125.957520, 0.0),
        127: (126.951630, 0.0),
        128: (127.948420, 0.0),
        129: (128.943160, 0.0),
        130: (129.940450, 0.0),
        131: (130.935870, 0.0),
        132: (131.933750, 0.0),
        133: (132.929780, 0.0),
        134: (133.928350, 0.0),
        135: (134.924880, 0.0),
        136: (135.923570, 0.0),
        137: (136.920479, 0.0),
        138: (137.919548, 0.0),
        139: (138.916804, 0.0),
        140: (139.916040, 0.0),
        141: (140.913555, 0.0),
        142: (141.912874, 0.0),
        143: (142.910933, 0.0),
        144: (143.912591, 0.0),
        145: (144.912749, 0.0),
        146: (145.914696, 0.0),
        147: (146.915139, 0.0),
        148: (147.917475, 0.0),
        149: (148.918334, 0.0),
        150: (149.920984, 0.0),
        151: (150.921207, 0.0),
        152: (151.923497, 0.0),
        153: (152.924117, 0.0),
        154: (153.926460, 0.0),
        155: (154.928100, 0.0),
        156: (155.931060, 0.0),
        157: (156.933040, 0.0),
        158: (157.936560, 0.0),
        159: (158.938970, 0.0),
        160: (159.942990, 0.0),
        161: (160.945860, 0.0),
        162: (161.950290, 0.0),
        163: (162.953680, 0.0),
    },
    'Po': {
        0: (209.000000, 1.000000),
        188: (187.999422, 0.0),
        189: (188.998481, 0.0),
        190: (189.995101, 0.0),
        191: (190.994574, 0.0),
        192: (191.991335, 0.0),
        193: (192.991030, 0.0),
        194: (193.988186, 0.0),
        195: (194.988110, 0.0),
        196: (195.985535, 0.0),
        197: (196.985660, 0.0),
        198: (197.983389, 0.0),
        199: (198.983666, 0.0),
        200: (199.981799, 0.0),
        201: (200.982260, 0.0),
        202: (201.980758, 0.0),
        203: (202.981420, 0.0),
        204: (203.980318, 0.0),
        205: (204.981203, 0.0),
        206: (205.980481, 0.0),
        207: (206.981593, 0.0),
        208: (207.981246, 0.0),
        209: (208.982430, 0.0),
        210: (209.982874, 0.0),
        211: (210.986653, 0.0),
        212: (211.988868, 0.0),
        213: (212.992857, 0.0),
        214: (213.995201, 0.0),
        215: (214.999420, 0.0),
        216: (216.001915, 0.0),
        217: (217.006335, 0.0),
        218: (218.008973, 0.0),
        219: (219.013740, 0.0),
        220: (220.016600, 0.0),
    },
    'Pr': {
        0: (140.907653, 1.000000),
        121: (120.955360, 0.0),
        122: (121.951810, 0.0),
        123: (122.945960, 0.0),
        124: (123.942960, 0.0),
        125: (124.937830, 0.0),
        126: (125.935310, 0.0),
        127: (126.930830, 0.0),
        128: (127.928790, 0.0),
        129: (128.925100, 0.0),
        130: (129.923590, 0.0),
        131: (130.920260, 0.0),
        132: (131.919260, 0.0),
        133: (132.916331, 0.0),
        134: (133.915710, 0.0),
        135: (134.913112, 0.0),
        136: (135.912692, 0.0),
        137: (136.910705, 0.0),
        138: (137.910755, 0.0),
        139: (138.908938, 0.0),
        140: (139.909076, 0.0),
        141: (140.907653, 1.000000),
        142: (141.910045, 0.0),
        143: (142.910817, 0.0),
        144: (143.913305, 0.0),
        145: (144.914512, 0.0),
        146: (145.917640, 0.0),
        147: (146.918996, 0.0),
        148: (147.922135, 0.0),
        149: (148.923720, 0.0),
        150: (149.926673, 0.0),
        151: (150.928319, 0.0),
        152: (151.931500, 0.0),
        153: (152.933840, 0.0),
        154: (153.937520, 0.0),
        155: (154.940120, 0.0),
        156: (155.944270, 0.0),
        157: (156.947430, 0.0),
        158: (157.951980, 0.0),
        159: (158.955500, 0.0),
    },
    'Pt': {
        0: (194.964791, 1.000000),
        166: (165.994860, 0.0),
        167: (166.992980, 0.0),
        168: (167.988150, 0.0),
        169: (168.986720, 0.0),
        170: (169.982495, 0.0),
        171: (170.981240, 0.0),
        172: (171.977347, 0.0),
        173: (172.976440, 0.0),
        174: (173.972819, 0.0),
        175: (174.972421, 0.0),
        176: (175.968945, 0.0),
        177: (176.968469, 0.0),
        178: (177.965649, 0.0),
        179: (178.965363, 0.0),
        180: (179.963031, 0.0),
        181: (180.963097, 0.0),
        182: (181.961171, 0.0),
        183: (182.961597, 0.0),
        184: (183.959922, 0.0),
        185: (184.960620, 0.0),
        186: (185.959351, 0.0),
        187: (186.960590, 0.0),
        188: (187.959395, 0.0),
        189: (188.960834, 0.0),
        190: (189.959932, 0.000140),
        191: (190.961677, 0.0),
        192: (191.961038, 0.007820),
        193: (192.962987, 0.0),
        194: (193.962680, 0.329670),
        195: (194.964791, 0.338320),
        196: (195.964952, 0.252420),
        197: (196.967340, 0.0),
        198: (197.967893, 0.071630),
        199: (198.970593, 0.0),
        200: (199.971441, 0.0),
        201: (200.974510, 0.0),
        202: (201.975740, 0.0),
    },
    'Pu': {
        0: (244.000000, 1.000000),
        228: (228.038740, 0.0),
        229: (229.040150, 0.0),
        230: (230.039650, 0.0),
        231: (231.041101, 0.0),
        232: (232.041187, 0.0),
        233: (233.043000, 0.0),
        234: (234.043317, 0.0),
        235: (235.045286, 0.0),
        236: (236.046058, 0.0),
        237: (237.048410, 0.0),
        238: (238.049560, 0.0),
        239: (239.052163, 0.0),
        240: (240.053813, 0.0),
        241: (241.056851, 0.0),
        242: (242.058743, 0.0),
        243: (243.062003, 0.0),
        244: (244.064204, 0.0),
        245: (245.067747, 0.0),
        246: (246.070205, 0.0),
        247: (247.074070, 0.0),
    },
    'Ra': {
        0: (226.000000, 1.000000),
        202: (202.009890, 0.0),
        203: (203.009270, 0.0),
        204: (204.006500, 0.0),
        205: (205.006270, 0.0),
        206: (206.003827, 0.0),
        207: (207.003800, 0.0),
        208: (208.001840, 0.0),
        209: (209.001990, 0.0),
        210: (210.000495, 0.0),
        211: (211.000898, 0.0),
        212: (211.999794, 0.0),
        213: (213.000384, 0.0),
        214: (214.000108, 0.0),
        215: (215.002720, 0.0),
        216: (216.003533, 0.0),
        217: (217.006320, 0.0),
        218: (218.007140, 0.0),
        219: (219.010085, 0.0),
        220: (220.011028, 0.0),
        221: (221.013917, 0.0),
        222: (222.015375, 0.0),
        223: (223.018502, 0.0),
        224: (224.020212, 0.0),
        225: (225.023612, 0.0),
        226: (226.025410, 0.0),
        227: (227.029178, 0.0),
        228: (228.031070, 0.0),
        229: (229.034958, 0.0),
        230: (230.037056, 0.0),
        231: (231.041220, 0.0),
        232: (232.043640, 0.0),
        233: (233.048060, 0.0),
        234: (234.050700, 0.0),
    },
    'Rb': {
        0: (84.911790, 1.000000),
        71: (70.965320, 0.0),
        72: (71.959080, 0.0),
        73: (72.950560, 0.0),
        74: (73.944265, 0.0),
        75: (74.938570, 0.0),
        76: (75.935072, 0.0),
        77: (76.930408, 0.0),
        78: (77.928141, 0.0),
        79: (78.923989, 0.0),
        80: (79.922519, 0.0),
        81: (80.918996, 0.0),
        82: (81.918209, 0.0),
        83: (82.915110, 0.0),
        84: (83.914385, 0.0),
        85: (84.911790, 0.721700),
        86: (85.911167, 0.0),
        87: (86.909181, 0.278300),
        88: (87.911316, 0.0),
        89: (88.912278, 0.0),
        90: (89.914802, 0.0),
        91: (90.916537, 0.0),
        92: (91.919729, 0.0),
        93: (92.922042, 0.0),
        94: (93.926405, 0.0),
        95: (94.929303, 0.0),
        96: (95.934270, 0.0),
        97: (96.937350, 0.0),
        98: (97.941790, 0.0),
        99: (98.945380, 0.0),
        100: (99.949870, 0.0),
        101: (100.953200, 0.0),
        102: (101.958870, 0.0),
    },
    'Re': {
        0: (186.955753, 1.000000),
        160: (159.982120, 0.0),
        161: (160.977590, 0.0),
        162: (161.976000, 0.0),
        163: (162.972081, 0.0),
        164: (163.970320, 0.0),
        165: (164.967089, 0.0),
        166: (165.965810, 0.0),
        167: (166.962600, 0.0),
        168: (167.961570, 0.0),
        169: (168.958790, 0.0),
        170: (169.958220, 0.0),
        171: (170.955720, 0.0),
        172: (171.955420, 0.0),
        173: (172.953240, 0.0),
        174: (173.953120, 0.0),
        175: (174.951380, 0.0),
        176: (175.951620, 0.0),
        177: (176.950330, 0.0),
        178: (177.950990, 0.0),
        179: (178.949988, 0.0),
        180: (179.950789, 0.0),
        181: (180.950068, 0.0),
        182: (181.951210, 0.0),
        183: (182.950820, 0.0),
        184: (183.952521, 0.0),
        185: (184.952955, 0.374000),
        186: (185.954986, 0.0),
        187: (186.955753, 0.626000),
        188: (187.958114, 0.0),
        189: (188.959229, 0.0),
        190: (189.961820, 0.0),
        191: (190.963125, 0.0),
        192: (191.965960, 0.0),
        193: (192.967470, 0.0),
        194: (193.970420, 0.0),
    },
    'Rf': {
        0: (265.000000, 1.000000),
        253: (253.100690, 0.0),
        254: (254.100180, 0.0),
        255: (255.101340, 0.0),
        256: (256.101166, 0.0),
        257: (257.102990, 0.0),
        258: (258.103490, 0.0),
        259: (259.105640, 0.0),
        260: (260.106440, 0.0),
        261: (261.108770, 0.0),
        262: (262.109930, 0.0),
        263: (263.112550, 0.0),
        264: (264.113990, 0.0),
        265: (265.116700, 0.0),
        266: (266.117960, 0.0),
        267: (267.121530, 0.0),
        268: (268.123640, 0.0),
    },
    'Rg': {
        0: (280.000000, 1.000000),
        272: (272.153620, 0.0),
        273: (273.153680, 0.0),
        274: (274.155710, 0.0),
        275: (275.156140, 0.0),
        276: (276.158490, 0.0),
        277: (277.159520, 0.0),
        278: (278.161600, 0.0),
        279: (279.162470, 0.0),
        280: (280.164470, 0.0),
        281: (281.165370, 0.0),
        282: (282.167490, 0.0),
        283: (283.168420, 0.0),
    },
    'Rh': {
        0: (102.905504, 1.000000),
        89: (88.948840, 0.0),
        90: (89.942870, 0.0),
        91: (90.936550, 0.0),
        92: (91.931980, 0.0),
        93: (92.925740, 0.0),
        94: (93.921700, 0.0),
        95: (94.915900, 0.0),
        96: (95.914461, 0.0),
        97: (96.911340, 0.0),
        98: (97.910708, 0.0),
        99: (98.908132, 0.0),
        100: (99.908122, 0.0),
        101: (100.906164, 0.0),
        102: (101.906843, 0.0),
        103: (102.905504, 1.000000),
        104: (103.906656, 0.0),
        105: (104.905694, 0.0),
        106: (105.907287, 0.0),
        107: (106.906748, 0.0),
        108: (107.908730, 0.0),
        109: (108.908737, 0.0),
        110: (109.911140, 0.0),
        111: (110.911590, 0.0),
        112: (111.914390, 0.0),
        113: (112.915530, 0.0),
        114: (113.918810, 0.0),
        115: (114.920330, 0.0),
        116: (115.924060, 0.0),
        117: (116.925980, 0.0),
        118: (117.930070, 0.0),
        119: (118.932110, 0.0),
        120: (119.936410, 0.0),
        121: (120.938720, 0.0),
        122: (121.943210, 0.0),
    },
    'Rn': {
        0: (222.000000, 1.000000),
        195: (195.005440, 0.0),
        196: (196.002115, 0.0),
        197: (197.001580, 0.0),
        198: (197.998679, 0.0),
        199: (198.998370, 0.0),
        200: (199.995699, 0.0),
        201: (200.995630, 0.0),
        202: (201.993263, 0.0),
        203: (202.993387, 0.0),
        204: (203.991429, 0.0),
        205: (204.991720, 0.0),
        206: (205.990214, 0.0),
        207: (206.990734, 0.0),
        208: (207.989642, 0.0),
        209: (208.990415, 0.0),
        210: (209.989696, 0.0),
        211: (210.990601, 0.0),
        212: (211.990704, 0.0),
        213: (212.993883, 0.0),
        214: (213.995363, 0.0),
        215: (214.998745, 0.0),
        216: (216.000274, 0.0),
        217: (217.003928, 0.0),
        218: (218.005601, 0.0),
        219: (219.009480, 0.0),
        220: (220.011394, 0.0),
        221: (221.015537, 0.0),
        222: (222.017578, 0.0),
        223: (223.021790, 0.0),
        224: (224.024090, 0.0),
        225: (225.028440, 0.0),
        226: (226.030890, 0.0),
        227: (227.035410, 0.0),
        228: (228.037990, 0.0),
    },
    'Ru': {
        0: (101.904349, 1.000000),
        87: (86.949180, 0.0),
        88: (87.940260, 0.0),
        89: (88.936110, 0.0),
        90: (89.929890, 0.0),
        91: (90.926290, 0.0),
        92: (91.920120, 0.0),
        93: (92.917050, 0.0),
        94: (93.911360, 0.0),
        95: (94.910413, 0.0),
        96: (95.907598, 0.055400),
        97: (96.907555, 0.0),
        98: (97.905287, 0.018700),
        99: (98.905939, 0.127600),
        100: (99.904219, 0.126000),
        101: (100.905582, 0.170600),
        102: (101.904349, 0.315500),
        103: (102.906324, 0.0),
        104: (103.905433, 0.186200),
        105: (104.907753, 0.0),
        106: (105.907329, 0.0),
        107: (106.909910, 0.0),
        108: (107.910170, 0.0),
        109: (108.913200, 0.0),
        110: (109.914140, 0.0),
        111: (110.917700, 0.0),
        112: (111.918970, 0.0),
        113: (112.922490, 0.0),
        114: (113.924280, 0.0),
        115: (114.928690, 0.0),
        116: (115.930810, 0.0),
        117: (116.935580, 0.0),
        118: (117.937820, 0.0),
        119: (118.942840, 0.0),
        120: (119.945310, 0.0),
    },
    'S': {
        0: (31.972071, 1.000000),
        26: (26.027880, 0.0),
        27: (27.018830, 0.0),
        28: (28.004370, 0.0),
        29: (28.996610, 0.0),
        30: (29.984903, 0.0),
        31: (30.979555, 0.0),
        32: (31.972071, 0.949900),
        33: (32.971459, 0.007500),
        34: (33.967867, 0.042500),
        35: (34.969032, 0.0),
        36: (35.967081, 0.000100),
        37: (36.971126, 0.0),
        38: (37.971163, 0.0),
        39: (38.975130, 0.0),
        40: (39.975450, 0.0),
        41: (40.979580, 0.0),
        42: (41.981020, 0.0),
        43: (42.987150, 0.0),
        44: (43.990210, 0.0),
        45: (44.996510, 0.0),
        46: (46.000750, 0.0),
        47: (47.008590, 0.0),
        48: (48.014170, 0.0),
        49: (49.023620, 0.0),
    },
    'Sb': {
        0: (120.903816, 1.000000),
        103: (102.939690, 0.0),
        104: (103.936470, 0.0),
        105: (104.931490, 0.0),
        106: (105.928790, 0.0),
        107: (106.924150, 0.0),
        108: (107.922160, 0.0),
        109: (108.918132, 0.0),
        110: (109.916750, 0.0),
        111: (110.913160, 0.0),
        112: (111.912398, 0.0),
        113: (112.909372, 0.0),
        114: (113.909270, 0.0),
        115: (114.906598, 0.0),
        116: (115.906794, 0.0),
        117: (116.904836, 0.0),
        118: (117.905529, 0.0),
        119: (118.903942, 0.0),
        120: (119.905072, 0.0),
        121: (120.903816, 0.572100),
        122: (121.905174, 0.0),
        123: (122.904214, 0.427900),
        124: (123.905936, 0.0),
        125: (124.905254, 0.0),
        126: (125.907250, 0.0),
        127: (126.906924, 0.0),
        128: (127.909169, 0.0),
        129: (128.909148, 0.0),
        130: (129.911656, 0.0),
        131: (130.911982, 0.0),
        132: (131.914467, 0.0),
        133: (132.915252, 0.0),
        134: (133.920380, 0.0),
        135: (134.925170, 0.0),
        136: (135.930350, 0.0),
        137: (136.935310, 0.0),
        138: (137.940790, 0.0),
        139: (138.945980, 0.0),
    },
    'Sc': {
        0: (44.955912, 1.000000),
        36: (36.014920, 0.0),
        37: (37.003050, 0.0),
        38: (37.994700, 0.0),
        39: (38.984790, 0.0),
        40: (39.977967, 0.0),
        41: (40.969251, 0.0),
        42: (41.965516, 0.0),
        43: (42.961151, 0.0),
        44: (43.959403, 0.0),
        45: (44.955912, 1.000000),
        46: (45.955172, 0.0),
        47: (46.952407, 0.0),
        48: (47.952231, 0.0),
        49: (48.950024, 0.0),
        50: (49.952188, 0.0),
        51: (50.953603, 0.0),
        52: (51.956680, 0.0),
        53: (52.959610, 0.0),
        54: (53.963260, 0.0),
        55: (54.968240, 0.0),
        56: (55.972870, 0.0),
        57: (56.977790, 0.0),
        58: (57.983710, 0.0),
        59: (58.989220, 0.0),
        60: (59.995710, 0.0),
    },
    'Se': {
        0: (79.916521, 1.000000),
        65: (64.964660, 0.0),
        66: (65.955210, 0.0),
        67: (66.950090, 0.0),
        68: (67.941800, 0.0),
        69: (68.939560, 0.0),
        70: (69.933390, 0.0),
        71: (70.932240, 0.0),
        72: (71.927112, 0.0),
        73: (72.926765, 0.0),
        74: (73.922476, 0.008900),
        75: (74.922523, 0.0),
        76: (75.919214, 0.093700),
        77: (76.919914, 0.076300),
        78: (77.917309, 0.237700),
        79: (78.918499, 0.0),
        80: (79.916521, 0.496100),
        81: (80.917992, 0.0),
        82: (81.916699, 0.087300),
        83: (82.919118, 0.0),
        84: (83.918462, 0.0),
        85: (84.922250, 0.0),
        86: (85.924272, 0.0),
        87: (86.928520, 0.0),
        88: (87.931420, 0.0),
        89: (88.936450, 0.0),
        90: (89.939960, 0.0),
        91: (90.945960, 0.0),
        92: (91.949920, 0.0),
        93: (92.956290, 0.0),
        94: (93.960490, 0.0),
    },
    'Sg': {
        0: (271.000000, 1.000000),
        258: (258.113170, 0.0),
        259: (259.114500, 0.0),
        260: (260.114420, 0.0),
        261: (261.116120, 0.0),
        262: (262.116400, 0.0),
        263: (263.118320, 0.0),
        264: (264.118930, 0.0),
        265: (265.121110, 0.0),
        266: (266.122070, 0.0),
        267: (267.124430, 0.0),
        268: (268.125610, 0.0),
        269: (269.128760, 0.0),
        270: (270.130330, 0.0),
        271: (271.133470, 0.0),
        272: (272.135160, 0.0),
        273: (273.138220, 0.0),
    },
    'Si': {
        0: (27.976927, 1.000000),
        22: (22.034530, 0.0),
        23: (23.025520, 0.0),
        24: (24.011546, 0.0),
        25: (25.004106, 0.0),
        26: (25.992330, 0.0),
        27: (26.986705, 0.0),
        28: (27.976927, 0.922230),
        29: (28.976495, 0.046850),
        30: (29.973770, 0.030920),
        31: (30.975363, 0.0),
        32: (31.974148, 0.0),
        33: (32.978000, 0.0),
        34: (33.978576, 0.0),
        35: (34.984580, 0.0),
        36: (35.986600, 0.0),
        37: (36.992940, 0.0),
        38: (37.995630, 0.0),
        39: (39.002070, 0.0),
        40: (40.005870, 0.0),
        41: (41.014560, 0.0),
        42: (42.019790, 0.0),
        43: (43.028660, 0.0),
        44: (44.035260, 0.0),
    },
    'Sm': {
        0: (151.919732, 1.000000),
        128: (127.958080, 0.0),
        129: (128.954640, 0.0),
        130: (129.948920, 0.0),
        131: (130.946110, 0.0),
        132: (131.940690, 0.0),
        133: (132.938670, 0.0),
        134: (133.933970, 0.0),
        135: (134.932520, 0.0),
        136: (135.928276, 0.0),
        137: (136.926970, 0.0),
        138: (137.923244, 0.0),
        139: (138.922297, 0.0),
        140: (139.918995, 0.0),
        141: (140.918476, 0.0),
        142: (141.915198, 0.0),
        143: (142.914628, 0.0),
        144: (143.911999, 0.030700),
        145: (144.913410, 0.0),
        146: (145.913041, 0.0),
        147: (146.914898, 0.149900),
        148: (147.914823, 0.112400),
        149: (148.917185, 0.138200),
        150: (149.917275, 0.073800),
        151: (150.919932, 0.0),
        152: (151.919732, 0.267500),
        153: (152.922097, 0.0),
        154: (153.922209, 0.227500),
        155: (154.924640, 0.0),
        156: (155.925528, 0.0),
        157: (156.928360, 0.0),
        158: (157.929990, 0.0),
        159: (158.933210, 0.0),
        160: (159.935140, 0.0),
        161: (160.938830, 0.0),
        162: (161.941220, 0.0),
        163: (162.945360, 0.0),
        164: (163.948280, 0.0),
        165: (164.952980, 0.0),
    },
    'Sn': {
        0: (119.902195, 1.000000),
        99: (98.949330, 0.0),
        100: (99.939040, 0.0),
        101: (100.936060, 0.0),
        102: (101.930300, 0.0),
        103: (102.928100, 0.0),
        104: (103.923140, 0.0),
        105: (104.921350, 0.0),
        106: (105.916880, 0.0),
        107: (106.915640, 0.0),
        108: (107.911925, 0.0),
        109: (108.911283, 0.0),
        110: (109.907843, 0.0),
        111: (110.907734, 0.0),
        112: (111.904818, 0.009700),
        113: (112.905171, 0.0),
        114: (113.902779, 0.006600),
        115: (114.903342, 0.003400),
        116: (115.901741, 0.145400),
        117: (116.902952, 0.076800),
        118: (117.901603, 0.242200),
        119: (118.903308, 0.085900),
        120: (119.902195, 0.325800),
        121: (120.904235, 0.0),
        122: (121.903439, 0.046300),
        123: (122.905721, 0.0),
        124: (123.905274, 0.057900),
        125: (124.907784, 0.0),
        126: (125.907653, 0.0),
        127: (126.910360, 0.0),
        128: (127.910537, 0.0),
        129: (128.913480, 0.0),
        130: (129.913967, 0.0),
        131: (130.917000, 0.0),
        132: (131.917816, 0.0),
        133: (132.923830, 0.0),
        134: (133.928290, 0.0),
        135: (134.934730, 0.0),
        136: (135.939340, 0.0),
        137: (136.945990, 0.0),
    },
    'Sr': {
        0: (87.905612, 1.000000),
        73: (72.965970, 0.0),
        74: (73.956310, 0.0),
        75: (74.949950, 0.0),
        76: (75.941770, 0.0),
        77: (76.937945, 0.0),
        78: (77.932180, 0.0),
        79: (78.929708, 0.0),
        80: (79.924521, 0.0),
        81: (80.923212, 0.0),
        82: (81.918402, 0.0),
        83: (82.917557, 0.0),
        84: (83.913425, 0.005600),
        85: (84.912933, 0.0),
        86: (85.909260, 0.098600),
        87: (86.908877, 0.070000),
        88: (87.905612, 0.825800),
        89: (88.907451, 0.0),
        90: (89.907738, 0.0),
        91: (90.910203, 0.0),
        92: (91.911038, 0.0),
        93: (92.914026, 0.0),
        94: (93.915361, 0.0),
        95: (94.919359, 0.0),
        96: (95.921697, 0.0),
        97: (96.926153, 0.0),
        98: (97.928453, 0.0),
        99: (98.933240, 0.0),
        100: (99.935350, 0.0),
        101: (100.940520, 0.0),
        102: (101.943020, 0.0),
        103: (102.948950, 0.0),
        104: (103.952330, 0.0),
        105: (104.958580, 0.0),
    },
    'Ta': {
        0: (180.947996, 1.000000),
        155: (154.974590, 0.0),
        156: (155.972300, 0.0),
        157: (156.968190, 0.0),
        158: (157.966700, 0.0),
        159: (158.963018, 0.0),
        160: (159.961490, 0.0),
        161: (160.958420, 0.0),
        162: (161.957290, 0.0),
        163: (162.954330, 0.0),
        164: (163.953530, 0.0),
        165: (164.950773, 0.0),
        166: (165.950510, 0.0),
        167: (166.948090, 0.0),
        168: (167.948050, 0.0),
        169: (168.946010, 0.0),
        170: (169.946180, 0.0),
        171: (170.944480, 0.0),
        172: (171.944900, 0.0),
        173: (172.943750, 0.0),
        174: (173.944450, 0.0),
        175: (174.943740, 0.0),
        176: (175.944860, 0.0),
        177: (176.944472, 0.0),
        178: (177.945778, 0.0),
        179: (178.945930, 0.0),
        180: (179.947465, 0.000120),
        181: (180.947996, 0.999880),
        182: (181.950152, 0.0),
        183: (182.951373, 0.0),
        184: (183.954008, 0.0),
        185: (184.955559, 0.0),
        186: (185.958550, 0.0),
        187: (186.960530, 0.0),
        188: (187.963700, 0.0),
        189: (188.965830, 0.0),
        190: (189.969230, 0.0),
    },
    'Tb': {
        0: (158.925347, 1.000000),
        136: (135.961380, 0.0),
        137: (136.955980, 0.0),
        138: (137.953160, 0.0),
        139: (138.948290, 0.0),
        140: (139.945810, 0.0),
        141: (140.941450, 0.0),
        142: (141.938740, 0.0),
        143: (142.935120, 0.0),
        144: (143.933050, 0.0),
        145: (144.929270, 0.0),
        146: (145.927250, 0.0),
        147: (146.924045, 0.0),
        148: (147.924272, 0.0),
        149: (148.923246, 0.0),
        150: (149.923660, 0.0),
        151: (150.923103, 0.0),
        152: (151.924070, 0.0),
        153: (152.923435, 0.0),
        154: (153.924680, 0.0),
        155: (154.923505, 0.0),
        156: (155.924747, 0.0),
        157: (156.924025, 0.0),
        158: (157.925413, 0.0),
        159: (158.925347, 1.000000),
        160: (159.927168, 0.0),
        161: (160.927570, 0.0),
        162: (161.929490, 0.0),
        163: (162.930648, 0.0),
        164: (163.933350, 0.0),
        165: (164.934880, 0.0),
        166: (165.937990, 0.0),
        167: (166.940050, 0.0),
        168: (167.943640, 0.0),
        169: (168.946220, 0.0),
        170: (169.950250, 0.0),
        171: (170.953300, 0.0),
    },
    'Tc': {
        0: (98.000000, 1.000000),
        85: (84.948830, 0.0),
        86: (85.942880, 0.0),
        87: (86.936530, 0.0),
        88: (87.932680, 0.0),
        89: (88.927170, 0.0),
        90: (89.923560, 0.0),
        91: (90.918430, 0.0),
        92: (91.915260, 0.0),
        93: (92.910249, 0.0),
        94: (93.909657, 0.0),
        95: (94.907657, 0.0),
        96: (95.907871, 0.0),
        97: (96.906365, 0.0),
        98: (97.907216, 0.0),
        99: (98.906255, 0.0),
        100: (99.907658, 0.0),
        101: (100.907315, 0.0),
        102: (101.909215, 0.0),
        103: (102.909181, 0.0),
        104: (103.911450, 0.0),
        105: (104.911660, 0.0),
        106: (105.914358, 0.0),
        107: (106.915080, 0.0),
        108: (107.918460, 0.0),
        109: (108.919980, 0.0),
        110: (109.923820, 0.0),
        111: (110.925690, 0.0),
        112: (111.929150, 0.0),
        113: (112.931590, 0.0),
        114: (113.935880, 0.0),
        115: (114.938690, 0.0),
        116: (115.943370, 0.0),
        117: (116.946480, 0.0),
        118: (117.951480, 0.0),
    },
    'Te': {
        0: (129.906224, 1.000000),
        105: (104.943640, 0.0),
        106: (105.937500, 0.0),
        107: (106.935010, 0.0),
        108: (107.929440, 0.0),
        109: (108.927420, 0.0),
        110: (109.922410, 0.0),
        111: (110.921110, 0.0),
        112: (111.917010, 0.0),
        113: (112.915890, 0.0),
        114: (113.912090, 0.0),
        115: (114.911900, 0.0),
        116: (115.908460, 0.0),
        117: (116.908645, 0.0),
        118: (117.905828, 0.0),
        119: (118.906404, 0.0),
        120: (119.904020, 0.000900),
        121: (120.904936, 0.0),
        122: (121.903044, 0.025500),
        123: (122.904270, 0.008900),
        124: (123.902818, 0.047400),
        125: (124.904431, 0.070700),
        126: (125.903312, 0.188400),
        127: (126.905226, 0.0),
        128: (127.904463, 0.317400),
        129: (128.906598, 0.0),
        130: (129.906224, 0.340800),
        131: (130.908524, 0.0),
        132: (131.908553, 0.0),
        133: (132.910955, 0.0),
        134: (133.911369, 0.0),
        135: (134.916450, 0.0),
        136: (135.920100, 0.0),
        137: (136.925320, 0.0),
        138: (137.929220, 0.0),
        139: (138.934730, 0.0),
        140: (139.938850, 0.0),
        141: (140.944650, 0.0),
        142: (141.949080, 0.0),
    },
    'Th': {
        0: (232.038055, 1.000000),
        209: (209.017720, 0.0),
        210: (210.015075, 0.0),
        211: (211.014930, 0.0),
        212: (212.012980, 0.0),
        213: (213.013010, 0.0),
        214: (214.011500, 0.0),
        215: (215.011730, 0.0),
        216: (216.011062, 0.0),
        217: (217.013114, 0.0),
        218: (218.013284, 0.0),
        219: (219.015540, 0.0),
        220: (220.015748, 0.0),
        221: (221.018184, 0.0),
        222: (222.018468, 0.0),
        223: (223.020811, 0.0),
        224: (224.021467, 0.0),
        225: (225.023951, 0.0),
        226: (226.024903, 0.0),
        227: (227.027704, 0.0),
        228: (228.028741, 0.0),
        229: (229.031762, 0.0),
        230: (230.033134, 0.0),
        231: (231.036304, 0.0),
        232: (232.038055, 1.000000),
        233: (233.041582, 0.0),
        234: (234.043601, 0.0),
        235: (235.047510, 0.0),
        236: (236.049870, 0.0),
        237: (237.053890, 0.0),
        238: (238.056500, 0.0),
    },
    'Ti': {
        0: (47.947946, 1.000000),
        38: (38.009770, 0.0),
        39: (39.001610, 0.0),
        40: (39.990500, 0.0),
        41: (40.983150, 0.0),
        42: (41.973031, 0.0),
        43: (42.968522, 0.0),
        44: (43.959690, 0.0),
        45: (44.958126, 0.0),
        46: (45.952632, 0.082500),
        47: (46.951763, 0.074400),
        48: (47.947946, 0.737200),
        49: (48.947870, 0.054100),
        50: (49.944791, 0.051800),
        51: (50.946615, 0.0),
        52: (51.946897, 0.0),
        53: (52.949730, 0.0),
        54: (53.951050, 0.0),
        55: (54.955270, 0.0),
        56: (55.958200, 0.0),
        57: (56.963990, 0.0),
        58: (57.966970, 0.0),
        59: (58.972930, 0.0),
        60: (59.976760, 0.0),
        61: (60.983200, 0.0),
        62: (61.987490, 0.0),
        63: (62.994420, 0.0),
    },
    'Tl': {
        0: (204.974427, 1.000000),
        176: (176.000590, 0.0),
        177: (176.996427, 0.0),
        178: (177.994900, 0.0),
        179: (178.991090, 0.0),
        180: (179.989910, 0.0),
        181: (180.986257, 0.0),
        182: (181.985670, 0.0),
        183: (182.982193, 0.0),
        184: (183.981870, 0.0),
        185: (184.978790, 0.0),
        186: (185.978330, 0.0),
        187: (186.975906, 0.0),
        188: (187.976010, 0.0),
        189: (188.973588, 0.0),
        190: (189.973880, 0.0),
        191: (190.971786, 0.0),
        192: (191.972230, 0.0),
        193: (192.970670, 0.0),
        194: (193.971200, 0.0),
        195: (194.969774, 0.0),
        196: (195.970481, 0.0),
        197: (196.969575, 0.0),
        198: (197.970480, 0.0),
        199: (198.969880, 0.0),
        200: (199.970963, 0.0),
        201: (200.970819, 0.0),
        202: (201.972106, 0.0),
        203: (202.972344, 0.295200),
        204: (203.973863, 0.0),
        205: (204.974427, 0.704800),
        206: (205.976110, 0.0),
        207: (206.977419, 0.0),
        208: (207.982019, 0.0),
        209: (208.985359, 0.0),
        210: (209.990074, 0.0),
        211: (210.993480, 0.0),
        212: (211.998230, 0.0),
    },
    'Tm': {
        0: (168.934213, 1.000000),
        145: (144.970070, 0.0),
        146: (145.966430, 0.0),
        147: (146.960960, 0.0),
        148: (147.957840, 0.0),
        149: (148.952720, 0.0),
        150: (149.949960, 0.0),
        151: (150.945483, 0.0),
        152: (151.944420, 0.0),
        153: (152.942012, 0.0),
        154: (153.941568, 0.0),
        155: (154.939199, 0.0),
        156: (155.938980, 0.0),
        157: (156.936970, 0.0),
        158: (157.936980, 0.0),
        159: (158.934980, 0.0),
        160: (159.935260, 0.0),
        161: (160.933550, 0.0),
        162: (161.933995, 0.0),
        163: (162.932651, 0.0),
        164: (163.933560, 0.0),
        165: (164.932435, 0.0),
        166: (165.933554, 0.0),
        167: (166.932852, 0.0),
        168: (167.934173, 0.0),
        169: (168.934213, 1.000000),
        170: (169.935801, 0.0),
        171: (170.936429, 0.0),
        172: (171.938400, 0.0),
        173: (172.939604, 0.0),
        174: (173.942170, 0.0),
        175: (174.943840, 0.0),
        176: (175.946990, 0.0),
        177: (176.949040, 0.0),
        178: (177.952640, 0.0),
        179: (178.955340, 0.0),
    },
    'U': {
        0: (238.050788, 1.000000),
        217: (217.024370, 0.0),
        218: (218.023540, 0.0),
        219: (219.024920, 0.0),
        220: (220.024720, 0.0),
        221: (221.026400, 0.0),
        222: (222.026090, 0.0),
        223: (223.027740, 0.0),
        224: (224.027605, 0.0),
        225: (225.029391, 0.0),
        226: (226.029339, 0.0),
        227: (227.031156, 0.0),
        228: (228.031374, 0.0),
        229: (229.033506, 0.0),
        230: (230.033940, 0.0),
        231: (231.036294, 0.0),
        232: (232.037156, 0.0),
        233: (233.039635, 0.0),
        234: (234.040952, 0.000054),
        235: (235.043930, 0.007204),
        236: (236.045568, 0.0),
        237: (237.048730, 0.0),
        238: (238.050788, 0.992742),
        239: (239.054293, 0.0),
        240: (240.056592, 0.0),
        241: (241.060330, 0.0),
        242: (242.062930, 0.0),
    },
    'Uuh': {
        0: (293.000000, 1.000000),
        289: (289.198860, 0.0),
        290: (290.198590, 0.0),
        291: (291.200010, 0.0),
        292: (292.199790, 0.0),
    },
    'Uuo': {
        0: (294.000000, 1.000000),
        293: (293.214670, 0.0),
    },
    'Uup': {
        0: (288.000000, 1.000000),
        287: (287.191190, 0.0),
        288: (288.192490, 0.0),
        289: (289.192720, 0.0),
        290: (290.194140, 0.0),
        291: (291.194380, 0.0),
    },
    'Uuq': {
        0: (289.000000, 1.000000),
        285: (285.183700, 0.0),
        286: (286.183860, 0.0),
        287: (287.185600, 0.0),
        288: (288.185690, 0.0),
        289: (289.187280, 0.0),
    },
    'Uus': {
        0: (292.000000, 1.000000),
        291: (291.206560, 0.0),
        292: (292.207550, 0.0),
    },
    'Uut': {
        0: (284.000000, 1.000000),
        283: (283.176450, 0.0),
        284: (284.178080, 0.0),
        285: (285.178730, 0.0),
        286: (286.180480, 0.0),
        287: (287.181050, 0.0),
    },
    'V': {
        0: (50.943959, 1.000000),
        40: (40.011090, 0.0),
        41: (40.999780, 0.0),
        42: (41.991230, 0.0),
        43: (42.980650, 0.0),
        44: (43.974110, 0.0),
        45: (44.965776, 0.0),
        46: (45.960200, 0.0),
        47: (46.954909, 0.0),
        48: (47.952254, 0.0),
        49: (48.948516, 0.0),
        50: (49.947159, 0.002500),
        51: (50.943959, 0.997500),
        52: (51.944775, 0.0),
        53: (52.944338, 0.0),
        54: (53.946440, 0.0),
        55: (54.947230, 0.0),
        56: (55.950530, 0.0),
        57: (56.952560, 0.0),
        58: (57.956830, 0.0),
        59: (58.960210, 0.0),
        60: (59.965030, 0.0),
        61: (60.968480, 0.0),
        62: (61.973780, 0.0),
        63: (62.977550, 0.0),
        64: (63.983470, 0.0),
        65: (64.987920, 0.0),
    },
    'W': {
        0: (183.950931, 1.000000),
        158: (157.974560, 0.0),
        159: (158.972920, 0.0),
        160: (159.968480, 0.0),
        161: (160.967360, 0.0),
        162: (161.963497, 0.0),
        163: (162.962520, 0.0),
        164: (163.958954, 0.0),
        165: (164.958280, 0.0),
        166: (165.955027, 0.0),
        167: (166.954816, 0.0),
        168: (167.951808, 0.0),
        169: (168.951779, 0.0),
        170: (169.949228, 0.0),
        171: (170.949450, 0.0),
        172: (171.947290, 0.0),
        173: (172.947690, 0.0),
        174: (173.946080, 0.0),
        175: (174.946720, 0.0),
        176: (175.945630, 0.0),
        177: (176.946640, 0.0),
        178: (177.945876, 0.0),
        179: (178.947070, 0.0),
        180: (179.946704, 0.001200),
        181: (180.948197, 0.0),
        182: (181.948204, 0.265000),
        183: (182.950223, 0.143100),
        184: (183.950931, 0.306400),
        185: (184.953419, 0.0),
        186: (185.954364, 0.284300),
        187: (186.957160, 0.0),
        188: (187.958489, 0.0),
        189: (188.961910, 0.0),
        190: (189.963180, 0.0),
        191: (190.966600, 0.0),
        192: (191.968170, 0.0),
    },
    'Xe': {
        0: (131.904154, 1.000000),
        110: (109.944280, 0.0),
        111: (110.941600, 0.0),
        112: (111.935620, 0.0),
        113: (112.933340, 0.0),
        114: (113.927980, 0.0),
        115: (114.926294, 0.0),
        116: (115.921581, 0.0),
        117: (116.920359, 0.0),
        118: (117.916179, 0.0),
        119: (118.915411, 0.0),
        120: (119.911784, 0.0),
        121: (120.911462, 0.0),
        122: (121.908368, 0.0),
        123: (122.908482, 0.0),
        124: (123.905893, 0.000952),
        125: (124.906396, 0.0),
        126: (125.904274, 0.000890),
        127: (126.905184, 0.0),
        128: (127.903531, 0.019102),
        129: (128.904779, 0.264006),
        130: (129.903508, 0.040710),
        131: (130.905082, 0.212324),
        132: (131.904154, 0.269086),
        133: (132.905911, 0.0),
        134: (133.905394, 0.104357),
        135: (134.907227, 0.0),
        136: (135.907219, 0.088573),
        137: (136.911562, 0.0),
        138: (137.913950, 0.0),
        139: (138.918793, 0.0),
        140: (139.921640, 0.0),
        141: (140.926650, 0.0),
        142: (141.929710, 0.0),
        143: (142.935110, 0.0),
        144: (143.938510, 0.0),
        145: (144.944070, 0.0),
        146: (145.947750, 0.0),
        147: (146.953560, 0.0),
    },
    'Y': {
        0: (88.905848, 1.000000),
        76: (75.958450, 0.0),
        77: (76.949650, 0.0),
        78: (77.943610, 0.0),
        79: (78.937350, 0.0),
        80: (79.934280, 0.0),
        81: (80.929130, 0.0),
        82: (81.926790, 0.0),
        83: (82.922350, 0.0),
        84: (83.920390, 0.0),
        85: (84.916433, 0.0),
        86: (85.914886, 0.0),
        87: (86.910876, 0.0),
        88: (87.909501, 0.0),
        89: (88.905848, 1.000000),
        90: (89.907152, 0.0),
        91: (90.907305, 0.0),
        92: (91.908949, 0.0),
        93: (92.909583, 0.0),
        94: (93.911595, 0.0),
        95: (94.912821, 0.0),
        96: (95.915891, 0.0),
        97: (96.918134, 0.0),
        98: (97.922203, 0.0),
        99: (98.924636, 0.0),
        100: (99.927760, 0.0),
        101: (100.930310, 0.0),
        102: (101.933560, 0.0),
        103: (102.936730, 0.0),
        104: (103.941050, 0.0),
        105: (104.944870, 0.0),
        106: (105.949790, 0.0),
        107: (106.954140, 0.0),
        108: (107.959480, 0.0),
    },
    'Yb': {
        0: (173.938862, 1.000000),
        148: (147.967420, 0.0),
        149: (148.964040, 0.0),
        150: (149.958420, 0.0),
        151: (150.955400, 0.0),
        152: (151.950290, 0.0),
        153: (152.949480, 0.0),
        154: (153.946394, 0.0),
        155: (154.945782, 0.0),
        156: (155.942818, 0.0),
        157: (156.942628, 0.0),
        158: (157.939866, 0.0),
        159: (158.940050, 0.0),
        160: (159.937552, 0.0),
        161: (160.937902, 0.0),
        162: (161.935768, 0.0),
        163: (162.936334, 0.0),
        164: (163.934489, 0.0),
        165: (164.935280, 0.0),
        166: (165.933882, 0.0),
        167: (166.934950, 0.0),
        168: (167.933897, 0.001300),
        169: (168.935190, 0.0),
        170: (169.934762, 0.030400),
        171: (170.936326, 0.142800),
        172: (171.936382, 0.218300),
        173: (172.938211, 0.161300),
        174: (173.938862, 0.318300),
        175: (174.941276, 0.0),
        176: (175.942572, 0.127600),
        177: (176.945261, 0.0),
        178: (177.946647, 0.0),
        179: (178.950170, 0.0),
        180: (179.952330, 0.0),
        181: (180.956150, 0.0),
    },
    'Zn': {
        0: (63.929142, 1.000000),
        54: (53.992950, 0.0),
        55: (54.983980, 0.0),
        56: (55.972380, 0.0),
        57: (56.964790, 0.0),
        58: (57.954590, 0.0),
        59: (58.949260, 0.0),
        60: (59.941827, 0.0),
        61: (60.939511, 0.0),
        62: (61.934330, 0.0),
        63: (62.933212, 0.0),
        64: (63.929142, 0.482680),
        65: (64.929241, 0.0),
        66: (65.926033, 0.279750),
        67: (66.927127, 0.041020),
        68: (67.924844, 0.190240),
        69: (68.926550, 0.0),
        70: (69.925319, 0.006310),
        71: (70.927722, 0.0),
        72: (71.926858, 0.0),
        73: (72.929780, 0.0),
        74: (73.929460, 0.0),
        75: (74.932940, 0.0),
        76: (75.933290, 0.0),
        77: (76.936960, 0.0),
        78: (77.938440, 0.0),
        79: (78.942650, 0.0),
        80: (79.944340, 0.0),
        81: (80.950480, 0.0),
        82: (81.954420, 0.0),
        83: (82.961030, 0.0),
    },
    'Zr': {
        0: (89.904704, 1.000000),
        78: (77.955230, 0.0),
        79: (78.949160, 0.0),
        80: (79.940400, 0.0),
        81: (80.937210, 0.0),
        82: (81.931090, 0.0),
        83: (82.928650, 0.0),
        84: (83.923250, 0.0),
        85: (84.921470, 0.0),
        86: (85.916470, 0.0),
        87: (86.914816, 0.0),
        88: (87.910227, 0.0),
        89: (88.908890, 0.0),
        90: (89.904704, 0.514500),
        91: (90.905646, 0.112200),
        92: (91.905041, 0.171500),
        93: (92.906476, 0.0),
        94: (93.906315, 0.173800),
        95: (94.908043, 0.0),
        96: (95.908273, 0.028000),
        97: (96.910953, 0.0),
        98: (97.912735, 0.0),
        99: (98.916512, 0.0),
        100: (99.917760, 0.0),
        101: (100.921140, 0.0),
        102: (101.922980, 0.0),
        103: (102.926600, 0.0),
        104: (103.928780, 0.0),
        105: (104.933050, 0.0),
        106: (105.935910, 0.0),
        107: (106.940750, 0.0),
        108: (107.943960, 0.0),
        109: (108.949240, 0.0),
        110: (109.952870, 0.0),
    },
    'e*': {
        0: (0.000549, 1.000000),
    },
}


if __name__ == '__main__':
    print(nist_mass['Pd'][0])
    print(crc_mass['Pd'][0])
