"""
This module has the highest level functions for fetching data from Edna BQ
"""
from concurrent.futures import ThreadPoolExecutor
from datetime import datetime
from typing import Tuple

import pandas as pd

from forecast_dataprep.data_fetching.data_models import BigQueryBundle
from forecast_dataprep.data_models import Dataframes, IngestedDataframes, MeteringPointList, ModelTargetList, SubstationList
from forecast_dataprep.data_fetching.common import get_hourly_data, get_national_holidays, get_prices, get_school_holidays, get_weekly_data
from forecast_dataprep.data_fetching.meteringpoints import get_meteringpoint_metadata
from forecast_dataprep.data_fetching.substations import get_substation_metadata


def get_dataframes(bq: BigQueryBundle, targets: ModelTargetList,
                   timespan: Tuple[datetime, datetime]) -> Dataframes:
    """
    Query Edna tables and fetch data, returned as dataframes.
    The substation/meteringpoint metadata is not part of this group of queries, in order to 
    optimise the overall execution order.
    """
    with ThreadPoolExecutor() as executor:
        dfh = executor.submit(get_hourly_data, bq, targets)
        dfw = executor.submit(get_weekly_data, bq, targets)
        dfnh = executor.submit(get_national_holidays, bq, timespan[0],
                               timespan[1])
        dfsh = executor.submit(get_school_holidays, bq, timespan[0],
                               timespan[1])
        dfp = executor.submit(get_prices, bq, timespan[0], timespan[1])

    dfh = dfh.result()
    dfw = dfw.result()
    dfnh = dfnh.result()
    dfsh = dfsh.result()
    dfp = dfp.result()

    return Dataframes(dfh, IngestedDataframes(dfw, dfsh, dfnh, dfp))


def get_metadata(bq: BigQueryBundle, targets: ModelTargetList) -> pd.DataFrame:
    """
    Get meteringpoint/substation metadata. 
    The result contains columns latitude and longitude that are needed to get the right forecast
    data from Weather API. It also contains column postalCode, which is needed to enrich with the
    location-dependant school holidays feature.
    """
    if isinstance(targets, MeteringPointList):
        return get_meteringpoint_metadata(bq, targets.identifiers)
    elif isinstance(targets, SubstationList):
        return get_substation_metadata(bq, targets.identifiers)
    raise TypeError
