"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cognito_1 = require("@aws-cdk/aws-cognito");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const core_1 = require("@aws-cdk/core");
const path_1 = require("path");
const lib_1 = require("../lib");
const app = new core_1.App();
const stack = new core_1.Stack(app, 'aws-appsync-integ');
const userPool = new aws_cognito_1.UserPool(stack, 'Pool', {
    userPoolName: 'myPool',
});
const api = new lib_1.GraphQLApi(stack, 'Api', {
    name: 'demoapi',
    schemaDefinitionFile: path_1.join(__dirname, 'schema.graphql'),
    authorizationConfig: {
        defaultAuthorization: {
            authorizationType: lib_1.AuthorizationType.USER_POOL,
            userPoolConfig: {
                userPool,
                defaultAction: lib_1.UserPoolDefaultAction.ALLOW,
            },
        },
        additionalAuthorizationModes: [
            {
                authorizationType: lib_1.AuthorizationType.API_KEY,
            },
        ],
    },
});
const noneDS = api.addNoneDataSource('None', 'Dummy data source');
noneDS.createResolver({
    typeName: 'Query',
    fieldName: 'getServiceVersion',
    requestMappingTemplate: lib_1.MappingTemplate.fromString(JSON.stringify({
        version: '2017-02-28',
    })),
    responseMappingTemplate: lib_1.MappingTemplate.fromString(JSON.stringify({
        version: 'v1',
    })),
});
const customerTable = new aws_dynamodb_1.Table(stack, 'CustomerTable', {
    billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
    partitionKey: {
        name: 'id',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
const orderTable = new aws_dynamodb_1.Table(stack, 'OrderTable', {
    billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
    partitionKey: {
        name: 'customer',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    sortKey: {
        name: 'order',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
const paymentTable = aws_dynamodb_1.Table.fromTableName(stack, 'PaymentTable', 'PaymentTable');
const customerDS = api.addDynamoDbDataSource('Customer', 'The customer data source', customerTable);
const orderDS = api.addDynamoDbDataSource('Order', 'The order data source', orderTable);
const paymentDS = api.addDynamoDbDataSource('Payment', 'The payment data source', paymentTable);
customerDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomers',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbScanTable(),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
customerDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbGetItem('id', 'id'),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'addCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey.partition('id').auto(), lib_1.Values.projecting('customer')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'saveCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey.partition('id').is('id'), lib_1.Values.projecting('customer')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'saveCustomerWithFirstOrder',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey
        .partition('order').auto()
        .sort('customer').is('customer.id'), lib_1.Values
        .projecting('order')
        .attribute('referral').is('referral')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'removeCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbDeleteItem('id', 'id'),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
const ops = [
    { suffix: 'Eq', op: lib_1.KeyCondition.eq },
    { suffix: 'Lt', op: lib_1.KeyCondition.lt },
    { suffix: 'Le', op: lib_1.KeyCondition.le },
    { suffix: 'Gt', op: lib_1.KeyCondition.gt },
    { suffix: 'Ge', op: lib_1.KeyCondition.ge },
];
for (const { suffix, op } of ops) {
    orderDS.createResolver({
        typeName: 'Query',
        fieldName: 'getCustomerOrders' + suffix,
        requestMappingTemplate: lib_1.MappingTemplate.dynamoDbQuery(op('customer', 'customer')),
        responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
    });
}
orderDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomerOrdersFilter',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbQuery(lib_1.KeyCondition.eq('customer', 'customer').and(lib_1.KeyCondition.beginsWith('order', 'order'))),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
orderDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomerOrdersBetween',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbQuery(lib_1.KeyCondition.eq('customer', 'customer').and(lib_1.KeyCondition.between('order', 'order1', 'order2'))),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
paymentDS.createResolver({
    typeName: 'Query',
    fieldName: 'getPayment',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbGetItem('id', 'id'),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
paymentDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'savePayment',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey.partition('id').is('id'), lib_1.Values.projecting('payment')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
const httpDS = api.addHttpDataSource('http', 'The http data source', 'https://aws.amazon.com/');
httpDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'doPostOnAws',
    requestMappingTemplate: lib_1.MappingTemplate.fromString(`{
    "version": "2018-05-29",
    "method": "POST",
    # if full path is https://api.xxxxxxxxx.com/posts then resourcePath would be /posts
    "resourcePath": "/path/123",
    "params":{
        "body": $util.toJson($ctx.args),
        "headers":{
            "Content-Type": "application/json",
            "Authorization": "$ctx.request.headers.Authorization"
        }
    }
  }`),
    responseMappingTemplate: lib_1.MappingTemplate.fromString(`
    ## Raise a GraphQL field error in case of a datasource invocation error
    #if($ctx.error)
      $util.error($ctx.error.message, $ctx.error.type)
    #end
    ## if the response status code is not 200, then return an error. Else return the body **
    #if($ctx.result.statusCode == 200)
        ## If response is 200, return the body.
        $ctx.result.body
    #else
        ## If response is not 200, append the response to error block.
        $utils.appendError($ctx.result.body, "$ctx.result.statusCode")
    #end
  `),
});
app.synth();
//# sourceMappingURL=data:application/json;base64,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