"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GraphQLApi = exports.FieldLogLevel = exports.UserPoolDefaultAction = exports.AuthorizationType = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const fs_1 = require("fs");
const appsync_generated_1 = require("./appsync.generated");
const data_source_1 = require("./data-source");
/**
 * enum with all possible values for AppSync authorization type
 */
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * API Key authorization type
     */
    AuthorizationType["API_KEY"] = "API_KEY";
    /**
     * AWS IAM authorization type. Can be used with Cognito Identity Pool federated credentials
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Cognito User Pool authorization type
     */
    AuthorizationType["USER_POOL"] = "AMAZON_COGNITO_USER_POOLS";
    /**
     * OpenID Connect authorization type
     */
    AuthorizationType["OIDC"] = "OPENID_CONNECT";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
/**
 * enum with all possible values for Cognito user-pool default actions
 */
var UserPoolDefaultAction;
(function (UserPoolDefaultAction) {
    /**
     * ALLOW access to API
     */
    UserPoolDefaultAction["ALLOW"] = "ALLOW";
    /**
     * DENY access to API
     */
    UserPoolDefaultAction["DENY"] = "DENY";
})(UserPoolDefaultAction = exports.UserPoolDefaultAction || (exports.UserPoolDefaultAction = {}));
/**
 * log-level for fields in AppSync
 */
var FieldLogLevel;
(function (FieldLogLevel) {
    /**
     * No logging
     */
    FieldLogLevel["NONE"] = "NONE";
    /**
     * Error logging
     */
    FieldLogLevel["ERROR"] = "ERROR";
    /**
     * All logging
     */
    FieldLogLevel["ALL"] = "ALL";
})(FieldLogLevel = exports.FieldLogLevel || (exports.FieldLogLevel = {}));
/**
 * An AppSync GraphQL API
 */
class GraphQLApi extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        super(scope, id);
        this.validateAuthorizationProps(props);
        const defaultAuthorizationType = ((_b = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.defaultAuthorization) === null || _b === void 0 ? void 0 : _b.authorizationType) ||
            AuthorizationType.API_KEY;
        let apiLogsRole;
        if (props.logConfig) {
            apiLogsRole = new aws_iam_1.Role(this, 'ApiLogsRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('appsync'),
            });
            apiLogsRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSAppSyncPushToCloudWatchLogs'));
        }
        this.api = new appsync_generated_1.CfnGraphQLApi(this, 'Resource', {
            name: props.name,
            authenticationType: defaultAuthorizationType,
            ...(props.logConfig && {
                logConfig: {
                    cloudWatchLogsRoleArn: apiLogsRole ? apiLogsRole.roleArn : undefined,
                    excludeVerboseContent: props.logConfig.excludeVerboseContent,
                    fieldLogLevel: props.logConfig.fieldLogLevel
                        ? props.logConfig.fieldLogLevel.toString()
                        : undefined,
                },
            }),
            openIdConnectConfig: ((_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.defaultAuthorization) === null || _d === void 0 ? void 0 : _d.authorizationType) ===
                AuthorizationType.OIDC
                ? this.formatOpenIdConnectConfig(props.authorizationConfig.defaultAuthorization
                    .openIdConnectConfig)
                : undefined,
            userPoolConfig: ((_f = (_e = props.authorizationConfig) === null || _e === void 0 ? void 0 : _e.defaultAuthorization) === null || _f === void 0 ? void 0 : _f.authorizationType) ===
                AuthorizationType.USER_POOL
                ? this.formatUserPoolConfig(props.authorizationConfig.defaultAuthorization.userPoolConfig)
                : undefined,
            additionalAuthenticationProviders: this.formatAdditionalAuthenticationProviders(props),
        });
        this.apiId = this.api.attrApiId;
        this.arn = this.api.attrArn;
        this.graphQlUrl = this.api.attrGraphQlUrl;
        this.name = this.api.name;
        if (defaultAuthorizationType === AuthorizationType.API_KEY ||
            ((_h = (_g = props.authorizationConfig) === null || _g === void 0 ? void 0 : _g.additionalAuthorizationModes) === null || _h === void 0 ? void 0 : _h.findIndex((authMode) => authMode.authorizationType === AuthorizationType.API_KEY)) !== -1) {
            const apiKeyConfig = ((_k = (_j = props.authorizationConfig) === null || _j === void 0 ? void 0 : _j.defaultAuthorization) === null || _k === void 0 ? void 0 : _k.apiKeyConfig) || {
                name: 'DefaultAPIKey',
                description: 'Default API Key created by CDK',
            };
            this.createAPIKey(apiKeyConfig);
        }
        let definition;
        if (props.schemaDefinition) {
            definition = props.schemaDefinition;
        }
        else if (props.schemaDefinitionFile) {
            definition = fs_1.readFileSync(props.schemaDefinitionFile).toString('UTF-8');
        }
        else {
            throw new Error('Missing Schema definition. Provide schemaDefinition or schemaDefinitionFile');
        }
        this.schema = new appsync_generated_1.CfnGraphQLSchema(this, 'Schema', {
            apiId: this.apiId,
            definition,
        });
    }
    /**
     * the configured API key, if present
     */
    get apiKey() {
        return this._apiKey;
    }
    /**
     * add a new dummy data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     */
    addNoneDataSource(name, description) {
        return new data_source_1.NoneDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
        });
    }
    /**
     * add a new DynamoDB data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param table The DynamoDB table backing this data source [disable-awslint:ref-via-interface]
     */
    addDynamoDbDataSource(name, description, table) {
        return new data_source_1.DynamoDbDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
            table,
        });
    }
    /**
     * add a new http data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param endpoint The http endpoint
     */
    addHttpDataSource(name, description, endpoint) {
        return new data_source_1.HttpDataSource(this, `${name}DS`, {
            api: this,
            description,
            endpoint,
            name,
        });
    }
    /**
     * add a new Lambda data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param lambdaFunction The Lambda function to call to interact with this data source
     */
    addLambdaDataSource(name, description, lambdaFunction) {
        return new data_source_1.LambdaDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
            lambdaFunction,
        });
    }
    validateAuthorizationProps(props) {
        var _a, _b, _c, _d, _e, _f, _g;
        const defaultAuthorizationType = ((_b = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.defaultAuthorization) === null || _b === void 0 ? void 0 : _b.authorizationType) ||
            AuthorizationType.API_KEY;
        if (defaultAuthorizationType === AuthorizationType.OIDC &&
            !((_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.defaultAuthorization) === null || _d === void 0 ? void 0 : _d.openIdConnectConfig)) {
            throw new Error('Missing default OIDC Configuration');
        }
        if (defaultAuthorizationType === AuthorizationType.USER_POOL &&
            !((_f = (_e = props.authorizationConfig) === null || _e === void 0 ? void 0 : _e.defaultAuthorization) === null || _f === void 0 ? void 0 : _f.userPoolConfig)) {
            throw new Error('Missing default User Pool Configuration');
        }
        if ((_g = props.authorizationConfig) === null || _g === void 0 ? void 0 : _g.additionalAuthorizationModes) {
            props.authorizationConfig.additionalAuthorizationModes.forEach((authorizationMode) => {
                if (authorizationMode.authorizationType === AuthorizationType.API_KEY &&
                    defaultAuthorizationType === AuthorizationType.API_KEY) {
                    throw new Error("You can't duplicate API_KEY in additional authorization config. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html");
                }
                if (authorizationMode.authorizationType === AuthorizationType.IAM &&
                    defaultAuthorizationType === AuthorizationType.IAM) {
                    throw new Error("You can't duplicate IAM in additional authorization config. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html");
                }
                if (authorizationMode.authorizationType === AuthorizationType.OIDC &&
                    !authorizationMode.openIdConnectConfig) {
                    throw new Error('Missing OIDC Configuration inside an additional authorization mode');
                }
                if (authorizationMode.authorizationType ===
                    AuthorizationType.USER_POOL &&
                    !authorizationMode.userPoolConfig) {
                    throw new Error('Missing User Pool Configuration inside an additional authorization mode');
                }
            });
        }
    }
    formatOpenIdConnectConfig(config) {
        return {
            authTtl: config.tokenExpiryFromAuth,
            clientId: config.clientId,
            iatTtl: config.tokenExpiryFromIssue,
            issuer: config.oidcProvider,
        };
    }
    formatUserPoolConfig(config) {
        return {
            userPoolId: config.userPool.userPoolId,
            awsRegion: config.userPool.stack.region,
            appIdClientRegex: config.appIdClientRegex,
            defaultAction: config.defaultAction || 'ALLOW',
        };
    }
    createAPIKey(config) {
        let expires;
        if (config.expires) {
            expires = new Date(config.expires).valueOf();
            const days = (d) => Date.now() + core_1.Duration.days(d).toMilliseconds();
            if (expires < days(1) || expires > days(365)) {
                throw Error('API key expiration must be between 1 and 365 days.');
            }
            expires = Math.round(expires / 1000);
        }
        const key = new appsync_generated_1.CfnApiKey(this, `${config.name || 'DefaultAPIKey'}ApiKey`, {
            expires,
            description: config.description || 'Default API Key created by CDK',
            apiId: this.apiId,
        });
        this._apiKey = key.attrApiKey;
    }
    formatAdditionalAuthorizationModes(authModes) {
        return authModes.reduce((acc, authMode) => [
            ...acc,
            {
                authenticationType: authMode.authorizationType,
                userPoolConfig: authMode.authorizationType === AuthorizationType.USER_POOL
                    ? this.formatUserPoolConfig(authMode.userPoolConfig)
                    : undefined,
                openIdConnectConfig: authMode.authorizationType === AuthorizationType.OIDC
                    ? this.formatOpenIdConnectConfig(authMode.openIdConnectConfig)
                    : undefined,
            },
        ], []);
    }
    formatAdditionalAuthenticationProviders(props) {
        var _a;
        const authModes = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.additionalAuthorizationModes;
        return authModes ? this.formatAdditionalAuthorizationModes(authModes) : undefined;
    }
}
exports.GraphQLApi = GraphQLApi;
//# sourceMappingURL=data:application/json;base64,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