"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaDataSource = exports.HttpDataSource = exports.DynamoDbDataSource = exports.NoneDataSource = exports.BackedDataSource = exports.BaseDataSource = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const appsync_generated_1 = require("./appsync.generated");
const resolver_1 = require("./resolver");
/**
 * Abstract AppSync datasource implementation. Do not use directly but use subclasses for concrete datasources
 */
class BaseDataSource extends core_1.Construct {
    constructor(scope, id, props, extended) {
        var _a;
        super(scope, id);
        if (extended.type !== 'NONE') {
            this.serviceRole = props.serviceRole || new aws_iam_1.Role(this, 'ServiceRole', { assumedBy: new aws_iam_1.ServicePrincipal('appsync') });
        }
        this.ds = new appsync_generated_1.CfnDataSource(this, 'Resource', {
            apiId: props.api.apiId,
            name: props.name,
            description: props.description,
            serviceRoleArn: (_a = this.serviceRole) === null || _a === void 0 ? void 0 : _a.roleArn,
            ...extended,
        });
        this.name = props.name;
        this.api = props.api;
    }
    /**
     * creates a new resolver for this datasource and API using the given properties
     */
    createResolver(props) {
        return new resolver_1.Resolver(this, `${props.typeName}${props.fieldName}Resolver`, {
            api: this.api,
            dataSource: this,
            ...props,
        });
    }
}
exports.BaseDataSource = BaseDataSource;
/**
 * Abstract AppSync datasource implementation. Do not use directly but use subclasses for resource backed datasources
 */
class BackedDataSource extends BaseDataSource {
    constructor(scope, id, props, extended) {
        super(scope, id, props, extended);
        this.grantPrincipal = this.serviceRole;
    }
}
exports.BackedDataSource = BackedDataSource;
/**
 * An AppSync dummy datasource
 */
class NoneDataSource extends BaseDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'NONE',
        });
    }
}
exports.NoneDataSource = NoneDataSource;
/**
 * An AppSync datasource backed by a DynamoDB table
 */
class DynamoDbDataSource extends BackedDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AMAZON_DYNAMODB',
            dynamoDbConfig: {
                tableName: props.table.tableName,
                awsRegion: props.table.stack.region,
                useCallerCredentials: props.useCallerCredentials,
            },
        });
        if (props.readOnlyAccess) {
            props.table.grantReadData(this);
        }
        else {
            props.table.grantReadWriteData(this);
        }
    }
}
exports.DynamoDbDataSource = DynamoDbDataSource;
/**
 * An AppSync datasource backed by a http endpoint
 */
class HttpDataSource extends BaseDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            httpConfig: {
                endpoint: props.endpoint,
            },
            type: 'HTTP',
        });
    }
}
exports.HttpDataSource = HttpDataSource;
/**
 * An AppSync datasource backed by a Lambda function
 */
class LambdaDataSource extends BackedDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AWS_LAMBDA',
            lambdaConfig: {
                lambdaFunctionArn: props.lambdaFunction.functionArn,
            },
        });
        props.lambdaFunction.grantInvoke(this);
    }
}
exports.LambdaDataSource = LambdaDataSource;
//# sourceMappingURL=data:application/json;base64,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